import * as cdk from 'aws-cdk-lib';
import * as s3 from 'aws-cdk-lib/aws-s3';
import * as iam from 'aws-cdk-lib/aws-iam';
import { Construct } from 'constructs';
export interface ImmuKVProps {
    /**
     * Name of the S3 bucket for ImmuKV storage
     * @default - Auto-generated bucket name
     */
    readonly bucketName?: string;
    /**
     * S3 prefix for all ImmuKV objects
     *
     * Controls where ImmuKV stores its data within the S3 bucket:
     * - Empty string or undefined: Files stored at bucket root (e.g., `_log.json`, `keys/mykey.json`)
     * - Without trailing slash (e.g., `myapp`): Flat prefix (e.g., `myapp_log.json`, `myappkeys/mykey.json`)
     * - With trailing slash (e.g., `myapp/`): Directory-style prefix (e.g., `myapp/_log.json`, `myapp/keys/mykey.json`)
     *
     * Note: S3 event notifications use prefix matching, so the filter will match any object
     * starting with `${s3Prefix}_log.json` (e.g., `_log.json`, `_log.json.backup`, etc.).
     * This is intentional behavior.
     *
     * @default - No prefix (root of bucket)
     */
    readonly s3Prefix?: string;
    /**
     * Duration to retain old log versions
     *
     * If specified, old log versions will be deleted after this duration.
     * Must be expressible in whole days (e.g., Duration.days(365)).
     * Can be used independently or combined with logVersionsToRetain.
     *
     * @default undefined - No time-based deletion (keep forever)
     */
    readonly logVersionRetention?: cdk.Duration;
    /**
     * Number of old log versions to retain
     *
     * If specified, only this many old log versions will be kept.
     * Can be used independently or combined with logVersionRetentionDays.
     *
     * @default undefined - No count-based deletion (keep all versions)
     */
    readonly logVersionsToRetain?: number;
    /**
     * Duration to retain old key object versions
     *
     * If specified, old key versions will be deleted after this duration.
     * Must be expressible in whole days (e.g., Duration.days(180)).
     * Can be used independently or combined with keyVersionsToRetain.
     *
     * @default undefined - No time-based deletion (keep forever)
     */
    readonly keyVersionRetention?: cdk.Duration;
    /**
     * Number of old key versions to retain per key
     *
     * If specified, only this many old versions will be kept per key.
     * Can be used independently or combined with keyVersionRetentionDays.
     *
     * @default undefined - No count-based deletion (keep all versions)
     */
    readonly keyVersionsToRetain?: number;
    /**
     * Enable KMS encryption instead of S3-managed encryption
     * @default false
     */
    readonly useKmsEncryption?: boolean;
    /**
     * Optional notification destination to trigger when log entries are created.
     * Supports Lambda functions, SNS topics, and SQS queues.
     *
     * Example with Lambda:
     * ```ts
     * import * as s3n from 'aws-cdk-lib/aws-s3-notifications';
     *
     * new ImmuKV(this, 'ImmuKV', {
     *   onLogEntryCreated: new s3n.LambdaDestination(myFunction)
     * });
     * ```
     *
     * Example with SNS:
     * ```ts
     * new ImmuKV(this, 'ImmuKV', {
     *   onLogEntryCreated: new s3n.SnsDestination(myTopic)
     * });
     * ```
     *
     * Example with SQS:
     * ```ts
     * new ImmuKV(this, 'ImmuKV', {
     *   onLogEntryCreated: new s3n.SqsDestination(myQueue)
     * });
     * ```
     *
     * @default - No event notification configured
     */
    readonly onLogEntryCreated?: s3.IBucketNotificationDestination;
}
/**
 * AWS CDK Construct for ImmuKV infrastructure
 *
 * Creates an S3 bucket with versioning enabled and IAM policies for
 * read/write and read-only access.
 */
export declare class ImmuKV extends Construct {
    /**
     * The S3 bucket used for ImmuKV storage
     */
    readonly bucket: s3.Bucket;
    /**
     * IAM managed policy for read/write access
     */
    readonly readWritePolicy: iam.ManagedPolicy;
    /**
     * IAM managed policy for read-only access
     */
    readonly readOnlyPolicy: iam.ManagedPolicy;
    constructor(scope: Construct, id: string, props?: ImmuKVProps);
}
