"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ImmuKV = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const s3 = require("aws-cdk-lib/aws-s3");
const iam = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
/**
 * Standard file names and patterns used by ImmuKV
 */
const LOG_FILE_PATTERN = '_log.json';
const KEYS_PREFIX = 'keys/';
/**
 * AWS CDK Construct for ImmuKV infrastructure
 *
 * Creates an S3 bucket with versioning enabled and IAM policies for
 * read/write and read-only access.
 */
class ImmuKV extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const logVersionRetention = props?.logVersionRetention;
        const logVersionsToRetain = props?.logVersionsToRetain;
        const keyVersionRetention = props?.keyVersionRetention;
        const keyVersionsToRetain = props?.keyVersionsToRetain;
        const s3Prefix = props?.s3Prefix ?? '';
        // Validate retention parameters if provided
        if (logVersionRetention !== undefined) {
            const days = logVersionRetention.toDays();
            if (!Number.isInteger(days) || days <= 0) {
                throw new Error('logVersionRetention must be expressible as a positive whole number of days');
            }
        }
        if (logVersionsToRetain !== undefined && (logVersionsToRetain < 0 || !Number.isInteger(logVersionsToRetain))) {
            throw new Error('logVersionsToRetain must be a non-negative integer');
        }
        if (keyVersionRetention !== undefined) {
            const days = keyVersionRetention.toDays();
            if (!Number.isInteger(days) || days <= 0) {
                throw new Error('keyVersionRetention must be expressible as a positive whole number of days');
            }
        }
        if (keyVersionsToRetain !== undefined && (keyVersionsToRetain < 0 || !Number.isInteger(keyVersionsToRetain))) {
            throw new Error('keyVersionsToRetain must be a non-negative integer');
        }
        // Validate s3Prefix
        if (s3Prefix && (s3Prefix.startsWith('/') || s3Prefix.includes('..'))) {
            throw new Error('s3Prefix must not start with "/" or contain ".."');
        }
        // Build lifecycle rules array conditionally
        const lifecycleRules = [];
        // Add log lifecycle rule if any retention parameter is specified
        if (logVersionRetention !== undefined || logVersionsToRetain !== undefined) {
            lifecycleRules.push({
                id: 'delete-old-log-versions',
                enabled: true,
                noncurrentVersionExpiration: logVersionRetention,
                noncurrentVersionsToRetain: logVersionsToRetain,
                prefix: `${s3Prefix}${LOG_FILE_PATTERN}`,
            });
        }
        // Add key lifecycle rule if any retention parameter is specified
        if (keyVersionRetention !== undefined || keyVersionsToRetain !== undefined) {
            lifecycleRules.push({
                id: 'delete-old-key-versions',
                enabled: true,
                noncurrentVersionExpiration: keyVersionRetention,
                noncurrentVersionsToRetain: keyVersionsToRetain,
                prefix: `${s3Prefix}${KEYS_PREFIX}`,
            });
        }
        // S3 Bucket with versioning
        this.bucket = new s3.Bucket(this, 'ImmuKVBucket', {
            bucketName: props?.bucketName,
            versioned: true,
            encryption: props?.useKmsEncryption
                ? s3.BucketEncryption.KMS_MANAGED
                : s3.BucketEncryption.S3_MANAGED,
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            lifecycleRules: lifecycleRules.length > 0 ? lifecycleRules : undefined,
            removalPolicy: cdk.RemovalPolicy.RETAIN,
        });
        // IAM Policy for read/write access (Lambda, EC2, ECS, etc.)
        this.readWritePolicy = new iam.ManagedPolicy(this, 'ImmuKVReadWritePolicy', {
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: [
                        's3:GetObject',
                        's3:GetObjectVersion',
                        's3:PutObject',
                        's3:ListBucket',
                        's3:ListBucketVersions',
                        's3:HeadObject',
                    ],
                    resources: [this.bucket.bucketArn, `${this.bucket.bucketArn}/*`],
                }),
            ],
        });
        // IAM Policy for read-only devices (sensors, IoT devices, etc.)
        this.readOnlyPolicy = new iam.ManagedPolicy(this, 'ImmuKVReadOnlyPolicy', {
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: [
                        's3:GetObject',
                        's3:GetObjectVersion',
                        's3:ListBucket',
                        's3:ListBucketVersions',
                        's3:HeadObject',
                    ],
                    resources: [this.bucket.bucketArn, `${this.bucket.bucketArn}/*`],
                }),
            ],
        });
        // S3 Event Notification (optional)
        if (props?.onLogEntryCreated) {
            this.bucket.addEventNotification(s3.EventType.OBJECT_CREATED, props.onLogEntryCreated, { prefix: `${s3Prefix}${LOG_FILE_PATTERN}` });
        }
    }
}
exports.ImmuKV = ImmuKV;
_a = JSII_RTTI_SYMBOL_1;
ImmuKV[_a] = { fqn: "cdk-immukv.ImmuKV", version: "0.1.15" };
//# sourceMappingURL=data:application/json;base64,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