# CLAUDE.md
This file provides guidance for Claude Code (claude.ai/code) when working with the code in this repository. Its purpose is to ensure consistency, maintainability, and high-quality code.

## 1. Project Documentation

The primary, high-level overview of this project is documented in `README.md`.

For more detailed explanations of specific components (e.g., a particular algorithm, data model, or API usage), create a dedicated Markdown file in the most relevant directory. List all such supplementary documentation files here for easy reference.

-   Add as needed (Example: `./src/model/usage.md`)

## 2. Development Commands

Use the following `uv` commands to manage common development tasks.

-   **Format Code**: `uv run ruff format`
    -   Formats all Python code according to the project's style configuration.
-   **Lint Code**: `uv run ruff check`
    -   Analyzes the code for potential errors, bugs, and stylistic issues.
-   **Auto-fix Lint Issues**: `uv run ruff check --fix`
    -   Automatically fixes all fixable linting issues.
-   **Run All Tests**: `uv run pytest`
    -   Executes the entire test suite using pytest.
-   **Run a Specific Test**: `uv run pytest tests/test_filename.py`
    -   Executes a single test file.
-   **Run Tests Verbose**: `uv run pytest -v`
    -   Runs tests with detailed, verbose output for debugging.

## 3. Project Architecture

This is a robotics project. Below is an overview of the key functional components. When modifying or adding code, please adhere to the described architecture.

### Transform Management (`tf`)

This component is responsible for managing spatial coordinate frame transformations. It provides utilities to compute and track the relationships between different parts of the robot (e.g., base, arm, gripper) and objects in the environment. All spatial calculations must use this system to ensure consistency.

### Robot Definition

This component defines the robot's physical structure, kinematics, and dynamics. The definition is typically loaded from a URDF (Unified Robot Description Format) or similar file. It includes specifications for links, joints, sensors, and actuators. Do not hard-code physical robot parameters; they should be accessed through this module.

### Motion Planning

This component handles the generation of trajectories for the robot to move from a starting point to a target destination. It includes forward/inverse kinematics solvers and algorithms for path planning while avoiding obstacles.

### Collision Management

This component provides tools for collision detection. It is used to check for potential collisions between different parts of the robot (self-collision) and between the robot and its environment. All motion plans must be validated through this system before execution.

### Project Directory Structure

The project follows a standard Python project layout.

-   `src/`: Contains the main source code for the application.
-   `tests/`: Contains all unit, integration, and functional tests. The structure should mirror the `src/` directory.
-   `docs/`: Contains documentation generated by tools like Sphinx. Do not edit manually.
-   `examples/`: Contains standalone example scripts demonstrating how to use the library/application.
-   `notebooks/`: Contains Jupyter notebooks for experimentation, analysis, and visualization.

## 4. Coding Guidelines

Follow these guidelines strictly to maintain code quality and consistency.

### Comments and Documentation

-   **Language**: All code comments, docstrings, and commit messages **must be written in English**.
-   **Clarity**: Write clear and descriptive comments to explain complex logic, business rules, or non-obvious implementations.
-   **Docstrings**: Write **Google-style docstrings** for all public modules, classes, methods, and functions. The docstring should describe the purpose, arguments (`Args:`), and return values (`Returns:`).

### Code Style and Best Practices

-   **Formatting**: Adhere to the `Ruff` formatter configuration. Always run the formatter before committing code.
-   **Idiomatic Naming**: Use clear and idiomatic Python naming conventions (e.g., `snake_case` for variables and functions, `PascalCase` for classes).
-   **Single Responsibility**: Aim for one primary class per file. Decompose complex modules into smaller, more focused files.
-   **Type Hinting**: **Always** use type hints for function arguments and return values. This is critical for static analysis and IDE support.
-   **Error Handling**: Implement robust error handling using `try...except` blocks where exceptions (e.g., `IOError`, `ValueError`) are likely to occur. Avoid catching generic `Exception`. However, the error handling for issues that can be prevented by type hints is not needed because that would be excessive.
-   **Logging**: Use the `logging` module for all output, including debugging, status information, and errors. Do not use `print()` statements in the library/application code.
-   **Custom Class `__repr__`**: All custom-defined classes MUST implement the `__repr__` special method for effective debugging and meaningful logging.

### Python Logging Specifics

-   **Logger Setup**: In each module, get a logger instance with `import logging` and `logger = logging.getLogger(__name__)`.
-   **No Cross-Module Imports**: Do not import logger instances from other modules. Each module should create its own to preserve the correct logger hierarchy.

## 5. Document Maintenance

-   **Stay Synchronized**: If you implement changes that conflict with or extend the information in this document (e.g., adding a new development command or architectural component), you **must** update this file accordingly.
-   **Update README**: If the project's setup, installation, or basic usage changes, update both `README.md` and this file. Documentation must always reflect the current state of the codebase.
