import glob
import _jsonnet
import json
import os
import os.path
import re
from dataclasses import dataclass, field
from datetime import timedelta
from enum import Enum
from pathlib import Path
from redun import File
from psij import (
    Job,
    JobAttributes,
    JobExecutor,
    JobExecutorConfig,
    JobSpec,
    JobState,
    JobStatus,
)
from typing import Any, Optional

from ._singleton import singleton

redun_namespace = "redun_psij"


class JobError(Exception):
    """Executor job error."""

    def __init__(self, message: str):
        super().__init__(message)


class JobFailure:
    """
    Returned by :func:`redun_psij.run_job_1_returning_failure` and :func:`redun_psij.run_job_n_returning_failure` on error, rather than throwing an exception.

    Contains stderr as an actual file
    so that deleting that file will trigger redun to rerun the failed task.
    """

    def __init__(
        self,
        exit_code: int,
        stderr: File,
    ):
        self.exit_code = exit_code
        self.stderr = stderr


class _FailureHandler(Enum):
    EXCEPTION = 1
    RETURN = 2


class ConfigError(Exception):
    """Executor configuration error."""

    def __init__(
        self,
        message: str,
        path: Optional[str] = None,
        tool: Optional[str] = None,
        cause: Optional[Exception] = None,
    ):
        self.message = message
        self.path = path
        self.tool = tool
        # cause is explicit so we can format it into the base Exception class
        # which is required for redun to show it in the console,
        # and this is the reason we don't use raise from and ex.__cause__
        self.cause = cause
        super().__init__(str(self))

    def __str__(self) -> str:
        result = "Configuration error"
        if self.tool is not None:
            result += f" for {self.tool}"
        if self.path is not None:
            result += f" in {self.path}"
        result += f": {self.message}"
        if self.cause is not None:
            result += f": {type(self.cause).__name__} {str(self.cause)}"
        return result


@dataclass(kw_only=True)
class CommonJobSpec:
    """The spec in common for :class:`redun_psij.Job1Spec` and :class:`redun_psij.JobNSpec`."""

    tool: str
    """The tool name, to be found in the job configuration file."""

    args: list[str]
    """Command and its arguments."""

    stdout_path: str
    """Path to file which captures stdout of the job."""

    stderr_path: str
    """Path to file which captures stderr of the job."""

    cwd: Optional[str] = None
    """Optional directory to change to before running the job."""

    custom_attributes: dict[str, str] = field(default_factory=dict)
    """Custom attributes to pass to PSI/J, e.g. from :class:`redun_psij.JobContext`."""


@dataclass(kw_only=True)
class Job1Spec(CommonJobSpec):
    """For jobs which produce a single file whose path is known in advance."""

    expected_path: str
    """Path of the single expected output file, it being an error if this is not created by the job."""


@dataclass
class FilteredGlob:
    """A file glob optionally without any paths matched by `reject_re`."""

    glob: str
    """File glob to match job output files."""

    reject_re: Optional[str] = None
    """Regular expression in Python regex syntax to exclude unwanted glob matches."""


@dataclass(kw_only=True)
class ExpectedPaths:
    required: dict[str, str] = field(default_factory=dict)
    """Paths of any expected output files which are required to be created by the job.  Any missing file constitutes an error.  Keys are arbitrary strings and may be used to match up output files with what was expected."""

    optional: dict[str, str] = field(default_factory=dict)
    """As per required where failure to create a file is not an error."""


@dataclass(kw_only=True)
class JobNSpec(CommonJobSpec):
    """
    For jobs which produce multiple files with paths matching the glob, exluding any matched by the regex.

    As many conbinations of globs and paths may be expected, in a dict whose keys are arbitrary strings.
    Result files will be returned in a dict with the same keys.
    """

    # each value is either a result path or a filtered glob
    expected_paths: ExpectedPaths = field(default_factory=ExpectedPaths)
    """Expected output files whose paths are known in advance."""

    expected_globs: dict[str, FilteredGlob] = field(default_factory=dict)
    """Expected output files whose paths are globbed after the event."""


def _create_job_attributes(
    configured_attributes: dict[str, Any],
    extra_custom_attributes: dict[str, str],
    executor_name: str,
    job_name: str,
) -> JobAttributes:
    augmented_custom_attributes = {
        "custom_attributes": (
            configured_attributes.get("custom_attributes", {})
            | {f"{executor_name}.job-name": job_name}
            | {f"{executor_name}.{k}": v for (k, v) in extra_custom_attributes.items()}
        )
    }

    if configured_duration := configured_attributes.get("duration"):
        try:
            duration = timedelta(**configured_duration)
        except TypeError as err:
            raise ConfigError(
                "invalid duration, must be dict with fields as per timedelta constructor with integer values",
                cause=err,
            )
        duration_attribute = {"duration": duration}
    else:
        duration_attribute = {}

    attributes = (
        configured_attributes | duration_attribute | augmented_custom_attributes
    )

    return JobAttributes(**attributes)


def _create_job_spec(
    spec: CommonJobSpec,
) -> tuple[JobSpec, str]:
    tool_config, config_path = _get_tool_config_and_path(spec.tool)
    try:
        executor_name = tool_config["executor"]
        job_prefix = tool_config.get("job_prefix", "")

        job_attributes = _create_job_attributes(
            configured_attributes=tool_config.get("job_attributes", {}),
            extra_custom_attributes=spec.custom_attributes,
            executor_name=executor_name,
            job_name=f"{job_prefix}{spec.tool}",
        )

        return (
            JobSpec(
                executable=spec.args[0],
                arguments=spec.args[1:],
                directory=spec.cwd or os.getcwd(),
                stdout_path=spec.stdout_path,
                stderr_path=spec.stderr_path,
                attributes=job_attributes,
            ),
            executor_name,
        )
    except ConfigError as err:
        if err.path is not None and err.tool is not None:
            raise
        else:
            raise ConfigError(
                message=err.message, path=config_path, tool=spec.tool, cause=err.cause
            )
    except Exception as ex:
        raise ConfigError(message=str(ex), path=config_path, cause=ex)


def _create_executor(executor_name: str):
    # PSI/J defaults to home directory, which is not what we want
    psij_dir = os.path.join(os.getcwd(), ".psij")
    os.makedirs(psij_dir, exist_ok=True)

    config = JobExecutorConfig(work_directory=Path(psij_dir))

    return JobExecutor.get_instance(executor_name, config=config)


def _job_description(
    job: Job,
    spec: CommonJobSpec,
    annotation: Optional[str] = None,
    multiline: bool = False,
) -> str:
    cwd = spec.cwd or os.getcwd()
    command_text = " ".join(spec.args)
    padded_annotation = f" {annotation}" if annotation is not None else ""
    if multiline:
        return (
            f"job {job.native_id}{padded_annotation}\ncwd: {cwd}\ncmd: {command_text}\n"
        )
    else:
        return f"job {job.native_id} {command_text}{padded_annotation} cwd={cwd}"


def _handle_failure(
    job: Job,
    status: JobStatus | None,
    spec: CommonJobSpec,
    failure_handler: _FailureHandler,
) -> Optional[JobFailure]:
    """On failure raise an exception, unless handle_job_failure, in which case, return the failure."""
    if status is None:
        raise JobError(f"job {job.native_id} status is None")
    elif status.state == JobState.CANCELED:
        raise JobError(f"job {job.native_id} canceled")
    elif status.state == JobState.FAILED:
        try:
            with open(spec.stderr_path, "r") as stderr_f:
                stderr_text = stderr_f.read()
                stderr_readable = True
        except:
            stderr_text = (
                f"(stderr unavailable because failed to read {spec.stderr_path})"
            )
            stderr_readable = False
        if status.exit_code is None:
            exit_text = ""
        elif status.exit_code > 128:
            signal = status.exit_code - 128
            exit_text = f" because {'killed' if signal == 9 else ' received signal %d' % signal}"
        else:
            exit_text = f" with exit code {status.exit_code}"
        failure_text = (
            f"failed ({status.message})" if status.message is not None else "failed"
        ) + exit_text
        metadata_text = (
            " ".join(["%s=%s" % (k, str(v)) for k, v in status.metadata])
            if status.metadata is not None
            else ""
        )

        if (
            failure_handler == _FailureHandler.RETURN
            and status.exit_code is not None
            and stderr_readable
        ):
            # It is important that we only return failure when the stderr file actually exists,
            # otherwise there's no way to trigger a rerun after fixing whatever was broken.
            # Simply deleting the stderr file should be enough of a trigger.
            return JobFailure(exit_code=status.exit_code, stderr=File(spec.stderr_path))
        raise JobError(
            f"{_job_description(job, spec, annotation=failure_text, multiline=True)}\nmetadata: {metadata_text}\n{stderr_text}"
        )
    return None


def _run_job_1(
    spec: Job1Spec,
    failure_handler: _FailureHandler,
) -> File | JobFailure:
    """
    Run a job which is expected to produce the single file `expected_path`
    """
    job_spec, executor_name = _create_job_spec(
        spec=spec,
    )

    job = Job(job_spec)
    _create_executor(executor_name).submit(job)

    status = job.wait()
    failure = _handle_failure(job, status, spec, failure_handler=failure_handler)
    if failure is not None:
        return failure
    else:
        if not os.path.exists(spec.expected_path):
            raise JobError(
                _job_description(
                    job,
                    spec,
                    annotation=f"failed to create {spec.expected_path}",
                    multiline=True,
                )
            )

        return File(spec.expected_path)


def run_job_1(
    spec: Job1Spec,
) -> File:
    """
    Run a job which is expected to produce the single file `expected_path`
    Errors are thrown as :class:`redun_psij.JobError`.

    Args:
        spec: defines the job parameters
    """
    result = _run_job_1(spec, failure_handler=_FailureHandler.EXCEPTION)
    assert isinstance(result, File)
    return result


def run_job_1_returning_failure(
    spec: Job1Spec,
) -> File | JobFailure:
    """
    Run a job which is expected to produce the single file `expected_path`
    Errors are returned as :class:`redun_psij.JobFailure`.

    Args:
        spec: defines the job parameters
    """
    return _run_job_1(spec, failure_handler=_FailureHandler.RETURN)


@dataclass
class ResultFiles:
    """The output files produced by :func:`redun_psij.run_job_n`."""

    expected_files: dict[str, File]
    """Files as defined by :class:`redun_psij.ExpectedPaths`."""

    globbed_files: dict[str, list[File]]
    """Files as defined by :class:`redun_psij.FilteredGlob`."""


def _result_files(
    job: Job,
    spec: CommonJobSpec,
    expected_paths: ExpectedPaths,
    expected_globs: dict[str, FilteredGlob],
) -> ResultFiles:
    """Return result files for expected paths, or those matching filtered glob."""
    # required items must all exist
    missing_required_paths = {}
    for k, path in expected_paths.required.items():
        if not os.path.exists(path):
            missing_required_paths[k] = path
    if missing_required_paths:
        raise JobError(
            _job_description(
                job,
                spec,
                annotation="failed to create %s"
                % ", ".join(
                    [f"{k}={path}" for (k, path) in missing_required_paths.items()]
                ),
                multiline=True,
            )
        )
    found_paths = expected_paths.required.copy()
    # optional paths are returned if they are found to exist
    for k, path in expected_paths.optional.items():
        if os.path.exists(path):
            found_paths[k] = path

    return ResultFiles(
        expected_files={k: File(path) for (k, path) in found_paths.items()},
        globbed_files={
            k: [
                File(path)
                for path in glob.glob(expected.glob)
                if expected.reject_re is None
                or re.search(expected.reject_re, path) is None
            ]
            for (k, expected) in expected_globs.items()
        },
    )


def _run_job_n(
    spec: JobNSpec,
    failure_handler: _FailureHandler,
) -> ResultFiles | JobFailure:
    """
    Run a job which is expected to produce files matching `result_glob`
    """

    # check no duplicates between required and optional expected paths
    duplicate_expected_keys = (
        spec.expected_paths.required.keys() & spec.expected_paths.optional.keys()
    )
    assert (
        len(duplicate_expected_keys) == 0
    ), f"duplicate expected keys: {', '.join(duplicate_expected_keys)}"

    job_spec, executor_name = _create_job_spec(
        spec=spec,
    )

    job = Job(job_spec)
    _create_executor(executor_name).submit(job)

    status = job.wait()
    failure = _handle_failure(job, status, spec, failure_handler=failure_handler)
    if failure is not None:
        return failure
    else:
        return _result_files(job, spec, spec.expected_paths, spec.expected_globs)


def run_job_n(
    spec: JobNSpec,
) -> ResultFiles:
    """
    Run a job which is expected to produce files matching :class:`redun_psij.ResultFiles`

    Errors are thrown as :class:`redun_psij.JobError`.
    """
    result_files = _run_job_n(spec, failure_handler=_FailureHandler.EXCEPTION)
    assert isinstance(result_files, ResultFiles)
    return result_files


def run_job_n_returning_failure(
    spec: JobNSpec,
) -> ResultFiles | JobFailure:
    """
    Run a job which is expected to produce files matching :class:`redun_psij.ResultFiles`

    Errors are returned as :class:`redun_psij.JobFailure`.

    Args:
        spec: defines the job parameters
    """
    return _run_job_n(spec, failure_handler=_FailureHandler.RETURN)


def _deep_get(values: Any, path: str, default: Any = None) -> Any:
    for selector in path.split("."):
        values = values.get(selector)
        if values is None:
            return default
    return values


PSIJ_EXECUTOR_CONFIG_PATH = os.environ.get("PSIJ_EXECUTOR_CONFIG_PATH", ".").split(":")


def _jsonnet_import_callback(base, rel):
    for import_dir in PSIJ_EXECUTOR_CONFIG_PATH:
        import_path = os.path.join(import_dir, rel)
        try:
            with open(import_path, "rb") as f:
                return import_path, f.read()
        except FileNotFoundError:
            pass
    raise FileNotFoundError(
        f"jsonnet import {rel} not found on path {PSIJ_EXECUTOR_CONFIG_PATH}"
    )


@singleton
class PsijExecutorConfig:
    def __init__(self):
        self._configured = False
        self._path = None

    @property
    def path(self) -> str:
        assert (
            self._path is not None
        ), "PsijExecutorConfig is not configured, need an early call to read_config()"

        return self._path

    def read_config(self):
        CONFIG = "psij-executor.jsonnet"
        # attempt to read config file from somewhere on the path
        for config_dir in PSIJ_EXECUTOR_CONFIG_PATH:
            path = os.path.join(config_dir, CONFIG)
            try:
                with open(path, "r") as config_f:
                    self._path = path
                    raw_config = config_f.read()
            except FileNotFoundError:
                continue

            try:
                json_config = _jsonnet.evaluate_snippet(
                    path, raw_config, import_callback=_jsonnet_import_callback
                )
                self._config = json.loads(json_config)
            except Exception as ex:
                raise ConfigError(
                    message="invalid Jsonnet configuration", path=path, cause=ex
                )

            self._configured = True
            return

        raise ConfigError(
            message=f"can't find {CONFIG} in {':'.join(PSIJ_EXECUTOR_CONFIG_PATH)}"
        )

    def get(self, path: str, default: Any = None) -> Any:
        if not self._configured:
            self.read_config()
        return _deep_get(self._config, path, default=default)


def _get_tool_config_and_path(tool: str) -> tuple[dict[str, Any], str]:
    """Get tool config and config path from the singleton config object."""
    config = PsijExecutorConfig()
    tool_config = config.get("tools.default", {}) | config.get(f"tools.{tool}", {})
    return tool_config, config.path


def get_tool_config(tool: str) -> dict[str, Any]:
    """Return tool config for the named tool from the Jsonnet configuration.

    Args:
        tool: name of tool
    """
    return _get_tool_config_and_path(tool)[0]
