"""Represent a token 'cell' for interlinears/reverse-interlinears."""

from dataclasses import dataclass, field
from typing import Optional

from ..burrito.source import Source
from ..burrito.target import Target


@dataclass
class AlignedToken:
    """An aligned set of source and target tokens.

    One or both
    """

    targettoken: Optional[Target] = None
    sourcetoken: Optional[Source] = None
    # if false, one or both tokens could be empty
    aligned: bool = False

    def __repr__(self) -> str:
        """Return a printed representation."""
        if self.targettoken:
            alignflag: str = ", aligned" if self.aligned else ""
            return f"<AlignedToken(targetid={self.targettoken.id}{alignflag})>"
        elif self.sourcetoken:
            return f"<AlignedToken(sourceid={self.sourcetoken.id})>"
        else:
            return "<AlignedToken(no token)>"

    #
    def __lt__(self: "AlignedToken", other: "AlignedToken") -> bool:
        if self.targettoken:
            if other.targettoken:
                return self.targettoken.id < other.targettoken.id
            elif self.sourcetoken:
                if other.sourcetoken:
                    return self.sourcetoken.id < other.sourcetoken.id
                else:
                    return True
            else:
                return False
        elif other.targettoken:
            if self.sourcetoken:
                if other.sourcetoken:
                    return self.sourcetoken.id < other.sourcetoken.id
                else:
                    return True
            elif other.sourcetoken:
                return True

    # def __gt__(self: "AlignedToken", other: "AlignedToken") -> bool:
    #     if self.targettoken and other.targettoken:
    #         return self.targettoken.id > other.targettoken.id
    #     elif self.sourcetoken and other.sourcetoken:
    #         return self.sourcetoken.id > other.sourcetoken.id
    #     elif self.targettoken:
    #         return True
    #     else:
    #         return False

    #  __lt__(), __le__(), __gt__(), and __ge__()

    def ids(self) -> str:
        """Return a string of the IDs of the tokens."""
        idstr = ""
        if self.sourcetoken:
            idstr = f"{self.sourcetoken.id}"
        else:
            idstr = "-" * 11
        if self.targettoken:
            idstr += f" {self.targettoken.id}"
        else:
            idstr += f" {'-' * 11}"
        return idstr

    def asdict(self) -> dict[str, str]:
        """Return a dictionary of the token details.

        Some attributes are renamed to keep them distinct.
        """
        atdict: dict[str, str] = {}
        if self.targettoken:
            atdict = self.targettoken.asdict()
            atdict["targetid"] = self.targettoken.id
            atdict["targettext"] = self.targettoken.text
            del atdict["id"], atdict["text"]
        if self.sourcetoken:
            atdict.update(self.sourcetoken.asdict())
            atdict["sourceid"] = self.sourcetoken.id
            atdict["sourcetext"] = self.sourcetoken.text
            del atdict["id"], atdict["text"]
        return atdict
