from dataclasses import dataclass
from typing import Union

from markupsafe import Markup

from strictdoc import __version__
from strictdoc.backend.sdoc.models.anchor import Anchor
from strictdoc.backend.sdoc.models.document import SDocDocument
from strictdoc.backend.sdoc.models.node import SDocNode
from strictdoc.core.document_tree import DocumentTree
from strictdoc.core.document_tree_iterator import DocumentTreeIterator
from strictdoc.core.file_tree import File, Folder
from strictdoc.core.project_config import ProjectConfig
from strictdoc.core.traceability_index import TraceabilityIndex
from strictdoc.export.html.generators.view_objects.helpers import (
    screen_should_display_file,
    screen_should_display_folder,
)
from strictdoc.export.html.html_templates import JinjaEnvironment
from strictdoc.export.html.renderers.link_renderer import LinkRenderer
from strictdoc.helpers.cast import assert_cast


@dataclass
class ProjectTreeViewObject:
    def __init__(
        self,
        *,
        traceability_index: TraceabilityIndex,
        project_config: ProjectConfig,
    ):
        self.traceability_index: TraceabilityIndex = traceability_index
        self.project_config: ProjectConfig = project_config

        link_renderer = LinkRenderer(
            root_path="", static_path=project_config.dir_for_sdoc_assets
        )
        self.link_renderer: LinkRenderer = link_renderer

        self.document_tree_iterator: DocumentTreeIterator = (
            DocumentTreeIterator(
                assert_cast(traceability_index.document_tree, DocumentTree)
            )
        )
        self.standalone: bool = False
        self.is_running_on_server: bool = project_config.is_running_on_server
        self.strictdoc_version = __version__
        self.contains_included_documents = (
            traceability_index.contains_included_documents
        )

    def render_local_anchor(
        self, node: Union[Anchor, SDocNode, SDocDocument]
    ) -> str:
        return self.link_renderer.render_local_anchor(node)

    def render_screen(self, jinja_environment: JinjaEnvironment) -> Markup:
        return jinja_environment.render_template_as_markup(
            "screens/project_index/index.jinja", view_object=self
        )

    def render_map(self, jinja_environment: JinjaEnvironment) -> Markup:
        return jinja_environment.render_template_as_markup(
            "screens/project_index/project_map.jinja.js", view_object=self
        )

    def render_static_url(self, url: str) -> str:
        return self.link_renderer.render_static_url(url)

    def render_url(self, url: str) -> str:
        return self.link_renderer.render_url(url)

    def render_static_url_with_prefix(self, url: str) -> str:
        return self.link_renderer.render_static_url_with_prefix(url)

    def is_empty_tree(self) -> bool:
        return self.document_tree_iterator.is_empty_tree()

    def get_document_level(self) -> int:
        return 0

    def should_display_fragments_toggle(self) -> bool:
        return self.project_config.export_included_documents

    def should_display_folder(self, folder: Folder) -> bool:
        return screen_should_display_folder(
            folder,
            self.traceability_index,
            self.project_config,
            must_only_include_non_included_sdoc=False,
        )

    def should_display_file(self, file: File) -> bool:
        return screen_should_display_file(
            file,
            self.traceability_index,
            self.project_config,
            must_only_include_non_included_sdoc=False,
        )
