"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Match = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const matcher_1 = require("./matcher");
const absent_1 = require("./private/matchers/absent");
const type_1 = require("./private/type");
/**
 * Partial and special matching during template assertions.
 */
class Match {
    /**
     * Use this matcher in the place of a field's value, if the field must not be present.
     */
    static absent() {
        return new absent_1.AbsentMatch('absent');
    }
    /**
     * Matches the specified pattern with the array found in the same relative path of the target.
     * The set of elements (or matchers) must be in the same order as would be found.
     * @param pattern the pattern to match
     */
    static arrayWith(pattern) {
        return new ArrayMatch('arrayWith', pattern);
    }
    /**
     * Matches the specified pattern with the array found in the same relative path of the target.
     * The set of elements (or matchers) must match exactly and in order.
     * @param pattern the pattern to match
     */
    static arrayEquals(pattern) {
        return new ArrayMatch('arrayEquals', pattern, { subsequence: false });
    }
    /**
     * Deep exact matching of the specified pattern to the target.
     * @param pattern the pattern to match
     */
    static exact(pattern) {
        return new LiteralMatch('exact', pattern, { partialObjects: false });
    }
    /**
     * Matches the specified pattern to an object found in the same relative path of the target.
     * The keys and their values (or matchers) must be present in the target but the target can be a superset.
     * @param pattern the pattern to match
     */
    static objectLike(pattern) {
        return new ObjectMatch('objectLike', pattern);
    }
    /**
     * Matches the specified pattern to an object found in the same relative path of the target.
     * The keys and their values (or matchers) must match exactly with the target.
     * @param pattern the pattern to match
     */
    static objectEquals(pattern) {
        return new ObjectMatch('objectEquals', pattern, { partial: false });
    }
    /**
     * Matches any target which does NOT follow the specified pattern.
     * @param pattern the pattern to NOT match
     */
    static not(pattern) {
        return new NotMatch('not', pattern);
    }
    /**
     * Matches any string-encoded JSON and applies the specified pattern after parsing it.
     * @param pattern the pattern to match after parsing the encoded JSON.
     */
    static serializedJson(pattern) {
        return new SerializedJson('serializedJson', pattern);
    }
    /**
     * Matches any non-null value at the target.
     */
    static anyValue() {
        return new AnyMatch('anyValue');
    }
    /**
     * Matches targets according to a regular expression
     */
    static stringLikeRegexp(pattern) {
        return new StringLikeRegexpMatch('stringLikeRegexp', pattern);
    }
}
exports.Match = Match;
_a = JSII_RTTI_SYMBOL_1;
Match[_a] = { fqn: "monocdk.assertions.Match", version: "1.191.0" };
/**
 * A Match class that expects the target to match with the pattern exactly.
 * The pattern may be nested with other matchers that are then deletegated to.
 */
class LiteralMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        super();
        this.name = name;
        this.pattern = pattern;
        this.partialObjects = options.partialObjects ?? false;
        if (matcher_1.Matcher.isMatcher(this.pattern)) {
            throw new Error('LiteralMatch cannot directly contain another matcher. ' +
                'Remove the top-level matcher or nest it more deeply.');
        }
    }
    test(actual) {
        if (Array.isArray(this.pattern)) {
            return new ArrayMatch(this.name, this.pattern, { subsequence: false, partialObjects: this.partialObjects }).test(actual);
        }
        if (typeof this.pattern === 'object') {
            return new ObjectMatch(this.name, this.pattern, { partial: this.partialObjects }).test(actual);
        }
        const result = new matcher_1.MatchResult(actual);
        if (typeof this.pattern !== typeof actual) {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Expected type ${typeof this.pattern} but received ${type_1.getType(actual)}`,
            });
            return result;
        }
        if (actual !== this.pattern) {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Expected ${this.pattern} but received ${actual}`,
            });
        }
        return result;
    }
}
/**
 * Match class that matches arrays.
 */
class ArrayMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        super();
        this.name = name;
        this.pattern = pattern;
        this.subsequence = options.subsequence ?? true;
        this.partialObjects = options.partialObjects ?? false;
    }
    test(actual) {
        if (!Array.isArray(actual)) {
            return new matcher_1.MatchResult(actual).recordFailure({
                matcher: this,
                path: [],
                message: `Expected type array but received ${type_1.getType(actual)}`,
            });
        }
        if (!this.subsequence && this.pattern.length !== actual.length) {
            return new matcher_1.MatchResult(actual).recordFailure({
                matcher: this,
                path: [],
                message: `Expected array of length ${this.pattern.length} but received ${actual.length}`,
            });
        }
        let patternIdx = 0;
        let actualIdx = 0;
        const result = new matcher_1.MatchResult(actual);
        while (patternIdx < this.pattern.length && actualIdx < actual.length) {
            const patternElement = this.pattern[patternIdx];
            const matcher = matcher_1.Matcher.isMatcher(patternElement)
                ? patternElement
                : new LiteralMatch(this.name, patternElement, { partialObjects: this.partialObjects });
            const matcherName = matcher.name;
            if (this.subsequence && (matcherName == 'absent' || matcherName == 'anyValue')) {
                // array subsequence matcher is not compatible with anyValue() or absent() matcher. They don't make sense to be used together.
                throw new Error(`The Matcher ${matcherName}() cannot be nested within arrayWith()`);
            }
            const innerResult = matcher.test(actual[actualIdx]);
            if (!this.subsequence || !innerResult.hasFailed()) {
                result.compose(`[${actualIdx}]`, innerResult);
                patternIdx++;
                actualIdx++;
            }
            else {
                actualIdx++;
            }
        }
        for (; patternIdx < this.pattern.length; patternIdx++) {
            const pattern = this.pattern[patternIdx];
            const element = (matcher_1.Matcher.isMatcher(pattern) || typeof pattern === 'object') ? ' ' : ` [${pattern}] `;
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Missing element${element}at pattern index ${patternIdx}`,
            });
        }
        return result;
    }
}
/**
 * Match class that matches objects.
 */
class ObjectMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        super();
        this.name = name;
        this.pattern = pattern;
        this.partial = options.partial ?? true;
    }
    test(actual) {
        if (typeof actual !== 'object' || Array.isArray(actual)) {
            return new matcher_1.MatchResult(actual).recordFailure({
                matcher: this,
                path: [],
                message: `Expected type object but received ${type_1.getType(actual)}`,
            });
        }
        const result = new matcher_1.MatchResult(actual);
        if (!this.partial) {
            for (const a of Object.keys(actual)) {
                if (!(a in this.pattern)) {
                    result.recordFailure({
                        matcher: this,
                        path: [`/${a}`],
                        message: 'Unexpected key',
                    });
                }
            }
        }
        for (const [patternKey, patternVal] of Object.entries(this.pattern)) {
            if (!(patternKey in actual) && !(patternVal instanceof absent_1.AbsentMatch)) {
                result.recordFailure({
                    matcher: this,
                    path: [`/${patternKey}`],
                    message: `Missing key '${patternKey}' among {${Object.keys(actual).join(',')}}`,
                });
                continue;
            }
            const matcher = matcher_1.Matcher.isMatcher(patternVal) ?
                patternVal :
                new LiteralMatch(this.name, patternVal, { partialObjects: this.partial });
            const inner = matcher.test(actual[patternKey]);
            result.compose(`/${patternKey}`, inner);
        }
        return result;
    }
}
class SerializedJson extends matcher_1.Matcher {
    constructor(name, pattern) {
        super();
        this.name = name;
        this.pattern = pattern;
    }
    ;
    test(actual) {
        const result = new matcher_1.MatchResult(actual);
        if (type_1.getType(actual) !== 'string') {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Expected JSON as a string but found ${type_1.getType(actual)}`,
            });
            return result;
        }
        let parsed;
        try {
            parsed = JSON.parse(actual);
        }
        catch (err) {
            if (err instanceof SyntaxError) {
                result.recordFailure({
                    matcher: this,
                    path: [],
                    message: `Invalid JSON string: ${actual}`,
                });
                return result;
            }
            else {
                throw err;
            }
        }
        const matcher = matcher_1.Matcher.isMatcher(this.pattern) ? this.pattern : new LiteralMatch(this.name, this.pattern);
        const innerResult = matcher.test(parsed);
        result.compose(`(${this.name})`, innerResult);
        return result;
    }
}
class NotMatch extends matcher_1.Matcher {
    constructor(name, pattern) {
        super();
        this.name = name;
        this.pattern = pattern;
    }
    test(actual) {
        const matcher = matcher_1.Matcher.isMatcher(this.pattern) ? this.pattern : new LiteralMatch(this.name, this.pattern);
        const innerResult = matcher.test(actual);
        const result = new matcher_1.MatchResult(actual);
        if (innerResult.failCount === 0) {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Found unexpected match: ${JSON.stringify(actual, undefined, 2)}`,
            });
        }
        return result;
    }
}
class AnyMatch extends matcher_1.Matcher {
    constructor(name) {
        super();
        this.name = name;
    }
    test(actual) {
        const result = new matcher_1.MatchResult(actual);
        if (actual == null) {
            result.recordFailure({
                matcher: this,
                path: [],
                message: 'Expected a value but found none',
            });
        }
        return result;
    }
}
class StringLikeRegexpMatch extends matcher_1.Matcher {
    constructor(name, pattern) {
        super();
        this.name = name;
        this.pattern = pattern;
    }
    test(actual) {
        const result = new matcher_1.MatchResult(actual);
        const regex = new RegExp(this.pattern, 'gm');
        if (typeof actual !== 'string') {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Expected a string, but got '${typeof actual}'`,
            });
        }
        if (!regex.test(actual)) {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `String '${actual}' did not match pattern '${this.pattern}'`,
            });
        }
        return result;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibWF0Y2guanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJtYXRjaC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLHVDQUFpRDtBQUNqRCxzREFBd0Q7QUFDeEQseUNBQXlDO0FBRXpDOztHQUVHO0FBQ0gsTUFBc0IsS0FBSztJQUN6Qjs7T0FFRztJQUNJLE1BQU0sQ0FBQyxNQUFNO1FBQ2xCLE9BQU8sSUFBSSxvQkFBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0tBQ2xDO0lBRUQ7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxTQUFTLENBQUMsT0FBYztRQUNwQyxPQUFPLElBQUksVUFBVSxDQUFDLFdBQVcsRUFBRSxPQUFPLENBQUMsQ0FBQztLQUM3QztJQUVEOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsV0FBVyxDQUFDLE9BQWM7UUFDdEMsT0FBTyxJQUFJLFVBQVUsQ0FBQyxhQUFhLEVBQUUsT0FBTyxFQUFFLEVBQUUsV0FBVyxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7S0FDdkU7SUFFRDs7O09BR0c7SUFDSSxNQUFNLENBQUMsS0FBSyxDQUFDLE9BQVk7UUFDOUIsT0FBTyxJQUFJLFlBQVksQ0FBQyxPQUFPLEVBQUUsT0FBTyxFQUFFLEVBQUUsY0FBYyxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7S0FDdEU7SUFFRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLFVBQVUsQ0FBQyxPQUE2QjtRQUNwRCxPQUFPLElBQUksV0FBVyxDQUFDLFlBQVksRUFBRSxPQUFPLENBQUMsQ0FBQztLQUMvQztJQUVEOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsWUFBWSxDQUFDLE9BQTZCO1FBQ3RELE9BQU8sSUFBSSxXQUFXLENBQUMsY0FBYyxFQUFFLE9BQU8sRUFBRSxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDO0tBQ3JFO0lBRUQ7OztPQUdHO0lBQ0ksTUFBTSxDQUFDLEdBQUcsQ0FBQyxPQUFZO1FBQzVCLE9BQU8sSUFBSSxRQUFRLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO0tBQ3JDO0lBRUQ7OztPQUdHO0lBQ0ksTUFBTSxDQUFDLGNBQWMsQ0FBQyxPQUFZO1FBQ3ZDLE9BQU8sSUFBSSxjQUFjLENBQUMsZ0JBQWdCLEVBQUUsT0FBTyxDQUFDLENBQUM7S0FDdEQ7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxRQUFRO1FBQ3BCLE9BQU8sSUFBSSxRQUFRLENBQUMsVUFBVSxDQUFDLENBQUM7S0FDakM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFlO1FBQzVDLE9BQU8sSUFBSSxxQkFBcUIsQ0FBQyxrQkFBa0IsRUFBRSxPQUFPLENBQUMsQ0FBQztLQUMvRDs7QUFoRkgsc0JBaUZDOzs7QUFhRDs7O0dBR0c7QUFDSCxNQUFNLFlBQWEsU0FBUSxpQkFBTztJQUdoQyxZQUNrQixJQUFZLEVBQ1gsT0FBWSxFQUM3QixVQUErQixFQUFFO1FBRWpDLEtBQUssRUFBRSxDQUFDO1FBSlEsU0FBSSxHQUFKLElBQUksQ0FBUTtRQUNYLFlBQU8sR0FBUCxPQUFPLENBQUs7UUFJN0IsSUFBSSxDQUFDLGNBQWMsR0FBRyxPQUFPLENBQUMsY0FBYyxJQUFJLEtBQUssQ0FBQztRQUV0RCxJQUFJLGlCQUFPLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsRUFBRTtZQUNuQyxNQUFNLElBQUksS0FBSyxDQUFDLHdEQUF3RDtnQkFDdEUsc0RBQXNELENBQUMsQ0FBQztTQUMzRDtLQUNGO0lBRU0sSUFBSSxDQUFDLE1BQVc7UUFDckIsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsRUFBRTtZQUMvQixPQUFPLElBQUksVUFBVSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLE9BQU8sRUFBRSxFQUFFLFdBQVcsRUFBRSxLQUFLLEVBQUUsY0FBYyxFQUFFLElBQUksQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztTQUMxSDtRQUVELElBQUksT0FBTyxJQUFJLENBQUMsT0FBTyxLQUFLLFFBQVEsRUFBRTtZQUNwQyxPQUFPLElBQUksV0FBVyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLE9BQU8sRUFBRSxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDaEc7UUFFRCxNQUFNLE1BQU0sR0FBRyxJQUFJLHFCQUFXLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDdkMsSUFBSSxPQUFPLElBQUksQ0FBQyxPQUFPLEtBQUssT0FBTyxNQUFNLEVBQUU7WUFDekMsTUFBTSxDQUFDLGFBQWEsQ0FBQztnQkFDbkIsT0FBTyxFQUFFLElBQUk7Z0JBQ2IsSUFBSSxFQUFFLEVBQUU7Z0JBQ1IsT0FBTyxFQUFFLGlCQUFpQixPQUFPLElBQUksQ0FBQyxPQUFPLGlCQUFpQixjQUFPLENBQUMsTUFBTSxDQUFDLEVBQUU7YUFDaEYsQ0FBQyxDQUFDO1lBQ0gsT0FBTyxNQUFNLENBQUM7U0FDZjtRQUVELElBQUksTUFBTSxLQUFLLElBQUksQ0FBQyxPQUFPLEVBQUU7WUFDM0IsTUFBTSxDQUFDLGFBQWEsQ0FBQztnQkFDbkIsT0FBTyxFQUFFLElBQUk7Z0JBQ2IsSUFBSSxFQUFFLEVBQUU7Z0JBQ1IsT0FBTyxFQUFFLFlBQVksSUFBSSxDQUFDLE9BQU8saUJBQWlCLE1BQU0sRUFBRTthQUMzRCxDQUFDLENBQUM7U0FDSjtRQUVELE9BQU8sTUFBTSxDQUFDO0tBQ2Y7Q0FDRjtBQXNCRDs7R0FFRztBQUNILE1BQU0sVUFBVyxTQUFRLGlCQUFPO0lBSTlCLFlBQ2tCLElBQVksRUFDWCxPQUFjLEVBQy9CLFVBQTZCLEVBQUU7UUFFL0IsS0FBSyxFQUFFLENBQUM7UUFKUSxTQUFJLEdBQUosSUFBSSxDQUFRO1FBQ1gsWUFBTyxHQUFQLE9BQU8sQ0FBTztRQUkvQixJQUFJLENBQUMsV0FBVyxHQUFHLE9BQU8sQ0FBQyxXQUFXLElBQUksSUFBSSxDQUFDO1FBQy9DLElBQUksQ0FBQyxjQUFjLEdBQUcsT0FBTyxDQUFDLGNBQWMsSUFBSSxLQUFLLENBQUM7S0FDdkQ7SUFFTSxJQUFJLENBQUMsTUFBVztRQUNyQixJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRTtZQUMxQixPQUFPLElBQUkscUJBQVcsQ0FBQyxNQUFNLENBQUMsQ0FBQyxhQUFhLENBQUM7Z0JBQzNDLE9BQU8sRUFBRSxJQUFJO2dCQUNiLElBQUksRUFBRSxFQUFFO2dCQUNSLE9BQU8sRUFBRSxvQ0FBb0MsY0FBTyxDQUFDLE1BQU0sQ0FBQyxFQUFFO2FBQy9ELENBQUMsQ0FBQztTQUNKO1FBQ0QsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLEtBQUssTUFBTSxDQUFDLE1BQU0sRUFBRTtZQUM5RCxPQUFPLElBQUkscUJBQVcsQ0FBQyxNQUFNLENBQUMsQ0FBQyxhQUFhLENBQUM7Z0JBQzNDLE9BQU8sRUFBRSxJQUFJO2dCQUNiLElBQUksRUFBRSxFQUFFO2dCQUNSLE9BQU8sRUFBRSw0QkFBNEIsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLGlCQUFpQixNQUFNLENBQUMsTUFBTSxFQUFFO2FBQ3pGLENBQUMsQ0FBQztTQUNKO1FBRUQsSUFBSSxVQUFVLEdBQUcsQ0FBQyxDQUFDO1FBQ25CLElBQUksU0FBUyxHQUFHLENBQUMsQ0FBQztRQUVsQixNQUFNLE1BQU0sR0FBRyxJQUFJLHFCQUFXLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDdkMsT0FBTyxVQUFVLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLElBQUksU0FBUyxHQUFHLE1BQU0sQ0FBQyxNQUFNLEVBQUU7WUFDcEUsTUFBTSxjQUFjLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsQ0FBQztZQUVoRCxNQUFNLE9BQU8sR0FBRyxpQkFBTyxDQUFDLFNBQVMsQ0FBQyxjQUFjLENBQUM7Z0JBQy9DLENBQUMsQ0FBQyxjQUFjO2dCQUNoQixDQUFDLENBQUMsSUFBSSxZQUFZLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxjQUFjLEVBQUUsRUFBRSxjQUFjLEVBQUUsSUFBSSxDQUFDLGNBQWMsRUFBRSxDQUFDLENBQUM7WUFFekYsTUFBTSxXQUFXLEdBQUcsT0FBTyxDQUFDLElBQUksQ0FBQztZQUNqQyxJQUFJLElBQUksQ0FBQyxXQUFXLElBQUksQ0FBQyxXQUFXLElBQUksUUFBUSxJQUFJLFdBQVcsSUFBSSxVQUFVLENBQUMsRUFBRTtnQkFDOUUsOEhBQThIO2dCQUM5SCxNQUFNLElBQUksS0FBSyxDQUFDLGVBQWUsV0FBVyx3Q0FBd0MsQ0FBQyxDQUFDO2FBQ3JGO1lBRUQsTUFBTSxXQUFXLEdBQUcsT0FBTyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQztZQUVwRCxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxTQUFTLEVBQUUsRUFBRTtnQkFDakQsTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFJLFNBQVMsR0FBRyxFQUFFLFdBQVcsQ0FBQyxDQUFDO2dCQUM5QyxVQUFVLEVBQUUsQ0FBQztnQkFDYixTQUFTLEVBQUUsQ0FBQzthQUNiO2lCQUFNO2dCQUNMLFNBQVMsRUFBRSxDQUFDO2FBQ2I7U0FDRjtRQUVELE9BQU8sVUFBVSxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxFQUFFLFVBQVUsRUFBRSxFQUFFO1lBQ3JELE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUM7WUFDekMsTUFBTSxPQUFPLEdBQUcsQ0FBQyxpQkFBTyxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsSUFBSSxPQUFPLE9BQU8sS0FBSyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxLQUFLLE9BQU8sSUFBSSxDQUFDO1lBQ3JHLE1BQU0sQ0FBQyxhQUFhLENBQUM7Z0JBQ25CLE9BQU8sRUFBRSxJQUFJO2dCQUNiLElBQUksRUFBRSxFQUFFO2dCQUNSLE9BQU8sRUFBRSxrQkFBa0IsT0FBTyxvQkFBb0IsVUFBVSxFQUFFO2FBQ25FLENBQUMsQ0FBQztTQUNKO1FBRUQsT0FBTyxNQUFNLENBQUM7S0FDZjtDQUNGO0FBY0Q7O0dBRUc7QUFDSCxNQUFNLFdBQVksU0FBUSxpQkFBTztJQUcvQixZQUNrQixJQUFZLEVBQ1gsT0FBNkIsRUFDOUMsVUFBOEIsRUFBRTtRQUVoQyxLQUFLLEVBQUUsQ0FBQztRQUpRLFNBQUksR0FBSixJQUFJLENBQVE7UUFDWCxZQUFPLEdBQVAsT0FBTyxDQUFzQjtRQUk5QyxJQUFJLENBQUMsT0FBTyxHQUFHLE9BQU8sQ0FBQyxPQUFPLElBQUksSUFBSSxDQUFDO0tBQ3hDO0lBRU0sSUFBSSxDQUFDLE1BQVc7UUFDckIsSUFBSSxPQUFPLE1BQU0sS0FBSyxRQUFRLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRTtZQUN2RCxPQUFPLElBQUkscUJBQVcsQ0FBQyxNQUFNLENBQUMsQ0FBQyxhQUFhLENBQUM7Z0JBQzNDLE9BQU8sRUFBRSxJQUFJO2dCQUNiLElBQUksRUFBRSxFQUFFO2dCQUNSLE9BQU8sRUFBRSxxQ0FBcUMsY0FBTyxDQUFDLE1BQU0sQ0FBQyxFQUFFO2FBQ2hFLENBQUMsQ0FBQztTQUNKO1FBRUQsTUFBTSxNQUFNLEdBQUcsSUFBSSxxQkFBVyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3ZDLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFO1lBQ2pCLEtBQUssTUFBTSxDQUFDLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsRUFBRTtnQkFDbkMsSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLElBQUksQ0FBQyxPQUFPLENBQUMsRUFBRTtvQkFDeEIsTUFBTSxDQUFDLGFBQWEsQ0FBQzt3QkFDbkIsT0FBTyxFQUFFLElBQUk7d0JBQ2IsSUFBSSxFQUFFLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQzt3QkFDZixPQUFPLEVBQUUsZ0JBQWdCO3FCQUMxQixDQUFDLENBQUM7aUJBQ0o7YUFDRjtTQUNGO1FBRUQsS0FBSyxNQUFNLENBQUMsVUFBVSxFQUFFLFVBQVUsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQ25FLElBQUksQ0FBQyxDQUFDLFVBQVUsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsVUFBVSxZQUFZLG9CQUFXLENBQUMsRUFBRTtnQkFDbkUsTUFBTSxDQUFDLGFBQWEsQ0FBQztvQkFDbkIsT0FBTyxFQUFFLElBQUk7b0JBQ2IsSUFBSSxFQUFFLENBQUMsSUFBSSxVQUFVLEVBQUUsQ0FBQztvQkFDeEIsT0FBTyxFQUFFLGdCQUFnQixVQUFVLFlBQVksTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUc7aUJBQ2hGLENBQUMsQ0FBQztnQkFDSCxTQUFTO2FBQ1Y7WUFDRCxNQUFNLE9BQU8sR0FBRyxpQkFBTyxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO2dCQUM3QyxVQUFVLENBQUMsQ0FBQztnQkFDWixJQUFJLFlBQVksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRSxFQUFFLGNBQWMsRUFBRSxJQUFJLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQztZQUM1RSxNQUFNLEtBQUssR0FBRyxPQUFPLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO1lBQy9DLE1BQU0sQ0FBQyxPQUFPLENBQUMsSUFBSSxVQUFVLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztTQUN6QztRQUVELE9BQU8sTUFBTSxDQUFDO0tBQ2Y7Q0FDRjtBQUVELE1BQU0sY0FBZSxTQUFRLGlCQUFPO0lBQ2xDLFlBQ2tCLElBQVksRUFDWCxPQUFZO1FBRTdCLEtBQUssRUFBRSxDQUFDO1FBSFEsU0FBSSxHQUFKLElBQUksQ0FBUTtRQUNYLFlBQU8sR0FBUCxPQUFPLENBQUs7S0FHOUI7SUFBQSxDQUFDO0lBRUssSUFBSSxDQUFDLE1BQVc7UUFDckIsTUFBTSxNQUFNLEdBQUcsSUFBSSxxQkFBVyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3ZDLElBQUksY0FBTyxDQUFDLE1BQU0sQ0FBQyxLQUFLLFFBQVEsRUFBRTtZQUNoQyxNQUFNLENBQUMsYUFBYSxDQUFDO2dCQUNuQixPQUFPLEVBQUUsSUFBSTtnQkFDYixJQUFJLEVBQUUsRUFBRTtnQkFDUixPQUFPLEVBQUUsdUNBQXVDLGNBQU8sQ0FBQyxNQUFNLENBQUMsRUFBRTthQUNsRSxDQUFDLENBQUM7WUFDSCxPQUFPLE1BQU0sQ0FBQztTQUNmO1FBQ0QsSUFBSSxNQUFNLENBQUM7UUFDWCxJQUFJO1lBQ0YsTUFBTSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDN0I7UUFBQyxPQUFPLEdBQUcsRUFBRTtZQUNaLElBQUksR0FBRyxZQUFZLFdBQVcsRUFBRTtnQkFDOUIsTUFBTSxDQUFDLGFBQWEsQ0FBQztvQkFDbkIsT0FBTyxFQUFFLElBQUk7b0JBQ2IsSUFBSSxFQUFFLEVBQUU7b0JBQ1IsT0FBTyxFQUFFLHdCQUF3QixNQUFNLEVBQUU7aUJBQzFDLENBQUMsQ0FBQztnQkFDSCxPQUFPLE1BQU0sQ0FBQzthQUNmO2lCQUFNO2dCQUNMLE1BQU0sR0FBRyxDQUFDO2FBQ1g7U0FDRjtRQUVELE1BQU0sT0FBTyxHQUFHLGlCQUFPLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsSUFBSSxZQUFZLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDM0csTUFBTSxXQUFXLEdBQUcsT0FBTyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUN6QyxNQUFNLENBQUMsT0FBTyxDQUFDLElBQUksSUFBSSxDQUFDLElBQUksR0FBRyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQzlDLE9BQU8sTUFBTSxDQUFDO0tBQ2Y7Q0FDRjtBQUVELE1BQU0sUUFBUyxTQUFRLGlCQUFPO0lBQzVCLFlBQ2tCLElBQVksRUFDWCxPQUE2QjtRQUU5QyxLQUFLLEVBQUUsQ0FBQztRQUhRLFNBQUksR0FBSixJQUFJLENBQVE7UUFDWCxZQUFPLEdBQVAsT0FBTyxDQUFzQjtLQUcvQztJQUVNLElBQUksQ0FBQyxNQUFXO1FBQ3JCLE1BQU0sT0FBTyxHQUFHLGlCQUFPLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsSUFBSSxZQUFZLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFM0csTUFBTSxXQUFXLEdBQUcsT0FBTyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUN6QyxNQUFNLE1BQU0sR0FBRyxJQUFJLHFCQUFXLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDdkMsSUFBSSxXQUFXLENBQUMsU0FBUyxLQUFLLENBQUMsRUFBRTtZQUMvQixNQUFNLENBQUMsYUFBYSxDQUFDO2dCQUNuQixPQUFPLEVBQUUsSUFBSTtnQkFDYixJQUFJLEVBQUUsRUFBRTtnQkFDUixPQUFPLEVBQUUsMkJBQTJCLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxFQUFFLFNBQVMsRUFBRSxDQUFDLENBQUMsRUFBRTthQUMzRSxDQUFDLENBQUM7U0FDSjtRQUNELE9BQU8sTUFBTSxDQUFDO0tBQ2Y7Q0FDRjtBQUVELE1BQU0sUUFBUyxTQUFRLGlCQUFPO0lBQzVCLFlBQTRCLElBQVk7UUFDdEMsS0FBSyxFQUFFLENBQUM7UUFEa0IsU0FBSSxHQUFKLElBQUksQ0FBUTtLQUV2QztJQUVNLElBQUksQ0FBQyxNQUFXO1FBQ3JCLE1BQU0sTUFBTSxHQUFHLElBQUkscUJBQVcsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUN2QyxJQUFJLE1BQU0sSUFBSSxJQUFJLEVBQUU7WUFDbEIsTUFBTSxDQUFDLGFBQWEsQ0FBQztnQkFDbkIsT0FBTyxFQUFFLElBQUk7Z0JBQ2IsSUFBSSxFQUFFLEVBQUU7Z0JBQ1IsT0FBTyxFQUFFLGlDQUFpQzthQUMzQyxDQUFDLENBQUM7U0FDSjtRQUNELE9BQU8sTUFBTSxDQUFDO0tBQ2Y7Q0FDRjtBQUVELE1BQU0scUJBQXNCLFNBQVEsaUJBQU87SUFDekMsWUFDa0IsSUFBWSxFQUNYLE9BQWU7UUFFaEMsS0FBSyxFQUFFLENBQUM7UUFIUSxTQUFJLEdBQUosSUFBSSxDQUFRO1FBQ1gsWUFBTyxHQUFQLE9BQU8sQ0FBUTtLQUdqQztJQUVELElBQUksQ0FBQyxNQUFXO1FBQ2QsTUFBTSxNQUFNLEdBQUcsSUFBSSxxQkFBVyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBRXZDLE1BQU0sS0FBSyxHQUFHLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFFN0MsSUFBSSxPQUFPLE1BQU0sS0FBSyxRQUFRLEVBQUU7WUFDOUIsTUFBTSxDQUFDLGFBQWEsQ0FBQztnQkFDbkIsT0FBTyxFQUFFLElBQUk7Z0JBQ2IsSUFBSSxFQUFFLEVBQUU7Z0JBQ1IsT0FBTyxFQUFFLCtCQUErQixPQUFPLE1BQU0sR0FBRzthQUN6RCxDQUFDLENBQUM7U0FDSjtRQUVELElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxFQUFFO1lBQ3ZCLE1BQU0sQ0FBQyxhQUFhLENBQUM7Z0JBQ25CLE9BQU8sRUFBRSxJQUFJO2dCQUNiLElBQUksRUFBRSxFQUFFO2dCQUNSLE9BQU8sRUFBRSxXQUFXLE1BQU0sNEJBQTRCLElBQUksQ0FBQyxPQUFPLEdBQUc7YUFDdEUsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxPQUFPLE1BQU0sQ0FBQztLQUNmO0NBRUYiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBNYXRjaGVyLCBNYXRjaFJlc3VsdCB9IGZyb20gJy4vbWF0Y2hlcic7XG5pbXBvcnQgeyBBYnNlbnRNYXRjaCB9IGZyb20gJy4vcHJpdmF0ZS9tYXRjaGVycy9hYnNlbnQnO1xuaW1wb3J0IHsgZ2V0VHlwZSB9IGZyb20gJy4vcHJpdmF0ZS90eXBlJztcblxuLyoqXG4gKiBQYXJ0aWFsIGFuZCBzcGVjaWFsIG1hdGNoaW5nIGR1cmluZyB0ZW1wbGF0ZSBhc3NlcnRpb25zLlxuICovXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgTWF0Y2gge1xuICAvKipcbiAgICogVXNlIHRoaXMgbWF0Y2hlciBpbiB0aGUgcGxhY2Ugb2YgYSBmaWVsZCdzIHZhbHVlLCBpZiB0aGUgZmllbGQgbXVzdCBub3QgYmUgcHJlc2VudC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgYWJzZW50KCk6IE1hdGNoZXIge1xuICAgIHJldHVybiBuZXcgQWJzZW50TWF0Y2goJ2Fic2VudCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIE1hdGNoZXMgdGhlIHNwZWNpZmllZCBwYXR0ZXJuIHdpdGggdGhlIGFycmF5IGZvdW5kIGluIHRoZSBzYW1lIHJlbGF0aXZlIHBhdGggb2YgdGhlIHRhcmdldC5cbiAgICogVGhlIHNldCBvZiBlbGVtZW50cyAob3IgbWF0Y2hlcnMpIG11c3QgYmUgaW4gdGhlIHNhbWUgb3JkZXIgYXMgd291bGQgYmUgZm91bmQuXG4gICAqIEBwYXJhbSBwYXR0ZXJuIHRoZSBwYXR0ZXJuIHRvIG1hdGNoXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGFycmF5V2l0aChwYXR0ZXJuOiBhbnlbXSk6IE1hdGNoZXIge1xuICAgIHJldHVybiBuZXcgQXJyYXlNYXRjaCgnYXJyYXlXaXRoJywgcGF0dGVybik7XG4gIH1cblxuICAvKipcbiAgICogTWF0Y2hlcyB0aGUgc3BlY2lmaWVkIHBhdHRlcm4gd2l0aCB0aGUgYXJyYXkgZm91bmQgaW4gdGhlIHNhbWUgcmVsYXRpdmUgcGF0aCBvZiB0aGUgdGFyZ2V0LlxuICAgKiBUaGUgc2V0IG9mIGVsZW1lbnRzIChvciBtYXRjaGVycykgbXVzdCBtYXRjaCBleGFjdGx5IGFuZCBpbiBvcmRlci5cbiAgICogQHBhcmFtIHBhdHRlcm4gdGhlIHBhdHRlcm4gdG8gbWF0Y2hcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgYXJyYXlFcXVhbHMocGF0dGVybjogYW55W10pOiBNYXRjaGVyIHtcbiAgICByZXR1cm4gbmV3IEFycmF5TWF0Y2goJ2FycmF5RXF1YWxzJywgcGF0dGVybiwgeyBzdWJzZXF1ZW5jZTogZmFsc2UgfSk7XG4gIH1cblxuICAvKipcbiAgICogRGVlcCBleGFjdCBtYXRjaGluZyBvZiB0aGUgc3BlY2lmaWVkIHBhdHRlcm4gdG8gdGhlIHRhcmdldC5cbiAgICogQHBhcmFtIHBhdHRlcm4gdGhlIHBhdHRlcm4gdG8gbWF0Y2hcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZXhhY3QocGF0dGVybjogYW55KTogTWF0Y2hlciB7XG4gICAgcmV0dXJuIG5ldyBMaXRlcmFsTWF0Y2goJ2V4YWN0JywgcGF0dGVybiwgeyBwYXJ0aWFsT2JqZWN0czogZmFsc2UgfSk7XG4gIH1cblxuICAvKipcbiAgICogTWF0Y2hlcyB0aGUgc3BlY2lmaWVkIHBhdHRlcm4gdG8gYW4gb2JqZWN0IGZvdW5kIGluIHRoZSBzYW1lIHJlbGF0aXZlIHBhdGggb2YgdGhlIHRhcmdldC5cbiAgICogVGhlIGtleXMgYW5kIHRoZWlyIHZhbHVlcyAob3IgbWF0Y2hlcnMpIG11c3QgYmUgcHJlc2VudCBpbiB0aGUgdGFyZ2V0IGJ1dCB0aGUgdGFyZ2V0IGNhbiBiZSBhIHN1cGVyc2V0LlxuICAgKiBAcGFyYW0gcGF0dGVybiB0aGUgcGF0dGVybiB0byBtYXRjaFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBvYmplY3RMaWtlKHBhdHRlcm46IHtba2V5OiBzdHJpbmddOiBhbnl9KTogTWF0Y2hlciB7XG4gICAgcmV0dXJuIG5ldyBPYmplY3RNYXRjaCgnb2JqZWN0TGlrZScsIHBhdHRlcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIE1hdGNoZXMgdGhlIHNwZWNpZmllZCBwYXR0ZXJuIHRvIGFuIG9iamVjdCBmb3VuZCBpbiB0aGUgc2FtZSByZWxhdGl2ZSBwYXRoIG9mIHRoZSB0YXJnZXQuXG4gICAqIFRoZSBrZXlzIGFuZCB0aGVpciB2YWx1ZXMgKG9yIG1hdGNoZXJzKSBtdXN0IG1hdGNoIGV4YWN0bHkgd2l0aCB0aGUgdGFyZ2V0LlxuICAgKiBAcGFyYW0gcGF0dGVybiB0aGUgcGF0dGVybiB0byBtYXRjaFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBvYmplY3RFcXVhbHMocGF0dGVybjoge1trZXk6IHN0cmluZ106IGFueX0pOiBNYXRjaGVyIHtcbiAgICByZXR1cm4gbmV3IE9iamVjdE1hdGNoKCdvYmplY3RFcXVhbHMnLCBwYXR0ZXJuLCB7IHBhcnRpYWw6IGZhbHNlIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIE1hdGNoZXMgYW55IHRhcmdldCB3aGljaCBkb2VzIE5PVCBmb2xsb3cgdGhlIHNwZWNpZmllZCBwYXR0ZXJuLlxuICAgKiBAcGFyYW0gcGF0dGVybiB0aGUgcGF0dGVybiB0byBOT1QgbWF0Y2hcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgbm90KHBhdHRlcm46IGFueSk6IE1hdGNoZXIge1xuICAgIHJldHVybiBuZXcgTm90TWF0Y2goJ25vdCcsIHBhdHRlcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIE1hdGNoZXMgYW55IHN0cmluZy1lbmNvZGVkIEpTT04gYW5kIGFwcGxpZXMgdGhlIHNwZWNpZmllZCBwYXR0ZXJuIGFmdGVyIHBhcnNpbmcgaXQuXG4gICAqIEBwYXJhbSBwYXR0ZXJuIHRoZSBwYXR0ZXJuIHRvIG1hdGNoIGFmdGVyIHBhcnNpbmcgdGhlIGVuY29kZWQgSlNPTi5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgc2VyaWFsaXplZEpzb24ocGF0dGVybjogYW55KTogTWF0Y2hlciB7XG4gICAgcmV0dXJuIG5ldyBTZXJpYWxpemVkSnNvbignc2VyaWFsaXplZEpzb24nLCBwYXR0ZXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBNYXRjaGVzIGFueSBub24tbnVsbCB2YWx1ZSBhdCB0aGUgdGFyZ2V0LlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBhbnlWYWx1ZSgpOiBNYXRjaGVyIHtcbiAgICByZXR1cm4gbmV3IEFueU1hdGNoKCdhbnlWYWx1ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIE1hdGNoZXMgdGFyZ2V0cyBhY2NvcmRpbmcgdG8gYSByZWd1bGFyIGV4cHJlc3Npb25cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgc3RyaW5nTGlrZVJlZ2V4cChwYXR0ZXJuOiBzdHJpbmcpOiBNYXRjaGVyIHtcbiAgICByZXR1cm4gbmV3IFN0cmluZ0xpa2VSZWdleHBNYXRjaCgnc3RyaW5nTGlrZVJlZ2V4cCcsIHBhdHRlcm4pO1xuICB9XG59XG5cbi8qKlxuICogT3B0aW9ucyB3aGVuIGluaXRpYWxpemluZyB0aGUgYExpdGVyYWxNYXRjaGAgY2xhc3MuXG4gKi9cbmludGVyZmFjZSBMaXRlcmFsTWF0Y2hPcHRpb25zIHtcbiAgLyoqXG4gICAqIFdoZXRoZXIgb2JqZWN0cyBuZXN0ZWQgYXQgYW55IGxldmVsIHNob3VsZCBiZSBtYXRjaGVkIHBhcnRpYWxseS5cbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHBhcnRpYWxPYmplY3RzPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBBIE1hdGNoIGNsYXNzIHRoYXQgZXhwZWN0cyB0aGUgdGFyZ2V0IHRvIG1hdGNoIHdpdGggdGhlIHBhdHRlcm4gZXhhY3RseS5cbiAqIFRoZSBwYXR0ZXJuIG1heSBiZSBuZXN0ZWQgd2l0aCBvdGhlciBtYXRjaGVycyB0aGF0IGFyZSB0aGVuIGRlbGV0ZWdhdGVkIHRvLlxuICovXG5jbGFzcyBMaXRlcmFsTWF0Y2ggZXh0ZW5kcyBNYXRjaGVyIHtcbiAgcHJpdmF0ZSByZWFkb25seSBwYXJ0aWFsT2JqZWN0czogYm9vbGVhbjtcblxuICBjb25zdHJ1Y3RvcihcbiAgICBwdWJsaWMgcmVhZG9ubHkgbmFtZTogc3RyaW5nLFxuICAgIHByaXZhdGUgcmVhZG9ubHkgcGF0dGVybjogYW55LFxuICAgIG9wdGlvbnM6IExpdGVyYWxNYXRjaE9wdGlvbnMgPSB7fSkge1xuXG4gICAgc3VwZXIoKTtcbiAgICB0aGlzLnBhcnRpYWxPYmplY3RzID0gb3B0aW9ucy5wYXJ0aWFsT2JqZWN0cyA/PyBmYWxzZTtcblxuICAgIGlmIChNYXRjaGVyLmlzTWF0Y2hlcih0aGlzLnBhdHRlcm4pKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0xpdGVyYWxNYXRjaCBjYW5ub3QgZGlyZWN0bHkgY29udGFpbiBhbm90aGVyIG1hdGNoZXIuICcgK1xuICAgICAgICAnUmVtb3ZlIHRoZSB0b3AtbGV2ZWwgbWF0Y2hlciBvciBuZXN0IGl0IG1vcmUgZGVlcGx5LicpO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyB0ZXN0KGFjdHVhbDogYW55KTogTWF0Y2hSZXN1bHQge1xuICAgIGlmIChBcnJheS5pc0FycmF5KHRoaXMucGF0dGVybikpIHtcbiAgICAgIHJldHVybiBuZXcgQXJyYXlNYXRjaCh0aGlzLm5hbWUsIHRoaXMucGF0dGVybiwgeyBzdWJzZXF1ZW5jZTogZmFsc2UsIHBhcnRpYWxPYmplY3RzOiB0aGlzLnBhcnRpYWxPYmplY3RzIH0pLnRlc3QoYWN0dWFsKTtcbiAgICB9XG5cbiAgICBpZiAodHlwZW9mIHRoaXMucGF0dGVybiA9PT0gJ29iamVjdCcpIHtcbiAgICAgIHJldHVybiBuZXcgT2JqZWN0TWF0Y2godGhpcy5uYW1lLCB0aGlzLnBhdHRlcm4sIHsgcGFydGlhbDogdGhpcy5wYXJ0aWFsT2JqZWN0cyB9KS50ZXN0KGFjdHVhbCk7XG4gICAgfVxuXG4gICAgY29uc3QgcmVzdWx0ID0gbmV3IE1hdGNoUmVzdWx0KGFjdHVhbCk7XG4gICAgaWYgKHR5cGVvZiB0aGlzLnBhdHRlcm4gIT09IHR5cGVvZiBhY3R1YWwpIHtcbiAgICAgIHJlc3VsdC5yZWNvcmRGYWlsdXJlKHtcbiAgICAgICAgbWF0Y2hlcjogdGhpcyxcbiAgICAgICAgcGF0aDogW10sXG4gICAgICAgIG1lc3NhZ2U6IGBFeHBlY3RlZCB0eXBlICR7dHlwZW9mIHRoaXMucGF0dGVybn0gYnV0IHJlY2VpdmVkICR7Z2V0VHlwZShhY3R1YWwpfWAsXG4gICAgICB9KTtcbiAgICAgIHJldHVybiByZXN1bHQ7XG4gICAgfVxuXG4gICAgaWYgKGFjdHVhbCAhPT0gdGhpcy5wYXR0ZXJuKSB7XG4gICAgICByZXN1bHQucmVjb3JkRmFpbHVyZSh7XG4gICAgICAgIG1hdGNoZXI6IHRoaXMsXG4gICAgICAgIHBhdGg6IFtdLFxuICAgICAgICBtZXNzYWdlOiBgRXhwZWN0ZWQgJHt0aGlzLnBhdHRlcm59IGJ1dCByZWNlaXZlZCAke2FjdHVhbH1gLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHJlc3VsdDtcbiAgfVxufVxuXG4vKipcbiAqIE9wdGlvbnMgd2hlbiBpbml0aWFsaXppbmcgdGhlIGBBcnJheU1hdGNoYCBjbGFzcy5cbiAqL1xuaW50ZXJmYWNlIEFycmF5TWF0Y2hPcHRpb25zIHtcbiAgLyoqXG4gICAqIFdoZXRoZXIgdGhlIHBhdHRlcm4gaXMgYSBzdWJzZXF1ZW5jZSBvZiB0aGUgdGFyZ2V0LlxuICAgKiBBIHN1YnNlcXVlbmNlIGlzIGEgc2VxdWVuY2UgdGhhdCBjYW4gYmUgZGVyaXZlZCBmcm9tIGFub3RoZXIgc2VxdWVuY2UgYnkgZGVsZXRpbmdcbiAgICogc29tZSBvciBubyBlbGVtZW50cyB3aXRob3V0IGNoYW5naW5nIHRoZSBvcmRlciBvZiB0aGUgcmVtYWluaW5nIGVsZW1lbnRzLlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBzdWJzZXF1ZW5jZT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdG8gY29udGludWUgbWF0Y2hpbmcgb2JqZWN0cyBpbnNpZGUgdGhlIGFycmF5IHBhcnRpYWxseVxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgcGFydGlhbE9iamVjdHM/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIE1hdGNoIGNsYXNzIHRoYXQgbWF0Y2hlcyBhcnJheXMuXG4gKi9cbmNsYXNzIEFycmF5TWF0Y2ggZXh0ZW5kcyBNYXRjaGVyIHtcbiAgcHJpdmF0ZSByZWFkb25seSBzdWJzZXF1ZW5jZTogYm9vbGVhbjtcbiAgcHJpdmF0ZSByZWFkb25seSBwYXJ0aWFsT2JqZWN0czogYm9vbGVhbjtcblxuICBjb25zdHJ1Y3RvcihcbiAgICBwdWJsaWMgcmVhZG9ubHkgbmFtZTogc3RyaW5nLFxuICAgIHByaXZhdGUgcmVhZG9ubHkgcGF0dGVybjogYW55W10sXG4gICAgb3B0aW9uczogQXJyYXlNYXRjaE9wdGlvbnMgPSB7fSkge1xuXG4gICAgc3VwZXIoKTtcbiAgICB0aGlzLnN1YnNlcXVlbmNlID0gb3B0aW9ucy5zdWJzZXF1ZW5jZSA/PyB0cnVlO1xuICAgIHRoaXMucGFydGlhbE9iamVjdHMgPSBvcHRpb25zLnBhcnRpYWxPYmplY3RzID8/IGZhbHNlO1xuICB9XG5cbiAgcHVibGljIHRlc3QoYWN0dWFsOiBhbnkpOiBNYXRjaFJlc3VsdCB7XG4gICAgaWYgKCFBcnJheS5pc0FycmF5KGFjdHVhbCkpIHtcbiAgICAgIHJldHVybiBuZXcgTWF0Y2hSZXN1bHQoYWN0dWFsKS5yZWNvcmRGYWlsdXJlKHtcbiAgICAgICAgbWF0Y2hlcjogdGhpcyxcbiAgICAgICAgcGF0aDogW10sXG4gICAgICAgIG1lc3NhZ2U6IGBFeHBlY3RlZCB0eXBlIGFycmF5IGJ1dCByZWNlaXZlZCAke2dldFR5cGUoYWN0dWFsKX1gLFxuICAgICAgfSk7XG4gICAgfVxuICAgIGlmICghdGhpcy5zdWJzZXF1ZW5jZSAmJiB0aGlzLnBhdHRlcm4ubGVuZ3RoICE9PSBhY3R1YWwubGVuZ3RoKSB7XG4gICAgICByZXR1cm4gbmV3IE1hdGNoUmVzdWx0KGFjdHVhbCkucmVjb3JkRmFpbHVyZSh7XG4gICAgICAgIG1hdGNoZXI6IHRoaXMsXG4gICAgICAgIHBhdGg6IFtdLFxuICAgICAgICBtZXNzYWdlOiBgRXhwZWN0ZWQgYXJyYXkgb2YgbGVuZ3RoICR7dGhpcy5wYXR0ZXJuLmxlbmd0aH0gYnV0IHJlY2VpdmVkICR7YWN0dWFsLmxlbmd0aH1gLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgbGV0IHBhdHRlcm5JZHggPSAwO1xuICAgIGxldCBhY3R1YWxJZHggPSAwO1xuXG4gICAgY29uc3QgcmVzdWx0ID0gbmV3IE1hdGNoUmVzdWx0KGFjdHVhbCk7XG4gICAgd2hpbGUgKHBhdHRlcm5JZHggPCB0aGlzLnBhdHRlcm4ubGVuZ3RoICYmIGFjdHVhbElkeCA8IGFjdHVhbC5sZW5ndGgpIHtcbiAgICAgIGNvbnN0IHBhdHRlcm5FbGVtZW50ID0gdGhpcy5wYXR0ZXJuW3BhdHRlcm5JZHhdO1xuXG4gICAgICBjb25zdCBtYXRjaGVyID0gTWF0Y2hlci5pc01hdGNoZXIocGF0dGVybkVsZW1lbnQpXG4gICAgICAgID8gcGF0dGVybkVsZW1lbnRcbiAgICAgICAgOiBuZXcgTGl0ZXJhbE1hdGNoKHRoaXMubmFtZSwgcGF0dGVybkVsZW1lbnQsIHsgcGFydGlhbE9iamVjdHM6IHRoaXMucGFydGlhbE9iamVjdHMgfSk7XG5cbiAgICAgIGNvbnN0IG1hdGNoZXJOYW1lID0gbWF0Y2hlci5uYW1lO1xuICAgICAgaWYgKHRoaXMuc3Vic2VxdWVuY2UgJiYgKG1hdGNoZXJOYW1lID09ICdhYnNlbnQnIHx8IG1hdGNoZXJOYW1lID09ICdhbnlWYWx1ZScpKSB7XG4gICAgICAgIC8vIGFycmF5IHN1YnNlcXVlbmNlIG1hdGNoZXIgaXMgbm90IGNvbXBhdGlibGUgd2l0aCBhbnlWYWx1ZSgpIG9yIGFic2VudCgpIG1hdGNoZXIuIFRoZXkgZG9uJ3QgbWFrZSBzZW5zZSB0byBiZSB1c2VkIHRvZ2V0aGVyLlxuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFRoZSBNYXRjaGVyICR7bWF0Y2hlck5hbWV9KCkgY2Fubm90IGJlIG5lc3RlZCB3aXRoaW4gYXJyYXlXaXRoKClgKTtcbiAgICAgIH1cblxuICAgICAgY29uc3QgaW5uZXJSZXN1bHQgPSBtYXRjaGVyLnRlc3QoYWN0dWFsW2FjdHVhbElkeF0pO1xuXG4gICAgICBpZiAoIXRoaXMuc3Vic2VxdWVuY2UgfHwgIWlubmVyUmVzdWx0Lmhhc0ZhaWxlZCgpKSB7XG4gICAgICAgIHJlc3VsdC5jb21wb3NlKGBbJHthY3R1YWxJZHh9XWAsIGlubmVyUmVzdWx0KTtcbiAgICAgICAgcGF0dGVybklkeCsrO1xuICAgICAgICBhY3R1YWxJZHgrKztcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIGFjdHVhbElkeCsrO1xuICAgICAgfVxuICAgIH1cblxuICAgIGZvciAoOyBwYXR0ZXJuSWR4IDwgdGhpcy5wYXR0ZXJuLmxlbmd0aDsgcGF0dGVybklkeCsrKSB7XG4gICAgICBjb25zdCBwYXR0ZXJuID0gdGhpcy5wYXR0ZXJuW3BhdHRlcm5JZHhdO1xuICAgICAgY29uc3QgZWxlbWVudCA9IChNYXRjaGVyLmlzTWF0Y2hlcihwYXR0ZXJuKSB8fCB0eXBlb2YgcGF0dGVybiA9PT0gJ29iamVjdCcpID8gJyAnIDogYCBbJHtwYXR0ZXJufV0gYDtcbiAgICAgIHJlc3VsdC5yZWNvcmRGYWlsdXJlKHtcbiAgICAgICAgbWF0Y2hlcjogdGhpcyxcbiAgICAgICAgcGF0aDogW10sXG4gICAgICAgIG1lc3NhZ2U6IGBNaXNzaW5nIGVsZW1lbnQke2VsZW1lbnR9YXQgcGF0dGVybiBpbmRleCAke3BhdHRlcm5JZHh9YCxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHJldHVybiByZXN1bHQ7XG4gIH1cbn1cblxuLyoqXG4gKiBPcHRpb25zIHdoZW4gaW5pdGlhbGl6aW5nIGBPYmplY3RNYXRjaGAgY2xhc3MuXG4gKi9cbmludGVyZmFjZSBPYmplY3RNYXRjaE9wdGlvbnMge1xuICAvKipcbiAgICogV2hldGhlciB0aGUgcGF0dGVybiBzaG91bGQgcGFydGlhbGx5IG1hdGNoIHdpdGggdGhlIHRhcmdldCBvYmplY3QuXG4gICAqIFRoZSB0YXJnZXQgb2JqZWN0IGNhbiBjb250YWluIG1vcmUga2V5cyB0aGFuIGV4cGVjdGVkIGJ5IHRoZSBwYXR0ZXJuLlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBwYXJ0aWFsPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBNYXRjaCBjbGFzcyB0aGF0IG1hdGNoZXMgb2JqZWN0cy5cbiAqL1xuY2xhc3MgT2JqZWN0TWF0Y2ggZXh0ZW5kcyBNYXRjaGVyIHtcbiAgcHJpdmF0ZSByZWFkb25seSBwYXJ0aWFsOiBib29sZWFuO1xuXG4gIGNvbnN0cnVjdG9yKFxuICAgIHB1YmxpYyByZWFkb25seSBuYW1lOiBzdHJpbmcsXG4gICAgcHJpdmF0ZSByZWFkb25seSBwYXR0ZXJuOiB7W2tleTogc3RyaW5nXTogYW55fSxcbiAgICBvcHRpb25zOiBPYmplY3RNYXRjaE9wdGlvbnMgPSB7fSkge1xuXG4gICAgc3VwZXIoKTtcbiAgICB0aGlzLnBhcnRpYWwgPSBvcHRpb25zLnBhcnRpYWwgPz8gdHJ1ZTtcbiAgfVxuXG4gIHB1YmxpYyB0ZXN0KGFjdHVhbDogYW55KTogTWF0Y2hSZXN1bHQge1xuICAgIGlmICh0eXBlb2YgYWN0dWFsICE9PSAnb2JqZWN0JyB8fCBBcnJheS5pc0FycmF5KGFjdHVhbCkpIHtcbiAgICAgIHJldHVybiBuZXcgTWF0Y2hSZXN1bHQoYWN0dWFsKS5yZWNvcmRGYWlsdXJlKHtcbiAgICAgICAgbWF0Y2hlcjogdGhpcyxcbiAgICAgICAgcGF0aDogW10sXG4gICAgICAgIG1lc3NhZ2U6IGBFeHBlY3RlZCB0eXBlIG9iamVjdCBidXQgcmVjZWl2ZWQgJHtnZXRUeXBlKGFjdHVhbCl9YCxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIGNvbnN0IHJlc3VsdCA9IG5ldyBNYXRjaFJlc3VsdChhY3R1YWwpO1xuICAgIGlmICghdGhpcy5wYXJ0aWFsKSB7XG4gICAgICBmb3IgKGNvbnN0IGEgb2YgT2JqZWN0LmtleXMoYWN0dWFsKSkge1xuICAgICAgICBpZiAoIShhIGluIHRoaXMucGF0dGVybikpIHtcbiAgICAgICAgICByZXN1bHQucmVjb3JkRmFpbHVyZSh7XG4gICAgICAgICAgICBtYXRjaGVyOiB0aGlzLFxuICAgICAgICAgICAgcGF0aDogW2AvJHthfWBdLFxuICAgICAgICAgICAgbWVzc2FnZTogJ1VuZXhwZWN0ZWQga2V5JyxcbiAgICAgICAgICB9KTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cblxuICAgIGZvciAoY29uc3QgW3BhdHRlcm5LZXksIHBhdHRlcm5WYWxdIG9mIE9iamVjdC5lbnRyaWVzKHRoaXMucGF0dGVybikpIHtcbiAgICAgIGlmICghKHBhdHRlcm5LZXkgaW4gYWN0dWFsKSAmJiAhKHBhdHRlcm5WYWwgaW5zdGFuY2VvZiBBYnNlbnRNYXRjaCkpIHtcbiAgICAgICAgcmVzdWx0LnJlY29yZEZhaWx1cmUoe1xuICAgICAgICAgIG1hdGNoZXI6IHRoaXMsXG4gICAgICAgICAgcGF0aDogW2AvJHtwYXR0ZXJuS2V5fWBdLFxuICAgICAgICAgIG1lc3NhZ2U6IGBNaXNzaW5nIGtleSAnJHtwYXR0ZXJuS2V5fScgYW1vbmcgeyR7T2JqZWN0LmtleXMoYWN0dWFsKS5qb2luKCcsJyl9fWAsXG4gICAgICAgIH0pO1xuICAgICAgICBjb250aW51ZTtcbiAgICAgIH1cbiAgICAgIGNvbnN0IG1hdGNoZXIgPSBNYXRjaGVyLmlzTWF0Y2hlcihwYXR0ZXJuVmFsKSA/XG4gICAgICAgIHBhdHRlcm5WYWwgOlxuICAgICAgICBuZXcgTGl0ZXJhbE1hdGNoKHRoaXMubmFtZSwgcGF0dGVyblZhbCwgeyBwYXJ0aWFsT2JqZWN0czogdGhpcy5wYXJ0aWFsIH0pO1xuICAgICAgY29uc3QgaW5uZXIgPSBtYXRjaGVyLnRlc3QoYWN0dWFsW3BhdHRlcm5LZXldKTtcbiAgICAgIHJlc3VsdC5jb21wb3NlKGAvJHtwYXR0ZXJuS2V5fWAsIGlubmVyKTtcbiAgICB9XG5cbiAgICByZXR1cm4gcmVzdWx0O1xuICB9XG59XG5cbmNsYXNzIFNlcmlhbGl6ZWRKc29uIGV4dGVuZHMgTWF0Y2hlciB7XG4gIGNvbnN0cnVjdG9yKFxuICAgIHB1YmxpYyByZWFkb25seSBuYW1lOiBzdHJpbmcsXG4gICAgcHJpdmF0ZSByZWFkb25seSBwYXR0ZXJuOiBhbnksXG4gICkge1xuICAgIHN1cGVyKCk7XG4gIH07XG5cbiAgcHVibGljIHRlc3QoYWN0dWFsOiBhbnkpOiBNYXRjaFJlc3VsdCB7XG4gICAgY29uc3QgcmVzdWx0ID0gbmV3IE1hdGNoUmVzdWx0KGFjdHVhbCk7XG4gICAgaWYgKGdldFR5cGUoYWN0dWFsKSAhPT0gJ3N0cmluZycpIHtcbiAgICAgIHJlc3VsdC5yZWNvcmRGYWlsdXJlKHtcbiAgICAgICAgbWF0Y2hlcjogdGhpcyxcbiAgICAgICAgcGF0aDogW10sXG4gICAgICAgIG1lc3NhZ2U6IGBFeHBlY3RlZCBKU09OIGFzIGEgc3RyaW5nIGJ1dCBmb3VuZCAke2dldFR5cGUoYWN0dWFsKX1gLFxuICAgICAgfSk7XG4gICAgICByZXR1cm4gcmVzdWx0O1xuICAgIH1cbiAgICBsZXQgcGFyc2VkO1xuICAgIHRyeSB7XG4gICAgICBwYXJzZWQgPSBKU09OLnBhcnNlKGFjdHVhbCk7XG4gICAgfSBjYXRjaCAoZXJyKSB7XG4gICAgICBpZiAoZXJyIGluc3RhbmNlb2YgU3ludGF4RXJyb3IpIHtcbiAgICAgICAgcmVzdWx0LnJlY29yZEZhaWx1cmUoe1xuICAgICAgICAgIG1hdGNoZXI6IHRoaXMsXG4gICAgICAgICAgcGF0aDogW10sXG4gICAgICAgICAgbWVzc2FnZTogYEludmFsaWQgSlNPTiBzdHJpbmc6ICR7YWN0dWFsfWAsXG4gICAgICAgIH0pO1xuICAgICAgICByZXR1cm4gcmVzdWx0O1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgdGhyb3cgZXJyO1xuICAgICAgfVxuICAgIH1cblxuICAgIGNvbnN0IG1hdGNoZXIgPSBNYXRjaGVyLmlzTWF0Y2hlcih0aGlzLnBhdHRlcm4pID8gdGhpcy5wYXR0ZXJuIDogbmV3IExpdGVyYWxNYXRjaCh0aGlzLm5hbWUsIHRoaXMucGF0dGVybik7XG4gICAgY29uc3QgaW5uZXJSZXN1bHQgPSBtYXRjaGVyLnRlc3QocGFyc2VkKTtcbiAgICByZXN1bHQuY29tcG9zZShgKCR7dGhpcy5uYW1lfSlgLCBpbm5lclJlc3VsdCk7XG4gICAgcmV0dXJuIHJlc3VsdDtcbiAgfVxufVxuXG5jbGFzcyBOb3RNYXRjaCBleHRlbmRzIE1hdGNoZXIge1xuICBjb25zdHJ1Y3RvcihcbiAgICBwdWJsaWMgcmVhZG9ubHkgbmFtZTogc3RyaW5nLFxuICAgIHByaXZhdGUgcmVhZG9ubHkgcGF0dGVybjoge1trZXk6IHN0cmluZ106IGFueX0pIHtcblxuICAgIHN1cGVyKCk7XG4gIH1cblxuICBwdWJsaWMgdGVzdChhY3R1YWw6IGFueSk6IE1hdGNoUmVzdWx0IHtcbiAgICBjb25zdCBtYXRjaGVyID0gTWF0Y2hlci5pc01hdGNoZXIodGhpcy5wYXR0ZXJuKSA/IHRoaXMucGF0dGVybiA6IG5ldyBMaXRlcmFsTWF0Y2godGhpcy5uYW1lLCB0aGlzLnBhdHRlcm4pO1xuXG4gICAgY29uc3QgaW5uZXJSZXN1bHQgPSBtYXRjaGVyLnRlc3QoYWN0dWFsKTtcbiAgICBjb25zdCByZXN1bHQgPSBuZXcgTWF0Y2hSZXN1bHQoYWN0dWFsKTtcbiAgICBpZiAoaW5uZXJSZXN1bHQuZmFpbENvdW50ID09PSAwKSB7XG4gICAgICByZXN1bHQucmVjb3JkRmFpbHVyZSh7XG4gICAgICAgIG1hdGNoZXI6IHRoaXMsXG4gICAgICAgIHBhdGg6IFtdLFxuICAgICAgICBtZXNzYWdlOiBgRm91bmQgdW5leHBlY3RlZCBtYXRjaDogJHtKU09OLnN0cmluZ2lmeShhY3R1YWwsIHVuZGVmaW5lZCwgMil9YCxcbiAgICAgIH0pO1xuICAgIH1cbiAgICByZXR1cm4gcmVzdWx0O1xuICB9XG59XG5cbmNsYXNzIEFueU1hdGNoIGV4dGVuZHMgTWF0Y2hlciB7XG4gIGNvbnN0cnVjdG9yKHB1YmxpYyByZWFkb25seSBuYW1lOiBzdHJpbmcpIHtcbiAgICBzdXBlcigpO1xuICB9XG5cbiAgcHVibGljIHRlc3QoYWN0dWFsOiBhbnkpOiBNYXRjaFJlc3VsdCB7XG4gICAgY29uc3QgcmVzdWx0ID0gbmV3IE1hdGNoUmVzdWx0KGFjdHVhbCk7XG4gICAgaWYgKGFjdHVhbCA9PSBudWxsKSB7XG4gICAgICByZXN1bHQucmVjb3JkRmFpbHVyZSh7XG4gICAgICAgIG1hdGNoZXI6IHRoaXMsXG4gICAgICAgIHBhdGg6IFtdLFxuICAgICAgICBtZXNzYWdlOiAnRXhwZWN0ZWQgYSB2YWx1ZSBidXQgZm91bmQgbm9uZScsXG4gICAgICB9KTtcbiAgICB9XG4gICAgcmV0dXJuIHJlc3VsdDtcbiAgfVxufVxuXG5jbGFzcyBTdHJpbmdMaWtlUmVnZXhwTWF0Y2ggZXh0ZW5kcyBNYXRjaGVyIHtcbiAgY29uc3RydWN0b3IoXG4gICAgcHVibGljIHJlYWRvbmx5IG5hbWU6IHN0cmluZyxcbiAgICBwcml2YXRlIHJlYWRvbmx5IHBhdHRlcm46IHN0cmluZykge1xuXG4gICAgc3VwZXIoKTtcbiAgfVxuXG4gIHRlc3QoYWN0dWFsOiBhbnkpOiBNYXRjaFJlc3VsdCB7XG4gICAgY29uc3QgcmVzdWx0ID0gbmV3IE1hdGNoUmVzdWx0KGFjdHVhbCk7XG5cbiAgICBjb25zdCByZWdleCA9IG5ldyBSZWdFeHAodGhpcy5wYXR0ZXJuLCAnZ20nKTtcblxuICAgIGlmICh0eXBlb2YgYWN0dWFsICE9PSAnc3RyaW5nJykge1xuICAgICAgcmVzdWx0LnJlY29yZEZhaWx1cmUoe1xuICAgICAgICBtYXRjaGVyOiB0aGlzLFxuICAgICAgICBwYXRoOiBbXSxcbiAgICAgICAgbWVzc2FnZTogYEV4cGVjdGVkIGEgc3RyaW5nLCBidXQgZ290ICcke3R5cGVvZiBhY3R1YWx9J2AsXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBpZiAoIXJlZ2V4LnRlc3QoYWN0dWFsKSkge1xuICAgICAgcmVzdWx0LnJlY29yZEZhaWx1cmUoe1xuICAgICAgICBtYXRjaGVyOiB0aGlzLFxuICAgICAgICBwYXRoOiBbXSxcbiAgICAgICAgbWVzc2FnZTogYFN0cmluZyAnJHthY3R1YWx9JyBkaWQgbm90IG1hdGNoIHBhdHRlcm4gJyR7dGhpcy5wYXR0ZXJufSdgLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHJlc3VsdDtcbiAgfVxuXG59XG4iXX0=