"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core");
const lib_1 = require("../lib");
/**
 * This file showcases how to split up a RestApi's Resources and Methods across nested stacks.
 *
 * The root stack 'RootStack' first defines a RestApi.
 * Two nested stacks BooksStack and PetsStack, create corresponding Resources '/books' and '/pets'.
 * They are then deployed to a 'prod' Stage via a third nested stack - DeployStack.
 *
 * To verify this worked, go to the APIGateway
 */
class RootStack extends core_1.Stack {
    constructor(scope) {
        super(scope, 'integ-restapi-import-RootStack');
        const restApi = new lib_1.RestApi(this, 'RestApi', {
            deploy: false,
        });
        restApi.root.addMethod('ANY');
        const petsStack = new PetsStack(this, {
            restApiId: restApi.restApiId,
            rootResourceId: restApi.restApiRootResourceId,
        });
        const booksStack = new BooksStack(this, {
            restApiId: restApi.restApiId,
            rootResourceId: restApi.restApiRootResourceId,
        });
        new DeployStack(this, {
            restApiId: restApi.restApiId,
            methods: petsStack.methods.concat(booksStack.methods),
        });
        new core_1.CfnOutput(this, 'PetsURL', {
            value: `https://${restApi.restApiId}.execute-api.${this.region}.amazonaws.com/prod/pets`,
        });
        new core_1.CfnOutput(this, 'BooksURL', {
            value: `https://${restApi.restApiId}.execute-api.${this.region}.amazonaws.com/prod/books`,
        });
    }
}
class PetsStack extends core_1.NestedStack {
    constructor(scope, props) {
        super(scope, 'integ-restapi-import-PetsStack', props);
        this.methods = [];
        const api = lib_1.RestApi.fromRestApiAttributes(this, 'RestApi', {
            restApiId: props.restApiId,
            rootResourceId: props.rootResourceId,
        });
        const method = api.root.addResource('pets').addMethod('GET', new lib_1.MockIntegration({
            integrationResponses: [{
                    statusCode: '200',
                }],
            passthroughBehavior: lib_1.PassthroughBehavior.NEVER,
            requestTemplates: {
                'application/json': '{ "statusCode": 200 }',
            },
        }), {
            methodResponses: [{ statusCode: '200' }],
        });
        this.methods.push(method);
    }
}
class BooksStack extends core_1.NestedStack {
    constructor(scope, props) {
        super(scope, 'integ-restapi-import-BooksStack', props);
        this.methods = [];
        const api = lib_1.RestApi.fromRestApiAttributes(this, 'RestApi', {
            restApiId: props.restApiId,
            rootResourceId: props.rootResourceId,
        });
        const method = api.root.addResource('books').addMethod('GET', new lib_1.MockIntegration({
            integrationResponses: [{
                    statusCode: '200',
                }],
            passthroughBehavior: lib_1.PassthroughBehavior.NEVER,
            requestTemplates: {
                'application/json': '{ "statusCode": 200 }',
            },
        }), {
            methodResponses: [{ statusCode: '200' }],
        });
        this.methods.push(method);
    }
}
class DeployStack extends core_1.NestedStack {
    constructor(scope, props) {
        super(scope, 'integ-restapi-import-DeployStack', props);
        const deployment = new lib_1.Deployment(this, 'Deployment', {
            api: lib_1.RestApi.fromRestApiId(this, 'RestApi', props.restApiId),
        });
        if (props.methods) {
            for (const method of props.methods) {
                deployment.node.addDependency(method);
            }
        }
        new lib_1.Stage(this, 'Stage', { deployment });
    }
}
new RootStack(new core_1.App());
//# sourceMappingURL=data:application/json;base64,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