import * as ecr from '../../aws-ecr';
import * as assets from '../../aws-ecr-assets';
import * as iam from '../../aws-iam';
import * as cdk from '../../core';
import { Construct } from 'constructs';
import { IVpcConnector } from './vpc-connector';
/**
 * The image repository types
 */
export declare enum ImageRepositoryType {
    /**
     * Amazon ECR Public
     */
    ECR_PUBLIC = "ECR_PUBLIC",
    /**
     * Amazon ECR
     */
    ECR = "ECR"
}
/**
 * The number of CPU units reserved for each instance of your App Runner service.
 *
 */
export declare class Cpu {
    readonly unit: string;
    /**
     * 1 vCPU
     */
    static readonly ONE_VCPU: Cpu;
    /**
     * 2 vCPU
     */
    static readonly TWO_VCPU: Cpu;
    /**
     * Custom CPU unit
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apprunner-service-instanceconfiguration.html#cfn-apprunner-service-instanceconfiguration-cpu
     *
     * @param unit custom CPU unit
     */
    static of(unit: string): Cpu;
    /**
     *
     * @param unit The unit of CPU.
     */
    private constructor();
}
/**
 * The amount of memory reserved for each instance of your App Runner service.
 */
export declare class Memory {
    readonly unit: string;
    /**
     * 2 GB(for 1 vCPU)
     */
    static readonly TWO_GB: Memory;
    /**
     * 3 GB(for 1 vCPU)
     */
    static readonly THREE_GB: Memory;
    /**
     * 4 GB(for 1 or 2 vCPU)
     */
    static readonly FOUR_GB: Memory;
    /**
     * Custom Memory unit
     *
     * @param unit custom Memory unit
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apprunner-service-instanceconfiguration.html#cfn-apprunner-service-instanceconfiguration-memory
     */
    static of(unit: string): Memory;
    /**
     *
     * @param unit The unit of memory.
     */
    private constructor();
}
/**
 * The code runtimes
 */
export declare class Runtime {
    readonly name: string;
    /**
     * NodeJS 12
     */
    static readonly NODEJS_12: Runtime;
    /**
     * Python 3
     */
    static readonly PYTHON_3: Runtime;
    /**
     * Other runtimes
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apprunner-service-codeconfigurationvalues.html#cfn-apprunner-service-codeconfigurationvalues-runtime for all available runtimes.
     *
     * @param name runtime name
     *
     */
    static of(name: string): Runtime;
    /**
     *
     * @param name The runtime name.
     */
    private constructor();
}
/**
 * Result of binding `Source` into a `Service`.
 */
export interface SourceConfig {
    /**
     * The image repository configuration (mutually exclusive  with `codeRepository`).
     *
     * @default - no image repository.
     */
    readonly imageRepository?: ImageRepository;
    /**
     * The ECR repository (required to grant the pull privileges for the iam role).
     *
     * @default - no ECR repository.
     */
    readonly ecrRepository?: ecr.IRepository;
    /**
     * The code repository configuration (mutually exclusive  with `imageRepository`).
     *
     * @default - no code repository.
     */
    readonly codeRepository?: CodeRepositoryProps;
}
/**
 * Properties of the Github repository for `Source.fromGitHub()`
 */
export interface GithubRepositoryProps {
    /**
     * The code configuration values. Will be ignored if configurationSource is `REPOSITORY`.
     * @default - no values will be passed. The `apprunner.yaml` from the github reopsitory will be used instead.
     */
    readonly codeConfigurationValues?: CodeConfigurationValues;
    /**
     * The source of the App Runner configuration.
     */
    readonly configurationSource: ConfigurationSourceType;
    /**
     * The location of the repository that contains the source code.
     */
    readonly repositoryUrl: string;
    /**
     * The branch name that represents a specific version for the repository.
     *
     * @default main
     */
    readonly branch?: string;
    /**
     * ARN of the connection to Github. Only required for Github source.
     */
    readonly connection: GitHubConnection;
}
/**
 * Properties of the image repository for `Source.fromEcrPublic()`
 */
export interface EcrPublicProps {
    /**
     * The image configuration for the image from ECR Public.
     * @default - no image configuration will be passed. The default `port` will be 8080.
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apprunner-service-imageconfiguration.html#cfn-apprunner-service-imageconfiguration-port
     */
    readonly imageConfiguration?: ImageConfiguration;
    /**
     * The ECR Public image URI.
     */
    readonly imageIdentifier: string;
}
/**
 * Properties of the image repository for `Source.fromEcr()`
 */
export interface EcrProps {
    /**
     * The image configuration for the image from ECR.
     * @default - no image configuration will be passed. The default `port` will be 8080.
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apprunner-service-imageconfiguration.html#cfn-apprunner-service-imageconfiguration-port
     */
    readonly imageConfiguration?: ImageConfiguration;
    /**
     * Represents the ECR repository.
     */
    readonly repository: ecr.IRepository;
    /**
     * Image tag.
     * @default - 'latest'
     * @deprecated use `tagOrDigest`
     */
    readonly tag?: string;
    /**
     * Image tag or digest (digests must start with `sha256:`).
     * @default - 'latest'
     */
    readonly tagOrDigest?: string;
}
/**
 * Properties of the image repository for `Source.fromAsset()`
 */
export interface AssetProps {
    /**
     * The image configuration for the image built from the asset.
     * @default - no image configuration will be passed. The default `port` will be 8080.
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apprunner-service-imageconfiguration.html#cfn-apprunner-service-imageconfiguration-port
     */
    readonly imageConfiguration?: ImageConfiguration;
    /**
     * Represents the docker image asset.
     */
    readonly asset: assets.DockerImageAsset;
}
/**
 * Represents the App Runner service source.
 */
export declare abstract class Source {
    /**
     * Source from the GitHub repository.
     */
    static fromGitHub(props: GithubRepositoryProps): GithubSource;
    /**
     * Source from the ECR repository.
     */
    static fromEcr(props: EcrProps): EcrSource;
    /**
     * Source from the ECR Public repository.
     */
    static fromEcrPublic(props: EcrPublicProps): EcrPublicSource;
    /**
     * Source from local assets.
     */
    static fromAsset(props: AssetProps): AssetSource;
    /**
      * Called when the Job is initialized to allow this object to bind.
      */
    abstract bind(scope: Construct): SourceConfig;
}
/**
 * Represents the service source from a Github repository.
 */
export declare class GithubSource extends Source {
    private readonly props;
    constructor(props: GithubRepositoryProps);
    bind(_scope: Construct): SourceConfig;
}
/**
 * Represents the service source from ECR.
 */
export declare class EcrSource extends Source {
    private readonly props;
    constructor(props: EcrProps);
    bind(_scope: Construct): SourceConfig;
}
/**
 * Represents the service source from ECR Public.
 */
export declare class EcrPublicSource extends Source {
    private readonly props;
    constructor(props: EcrPublicProps);
    bind(_scope: Construct): SourceConfig;
}
/**
 * Represents the source from local assets.
 */
export declare class AssetSource extends Source {
    private readonly props;
    constructor(props: AssetProps);
    bind(_scope: Construct): SourceConfig;
}
/**
 * Describes the configuration that AWS App Runner uses to run an App Runner service
 * using an image pulled from a source image repository.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apprunner-service-imageconfiguration.html
 */
export interface ImageConfiguration {
    /**
     * The port that your application listens to in the container.
     *
     * @default 8080
     */
    readonly port?: number;
    /**
     * Environment variables that are available to your running App Runner service.
     *
     * @default - no environment variables
     */
    readonly environment?: {
        [key: string]: string;
    };
    /**
     * An optional command that App Runner runs to start the application in the source image.
     * If specified, this command overrides the Docker image’s default start command.
     *
     * @default - no start command
     */
    readonly startCommand?: string;
}
/**
 * Describes a source image repository.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apprunner-service-imagerepository.html
 */
export interface ImageRepository {
    /**
     * The identifier of the image. For `ECR_PUBLIC` imageRepositoryType, the identifier domain should
     * always be `public.ecr.aws`. For `ECR`, the pattern should be
     * `([0-9]{12}.dkr.ecr.[a-z\-]+-[0-9]{1}.amazonaws.com\/.*)`.
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apprunner-service-imagerepository.html for more details.
     */
    readonly imageIdentifier: string;
    /**
     * The type of the image repository. This reflects the repository provider and whether
     * the repository is private or public.
     */
    readonly imageRepositoryType: ImageRepositoryType;
    /**
     * Configuration for running the identified image.
     * @default - no image configuration will be passed. The default `port` will be 8080.
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apprunner-service-imageconfiguration.html#cfn-apprunner-service-imageconfiguration-port
     */
    readonly imageConfiguration?: ImageConfiguration;
}
/**
 * Identifies a version of code that AWS App Runner refers to within a source code repository.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apprunner-service-sourcecodeversion.html
 */
export interface SourceCodeVersion {
    /**
     * The type of version identifier.
     */
    readonly type: string;
    /**
     * A source code version.
     */
    readonly value: string;
}
/**
 * Properties of the CodeRepository.
 */
export interface CodeRepositoryProps {
    /**
     * Configuration for building and running the service from a source code repository.
     */
    readonly codeConfiguration: CodeConfiguration;
    /**
     * The location of the repository that contains the source code.
     */
    readonly repositoryUrl: string;
    /**
     * The version that should be used within the source code repository.
     */
    readonly sourceCodeVersion: SourceCodeVersion;
    /**
     * The App Runner connection for GitHub.
     */
    readonly connection: GitHubConnection;
}
/**
 * Properties of the AppRunner Service
 */
export interface ServiceProps {
    /**
     * The source of the repository for the service.
     */
    readonly source: Source;
    /**
     * The number of CPU units reserved for each instance of your App Runner service.
     *
     * @default Cpu.ONE_VCPU
     */
    readonly cpu?: Cpu;
    /**
     * The amount of memory reserved for each instance of your App Runner service.
     *
     * @default Memory.TWO_GB
     */
    readonly memory?: Memory;
    /**
     * The IAM role that grants the App Runner service access to a source repository.
     * It's required for ECR image repositories (but not for ECR Public repositories).
     *
     * The role must be assumable by the 'build.apprunner.amazonaws.com' service principal.
     *
     * @see https://docs.aws.amazon.com/apprunner/latest/dg/security_iam_service-with-iam.html#security_iam_service-with-iam-roles-service.access
     *
     * @default - generate a new access role.
     */
    readonly accessRole?: iam.IRole;
    /**
     * The IAM role that provides permissions to your App Runner service.
     * These are permissions that your code needs when it calls any AWS APIs.
     *
     * The role must be assumable by the 'tasks.apprunner.amazonaws.com' service principal.
     *
     * @see https://docs.aws.amazon.com/apprunner/latest/dg/security_iam_service-with-iam.html#security_iam_service-with-iam-roles-service.instance
     *
     * @default - no instance role attached.
     */
    readonly instanceRole?: iam.IRole;
    /**
     * Name of the service.
     *
     * @default - auto-generated if undefined.
     */
    readonly serviceName?: string;
    /**
     * Settings for an App Runner VPC connector to associate with the service.
     *
     * @default - no VPC connector, uses the DEFAULT egress type instead
     */
    readonly vpcConnector?: IVpcConnector;
}
/**
 * The source of the App Runner configuration.
 */
export declare enum ConfigurationSourceType {
    /**
     * App Runner reads configuration values from `the apprunner.yaml` file in the source code repository
     * and ignores `configurationValues`.
     */
    REPOSITORY = "REPOSITORY",
    /**
     * App Runner uses configuration values provided in `configurationValues` and ignores the `apprunner.yaml`
     * file in the source code repository.
     */
    API = "API"
}
/**
 * Describes the configuration that AWS App Runner uses to build and run an App Runner service
 * from a source code repository.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apprunner-service-codeconfiguration.html
 */
export interface CodeConfiguration {
    /**
     * The basic configuration for building and running the App Runner service.
     * Use it to quickly launch an App Runner service without providing a apprunner.yaml file in the
     * source code repository (or ignoring the file if it exists).
     *
     * @default - not specified. Use `apprunner.yaml` instead.
     */
    readonly configurationValues?: CodeConfigurationValues;
    /**
     * The source of the App Runner configuration.
     */
    readonly configurationSource: ConfigurationSourceType;
}
/**
 * Describes the basic configuration needed for building and running an AWS App Runner service.
 * This type doesn't support the full set of possible configuration options. Fur full configuration capabilities,
 * use a `apprunner.yaml` file in the source code repository.
 */
export interface CodeConfigurationValues {
    /**
     * The command App Runner runs to build your application.
     *
     * @default - no build command.
     */
    readonly buildCommand?: string;
    /**
     * The port that your application listens to in the container.
     *
     * @default 8080
     */
    readonly port?: string;
    /**
     * A runtime environment type for building and running an App Runner service. It represents
     * a programming language runtime.
     */
    readonly runtime: Runtime;
    /**
     * The environment variables that are available to your running App Runner service.
     *
     * @default - no environment variables.
     */
    readonly environment?: {
        [key: string]: string;
    };
    /**
     * The command App Runner runs to start your application.
     *
     * @default - no start command.
     */
    readonly startCommand?: string;
}
/**
 * Represents the App Runner connection that enables the App Runner service to connect
 * to a source repository. It's required for GitHub code repositories.
 */
export declare class GitHubConnection {
    /**
     * Using existing App Runner connection by specifying the connection ARN.
     * @param arn connection ARN
     * @returns Connection
     */
    static fromConnectionArn(arn: string): GitHubConnection;
    /**
     * The ARN of the Connection for App Runner service to connect to the repository.
     */
    readonly connectionArn: string;
    constructor(arn: string);
}
/**
 * Attributes for the App Runner Service
 */
export interface ServiceAttributes {
    /**
     * The name of the service.
     */
    readonly serviceName: string;
    /**
     * The ARN of the service.
     */
    readonly serviceArn: string;
    /**
     * The URL of the service.
     */
    readonly serviceUrl: string;
    /**
     * The status of the service.
     */
    readonly serviceStatus: string;
}
/**
 * Represents the App Runner Service.
 */
export interface IService extends cdk.IResource {
    /**
     * The Name of the service.
     */
    readonly serviceName: string;
    /**
     * The ARN of the service.
     */
    readonly serviceArn: string;
}
/**
 * The App Runner Service.
 */
export declare class Service extends cdk.Resource {
    /**
     * Import from service name.
     */
    static fromServiceName(scope: Construct, id: string, serviceName: string): IService;
    /**
     * Import from service attributes.
     */
    static fromServiceAttributes(scope: Construct, id: string, attrs: ServiceAttributes): IService;
    private readonly props;
    private accessRole?;
    private source;
    /**
     * Environment variables for this service
     */
    private environment?;
    /**
     * The ARN of the Service.
     * @attribute
     */
    readonly serviceArn: string;
    /**
     * The ID of the Service.
     * @attribute
     */
    readonly serviceId: string;
    /**
     * The URL of the Service.
     * @attribute
     */
    readonly serviceUrl: string;
    /**
     * The status of the Service.
     * @attribute
     */
    readonly serviceStatus: string;
    /**
     * The name of the service.
     * @attribute
     */
    readonly serviceName: string;
    constructor(scope: Construct, id: string, props: ServiceProps);
    private renderAuthenticationConfiguration;
    private renderCodeConfiguration;
    private renderCodeConfigurationValues;
    private renderImageRepository;
    private renderEnvironmentVariables;
    private generateDefaultRole;
}
