import { ITable } from '../../aws-dynamodb';
import { IDomain as IElasticsearchDomain } from '../../aws-elasticsearch';
import { IFunction } from '../../aws-lambda';
import { IDomain as IOpenSearchDomain } from '../../aws-opensearchservice';
import { IServerlessCluster } from '../../aws-rds';
import { ISecret } from '../../aws-secretsmanager';
import { CfnResource, IResource, Resource } from '../../core';
import { DynamoDbDataSource, HttpDataSource, LambdaDataSource, NoneDataSource, RdsDataSource, AwsIamConfig, ElasticsearchDataSource, OpenSearchDataSource } from './data-source';
import { Resolver, ExtendedResolverProps } from './resolver';
/**
 * Optional configuration for data sources
 */
export interface DataSourceOptions {
    /**
     * The name of the data source, overrides the id given by cdk
     *
     * @default - generated by cdk given the id
     */
    readonly name?: string;
    /**
     * The description of the data source
     *
     * @default - No description
     */
    readonly description?: string;
}
/**
 * Optional configuration for Http data sources
 */
export interface HttpDataSourceOptions extends DataSourceOptions {
    /**
     * The authorization config in case the HTTP endpoint requires authorization
     *
     * @default - none
     */
    readonly authorizationConfig?: AwsIamConfig;
}
/**
 * Interface for GraphQL
 */
export interface IGraphqlApi extends IResource {
    /**
     * an unique AWS AppSync GraphQL API identifier
     * i.e. 'lxz775lwdrgcndgz3nurvac7oa'
     *
     * @attribute
     */
    readonly apiId: string;
    /**
     * the ARN of the API
     *
     * @attribute
     */
    readonly arn: string;
    /**
     * add a new dummy data source to this API. Useful for pipeline resolvers
     * and for backend changes that don't require a data source.
     *
     * @param id The data source's id
     * @param options The optional configuration for this data source
     */
    addNoneDataSource(id: string, options?: DataSourceOptions): NoneDataSource;
    /**
     * add a new DynamoDB data source to this API
     *
     * @param id The data source's id
     * @param table The DynamoDB table backing this data source
     * @param options The optional configuration for this data source
     */
    addDynamoDbDataSource(id: string, table: ITable, options?: DataSourceOptions): DynamoDbDataSource;
    /**
     * add a new http data source to this API
     *
     * @param id The data source's id
     * @param endpoint The http endpoint
     * @param options The optional configuration for this data source
     */
    addHttpDataSource(id: string, endpoint: string, options?: HttpDataSourceOptions): HttpDataSource;
    /**
     * add a new Lambda data source to this API
     *
     * @param id The data source's id
     * @param lambdaFunction The Lambda function to call to interact with this data source
     * @param options The optional configuration for this data source
     */
    addLambdaDataSource(id: string, lambdaFunction: IFunction, options?: DataSourceOptions): LambdaDataSource;
    /**
     * add a new Rds data source to this API
     *
     * @param id The data source's id
     * @param serverlessCluster The serverless cluster to interact with this data source
     * @param secretStore The secret store that contains the username and password for the serverless cluster
     * @param databaseName The optional name of the database to use within the cluster
     * @param options The optional configuration for this data source
     */
    addRdsDataSource(id: string, serverlessCluster: IServerlessCluster, secretStore: ISecret, databaseName?: string, options?: DataSourceOptions): RdsDataSource;
    /**
     * add a new elasticsearch data source to this API
     *
     * @deprecated - use `addOpenSearchDataSource`
     * @param id The data source's id
     * @param domain The elasticsearch domain for this data source
     * @param options The optional configuration for this data source
     */
    addElasticsearchDataSource(id: string, domain: IElasticsearchDomain, options?: DataSourceOptions): ElasticsearchDataSource;
    /**
     * Add a new OpenSearch data source to this API
     *
     * @param id The data source's id
     * @param domain The OpenSearch domain for this data source
     * @param options The optional configuration for this data source
     */
    addOpenSearchDataSource(id: string, domain: IOpenSearchDomain, options?: DataSourceOptions): OpenSearchDataSource;
    /**
     * creates a new resolver for this datasource and API using the given properties
     */
    createResolver(props: ExtendedResolverProps): Resolver;
    /**
     * Add schema dependency if not imported
     *
     * @param construct the dependee
     */
    addSchemaDependency(construct: CfnResource): boolean;
}
/**
 * Base Class for GraphQL API
 */
export declare abstract class GraphqlApiBase extends Resource implements IGraphqlApi {
    /**
     * an unique AWS AppSync GraphQL API identifier
     * i.e. 'lxz775lwdrgcndgz3nurvac7oa'
     */
    abstract readonly apiId: string;
    /**
     * the ARN of the API
     */
    abstract readonly arn: string;
    /**
     * add a new dummy data source to this API. Useful for pipeline resolvers
     * and for backend changes that don't require a data source.
     *
     * @param id The data source's id
     * @param options The optional configuration for this data source
     */
    addNoneDataSource(id: string, options?: DataSourceOptions): NoneDataSource;
    /**
     * add a new DynamoDB data source to this API
     *
     * @param id The data source's id
     * @param table The DynamoDB table backing this data source
     * @param options The optional configuration for this data source
     */
    addDynamoDbDataSource(id: string, table: ITable, options?: DataSourceOptions): DynamoDbDataSource;
    /**
     * add a new http data source to this API
     *
     * @param id The data source's id
     * @param endpoint The http endpoint
     * @param options The optional configuration for this data source
     */
    addHttpDataSource(id: string, endpoint: string, options?: HttpDataSourceOptions): HttpDataSource;
    /**
     * add a new Lambda data source to this API
     *
     * @param id The data source's id
     * @param lambdaFunction The Lambda function to call to interact with this data source
     * @param options The optional configuration for this data source
     */
    addLambdaDataSource(id: string, lambdaFunction: IFunction, options?: DataSourceOptions): LambdaDataSource;
    /**
     * add a new Rds data source to this API
     * @param id The data source's id
     * @param serverlessCluster The serverless cluster to interact with this data source
     * @param secretStore The secret store that contains the username and password for the serverless cluster
     * @param databaseName The optional name of the database to use within the cluster
     * @param options The optional configuration for this data source
     */
    addRdsDataSource(id: string, serverlessCluster: IServerlessCluster, secretStore: ISecret, databaseName?: string, options?: DataSourceOptions): RdsDataSource;
    /**
     * add a new elasticsearch data source to this API
     *
     * @deprecated - use `addOpenSearchDataSource`
     * @param id The data source's id
     * @param domain The elasticsearch domain for this data source
     * @param options The optional configuration for this data source
     */
    addElasticsearchDataSource(id: string, domain: IElasticsearchDomain, options?: DataSourceOptions): ElasticsearchDataSource;
    /**
     * add a new OpenSearch data source to this API
     *
     * @param id The data source's id
     * @param domain The OpenSearch domain for this data source
     * @param options The optional configuration for this data source
     */
    addOpenSearchDataSource(id: string, domain: IOpenSearchDomain, options?: DataSourceOptions): OpenSearchDataSource;
    /**
     * creates a new resolver for this datasource and API using the given properties
     */
    createResolver(props: ExtendedResolverProps): Resolver;
    /**
     * Add schema dependency if not imported
     *
     * @param construct the dependee
     */
    addSchemaDependency(construct: CfnResource): boolean;
}
