"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Between = exports.BinaryCondition = exports.BeginsWith = exports.BaseKeyCondition = exports.shapeAddition = exports.SchemaMode = void 0;
/**
 * Utility enum for Schema class
 */
var SchemaMode;
(function (SchemaMode) {
    SchemaMode["FILE"] = "FILE";
    SchemaMode["CODE"] = "CODE";
})(SchemaMode = exports.SchemaMode || (exports.SchemaMode = {}));
;
/**
 * Generates an addition to the schema
 *
 * @param options the options to produced a stringfied addition
 *
 * @returns the following shape:
 *
 * ```
 * prefix name interfaces directives {
 *   field
 *   field
 *   ...
 * }
 * ```
 */
function shapeAddition(options) {
    const typeName = () => { return options.name ? ` ${options.name}` : ''; };
    const interfaces = generateInterfaces(options.interfaceTypes);
    const directives = generateDirectives({
        directives: options.directives,
        modes: options.modes,
    });
    return options.fields.reduce((acc, field) => `${acc}  ${field}\n`, `${options.prefix}${typeName()}${interfaces}${directives} {\n`) + '}';
}
exports.shapeAddition = shapeAddition;
/**
 * Utility class to represent DynamoDB key conditions.
 */
class BaseKeyCondition {
    and(cond) {
        return new (class extends BaseKeyCondition {
            constructor(left, right) {
                super();
                this.left = left;
                this.right = right;
            }
            renderCondition() {
                return `${this.left.renderCondition()} AND ${this.right.renderCondition()}`;
            }
            keyNames() {
                return concatAndDedup(this.left.keyNames(), this.right.keyNames());
            }
            args() {
                return concatAndDedup(this.left.args(), this.right.args());
            }
        })(this, cond);
    }
    renderExpressionNames() {
        return this.keyNames()
            .map((keyName) => {
            return `"#${keyName}" : "${keyName}"`;
        })
            .join(', ');
    }
    renderExpressionValues() {
        return this.args()
            .map((arg) => {
            return `":${arg}" : $util.dynamodb.toDynamoDBJson($ctx.args.${arg})`;
        })
            .join(', ');
    }
}
exports.BaseKeyCondition = BaseKeyCondition;
/**
 * Utility class to represent DynamoDB "begins_with" key conditions.
 */
class BeginsWith extends BaseKeyCondition {
    constructor(keyName, arg) {
        super();
        this.keyName = keyName;
        this.arg = arg;
    }
    renderCondition() {
        return `begins_with(#${this.keyName}, :${this.arg})`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
exports.BeginsWith = BeginsWith;
/**
 * Utility class to represent DynamoDB binary key conditions.
 */
class BinaryCondition extends BaseKeyCondition {
    constructor(keyName, op, arg) {
        super();
        this.keyName = keyName;
        this.op = op;
        this.arg = arg;
    }
    renderCondition() {
        return `#${this.keyName} ${this.op} :${this.arg}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
exports.BinaryCondition = BinaryCondition;
/**
 * Utility class to represent DynamoDB "between" key conditions.
 */
class Between extends BaseKeyCondition {
    constructor(keyName, arg1, arg2) {
        super();
        this.keyName = keyName;
        this.arg1 = arg1;
        this.arg2 = arg2;
    }
    renderCondition() {
        return `#${this.keyName} BETWEEN :${this.arg1} AND :${this.arg2}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg1, this.arg2];
    }
}
exports.Between = Between;
function concatAndDedup(left, right) {
    return left.concat(right).filter((elem, index, self) => {
        return index === self.indexOf(elem);
    });
}
/**
 * Utility function to generate interfaces for object types
 *
 * @param interfaceTypes the interfaces this object type implements
 */
function generateInterfaces(interfaceTypes) {
    if (!interfaceTypes || interfaceTypes.length === 0)
        return '';
    return interfaceTypes.reduce((acc, interfaceType) => `${acc} ${interfaceType.name} &`, ' implements').slice(0, -2);
}
/**
 * Utility function to generate directives
 */
function generateDirectives(options) {
    if (!options.directives || options.directives.length === 0)
        return '';
    // reduce over all directives and get string version of the directive
    // pass in the auth modes for checks to happen on compile time
    return options.directives.reduce((acc, directive) => `${acc}${directive._bindToAuthModes(options.modes).toString()}${options.delimiter ?? ' '}`, ' ').slice(0, -1);
}
//# sourceMappingURL=data:application/json;base64,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