"use strict";
var _a, _b, _c, _d, _e, _f, _g;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthCheck = exports.ScalingProcess = exports.ScalingEvents = exports.ScalingEvent = exports.UpdateType = exports.AutoScalingGroup = exports.GroupMetric = exports.GroupMetrics = exports.UpdatePolicy = exports.Signals = exports.SpotAllocationStrategy = exports.OnDemandAllocationStrategy = exports.Monitoring = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const elbv2 = require("../../aws-elasticloadbalancingv2");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const aspects_1 = require("./aspects");
const autoscaling_generated_1 = require("./autoscaling.generated");
const lifecycle_hook_1 = require("./lifecycle-hook");
const scheduled_action_1 = require("./scheduled-action");
const step_scaling_policy_1 = require("./step-scaling-policy");
const target_tracking_scaling_policy_1 = require("./target-tracking-scaling-policy");
const volume_1 = require("./volume");
const warm_pool_1 = require("./warm-pool");
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * The monitoring mode for instances launched in an autoscaling group
 */
var Monitoring;
(function (Monitoring) {
    /**
     * Generates metrics every 5 minutes
     */
    Monitoring[Monitoring["BASIC"] = 0] = "BASIC";
    /**
     * Generates metrics every minute
     */
    Monitoring[Monitoring["DETAILED"] = 1] = "DETAILED";
})(Monitoring = exports.Monitoring || (exports.Monitoring = {}));
/**
 * Indicates how to allocate instance types to fulfill On-Demand capacity.
 */
var OnDemandAllocationStrategy;
(function (OnDemandAllocationStrategy) {
    /**
     * This strategy uses the order of instance types in the LaunchTemplateOverrides to define the launch
     * priority of each instance type. The first instance type in the array is prioritized higher than the
     * last. If all your On-Demand capacity cannot be fulfilled using your highest priority instance, then
     * the Auto Scaling group launches the remaining capacity using the second priority instance type, and
     * so on.
     */
    OnDemandAllocationStrategy["PRIORITIZED"] = "prioritized";
})(OnDemandAllocationStrategy = exports.OnDemandAllocationStrategy || (exports.OnDemandAllocationStrategy = {}));
/**
 * Indicates how to allocate instance types to fulfill Spot capacity.
 */
var SpotAllocationStrategy;
(function (SpotAllocationStrategy) {
    /**
     * The Auto Scaling group launches instances using the Spot pools with the lowest price, and evenly
     * allocates your instances across the number of Spot pools that you specify.
     */
    SpotAllocationStrategy["LOWEST_PRICE"] = "lowest-price";
    /**
     * The Auto Scaling group launches instances using Spot pools that are optimally chosen based on the
     * available Spot capacity.
     *
     * Recommended.
     */
    SpotAllocationStrategy["CAPACITY_OPTIMIZED"] = "capacity-optimized";
    /**
     * When you use this strategy, you need to set the order of instance types in the list of launch template
     * overrides from highest to lowest priority (from first to last in the list). Amazon EC2 Auto Scaling
     * honors the instance type priorities on a best-effort basis but optimizes for capacity first.
     */
    SpotAllocationStrategy["CAPACITY_OPTIMIZED_PRIORITIZED"] = "capacity-optimized-prioritized";
})(SpotAllocationStrategy = exports.SpotAllocationStrategy || (exports.SpotAllocationStrategy = {}));
/**
 * Configure whether the AutoScalingGroup waits for signals
 *
 * If you do configure waiting for signals, you should make sure the instances
 * invoke `cfn-signal` somewhere in their UserData to signal that they have
 * started up (either successfully or unsuccessfully).
 *
 * Signals are used both during intial creation and subsequent updates.
 */
class Signals {
    /**
     * Wait for the desiredCapacity of the AutoScalingGroup amount of signals to have been received
     *
     * If no desiredCapacity has been configured, wait for minCapacity signals intead.
     *
     * This number is used during initial creation and during replacing updates.
     * During rolling updates, all updated instances must send a signal.
     */
    static waitForAll(options = {}) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_autoscaling_SignalsOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.waitForAll);
            }
            throw error;
        }
        validatePercentage(options.minSuccessPercentage);
        return new class extends Signals {
            renderCreationPolicy(renderOptions) {
                return this.doRender(options, renderOptions.desiredCapacity ?? renderOptions.minCapacity);
            }
        }();
    }
    /**
     * Wait for the minCapacity of the AutoScalingGroup amount of signals to have been received
     *
     * This number is used during initial creation and during replacing updates.
     * During rolling updates, all updated instances must send a signal.
     */
    static waitForMinCapacity(options = {}) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_autoscaling_SignalsOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.waitForMinCapacity);
            }
            throw error;
        }
        validatePercentage(options.minSuccessPercentage);
        return new class extends Signals {
            renderCreationPolicy(renderOptions) {
                return this.doRender(options, renderOptions.minCapacity);
            }
        }();
    }
    /**
     * Wait for a specific amount of signals to have been received
     *
     * You should send one signal per instance, so this represents the number of
     * instances to wait for.
     *
     * This number is used during initial creation and during replacing updates.
     * During rolling updates, all updated instances must send a signal.
     */
    static waitForCount(count, options = {}) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_autoscaling_SignalsOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.waitForCount);
            }
            throw error;
        }
        validatePercentage(options.minSuccessPercentage);
        return new class extends Signals {
            renderCreationPolicy() {
                return this.doRender(options, count);
            }
        }();
    }
    /**
     * Helper to render the actual creation policy, as the logic between them is quite similar
     */
    doRender(options, count) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_autoscaling_SignalsOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.doRender);
            }
            throw error;
        }
        const minSuccessfulInstancesPercent = validatePercentage(options.minSuccessPercentage);
        return {
            ...options.minSuccessPercentage !== undefined ? { autoScalingCreationPolicy: { minSuccessfulInstancesPercent } } : {},
            resourceSignal: {
                count,
                timeout: options.timeout?.toIsoString(),
            },
        };
    }
}
exports.Signals = Signals;
_a = JSII_RTTI_SYMBOL_1;
Signals[_a] = { fqn: "monocdk.aws_autoscaling.Signals", version: "1.191.0" };
/**
 * How existing instances should be updated
 */
class UpdatePolicy {
    /**
     * Create a new AutoScalingGroup and switch over to it
     */
    static replacingUpdate() {
        return new class extends UpdatePolicy {
            _renderUpdatePolicy() {
                return {
                    autoScalingReplacingUpdate: { willReplace: true },
                };
            }
        }();
    }
    /**
     * Replace the instances in the AutoScalingGroup one by one, or in batches
     */
    static rollingUpdate(options = {}) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_autoscaling_RollingUpdateOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.rollingUpdate);
            }
            throw error;
        }
        const minSuccessPercentage = validatePercentage(options.minSuccessPercentage);
        return new class extends UpdatePolicy {
            _renderUpdatePolicy(renderOptions) {
                return {
                    autoScalingRollingUpdate: {
                        maxBatchSize: options.maxBatchSize,
                        minInstancesInService: options.minInstancesInService,
                        suspendProcesses: options.suspendProcesses ?? DEFAULT_SUSPEND_PROCESSES,
                        minSuccessfulInstancesPercent: minSuccessPercentage ?? renderOptions.creationPolicy?.autoScalingCreationPolicy?.minSuccessfulInstancesPercent,
                        waitOnResourceSignals: options.waitOnResourceSignals ?? renderOptions.creationPolicy?.resourceSignal !== undefined ? true : undefined,
                        pauseTime: options.pauseTime?.toIsoString() ?? renderOptions.creationPolicy?.resourceSignal?.timeout,
                    },
                };
            }
        }();
    }
}
exports.UpdatePolicy = UpdatePolicy;
_b = JSII_RTTI_SYMBOL_1;
UpdatePolicy[_b] = { fqn: "monocdk.aws_autoscaling.UpdatePolicy", version: "1.191.0" };
/**
 * A set of group metrics
 */
class GroupMetrics {
    constructor(...metrics) {
        /**
         * @internal
         */
        this._metrics = new Set();
        try {
            jsiiDeprecationWarnings.monocdk_aws_autoscaling_GroupMetric(metrics);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, GroupMetrics);
            }
            throw error;
        }
        metrics?.forEach(metric => this._metrics.add(metric));
    }
    /**
     * Report all group metrics.
     */
    static all() {
        return new GroupMetrics();
    }
}
exports.GroupMetrics = GroupMetrics;
_c = JSII_RTTI_SYMBOL_1;
GroupMetrics[_c] = { fqn: "monocdk.aws_autoscaling.GroupMetrics", version: "1.191.0" };
/**
 * Group metrics that an Auto Scaling group sends to Amazon CloudWatch.
 */
class GroupMetric {
    constructor(name) {
        this.name = name;
    }
}
exports.GroupMetric = GroupMetric;
_d = JSII_RTTI_SYMBOL_1;
GroupMetric[_d] = { fqn: "monocdk.aws_autoscaling.GroupMetric", version: "1.191.0" };
/**
 * The minimum size of the Auto Scaling group
 */
GroupMetric.MIN_SIZE = new GroupMetric('GroupMinSize');
/**
 * The maximum size of the Auto Scaling group
 */
GroupMetric.MAX_SIZE = new GroupMetric('GroupMaxSize');
/**
 * The number of instances that the Auto Scaling group attempts to maintain
 */
GroupMetric.DESIRED_CAPACITY = new GroupMetric('GroupDesiredCapacity');
/**
 * The number of instances that are running as part of the Auto Scaling group
 * This metric does not include instances that are pending or terminating
 */
GroupMetric.IN_SERVICE_INSTANCES = new GroupMetric('GroupInServiceInstances');
/**
 * The number of instances that are pending
 * A pending instance is not yet in service, this metric does not include instances that are in service or terminating
 */
GroupMetric.PENDING_INSTANCES = new GroupMetric('GroupPendingInstances');
/**
 * The number of instances that are in a Standby state
 * Instances in this state are still running but are not actively in service
 */
GroupMetric.STANDBY_INSTANCES = new GroupMetric('GroupStandbyInstances');
/**
 * The number of instances that are in the process of terminating
 * This metric does not include instances that are in service or pending
 */
GroupMetric.TERMINATING_INSTANCES = new GroupMetric('GroupTerminatingInstances');
/**
 * The total number of instances in the Auto Scaling group
 * This metric identifies the number of instances that are in service, pending, and terminating
 */
GroupMetric.TOTAL_INSTANCES = new GroupMetric('GroupTotalInstances');
class AutoScalingGroupBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
    }
    /**
     * Send a message to either an SQS queue or SNS topic when instances launch or terminate
     */
    addLifecycleHook(id, props) {
        return new lifecycle_hook_1.LifecycleHook(this, `LifecycleHook${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * Add a pool of pre-initialized EC2 instances that sits alongside an Auto Scaling group
     */
    addWarmPool(options) {
        return new warm_pool_1.WarmPool(this, 'WarmPool', {
            autoScalingGroup: this,
            ...options,
        });
    }
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id, props) {
        return new scheduled_action_1.ScheduledAction(this, `ScheduledAction${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target CPU utilization
     */
    scaleOnCpuUtilization(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_CPU_UTILIZATION,
            targetValue: props.targetUtilizationPercent,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target network ingress rate
     */
    scaleOnIncomingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_IN,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target network egress rate
     */
    scaleOnOutgoingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_OUT,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target request handling rate
     *
     * The AutoScalingGroup must have been attached to an Application Load Balancer
     * in order to be able to call this.
     */
    scaleOnRequestCount(id, props) {
        if (this.albTargetGroup === undefined) {
            throw new Error('Attach the AutoScalingGroup to a non-imported Application Load Balancer before calling scaleOnRequestCount()');
        }
        const resourceLabel = `${this.albTargetGroup.firstLoadBalancerFullName}/${this.albTargetGroup.targetGroupFullName}`;
        if ((props.targetRequestsPerMinute === undefined) === (props.targetRequestsPerSecond === undefined)) {
            throw new Error('Specify exactly one of \'targetRequestsPerMinute\' or \'targetRequestsPerSecond\'');
        }
        let rpm;
        if (props.targetRequestsPerSecond !== undefined) {
            if (core_1.Token.isUnresolved(props.targetRequestsPerSecond)) {
                throw new Error('\'targetRequestsPerSecond\' cannot be an unresolved value; use \'targetRequestsPerMinute\' instead.');
            }
            rpm = props.targetRequestsPerSecond * 60;
        }
        else {
            rpm = props.targetRequestsPerMinute;
        }
        const policy = new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ALB_REQUEST_COUNT_PER_TARGET,
            targetValue: rpm,
            resourceLabel,
            ...props,
        });
        policy.node.addDependency(this.albTargetGroup.loadBalancerAttached);
        return policy;
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    scaleToTrackMetric(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            customMetric: props.metric,
            ...props,
        });
    }
    /**
     * Scale out or in, in response to a metric
     */
    scaleOnMetric(id, props) {
        return new step_scaling_policy_1.StepScalingPolicy(this, id, { ...props, autoScalingGroup: this });
    }
    addUserData(..._commands) {
    }
}
/**
 * A Fleet represents a managed set of EC2 instances
 *
 * The Fleet models a number of AutoScalingGroups, a launch configuration, a
 * security group and an instance role.
 *
 * It allows adding arbitrary commands to the startup scripts of the instances
 * in the fleet.
 *
 * The ASG spans the availability zones specified by vpcSubnets, falling back to
 * the Vpc default strategy if not specified.
 */
class AutoScalingGroup extends AutoScalingGroupBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.autoScalingGroupName,
        });
        this.loadBalancerNames = [];
        this.targetGroupArns = [];
        this.groupMetrics = [];
        this.notifications = [];
        try {
            jsiiDeprecationWarnings.monocdk_aws_autoscaling_AutoScalingGroupProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, AutoScalingGroup);
            }
            throw error;
        }
        this.newInstancesProtectedFromScaleIn = props.newInstancesProtectedFromScaleIn;
        if (props.initOptions && !props.init) {
            throw new Error('Setting \'initOptions\' requires that \'init\' is also set');
        }
        if (props.groupMetrics) {
            this.groupMetrics.push(...props.groupMetrics);
        }
        let launchConfig = undefined;
        if (props.launchTemplate || props.mixedInstancesPolicy) {
            this.verifyNoLaunchConfigPropIsGiven(props);
            const bareLaunchTemplate = props.launchTemplate;
            const mixedInstancesPolicy = props.mixedInstancesPolicy;
            if (bareLaunchTemplate && mixedInstancesPolicy) {
                throw new Error('Setting \'mixedInstancesPolicy\' must not be set when \'launchTemplate\' is set');
            }
            if (bareLaunchTemplate && bareLaunchTemplate instanceof ec2.LaunchTemplate) {
                if (!bareLaunchTemplate.instanceType) {
                    throw new Error('Setting \'launchTemplate\' requires its \'instanceType\' to be set');
                }
                if (!bareLaunchTemplate.imageId) {
                    throw new Error('Setting \'launchTemplate\' requires its \'machineImage\' to be set');
                }
                this.launchTemplate = bareLaunchTemplate;
            }
            if (mixedInstancesPolicy && mixedInstancesPolicy.launchTemplate instanceof ec2.LaunchTemplate) {
                if (!mixedInstancesPolicy.launchTemplate.imageId) {
                    throw new Error('Setting \'mixedInstancesPolicy.launchTemplate\' requires its \'machineImage\' to be set');
                }
                this.launchTemplate = mixedInstancesPolicy.launchTemplate;
            }
            this._role = this.launchTemplate?.role;
            this.grantPrincipal = this._role || new iam.UnknownPrincipal({ resource: this });
            this.osType = this.launchTemplate?.osType || ec2.OperatingSystemType.UNKNOWN;
        }
        else {
            if (!props.machineImage) {
                throw new Error('Setting \'machineImage\' is required when \'launchTemplate\' and \'mixedInstancesPolicy\' is not set');
            }
            if (!props.instanceType) {
                throw new Error('Setting \'instanceType\' is required when \'launchTemplate\' and \'mixedInstancesPolicy\' is not set');
            }
            this.securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'InstanceSecurityGroup', {
                vpc: props.vpc,
                allowAllOutbound: props.allowAllOutbound !== false,
            });
            this._connections = new ec2.Connections({ securityGroups: [this.securityGroup] });
            this.securityGroups = [this.securityGroup];
            core_1.Tags.of(this).add(NAME_TAG, this.node.path);
            this._role = props.role || new iam.Role(this, 'InstanceRole', {
                roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
            });
            this.grantPrincipal = this._role;
            const iamProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
                roles: [this.role.roleName],
            });
            // use delayed evaluation
            const imageConfig = props.machineImage.getImage(this);
            this._userData = props.userData ?? imageConfig.userData;
            const userDataToken = core_1.Lazy.string({ produce: () => core_1.Fn.base64(this.userData.render()) });
            const securityGroupsToken = core_1.Lazy.list({ produce: () => this.securityGroups.map(sg => sg.securityGroupId) });
            launchConfig = new autoscaling_generated_1.CfnLaunchConfiguration(this, 'LaunchConfig', {
                imageId: imageConfig.imageId,
                keyName: props.keyName,
                instanceType: props.instanceType.toString(),
                instanceMonitoring: (props.instanceMonitoring !== undefined ? (props.instanceMonitoring === Monitoring.DETAILED) : undefined),
                securityGroups: securityGroupsToken,
                iamInstanceProfile: iamProfile.ref,
                userData: userDataToken,
                associatePublicIpAddress: props.associatePublicIpAddress,
                spotPrice: props.spotPrice,
                blockDeviceMappings: (props.blockDevices !== undefined ?
                    synthesizeBlockDeviceMappings(this, props.blockDevices) : undefined),
            });
            launchConfig.node.addDependency(this.role);
            this.osType = imageConfig.osType;
        }
        // desiredCapacity just reflects what the user has supplied.
        const desiredCapacity = props.desiredCapacity;
        const minCapacity = props.minCapacity ?? 1;
        const maxCapacity = props.maxCapacity ?? desiredCapacity ?? Math.max(minCapacity, 1);
        core_1.withResolved(minCapacity, maxCapacity, (min, max) => {
            if (min > max) {
                throw new Error(`minCapacity (${min}) should be <= maxCapacity (${max})`);
            }
        });
        core_1.withResolved(desiredCapacity, minCapacity, (desired, min) => {
            if (desired === undefined) {
                return;
            }
            if (desired < min) {
                throw new Error(`Should have minCapacity (${min}) <= desiredCapacity (${desired})`);
            }
        });
        core_1.withResolved(desiredCapacity, maxCapacity, (desired, max) => {
            if (desired === undefined) {
                return;
            }
            if (max < desired) {
                throw new Error(`Should have desiredCapacity (${desired}) <= maxCapacity (${max})`);
            }
        });
        if (desiredCapacity !== undefined) {
            core_1.Annotations.of(this).addWarning('desiredCapacity has been configured. Be aware this will reset the size of your AutoScalingGroup on every deployment. See https://github.com/aws/aws-cdk/issues/5215');
        }
        this.maxInstanceLifetime = props.maxInstanceLifetime;
        // See https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-max-instance-lifetime.html for details on max instance lifetime.
        if (this.maxInstanceLifetime && !this.maxInstanceLifetime.isUnresolved() &&
            (this.maxInstanceLifetime.toSeconds() !== 0) &&
            (this.maxInstanceLifetime.toSeconds() < 86400 || this.maxInstanceLifetime.toSeconds() > 31536000)) {
            throw new Error('maxInstanceLifetime must be between 1 and 365 days (inclusive)');
        }
        if (props.notificationsTopic && props.notifications) {
            throw new Error('Cannot set \'notificationsTopic\' and \'notifications\', \'notificationsTopic\' is deprecated use \'notifications\' instead');
        }
        if (props.notificationsTopic) {
            this.notifications = [{
                    topic: props.notificationsTopic,
                }];
        }
        if (props.notifications) {
            this.notifications = props.notifications.map(nc => ({
                topic: nc.topic,
                scalingEvents: nc.scalingEvents ?? ScalingEvents.ALL,
            }));
        }
        const { subnetIds, hasPublic } = props.vpc.selectSubnets(props.vpcSubnets);
        const asgProps = {
            autoScalingGroupName: this.physicalName,
            cooldown: props.cooldown?.toSeconds().toString(),
            minSize: core_1.Tokenization.stringifyNumber(minCapacity),
            maxSize: core_1.Tokenization.stringifyNumber(maxCapacity),
            desiredCapacity: desiredCapacity !== undefined ? core_1.Tokenization.stringifyNumber(desiredCapacity) : undefined,
            loadBalancerNames: core_1.Lazy.list({ produce: () => this.loadBalancerNames }, { omitEmpty: true }),
            targetGroupArns: core_1.Lazy.list({ produce: () => this.targetGroupArns }, { omitEmpty: true }),
            notificationConfigurations: this.renderNotificationConfiguration(),
            metricsCollection: core_1.Lazy.any({ produce: () => this.renderMetricsCollection() }),
            vpcZoneIdentifier: subnetIds,
            healthCheckType: props.healthCheck && props.healthCheck.type,
            healthCheckGracePeriod: props.healthCheck && props.healthCheck.gracePeriod && props.healthCheck.gracePeriod.toSeconds(),
            maxInstanceLifetime: this.maxInstanceLifetime ? this.maxInstanceLifetime.toSeconds() : undefined,
            newInstancesProtectedFromScaleIn: core_1.Lazy.any({ produce: () => this.newInstancesProtectedFromScaleIn }),
            terminationPolicies: props.terminationPolicies,
            ...this.getLaunchSettings(launchConfig, props.launchTemplate, props.mixedInstancesPolicy),
        };
        if (!hasPublic && props.associatePublicIpAddress) {
            throw new Error("To set 'associatePublicIpAddress: true' you must select Public subnets (vpcSubnets: { subnetType: SubnetType.PUBLIC })");
        }
        this.autoScalingGroup = new autoscaling_generated_1.CfnAutoScalingGroup(this, 'ASG', asgProps);
        this.autoScalingGroupName = this.getResourceNameAttribute(this.autoScalingGroup.ref),
            this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
                service: 'autoscaling',
                resource: 'autoScalingGroup:*:autoScalingGroupName',
                resourceName: this.autoScalingGroupName,
            });
        this.node.defaultChild = this.autoScalingGroup;
        this.applyUpdatePolicies(props, { desiredCapacity, minCapacity });
        if (props.init) {
            this.applyCloudFormationInit(props.init, props.initOptions);
        }
        this.spotPrice = props.spotPrice;
        if (props.requireImdsv2) {
            core_1.Aspects.of(this).add(new aspects_1.AutoScalingGroupRequireImdsv2Aspect());
        }
    }
    static fromAutoScalingGroupName(scope, id, autoScalingGroupName) {
        class Import extends AutoScalingGroupBase {
            constructor() {
                super(...arguments);
                this.autoScalingGroupName = autoScalingGroupName;
                this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
                    service: 'autoscaling',
                    resource: 'autoScalingGroup:*:autoScalingGroupName',
                    resourceName: this.autoScalingGroupName,
                });
                this.osType = ec2.OperatingSystemType.UNKNOWN;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add the security group to all instances via the launch configuration
     * security groups array.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ec2_ISecurityGroup(securityGroup);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addSecurityGroup);
            }
            throw error;
        }
        if (!this.securityGroups) {
            throw new Error('You cannot add security groups when the Auto Scaling Group is created from a Launch Template.');
        }
        this.securityGroups.push(securityGroup);
    }
    /**
     * Attach to a classic load balancer
     */
    attachToClassicLB(loadBalancer) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancing_LoadBalancer(loadBalancer);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.attachToClassicLB);
            }
            throw error;
        }
        this.loadBalancerNames.push(loadBalancer.loadBalancerName);
    }
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToApplicationTargetGroup(targetGroup) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_IApplicationTargetGroup(targetGroup);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.attachToApplicationTargetGroup);
            }
            throw error;
        }
        if (this.albTargetGroup !== undefined) {
            throw new Error('Cannot add AutoScalingGroup to 2nd Target Group');
        }
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        if (targetGroup instanceof elbv2.ApplicationTargetGroup) {
            // Copy onto self if it's a concrete type. We need this for autoscaling
            // based on request count, which we cannot do with an imported TargetGroup.
            this.albTargetGroup = targetGroup;
        }
        targetGroup.registerConnectable(this);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToNetworkTargetGroup(targetGroup) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_INetworkTargetGroup(targetGroup);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.attachToNetworkTargetGroup);
            }
            throw error;
        }
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    addUserData(...commands) {
        this.userData.addCommands(...commands);
    }
    /**
     * Adds a statement to the IAM role assumed by instances of this fleet.
     */
    addToRolePolicy(statement) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_iam_PolicyStatement(statement);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToRolePolicy);
            }
            throw error;
        }
        this.role.addToPrincipalPolicy(statement);
    }
    /**
     * Use a CloudFormation Init configuration at instance startup
     *
     * This does the following:
     *
     * - Attaches the CloudFormation Init metadata to the AutoScalingGroup resource.
     * - Add commands to the UserData to run `cfn-init` and `cfn-signal`.
     * - Update the instance's CreationPolicy to wait for `cfn-init` to finish
     *   before reporting success.
     */
    applyCloudFormationInit(init, options = {}) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ec2_CloudFormationInit(init);
            jsiiDeprecationWarnings.monocdk_aws_autoscaling_ApplyCloudFormationInitOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.applyCloudFormationInit);
            }
            throw error;
        }
        if (!this.autoScalingGroup.cfnOptions.creationPolicy?.resourceSignal) {
            throw new Error('When applying CloudFormationInit, you must also configure signals by supplying \'signals\' at instantiation time.');
        }
        init.attach(this.autoScalingGroup, {
            platform: this.osType,
            instanceRole: this.role,
            userData: this.userData,
            configSets: options.configSets,
            embedFingerprint: options.embedFingerprint,
            printLog: options.printLog,
            ignoreFailures: options.ignoreFailures,
            includeRole: options.includeRole,
            includeUrl: options.includeUrl,
        });
    }
    /**
     * Ensures newly-launched instances are protected from scale-in.
     */
    protectNewInstancesFromScaleIn() {
        this.newInstancesProtectedFromScaleIn = true;
    }
    /**
     * Returns `true` if newly-launched instances are protected from scale-in.
     */
    areNewInstancesProtectedFromScaleIn() {
        return this.newInstancesProtectedFromScaleIn === true;
    }
    /**
     * The network connections associated with this resource.
     */
    get connections() {
        if (this._connections) {
            return this._connections;
        }
        if (this.launchTemplate) {
            return this.launchTemplate.connections;
        }
        throw new Error('AutoScalingGroup can only be used as IConnectable if it is not created from an imported Launch Template.');
    }
    /**
     * The Base64-encoded user data to make available to the launched EC2 instances.
     *
     * @throws an error if a launch template is given and it does not provide a non-null `userData`
     */
    get userData() {
        if (this._userData) {
            return this._userData;
        }
        if (this.launchTemplate?.userData) {
            return this.launchTemplate.userData;
        }
        throw new Error('The provided launch template does not expose its user data.');
    }
    /**
     * The IAM Role in the instance profile
     *
     * @throws an error if a launch template is given
     */
    get role() {
        if (this._role) {
            return this._role;
        }
        throw new Error('The provided launch template does not expose or does not define its role.');
    }
    verifyNoLaunchConfigPropIsGiven(props) {
        if (props.machineImage) {
            throw new Error('Setting \'machineImage\' must not be set when \'launchTemplate\' or \'mixedInstancesPolicy\' is set');
        }
        if (props.instanceType) {
            throw new Error('Setting \'instanceType\' must not be set when \'launchTemplate\' or \'mixedInstancesPolicy\' is set');
        }
        if (props.role) {
            throw new Error('Setting \'role\' must not be set when \'launchTemplate\' or \'mixedInstancesPolicy\' is set');
        }
        if (props.userData) {
            throw new Error('Setting \'userData\' must not be set when \'launchTemplate\' or \'mixedInstancesPolicy\' is set');
        }
        if (props.securityGroup) {
            throw new Error('Setting \'securityGroup\' must not be set when \'launchTemplate\' or \'mixedInstancesPolicy\' is set');
        }
        if (props.keyName) {
            throw new Error('Setting \'keyName\' must not be set when \'launchTemplate\' or \'mixedInstancesPolicy\' is set');
        }
        if (props.instanceMonitoring) {
            throw new Error('Setting \'instanceMonitoring\' must not be set when \'launchTemplate\' or \'mixedInstancesPolicy\' is set');
        }
        if (props.associatePublicIpAddress) {
            throw new Error('Setting \'associatePublicIpAddress\' must not be set when \'launchTemplate\' or \'mixedInstancesPolicy\' is set');
        }
        if (props.spotPrice) {
            throw new Error('Setting \'spotPrice\' must not be set when \'launchTemplate\' or \'mixedInstancesPolicy\' is set');
        }
        if (props.blockDevices) {
            throw new Error('Setting \'blockDevices\' must not be set when \'launchTemplate\' or \'mixedInstancesPolicy\' is set');
        }
    }
    /**
     * Apply CloudFormation update policies for the AutoScalingGroup
     */
    applyUpdatePolicies(props, signalOptions) {
        // Make sure people are not using the old and new properties together
        const oldProps = [
            'updateType',
            'rollingUpdateConfiguration',
            'resourceSignalCount',
            'resourceSignalTimeout',
            'replacingUpdateMinSuccessfulInstancesPercent',
        ];
        for (const prop of oldProps) {
            if ((props.signals || props.updatePolicy) && props[prop] !== undefined) {
                throw new Error(`Cannot set 'signals'/'updatePolicy' and '${prop}' together. Prefer 'signals'/'updatePolicy'`);
            }
        }
        // Reify updatePolicy to `rollingUpdate` default in case it is combined with `init`
        props = {
            ...props,
            updatePolicy: props.updatePolicy ?? (props.init ? UpdatePolicy.rollingUpdate() : undefined),
        };
        if (props.signals || props.updatePolicy) {
            this.applyNewSignalUpdatePolicies(props, signalOptions);
        }
        else {
            this.applyLegacySignalUpdatePolicies(props);
        }
        // The following is technically part of the "update policy" but it's also a completely
        // separate aspect of rolling/replacing update, so it's just its own top-level property.
        // Default is 'true' because that's what you're most likely to want
        if (props.ignoreUnmodifiedSizeProperties !== false) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingScheduledAction: { ignoreUnmodifiedGroupSizeProperties: true },
            };
        }
        if (props.signals && !props.init) {
            // To be able to send a signal using `cfn-init`, the execution role needs
            // `cloudformation:SignalResource`. Normally the binding of CfnInit would
            // grant that permissions and another one, but if the user wants to use
            // `signals` without `init`, add the permissions here.
            //
            // If they call `applyCloudFormationInit()` after construction, nothing bad
            // happens either, we'll just have a duplicate statement which doesn't hurt.
            this.addToRolePolicy(new iam.PolicyStatement({
                actions: ['cloudformation:SignalResource'],
                resources: [core_1.Aws.STACK_ID],
            }));
        }
    }
    /**
     * Use 'signals' and 'updatePolicy' to determine the creation and update policies
     */
    applyNewSignalUpdatePolicies(props, signalOptions) {
        this.autoScalingGroup.cfnOptions.creationPolicy = props.signals?.renderCreationPolicy(signalOptions);
        this.autoScalingGroup.cfnOptions.updatePolicy = props.updatePolicy?._renderUpdatePolicy({
            creationPolicy: this.autoScalingGroup.cfnOptions.creationPolicy,
        });
    }
    applyLegacySignalUpdatePolicies(props) {
        if (props.updateType === UpdateType.REPLACING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingReplacingUpdate: {
                    willReplace: true,
                },
            };
            if (props.replacingUpdateMinSuccessfulInstancesPercent !== undefined) {
                // Yes, this goes on CreationPolicy, not as a process parameter to ReplacingUpdate.
                // It's a little confusing, but the docs seem to explicitly state it will only be used
                // during the update?
                //
                // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-creationpolicy.html
                this.autoScalingGroup.cfnOptions.creationPolicy = {
                    ...this.autoScalingGroup.cfnOptions.creationPolicy,
                    autoScalingCreationPolicy: {
                        minSuccessfulInstancesPercent: validatePercentage(props.replacingUpdateMinSuccessfulInstancesPercent),
                    },
                };
            }
        }
        else if (props.updateType === UpdateType.ROLLING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingRollingUpdate: renderRollingUpdateConfig(props.rollingUpdateConfiguration),
            };
        }
        if (props.resourceSignalCount !== undefined || props.resourceSignalTimeout !== undefined) {
            this.autoScalingGroup.cfnOptions.creationPolicy = {
                ...this.autoScalingGroup.cfnOptions.creationPolicy,
                resourceSignal: {
                    count: props.resourceSignalCount,
                    timeout: props.resourceSignalTimeout && props.resourceSignalTimeout.toIsoString(),
                },
            };
        }
    }
    renderNotificationConfiguration() {
        if (this.notifications.length === 0) {
            return undefined;
        }
        return this.notifications.map(notification => ({
            topicArn: notification.topic.topicArn,
            notificationTypes: notification.scalingEvents ? notification.scalingEvents._types : ScalingEvents.ALL._types,
        }));
    }
    renderMetricsCollection() {
        if (this.groupMetrics.length === 0) {
            return undefined;
        }
        return this.groupMetrics.map(group => ({
            granularity: '1Minute',
            metrics: group._metrics?.size !== 0 ? [...group._metrics].map(m => m.name) : undefined,
        }));
    }
    getLaunchSettings(launchConfig, launchTemplate, mixedInstancesPolicy) {
        if (launchConfig) {
            return {
                launchConfigurationName: launchConfig.ref,
            };
        }
        if (launchTemplate) {
            return {
                launchTemplate: this.convertILaunchTemplateToSpecification(launchTemplate),
            };
        }
        if (mixedInstancesPolicy) {
            let instancesDistribution = undefined;
            if (mixedInstancesPolicy.instancesDistribution) {
                const dist = mixedInstancesPolicy.instancesDistribution;
                instancesDistribution = {
                    onDemandAllocationStrategy: dist.onDemandAllocationStrategy?.toString(),
                    onDemandBaseCapacity: dist.onDemandBaseCapacity,
                    onDemandPercentageAboveBaseCapacity: dist.onDemandPercentageAboveBaseCapacity,
                    spotAllocationStrategy: dist.spotAllocationStrategy?.toString(),
                    spotInstancePools: dist.spotInstancePools,
                    spotMaxPrice: dist.spotMaxPrice,
                };
            }
            return {
                mixedInstancesPolicy: {
                    instancesDistribution,
                    launchTemplate: {
                        launchTemplateSpecification: this.convertILaunchTemplateToSpecification(mixedInstancesPolicy.launchTemplate),
                        ...(mixedInstancesPolicy.launchTemplateOverrides ? {
                            overrides: mixedInstancesPolicy.launchTemplateOverrides.map(override => {
                                if (override.weightedCapacity && Math.floor(override.weightedCapacity) !== override.weightedCapacity) {
                                    throw new Error('Weight must be an integer');
                                }
                                return {
                                    instanceType: override.instanceType.toString(),
                                    launchTemplateSpecification: override.launchTemplate
                                        ? this.convertILaunchTemplateToSpecification(override.launchTemplate)
                                        : undefined,
                                    weightedCapacity: override.weightedCapacity?.toString(),
                                };
                            }),
                        } : {}),
                    },
                },
            };
        }
        throw new Error('Either launchConfig, launchTemplate or mixedInstancesPolicy needs to be specified.');
    }
    convertILaunchTemplateToSpecification(launchTemplate) {
        if (launchTemplate.launchTemplateId) {
            return {
                launchTemplateId: launchTemplate.launchTemplateId,
                version: launchTemplate.versionNumber,
            };
        }
        else {
            return {
                launchTemplateName: launchTemplate.launchTemplateName,
                version: launchTemplate.versionNumber,
            };
        }
    }
}
exports.AutoScalingGroup = AutoScalingGroup;
_e = JSII_RTTI_SYMBOL_1;
AutoScalingGroup[_e] = { fqn: "monocdk.aws_autoscaling.AutoScalingGroup", version: "1.191.0" };
/**
 * The type of update to perform on instances in this AutoScalingGroup
 *
 * @deprecated Use UpdatePolicy instead
 */
var UpdateType;
(function (UpdateType) {
    /**
     * Don't do anything
     */
    UpdateType["NONE"] = "None";
    /**
     * Replace the entire AutoScalingGroup
     *
     * Builds a new AutoScalingGroup first, then delete the old one.
     */
    UpdateType["REPLACING_UPDATE"] = "Replace";
    /**
     * Replace the instances in the AutoScalingGroup.
     */
    UpdateType["ROLLING_UPDATE"] = "RollingUpdate";
})(UpdateType = exports.UpdateType || (exports.UpdateType = {}));
/**
 * Fleet scaling events
 */
var ScalingEvent;
(function (ScalingEvent) {
    /**
     * Notify when an instance was launched
     */
    ScalingEvent["INSTANCE_LAUNCH"] = "autoscaling:EC2_INSTANCE_LAUNCH";
    /**
     * Notify when an instance was terminated
     */
    ScalingEvent["INSTANCE_TERMINATE"] = "autoscaling:EC2_INSTANCE_TERMINATE";
    /**
     * Notify when an instance failed to terminate
     */
    ScalingEvent["INSTANCE_TERMINATE_ERROR"] = "autoscaling:EC2_INSTANCE_TERMINATE_ERROR";
    /**
     * Notify when an instance failed to launch
     */
    ScalingEvent["INSTANCE_LAUNCH_ERROR"] = "autoscaling:EC2_INSTANCE_LAUNCH_ERROR";
    /**
     * Send a test notification to the topic
     */
    ScalingEvent["TEST_NOTIFICATION"] = "autoscaling:TEST_NOTIFICATION";
})(ScalingEvent = exports.ScalingEvent || (exports.ScalingEvent = {}));
/**
 * A list of ScalingEvents, you can use one of the predefined lists, such as ScalingEvents.ERRORS
 * or create a custom group by instantiating a `NotificationTypes` object, e.g: `new NotificationTypes(`NotificationType.INSTANCE_LAUNCH`)`.
 */
class ScalingEvents {
    constructor(...types) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_autoscaling_ScalingEvent(types);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ScalingEvents);
            }
            throw error;
        }
        this._types = types;
    }
}
exports.ScalingEvents = ScalingEvents;
_f = JSII_RTTI_SYMBOL_1;
ScalingEvents[_f] = { fqn: "monocdk.aws_autoscaling.ScalingEvents", version: "1.191.0" };
/**
 * Fleet scaling errors
 */
ScalingEvents.ERRORS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * All fleet scaling events
 */
ScalingEvents.ALL = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * Fleet scaling launch events
 */
ScalingEvents.LAUNCH_EVENTS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR);
/**
 * Fleet termination launch events
 */
ScalingEvents.TERMINATION_EVENTS = new ScalingEvents(ScalingEvent.INSTANCE_TERMINATE, ScalingEvent.INSTANCE_TERMINATE_ERROR);
var ScalingProcess;
(function (ScalingProcess) {
    ScalingProcess["LAUNCH"] = "Launch";
    ScalingProcess["TERMINATE"] = "Terminate";
    ScalingProcess["HEALTH_CHECK"] = "HealthCheck";
    ScalingProcess["REPLACE_UNHEALTHY"] = "ReplaceUnhealthy";
    ScalingProcess["AZ_REBALANCE"] = "AZRebalance";
    ScalingProcess["ALARM_NOTIFICATION"] = "AlarmNotification";
    ScalingProcess["SCHEDULED_ACTIONS"] = "ScheduledActions";
    ScalingProcess["ADD_TO_LOAD_BALANCER"] = "AddToLoadBalancer";
})(ScalingProcess = exports.ScalingProcess || (exports.ScalingProcess = {}));
// Recommended list of processes to suspend from here:
// https://aws.amazon.com/premiumsupport/knowledge-center/auto-scaling-group-rolling-updates/
const DEFAULT_SUSPEND_PROCESSES = [ScalingProcess.HEALTH_CHECK, ScalingProcess.REPLACE_UNHEALTHY, ScalingProcess.AZ_REBALANCE,
    ScalingProcess.ALARM_NOTIFICATION, ScalingProcess.SCHEDULED_ACTIONS];
/**
 * Health check settings
 */
class HealthCheck {
    constructor(type, gracePeriod) {
        this.type = type;
        this.gracePeriod = gracePeriod;
    }
    /**
     * Use EC2 for health checks
     *
     * @param options EC2 health check options
     */
    static ec2(options = {}) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_autoscaling_Ec2HealthCheckOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.ec2);
            }
            throw error;
        }
        return new HealthCheck(HealthCheckType.EC2, options.grace);
    }
    /**
     * Use ELB for health checks.
     * It considers the instance unhealthy if it fails either the EC2 status checks or the load balancer health checks.
     *
     * @param options ELB health check options
     */
    static elb(options) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_autoscaling_ElbHealthCheckOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.elb);
            }
            throw error;
        }
        return new HealthCheck(HealthCheckType.ELB, options.grace);
    }
}
exports.HealthCheck = HealthCheck;
_g = JSII_RTTI_SYMBOL_1;
HealthCheck[_g] = { fqn: "monocdk.aws_autoscaling.HealthCheck", version: "1.191.0" };
var HealthCheckType;
(function (HealthCheckType) {
    HealthCheckType["EC2"] = "EC2";
    HealthCheckType["ELB"] = "ELB";
})(HealthCheckType || (HealthCheckType = {}));
/**
 * Render the rolling update configuration into the appropriate object
 */
function renderRollingUpdateConfig(config = {}) {
    const waitOnResourceSignals = config.minSuccessfulInstancesPercent !== undefined;
    const pauseTime = config.pauseTime || (waitOnResourceSignals ? core_1.Duration.minutes(5) : core_1.Duration.seconds(0));
    return {
        maxBatchSize: config.maxBatchSize,
        minInstancesInService: config.minInstancesInService,
        minSuccessfulInstancesPercent: validatePercentage(config.minSuccessfulInstancesPercent),
        waitOnResourceSignals,
        pauseTime: pauseTime && pauseTime.toIsoString(),
        suspendProcesses: config.suspendProcesses ?? DEFAULT_SUSPEND_PROCESSES,
    };
}
function validatePercentage(x) {
    if (x === undefined || (0 <= x && x <= 100)) {
        return x;
    }
    throw new Error(`Expected: a percentage 0..100, got: ${x}`);
}
/**
 * Synthesize an array of block device mappings from a list of block device
 *
 * @param construct the instance/asg construct, used to host any warning
 * @param blockDevices list of block devices
 */
function synthesizeBlockDeviceMappings(construct, blockDevices) {
    return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
        const { virtualName, ebsDevice: ebs } = volume;
        if (volume === volume_1.BlockDeviceVolume._NO_DEVICE || mappingEnabled === false) {
            return {
                deviceName,
                noDevice: true,
            };
        }
        if (ebs) {
            const { iops, volumeType } = ebs;
            if (!iops) {
                if (volumeType === volume_1.EbsDeviceVolumeType.IO1) {
                    throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                }
            }
            else if (volumeType !== volume_1.EbsDeviceVolumeType.IO1) {
                core_1.Annotations.of(construct).addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            }
        }
        return {
            deviceName, ebs, virtualName,
        };
    });
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXV0by1zY2FsaW5nLWdyb3VwLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiYXV0by1zY2FsaW5nLWdyb3VwLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7OztBQUNBLHFDQUFxQztBQUVyQywwREFBMEQ7QUFDMUQscUNBQXFDO0FBR3JDLHFDQVFvQjtBQUVwQix1Q0FBZ0U7QUFDaEUsbUVBQWdIO0FBQ2hILHFEQUEwRTtBQUMxRSx5REFBZ0Y7QUFDaEYsK0RBQXVGO0FBQ3ZGLHFGQUEwSDtBQUUxSCxxQ0FBK0U7QUFDL0UsMkNBQXdEO0FBRXhEOztHQUVHO0FBQ0gsTUFBTSxRQUFRLEdBQVcsTUFBTSxDQUFDO0FBRWhDOztHQUVHO0FBQ0gsSUFBWSxVQVVYO0FBVkQsV0FBWSxVQUFVO0lBQ3BCOztPQUVHO0lBQ0gsNkNBQUssQ0FBQTtJQUVMOztPQUVHO0lBQ0gsbURBQVEsQ0FBQTtBQUNWLENBQUMsRUFWVyxVQUFVLEdBQVYsa0JBQVUsS0FBVixrQkFBVSxRQVVyQjtBQTZURDs7R0FFRztBQUNILElBQVksMEJBU1g7QUFURCxXQUFZLDBCQUEwQjtJQUNwQzs7Ozs7O09BTUc7SUFDSCx5REFBMkIsQ0FBQTtBQUM3QixDQUFDLEVBVFcsMEJBQTBCLEdBQTFCLGtDQUEwQixLQUExQixrQ0FBMEIsUUFTckM7QUFFRDs7R0FFRztBQUNILElBQVksc0JBcUJYO0FBckJELFdBQVksc0JBQXNCO0lBQ2hDOzs7T0FHRztJQUNILHVEQUE2QixDQUFBO0lBRTdCOzs7OztPQUtHO0lBQ0gsbUVBQXlDLENBQUE7SUFFekM7Ozs7T0FJRztJQUNILDJGQUFpRSxDQUFBO0FBQ25FLENBQUMsRUFyQlcsc0JBQXNCLEdBQXRCLDhCQUFzQixLQUF0Qiw4QkFBc0IsUUFxQmpDO0FBa09EOzs7Ozs7OztHQVFHO0FBQ0gsTUFBc0IsT0FBTztJQUMzQjs7Ozs7OztPQU9HO0lBQ0ksTUFBTSxDQUFDLFVBQVUsQ0FBQyxVQUEwQixFQUFFOzs7Ozs7Ozs7O1FBQ25ELGtCQUFrQixDQUFDLE9BQU8sQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBQ2pELE9BQU8sSUFBSSxLQUFNLFNBQVEsT0FBTztZQUN2QixvQkFBb0IsQ0FBQyxhQUFtQztnQkFDN0QsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRSxhQUFhLENBQUMsZUFBZSxJQUFJLGFBQWEsQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUM1RixDQUFDO1NBQ0YsRUFBRSxDQUFDO0tBQ0w7SUFFRDs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyxrQkFBa0IsQ0FBQyxVQUEwQixFQUFFOzs7Ozs7Ozs7O1FBQzNELGtCQUFrQixDQUFDLE9BQU8sQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBQ2pELE9BQU8sSUFBSSxLQUFNLFNBQVEsT0FBTztZQUN2QixvQkFBb0IsQ0FBQyxhQUFtQztnQkFDN0QsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRSxhQUFhLENBQUMsV0FBVyxDQUFDLENBQUM7WUFDM0QsQ0FBQztTQUNGLEVBQUUsQ0FBQztLQUNMO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSSxNQUFNLENBQUMsWUFBWSxDQUFDLEtBQWEsRUFBRSxVQUEwQixFQUFFOzs7Ozs7Ozs7O1FBQ3BFLGtCQUFrQixDQUFDLE9BQU8sQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBQ2pELE9BQU8sSUFBSSxLQUFNLFNBQVEsT0FBTztZQUN2QixvQkFBb0I7Z0JBQ3pCLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFDdkMsQ0FBQztTQUNGLEVBQUUsQ0FBQztLQUNMO0lBT0Q7O09BRUc7SUFDTyxRQUFRLENBQUMsT0FBdUIsRUFBRSxLQUFjOzs7Ozs7Ozs7O1FBQ3hELE1BQU0sNkJBQTZCLEdBQUcsa0JBQWtCLENBQUMsT0FBTyxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDdkYsT0FBTztZQUNMLEdBQUcsT0FBTyxDQUFDLG9CQUFvQixLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRSx5QkFBeUIsRUFBRSxFQUFFLDZCQUE2QixFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRztZQUN0SCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSztnQkFDTCxPQUFPLEVBQUUsT0FBTyxDQUFDLE9BQU8sRUFBRSxXQUFXLEVBQUU7YUFDeEM7U0FDRixDQUFDO0tBQ0g7O0FBcEVILDBCQXNFQzs7O0FBOENEOztHQUVHO0FBQ0gsTUFBc0IsWUFBWTtJQUNoQzs7T0FFRztJQUNJLE1BQU0sQ0FBQyxlQUFlO1FBQzNCLE9BQU8sSUFBSSxLQUFNLFNBQVEsWUFBWTtZQUM1QixtQkFBbUI7Z0JBQ3hCLE9BQU87b0JBQ0wsMEJBQTBCLEVBQUUsRUFBRSxXQUFXLEVBQUUsSUFBSSxFQUFFO2lCQUNsRCxDQUFDO1lBQ0osQ0FBQztTQUNGLEVBQUUsQ0FBQztLQUNMO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsYUFBYSxDQUFDLFVBQWdDLEVBQUU7Ozs7Ozs7Ozs7UUFDNUQsTUFBTSxvQkFBb0IsR0FBRyxrQkFBa0IsQ0FBQyxPQUFPLENBQUMsb0JBQW9CLENBQUMsQ0FBQztRQUU5RSxPQUFPLElBQUksS0FBTSxTQUFRLFlBQVk7WUFDNUIsbUJBQW1CLENBQUMsYUFBa0M7Z0JBQzNELE9BQU87b0JBQ0wsd0JBQXdCLEVBQUU7d0JBQ3hCLFlBQVksRUFBRSxPQUFPLENBQUMsWUFBWTt3QkFDbEMscUJBQXFCLEVBQUUsT0FBTyxDQUFDLHFCQUFxQjt3QkFDcEQsZ0JBQWdCLEVBQUUsT0FBTyxDQUFDLGdCQUFnQixJQUFJLHlCQUF5Qjt3QkFDdkUsNkJBQTZCLEVBQzNCLG9CQUFvQixJQUFJLGFBQWEsQ0FBQyxjQUFjLEVBQUUseUJBQXlCLEVBQUUsNkJBQTZCO3dCQUNoSCxxQkFBcUIsRUFBRSxPQUFPLENBQUMscUJBQXFCLElBQUksYUFBYSxDQUFDLGNBQWMsRUFBRSxjQUFjLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLFNBQVM7d0JBQ3JJLFNBQVMsRUFBRSxPQUFPLENBQUMsU0FBUyxFQUFFLFdBQVcsRUFBRSxJQUFJLGFBQWEsQ0FBQyxjQUFjLEVBQUUsY0FBYyxFQUFFLE9BQU87cUJBQ3JHO2lCQUNGLENBQUM7WUFDSixDQUFDO1NBQ0YsRUFBRSxDQUFDO0tBQ0w7O0FBbkNILG9DQTBDQzs7O0FBb0VEOztHQUVHO0FBQ0gsTUFBYSxZQUFZO0lBY3ZCLFlBQVksR0FBRyxPQUFzQjtRQUxyQzs7V0FFRztRQUNJLGFBQVEsR0FBRyxJQUFJLEdBQUcsRUFBZSxDQUFDOzs7Ozs7K0NBWjlCLFlBQVk7Ozs7UUFlckIsT0FBTyxFQUFFLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7S0FDdkQ7SUFkRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxHQUFHO1FBQ2YsT0FBTyxJQUFJLFlBQVksRUFBRSxDQUFDO0tBQzNCOztBQVBILG9DQWlCQzs7O0FBRUQ7O0dBRUc7QUFDSCxNQUFhLFdBQVc7SUFvRHRCLFlBQVksSUFBWTtRQUN0QixJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQztLQUNsQjs7QUF0REgsa0NBdURDOzs7QUFyREM7O0dBRUc7QUFDb0Isb0JBQVEsR0FBRyxJQUFJLFdBQVcsQ0FBQyxjQUFjLENBQUMsQ0FBQztBQUVsRTs7R0FFRztBQUNvQixvQkFBUSxHQUFHLElBQUksV0FBVyxDQUFDLGNBQWMsQ0FBQyxDQUFDO0FBRWxFOztHQUVHO0FBQ29CLDRCQUFnQixHQUFHLElBQUksV0FBVyxDQUFDLHNCQUFzQixDQUFDLENBQUM7QUFFbEY7OztHQUdHO0FBQ29CLGdDQUFvQixHQUFHLElBQUksV0FBVyxDQUFDLHlCQUF5QixDQUFDLENBQUM7QUFFekY7OztHQUdHO0FBQ29CLDZCQUFpQixHQUFHLElBQUksV0FBVyxDQUFDLHVCQUF1QixDQUFDLENBQUM7QUFFcEY7OztHQUdHO0FBQ29CLDZCQUFpQixHQUFHLElBQUksV0FBVyxDQUFDLHVCQUF1QixDQUFDLENBQUM7QUFFcEY7OztHQUdHO0FBQ29CLGlDQUFxQixHQUFHLElBQUksV0FBVyxDQUFDLDJCQUEyQixDQUFDLENBQUM7QUFFNUY7OztHQUdHO0FBQ29CLDJCQUFlLEdBQUcsSUFBSSxXQUFXLENBQUMscUJBQXFCLENBQUMsQ0FBQztBQVlsRixNQUFlLG9CQUFxQixTQUFRLGVBQVE7SUFBcEQ7O1FBTWtCLG1CQUFjLEdBQW1CLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7SUFnSWhHLENBQUM7SUE5SEM7O09BRUc7SUFDSSxnQkFBZ0IsQ0FBQyxFQUFVLEVBQUUsS0FBOEI7UUFDaEUsT0FBTyxJQUFJLDhCQUFhLENBQUMsSUFBSSxFQUFFLGdCQUFnQixFQUFFLEVBQUUsRUFBRTtZQUNuRCxnQkFBZ0IsRUFBRSxJQUFJO1lBQ3RCLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztLQUNKO0lBRUQ7O09BRUc7SUFDSSxXQUFXLENBQUMsT0FBeUI7UUFDMUMsT0FBTyxJQUFJLG9CQUFRLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNwQyxnQkFBZ0IsRUFBRSxJQUFJO1lBQ3RCLEdBQUcsT0FBTztTQUNYLENBQUMsQ0FBQztLQUNKO0lBRUQ7O09BRUc7SUFDSSxlQUFlLENBQUMsRUFBVSxFQUFFLEtBQWdDO1FBQ2pFLE9BQU8sSUFBSSxrQ0FBZSxDQUFDLElBQUksRUFBRSxrQkFBa0IsRUFBRSxFQUFFLEVBQUU7WUFDdkQsZ0JBQWdCLEVBQUUsSUFBSTtZQUN0QixHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7S0FDSjtJQUVEOztPQUVHO0lBQ0kscUJBQXFCLENBQUMsRUFBVSxFQUFFLEtBQWlDO1FBQ3hFLE9BQU8sSUFBSSw0REFBMkIsQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsRUFBRSxFQUFFO1lBQ2pFLGdCQUFnQixFQUFFLElBQUk7WUFDdEIsZ0JBQWdCLEVBQUUsaURBQWdCLENBQUMsMkJBQTJCO1lBQzlELFdBQVcsRUFBRSxLQUFLLENBQUMsd0JBQXdCO1lBQzNDLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztLQUNKO0lBRUQ7O09BRUc7SUFDSSxvQkFBb0IsQ0FBQyxFQUFVLEVBQUUsS0FBcUM7UUFDM0UsT0FBTyxJQUFJLDREQUEyQixDQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRSxFQUFFLEVBQUU7WUFDakUsZ0JBQWdCLEVBQUUsSUFBSTtZQUN0QixnQkFBZ0IsRUFBRSxpREFBZ0IsQ0FBQyxzQkFBc0I7WUFDekQsV0FBVyxFQUFFLEtBQUssQ0FBQyxvQkFBb0I7WUFDdkMsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO0tBQ0o7SUFFRDs7T0FFRztJQUNJLG9CQUFvQixDQUFDLEVBQVUsRUFBRSxLQUFxQztRQUMzRSxPQUFPLElBQUksNERBQTJCLENBQUMsSUFBSSxFQUFFLGdCQUFnQixFQUFFLEVBQUUsRUFBRTtZQUNqRSxnQkFBZ0IsRUFBRSxJQUFJO1lBQ3RCLGdCQUFnQixFQUFFLGlEQUFnQixDQUFDLHVCQUF1QjtZQUMxRCxXQUFXLEVBQUUsS0FBSyxDQUFDLG9CQUFvQjtZQUN2QyxHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7S0FDSjtJQUVEOzs7OztPQUtHO0lBQ0ksbUJBQW1CLENBQUMsRUFBVSxFQUFFLEtBQStCO1FBQ3BFLElBQUksSUFBSSxDQUFDLGNBQWMsS0FBSyxTQUFTLEVBQUU7WUFDckMsTUFBTSxJQUFJLEtBQUssQ0FBQyw4R0FBOEcsQ0FBQyxDQUFDO1NBQ2pJO1FBRUQsTUFBTSxhQUFhLEdBQUcsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLHlCQUF5QixJQUFJLElBQUksQ0FBQyxjQUFjLENBQUMsbUJBQW1CLEVBQUUsQ0FBQztRQUVwSCxJQUFJLENBQUMsS0FBSyxDQUFDLHVCQUF1QixLQUFLLFNBQVMsQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLHVCQUF1QixLQUFLLFNBQVMsQ0FBQyxFQUFFO1lBQ25HLE1BQU0sSUFBSSxLQUFLLENBQUMsbUZBQW1GLENBQUMsQ0FBQztTQUN0RztRQUVELElBQUksR0FBVyxDQUFDO1FBQ2hCLElBQUksS0FBSyxDQUFDLHVCQUF1QixLQUFLLFNBQVMsRUFBRTtZQUMvQyxJQUFJLFlBQUssQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLHVCQUF1QixDQUFDLEVBQUU7Z0JBQ3JELE1BQU0sSUFBSSxLQUFLLENBQUMscUdBQXFHLENBQUMsQ0FBQzthQUN4SDtZQUNELEdBQUcsR0FBRyxLQUFLLENBQUMsdUJBQXVCLEdBQUcsRUFBRSxDQUFDO1NBQzFDO2FBQU07WUFDTCxHQUFHLEdBQUcsS0FBSyxDQUFDLHVCQUF3QixDQUFDO1NBQ3RDO1FBRUQsTUFBTSxNQUFNLEdBQUcsSUFBSSw0REFBMkIsQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsRUFBRSxFQUFFO1lBQ3pFLGdCQUFnQixFQUFFLElBQUk7WUFDdEIsZ0JBQWdCLEVBQUUsaURBQWdCLENBQUMsNEJBQTRCO1lBQy9ELFdBQVcsRUFBRSxHQUFHO1lBQ2hCLGFBQWE7WUFDYixHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7UUFFSCxNQUFNLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDcEUsT0FBTyxNQUFNLENBQUM7S0FDZjtJQUVEOztPQUVHO0lBQ0ksa0JBQWtCLENBQUMsRUFBVSxFQUFFLEtBQWdDO1FBQ3BFLE9BQU8sSUFBSSw0REFBMkIsQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsRUFBRSxFQUFFO1lBQ2pFLGdCQUFnQixFQUFFLElBQUk7WUFDdEIsWUFBWSxFQUFFLEtBQUssQ0FBQyxNQUFNO1lBQzFCLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztLQUNKO0lBRUQ7O09BRUc7SUFDSSxhQUFhLENBQUMsRUFBVSxFQUFFLEtBQWtDO1FBQ2pFLE9BQU8sSUFBSSx1Q0FBaUIsQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFLEVBQUUsR0FBRyxLQUFLLEVBQUUsZ0JBQWdCLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztLQUM5RTtJQUVNLFdBQVcsQ0FBQyxHQUFHLFNBQW1CO0tBRXhDO0NBQ0Y7QUFFRDs7Ozs7Ozs7Ozs7R0FXRztBQUNILE1BQWEsZ0JBQWlCLFNBQVEsb0JBQW9CO0lBaUV4RCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTRCO1FBQ3BFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsWUFBWSxFQUFFLEtBQUssQ0FBQyxvQkFBb0I7U0FDekMsQ0FBQyxDQUFDO1FBZFksc0JBQWlCLEdBQWEsRUFBRSxDQUFDO1FBQ2pDLG9CQUFlLEdBQWEsRUFBRSxDQUFDO1FBQy9CLGlCQUFZLEdBQW1CLEVBQUUsQ0FBQztRQUNsQyxrQkFBYSxHQUFnQyxFQUFFLENBQUM7Ozs7OzsrQ0F6RHRELGdCQUFnQjs7OztRQXNFekIsSUFBSSxDQUFDLGdDQUFnQyxHQUFHLEtBQUssQ0FBQyxnQ0FBZ0MsQ0FBQztRQUUvRSxJQUFJLEtBQUssQ0FBQyxXQUFXLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxFQUFFO1lBQ3BDLE1BQU0sSUFBSSxLQUFLLENBQUMsNERBQTRELENBQUMsQ0FBQztTQUMvRTtRQUVELElBQUksS0FBSyxDQUFDLFlBQVksRUFBRTtZQUN0QixJQUFJLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQztTQUMvQztRQUVELElBQUksWUFBWSxHQUF1QyxTQUFTLENBQUM7UUFDakUsSUFBSSxLQUFLLENBQUMsY0FBYyxJQUFJLEtBQUssQ0FBQyxvQkFBb0IsRUFBRTtZQUN0RCxJQUFJLENBQUMsK0JBQStCLENBQUMsS0FBSyxDQUFDLENBQUM7WUFFNUMsTUFBTSxrQkFBa0IsR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDO1lBQ2hELE1BQU0sb0JBQW9CLEdBQUcsS0FBSyxDQUFDLG9CQUFvQixDQUFDO1lBRXhELElBQUksa0JBQWtCLElBQUksb0JBQW9CLEVBQUU7Z0JBQzlDLE1BQU0sSUFBSSxLQUFLLENBQUMsaUZBQWlGLENBQUMsQ0FBQzthQUNwRztZQUVELElBQUksa0JBQWtCLElBQUksa0JBQWtCLFlBQVksR0FBRyxDQUFDLGNBQWMsRUFBRTtnQkFDMUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLFlBQVksRUFBRTtvQkFDcEMsTUFBTSxJQUFJLEtBQUssQ0FBQyxvRUFBb0UsQ0FBQyxDQUFDO2lCQUN2RjtnQkFFRCxJQUFJLENBQUMsa0JBQWtCLENBQUMsT0FBTyxFQUFFO29CQUMvQixNQUFNLElBQUksS0FBSyxDQUFDLG9FQUFvRSxDQUFDLENBQUM7aUJBQ3ZGO2dCQUVELElBQUksQ0FBQyxjQUFjLEdBQUcsa0JBQWtCLENBQUM7YUFDMUM7WUFFRCxJQUFJLG9CQUFvQixJQUFJLG9CQUFvQixDQUFDLGNBQWMsWUFBWSxHQUFHLENBQUMsY0FBYyxFQUFFO2dCQUM3RixJQUFJLENBQUMsb0JBQW9CLENBQUMsY0FBYyxDQUFDLE9BQU8sRUFBRTtvQkFDaEQsTUFBTSxJQUFJLEtBQUssQ0FBQyx5RkFBeUYsQ0FBQyxDQUFDO2lCQUM1RztnQkFFRCxJQUFJLENBQUMsY0FBYyxHQUFHLG9CQUFvQixDQUFDLGNBQWMsQ0FBQzthQUMzRDtZQUVELElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxDQUFDLGNBQWMsRUFBRSxJQUFJLENBQUM7WUFDdkMsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUMsS0FBSyxJQUFJLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7WUFFakYsSUFBSSxDQUFDLE1BQU0sR0FBRyxJQUFJLENBQUMsY0FBYyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsbUJBQW1CLENBQUMsT0FBTyxDQUFDO1NBQzlFO2FBQU07WUFDTCxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksRUFBRTtnQkFDdkIsTUFBTSxJQUFJLEtBQUssQ0FBQyxzR0FBc0csQ0FBQyxDQUFDO2FBQ3pIO1lBQ0QsSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLEVBQUU7Z0JBQ3ZCLE1BQU0sSUFBSSxLQUFLLENBQUMsc0dBQXNHLENBQUMsQ0FBQzthQUN6SDtZQUVELElBQUksQ0FBQyxhQUFhLEdBQUcsS0FBSyxDQUFDLGFBQWEsSUFBSSxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLHVCQUF1QixFQUFFO2dCQUMvRixHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7Z0JBQ2QsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixLQUFLLEtBQUs7YUFDbkQsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLEdBQUcsQ0FBQyxXQUFXLENBQUMsRUFBRSxjQUFjLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1lBQ2xGLElBQUksQ0FBQyxjQUFjLEdBQUcsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUM7WUFDM0MsV0FBSSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxHQUFHLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7WUFFNUMsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUMsSUFBSSxJQUFJLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO2dCQUM1RCxRQUFRLEVBQUUsbUJBQVksQ0FBQyxrQkFBa0I7Z0JBQ3pDLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxtQkFBbUIsQ0FBQzthQUN6RCxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUM7WUFFakMsTUFBTSxVQUFVLEdBQUcsSUFBSSxHQUFHLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLGlCQUFpQixFQUFFO2dCQUNyRSxLQUFLLEVBQUUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQzthQUM1QixDQUFDLENBQUM7WUFFSCx5QkFBeUI7WUFDekIsTUFBTSxXQUFXLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDdEQsSUFBSSxDQUFDLFNBQVMsR0FBRyxLQUFLLENBQUMsUUFBUSxJQUFJLFdBQVcsQ0FBQyxRQUFRLENBQUM7WUFDeEQsTUFBTSxhQUFhLEdBQUcsV0FBSSxDQUFDLE1BQU0sQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxTQUFFLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFTLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUM7WUFDekYsTUFBTSxtQkFBbUIsR0FBRyxXQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxjQUFlLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUU3RyxZQUFZLEdBQUcsSUFBSSw4Q0FBc0IsQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO2dCQUM5RCxPQUFPLEVBQUUsV0FBVyxDQUFDLE9BQU87Z0JBQzVCLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTztnQkFDdEIsWUFBWSxFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsUUFBUSxFQUFFO2dCQUMzQyxrQkFBa0IsRUFBRSxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLGtCQUFrQixLQUFLLFVBQVUsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO2dCQUM3SCxjQUFjLEVBQUUsbUJBQW1CO2dCQUNuQyxrQkFBa0IsRUFBRSxVQUFVLENBQUMsR0FBRztnQkFDbEMsUUFBUSxFQUFFLGFBQWE7Z0JBQ3ZCLHdCQUF3QixFQUFFLEtBQUssQ0FBQyx3QkFBd0I7Z0JBQ3hELFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUztnQkFDMUIsbUJBQW1CLEVBQUUsQ0FBQyxLQUFLLENBQUMsWUFBWSxLQUFLLFNBQVMsQ0FBQyxDQUFDO29CQUN0RCw2QkFBNkIsQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7YUFDdkUsQ0FBQyxDQUFDO1lBRUgsWUFBWSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzNDLElBQUksQ0FBQyxNQUFNLEdBQUcsV0FBVyxDQUFDLE1BQU0sQ0FBQztTQUNsQztRQUVELDREQUE0RDtRQUM1RCxNQUFNLGVBQWUsR0FBRyxLQUFLLENBQUMsZUFBZSxDQUFDO1FBQzlDLE1BQU0sV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLElBQUksQ0FBQyxDQUFDO1FBQzNDLE1BQU0sV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLElBQUksZUFBZSxJQUFJLElBQUksQ0FBQyxHQUFHLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBRXJGLG1CQUFZLENBQUMsV0FBVyxFQUFFLFdBQVcsRUFBRSxDQUFDLEdBQUcsRUFBRSxHQUFHLEVBQUUsRUFBRTtZQUNsRCxJQUFJLEdBQUcsR0FBRyxHQUFHLEVBQUU7Z0JBQ2IsTUFBTSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsR0FBRywrQkFBK0IsR0FBRyxHQUFHLENBQUMsQ0FBQzthQUMzRTtRQUNILENBQUMsQ0FBQyxDQUFDO1FBQ0gsbUJBQVksQ0FBQyxlQUFlLEVBQUUsV0FBVyxFQUFFLENBQUMsT0FBTyxFQUFFLEdBQUcsRUFBRSxFQUFFO1lBQzFELElBQUksT0FBTyxLQUFLLFNBQVMsRUFBRTtnQkFBRSxPQUFPO2FBQUU7WUFDdEMsSUFBSSxPQUFPLEdBQUcsR0FBRyxFQUFFO2dCQUNqQixNQUFNLElBQUksS0FBSyxDQUFDLDRCQUE0QixHQUFHLHlCQUF5QixPQUFPLEdBQUcsQ0FBQyxDQUFDO2FBQ3JGO1FBQ0gsQ0FBQyxDQUFDLENBQUM7UUFDSCxtQkFBWSxDQUFDLGVBQWUsRUFBRSxXQUFXLEVBQUUsQ0FBQyxPQUFPLEVBQUUsR0FBRyxFQUFFLEVBQUU7WUFDMUQsSUFBSSxPQUFPLEtBQUssU0FBUyxFQUFFO2dCQUFFLE9BQU87YUFBRTtZQUN0QyxJQUFJLEdBQUcsR0FBRyxPQUFPLEVBQUU7Z0JBQ2pCLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0NBQWdDLE9BQU8scUJBQXFCLEdBQUcsR0FBRyxDQUFDLENBQUM7YUFDckY7UUFDSCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksZUFBZSxLQUFLLFNBQVMsRUFBRTtZQUNqQyxrQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxVQUFVLENBQUMscUtBQXFLLENBQUMsQ0FBQztTQUN4TTtRQUVELElBQUksQ0FBQyxtQkFBbUIsR0FBRyxLQUFLLENBQUMsbUJBQW1CLENBQUM7UUFDckQsaUlBQWlJO1FBQ2pJLElBQUksSUFBSSxDQUFDLG1CQUFtQixJQUFJLENBQUMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLFlBQVksRUFBRTtZQUN0RSxDQUFDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxTQUFTLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFDNUMsQ0FBQyxJQUFJLENBQUMsbUJBQW1CLENBQUMsU0FBUyxFQUFFLEdBQUcsS0FBSyxJQUFJLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxTQUFTLEVBQUUsR0FBRyxRQUFRLENBQUMsRUFBRTtZQUNuRyxNQUFNLElBQUksS0FBSyxDQUFDLGdFQUFnRSxDQUFDLENBQUM7U0FDbkY7UUFFRCxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsSUFBSSxLQUFLLENBQUMsYUFBYSxFQUFFO1lBQ25ELE1BQU0sSUFBSSxLQUFLLENBQUMsNkhBQTZILENBQUMsQ0FBQztTQUNoSjtRQUVELElBQUksS0FBSyxDQUFDLGtCQUFrQixFQUFFO1lBQzVCLElBQUksQ0FBQyxhQUFhLEdBQUcsQ0FBQztvQkFDcEIsS0FBSyxFQUFFLEtBQUssQ0FBQyxrQkFBa0I7aUJBQ2hDLENBQUMsQ0FBQztTQUNKO1FBRUQsSUFBSSxLQUFLLENBQUMsYUFBYSxFQUFFO1lBQ3ZCLElBQUksQ0FBQyxhQUFhLEdBQUcsS0FBSyxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO2dCQUNsRCxLQUFLLEVBQUUsRUFBRSxDQUFDLEtBQUs7Z0JBQ2YsYUFBYSxFQUFFLEVBQUUsQ0FBQyxhQUFhLElBQUksYUFBYSxDQUFDLEdBQUc7YUFDckQsQ0FBQyxDQUFDLENBQUM7U0FDTDtRQUVELE1BQU0sRUFBRSxTQUFTLEVBQUUsU0FBUyxFQUFFLEdBQUcsS0FBSyxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQzNFLE1BQU0sUUFBUSxHQUE2QjtZQUN6QyxvQkFBb0IsRUFBRSxJQUFJLENBQUMsWUFBWTtZQUN2QyxRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVEsRUFBRSxTQUFTLEVBQUUsQ0FBQyxRQUFRLEVBQUU7WUFDaEQsT0FBTyxFQUFFLG1CQUFZLENBQUMsZUFBZSxDQUFDLFdBQVcsQ0FBQztZQUNsRCxPQUFPLEVBQUUsbUJBQVksQ0FBQyxlQUFlLENBQUMsV0FBVyxDQUFDO1lBQ2xELGVBQWUsRUFBRSxlQUFlLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxtQkFBWSxDQUFDLGVBQWUsQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUMxRyxpQkFBaUIsRUFBRSxXQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsRUFBRSxFQUFFLEVBQUUsU0FBUyxFQUFFLElBQUksRUFBRSxDQUFDO1lBQzVGLGVBQWUsRUFBRSxXQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxlQUFlLEVBQUUsRUFBRSxFQUFFLFNBQVMsRUFBRSxJQUFJLEVBQUUsQ0FBQztZQUN4RiwwQkFBMEIsRUFBRSxJQUFJLENBQUMsK0JBQStCLEVBQUU7WUFDbEUsaUJBQWlCLEVBQUUsV0FBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsdUJBQXVCLEVBQUUsRUFBRSxDQUFDO1lBQzlFLGlCQUFpQixFQUFFLFNBQVM7WUFDNUIsZUFBZSxFQUFFLEtBQUssQ0FBQyxXQUFXLElBQUksS0FBSyxDQUFDLFdBQVcsQ0FBQyxJQUFJO1lBQzVELHNCQUFzQixFQUFFLEtBQUssQ0FBQyxXQUFXLElBQUksS0FBSyxDQUFDLFdBQVcsQ0FBQyxXQUFXLElBQUksS0FBSyxDQUFDLFdBQVcsQ0FBQyxXQUFXLENBQUMsU0FBUyxFQUFFO1lBQ3ZILG1CQUFtQixFQUFFLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ2hHLGdDQUFnQyxFQUFFLFdBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLGdDQUFnQyxFQUFFLENBQUM7WUFDcEcsbUJBQW1CLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjtZQUM5QyxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxZQUFZLEVBQUUsS0FBSyxDQUFDLGNBQWMsRUFBRSxLQUFLLENBQUMsb0JBQW9CLENBQUM7U0FDMUYsQ0FBQztRQUVGLElBQUksQ0FBQyxTQUFTLElBQUksS0FBSyxDQUFDLHdCQUF3QixFQUFFO1lBQ2hELE1BQU0sSUFBSSxLQUFLLENBQUMsd0hBQXdILENBQUMsQ0FBQztTQUMzSTtRQUVELElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxJQUFJLDJDQUFtQixDQUFDLElBQUksRUFBRSxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDdkUsSUFBSSxDQUFDLG9CQUFvQixHQUFHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxDQUFDO1lBQ3BGLElBQUksQ0FBQyxtQkFBbUIsR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVMsQ0FBQztnQkFDbEQsT0FBTyxFQUFFLGFBQWE7Z0JBQ3RCLFFBQVEsRUFBRSx5Q0FBeUM7Z0JBQ25ELFlBQVksRUFBRSxJQUFJLENBQUMsb0JBQW9CO2FBQ3hDLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQztRQUUvQyxJQUFJLENBQUMsbUJBQW1CLENBQUMsS0FBSyxFQUFFLEVBQUUsZUFBZSxFQUFFLFdBQVcsRUFBRSxDQUFDLENBQUM7UUFDbEUsSUFBSSxLQUFLLENBQUMsSUFBSSxFQUFFO1lBQ2QsSUFBSSxDQUFDLHVCQUF1QixDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDO1NBQzdEO1FBRUQsSUFBSSxDQUFDLFNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO1FBRWpDLElBQUksS0FBSyxDQUFDLGFBQWEsRUFBRTtZQUN2QixjQUFPLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLEdBQUcsQ0FBQyxJQUFJLDZDQUFtQyxFQUFFLENBQUMsQ0FBQztTQUNqRTtLQUNGO0lBL1BNLE1BQU0sQ0FBQyx3QkFBd0IsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxvQkFBNEI7UUFDL0YsTUFBTSxNQUFPLFNBQVEsb0JBQW9CO1lBQXpDOztnQkFDUyx5QkFBb0IsR0FBRyxvQkFBb0IsQ0FBQztnQkFDNUMsd0JBQW1CLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7b0JBQ3BELE9BQU8sRUFBRSxhQUFhO29CQUN0QixRQUFRLEVBQUUseUNBQXlDO29CQUNuRCxZQUFZLEVBQUUsSUFBSSxDQUFDLG9CQUFvQjtpQkFDeEMsQ0FBQyxDQUFDO2dCQUNhLFdBQU0sR0FBRyxHQUFHLENBQUMsbUJBQW1CLENBQUMsT0FBTyxDQUFDO1lBQzNELENBQUM7U0FBQTtRQUVELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0tBQzlCO0lBcVBEOzs7OztPQUtHO0lBQ0ksZ0JBQWdCLENBQUMsYUFBaUM7Ozs7Ozs7Ozs7UUFDdkQsSUFBSSxDQUFDLElBQUksQ0FBQyxjQUFjLEVBQUU7WUFDeEIsTUFBTSxJQUFJLEtBQUssQ0FBQywrRkFBK0YsQ0FBQyxDQUFDO1NBQ2xIO1FBRUQsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUM7S0FDekM7SUFFRDs7T0FFRztJQUNJLGlCQUFpQixDQUFDLFlBQThCOzs7Ozs7Ozs7O1FBQ3JELElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLGdCQUFnQixDQUFDLENBQUM7S0FDNUQ7SUFFRDs7T0FFRztJQUNJLDhCQUE4QixDQUFDLFdBQTBDOzs7Ozs7Ozs7O1FBQzlFLElBQUksSUFBSSxDQUFDLGNBQWMsS0FBSyxTQUFTLEVBQUU7WUFDckMsTUFBTSxJQUFJLEtBQUssQ0FBQyxpREFBaUQsQ0FBQyxDQUFDO1NBQ3BFO1FBRUQsSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBQ3RELElBQUksV0FBVyxZQUFZLEtBQUssQ0FBQyxzQkFBc0IsRUFBRTtZQUN2RCx1RUFBdUU7WUFDdkUsMkVBQTJFO1lBQzNFLElBQUksQ0FBQyxjQUFjLEdBQUcsV0FBVyxDQUFDO1NBQ25DO1FBRUQsV0FBVyxDQUFDLG1CQUFtQixDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sRUFBRSxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxRQUFRLEVBQUUsQ0FBQztLQUNsRDtJQUVEOztPQUVHO0lBQ0ksMEJBQTBCLENBQUMsV0FBc0M7Ozs7Ozs7Ozs7UUFDdEUsSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBQ3RELE9BQU8sRUFBRSxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxRQUFRLEVBQUUsQ0FBQztLQUNsRDtJQUVNLFdBQVcsQ0FBQyxHQUFHLFFBQWtCO1FBQ3RDLElBQUksQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLEdBQUcsUUFBUSxDQUFDLENBQUM7S0FDeEM7SUFFRDs7T0FFRztJQUNJLGVBQWUsQ0FBQyxTQUE4Qjs7Ozs7Ozs7OztRQUNuRCxJQUFJLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLFNBQVMsQ0FBQyxDQUFDO0tBQzNDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksdUJBQXVCLENBQUMsSUFBNEIsRUFBRSxVQUEwQyxFQUFFOzs7Ozs7Ozs7OztRQUN2RyxJQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLFVBQVUsQ0FBQyxjQUFjLEVBQUUsY0FBYyxFQUFFO1lBQ3BFLE1BQU0sSUFBSSxLQUFLLENBQUMsbUhBQW1ILENBQUMsQ0FBQztTQUN0STtRQUVELElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLGdCQUFnQixFQUFFO1lBQ2pDLFFBQVEsRUFBRSxJQUFJLENBQUMsTUFBTTtZQUNyQixZQUFZLEVBQUUsSUFBSSxDQUFDLElBQUk7WUFDdkIsUUFBUSxFQUFFLElBQUksQ0FBQyxRQUFRO1lBQ3ZCLFVBQVUsRUFBRSxPQUFPLENBQUMsVUFBVTtZQUM5QixnQkFBZ0IsRUFBRSxPQUFPLENBQUMsZ0JBQWdCO1lBQzFDLFFBQVEsRUFBRSxPQUFPLENBQUMsUUFBUTtZQUMxQixjQUFjLEVBQUUsT0FBTyxDQUFDLGNBQWM7WUFDdEMsV0FBVyxFQUFFLE9BQU8sQ0FBQyxXQUFXO1lBQ2hDLFVBQVUsRUFBRSxPQUFPLENBQUMsVUFBVTtTQUMvQixDQUFDLENBQUM7S0FDSjtJQUVEOztPQUVHO0lBQ0ksOEJBQThCO1FBQ25DLElBQUksQ0FBQyxnQ0FBZ0MsR0FBRyxJQUFJLENBQUM7S0FDOUM7SUFFRDs7T0FFRztJQUNJLG1DQUFtQztRQUN4QyxPQUFPLElBQUksQ0FBQyxnQ0FBZ0MsS0FBSyxJQUFJLENBQUM7S0FDdkQ7SUFFRDs7T0FFRztJQUNILElBQVcsV0FBVztRQUNwQixJQUFJLElBQUksQ0FBQyxZQUFZLEVBQUU7WUFDckIsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDO1NBQzFCO1FBRUQsSUFBSSxJQUFJLENBQUMsY0FBYyxFQUFFO1lBQ3ZCLE9BQU8sSUFBSSxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQUM7U0FDeEM7UUFFRCxNQUFNLElBQUksS0FBSyxDQUFDLDBHQUEwRyxDQUFDLENBQUM7S0FDN0g7SUFFRDs7OztPQUlHO0lBQ0gsSUFBVyxRQUFRO1FBQ2pCLElBQUksSUFBSSxDQUFDLFNBQVMsRUFBRTtZQUNsQixPQUFPLElBQUksQ0FBQyxTQUFTLENBQUM7U0FDdkI7UUFFRCxJQUFJLElBQUksQ0FBQyxjQUFjLEVBQUUsUUFBUSxFQUFFO1lBQ2pDLE9BQU8sSUFBSSxDQUFDLGNBQWMsQ0FBQyxRQUFRLENBQUM7U0FDckM7UUFFRCxNQUFNLElBQUksS0FBSyxDQUFDLDZEQUE2RCxDQUFDLENBQUM7S0FDaEY7SUFFRDs7OztPQUlHO0lBQ0gsSUFBVyxJQUFJO1FBQ2IsSUFBSSxJQUFJLENBQUMsS0FBSyxFQUFFO1lBQ2QsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDO1NBQ25CO1FBRUQsTUFBTSxJQUFJLEtBQUssQ0FBQywyRUFBMkUsQ0FBQyxDQUFDO0tBQzlGO0lBRU8sK0JBQStCLENBQUMsS0FBNEI7UUFDbEUsSUFBSSxLQUFLLENBQUMsWUFBWSxFQUFFO1lBQ3RCLE1BQU0sSUFBSSxLQUFLLENBQUMscUdBQXFHLENBQUMsQ0FBQztTQUN4SDtRQUNELElBQUksS0FBSyxDQUFDLFlBQVksRUFBRTtZQUN0QixNQUFNLElBQUksS0FBSyxDQUFDLHFHQUFxRyxDQUFDLENBQUM7U0FDeEg7UUFDRCxJQUFJLEtBQUssQ0FBQyxJQUFJLEVBQUU7WUFDZCxNQUFNLElBQUksS0FBSyxDQUFDLDZGQUE2RixDQUFDLENBQUM7U0FDaEg7UUFDRCxJQUFJLEtBQUssQ0FBQyxRQUFRLEVBQUU7WUFDbEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxpR0FBaUcsQ0FBQyxDQUFDO1NBQ3BIO1FBQ0QsSUFBSSxLQUFLLENBQUMsYUFBYSxFQUFFO1lBQ3ZCLE1BQU0sSUFBSSxLQUFLLENBQUMsc0dBQXNHLENBQUMsQ0FBQztTQUN6SDtRQUNELElBQUksS0FBSyxDQUFDLE9BQU8sRUFBRTtZQUNqQixNQUFNLElBQUksS0FBSyxDQUFDLGdHQUFnRyxDQUFDLENBQUM7U0FDbkg7UUFDRCxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsRUFBRTtZQUM1QixNQUFNLElBQUksS0FBSyxDQUFDLDJHQUEyRyxDQUFDLENBQUM7U0FDOUg7UUFDRCxJQUFJLEtBQUssQ0FBQyx3QkFBd0IsRUFBRTtZQUNsQyxNQUFNLElBQUksS0FBSyxDQUFDLGlIQUFpSCxDQUFDLENBQUM7U0FDcEk7UUFDRCxJQUFJLEtBQUssQ0FBQyxTQUFTLEVBQUU7WUFDbkIsTUFBTSxJQUFJLEtBQUssQ0FBQyxrR0FBa0csQ0FBQyxDQUFDO1NBQ3JIO1FBQ0QsSUFBSSxLQUFLLENBQUMsWUFBWSxFQUFFO1lBQ3RCLE1BQU0sSUFBSSxLQUFLLENBQUMscUdBQXFHLENBQUMsQ0FBQztTQUN4SDtLQUNGO0lBRUQ7O09BRUc7SUFDSyxtQkFBbUIsQ0FBQyxLQUE0QixFQUFFLGFBQW1DO1FBQzNGLHFFQUFxRTtRQUNyRSxNQUFNLFFBQVEsR0FBdUM7WUFDbkQsWUFBWTtZQUNaLDRCQUE0QjtZQUM1QixxQkFBcUI7WUFDckIsdUJBQXVCO1lBQ3ZCLDhDQUE4QztTQUMvQyxDQUFDO1FBQ0YsS0FBSyxNQUFNLElBQUksSUFBSSxRQUFRLEVBQUU7WUFDM0IsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLElBQUksS0FBSyxDQUFDLFlBQVksQ0FBQyxJQUFJLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxTQUFTLEVBQUU7Z0JBQ3RFLE1BQU0sSUFBSSxLQUFLLENBQUMsNENBQTRDLElBQUksNkNBQTZDLENBQUMsQ0FBQzthQUNoSDtTQUNGO1FBRUQsbUZBQW1GO1FBQ25GLEtBQUssR0FBRztZQUNOLEdBQUcsS0FBSztZQUNSLFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWSxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsWUFBWSxDQUFDLGFBQWEsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7U0FDNUYsQ0FBQztRQUVGLElBQUksS0FBSyxDQUFDLE9BQU8sSUFBSSxLQUFLLENBQUMsWUFBWSxFQUFFO1lBQ3ZDLElBQUksQ0FBQyw0QkFBNEIsQ0FBQyxLQUFLLEVBQUUsYUFBYSxDQUFDLENBQUM7U0FDekQ7YUFBTTtZQUNMLElBQUksQ0FBQywrQkFBK0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztTQUM3QztRQUVELHNGQUFzRjtRQUN0Rix3RkFBd0Y7UUFDeEYsbUVBQW1FO1FBQ25FLElBQUksS0FBSyxDQUFDLDhCQUE4QixLQUFLLEtBQUssRUFBRTtZQUNsRCxJQUFJLENBQUMsZ0JBQWdCLENBQUMsVUFBVSxDQUFDLFlBQVksR0FBRztnQkFDOUMsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsVUFBVSxDQUFDLFlBQVk7Z0JBQ2hELDBCQUEwQixFQUFFLEVBQUUsbUNBQW1DLEVBQUUsSUFBSSxFQUFFO2FBQzFFLENBQUM7U0FDSDtRQUVELElBQUksS0FBSyxDQUFDLE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUU7WUFDaEMseUVBQXlFO1lBQ3pFLHlFQUF5RTtZQUN6RSx1RUFBdUU7WUFDdkUsc0RBQXNEO1lBQ3RELEVBQUU7WUFDRiwyRUFBMkU7WUFDM0UsNEVBQTRFO1lBQzVFLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUMzQyxPQUFPLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQztnQkFDMUMsU0FBUyxFQUFFLENBQUMsVUFBRyxDQUFDLFFBQVEsQ0FBQzthQUMxQixDQUFDLENBQUMsQ0FBQztTQUNMO0tBQ0Y7SUFFRDs7T0FFRztJQUNLLDRCQUE0QixDQUFDLEtBQTRCLEVBQUUsYUFBbUM7UUFDcEcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLFVBQVUsQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDLE9BQU8sRUFBRSxvQkFBb0IsQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUNyRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsVUFBVSxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxFQUFFLG1CQUFtQixDQUFDO1lBQ3RGLGNBQWMsRUFBRSxJQUFJLENBQUMsZ0JBQWdCLENBQUMsVUFBVSxDQUFDLGNBQWM7U0FDaEUsQ0FBQyxDQUFDO0tBQ0o7SUFFTywrQkFBK0IsQ0FBQyxLQUE0QjtRQUNsRSxJQUFJLEtBQUssQ0FBQyxVQUFVLEtBQUssVUFBVSxDQUFDLGdCQUFnQixFQUFFO1lBQ3BELElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVLENBQUMsWUFBWSxHQUFHO2dCQUM5QyxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVLENBQUMsWUFBWTtnQkFDaEQsMEJBQTBCLEVBQUU7b0JBQzFCLFdBQVcsRUFBRSxJQUFJO2lCQUNsQjthQUNGLENBQUM7WUFFRixJQUFJLEtBQUssQ0FBQyw0Q0FBNEMsS0FBSyxTQUFTLEVBQUU7Z0JBQ3BFLG1GQUFtRjtnQkFDbkYsc0ZBQXNGO2dCQUN0RixxQkFBcUI7Z0JBQ3JCLEVBQUU7Z0JBQ0YsbUdBQW1HO2dCQUNuRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsVUFBVSxDQUFDLGNBQWMsR0FBRztvQkFDaEQsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsVUFBVSxDQUFDLGNBQWM7b0JBQ2xELHlCQUF5QixFQUFFO3dCQUN6Qiw2QkFBNkIsRUFBRSxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsNENBQTRDLENBQUM7cUJBQ3RHO2lCQUNGLENBQUM7YUFDSDtTQUNGO2FBQU0sSUFBSSxLQUFLLENBQUMsVUFBVSxLQUFLLFVBQVUsQ0FBQyxjQUFjLEVBQUU7WUFDekQsSUFBSSxDQUFDLGdCQUFnQixDQUFDLFVBQVUsQ0FBQyxZQUFZLEdBQUc7Z0JBQzlDLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLFVBQVUsQ0FBQyxZQUFZO2dCQUNoRCx3QkFBd0IsRUFBRSx5QkFBeUIsQ0FBQyxLQUFLLENBQUMsMEJBQTBCLENBQUM7YUFDdEYsQ0FBQztTQUNIO1FBRUQsSUFBSSxLQUFLLENBQUMsbUJBQW1CLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyxxQkFBcUIsS0FBSyxTQUFTLEVBQUU7WUFDeEYsSUFBSSxDQUFDLGdCQUFnQixDQUFDLFVBQVUsQ0FBQyxjQUFjLEdBQUc7Z0JBQ2hELEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLFVBQVUsQ0FBQyxjQUFjO2dCQUNsRCxjQUFjLEVBQUU7b0JBQ2QsS0FBSyxFQUFFLEtBQUssQ0FBQyxtQkFBbUI7b0JBQ2hDLE9BQU8sRUFBRSxLQUFLLENBQUMscUJBQXFCLElBQUksS0FBSyxDQUFDLHFCQUFxQixDQUFDLFdBQVcsRUFBRTtpQkFDbEY7YUFDRixDQUFDO1NBQ0g7S0FDRjtJQUVPLCtCQUErQjtRQUNyQyxJQUFJLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUNuQyxPQUFPLFNBQVMsQ0FBQztTQUNsQjtRQUVELE9BQU8sSUFBSSxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1lBQzdDLFFBQVEsRUFBRSxZQUFZLENBQUMsS0FBSyxDQUFDLFFBQVE7WUFDckMsaUJBQWlCLEVBQUUsWUFBWSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsWUFBWSxDQUFDLGFBQWEsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsTUFBTTtTQUM3RyxDQUFDLENBQUMsQ0FBQztLQUNMO0lBRU8sdUJBQXVCO1FBQzdCLElBQUksSUFBSSxDQUFDLFlBQVksQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ2xDLE9BQU8sU0FBUyxDQUFDO1NBQ2xCO1FBRUQsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLENBQUM7WUFDckMsV0FBVyxFQUFFLFNBQVM7WUFDdEIsT0FBTyxFQUFFLEtBQUssQ0FBQyxRQUFRLEVBQUUsSUFBSSxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVM7U0FDdkYsQ0FBQyxDQUFDLENBQUM7S0FDTDtJQUVPLGlCQUFpQixDQUFDLFlBQXFDLEVBQUUsY0FBb0MsRUFBRSxvQkFBMkM7UUFJaEosSUFBSSxZQUFZLEVBQUU7WUFDaEIsT0FBTztnQkFDTCx1QkFBdUIsRUFBRSxZQUFZLENBQUMsR0FBRzthQUMxQyxDQUFDO1NBQ0g7UUFFRCxJQUFJLGNBQWMsRUFBRTtZQUNsQixPQUFPO2dCQUNMLGNBQWMsRUFBRSxJQUFJLENBQUMscUNBQXFDLENBQUMsY0FBYyxDQUFDO2FBQzNFLENBQUM7U0FDSDtRQUVELElBQUksb0JBQW9CLEVBQUU7WUFDeEIsSUFBSSxxQkFBcUIsR0FBa0UsU0FBUyxDQUFDO1lBQ3JHLElBQUksb0JBQW9CLENBQUMscUJBQXFCLEVBQUU7Z0JBQzlDLE1BQU0sSUFBSSxHQUFHLG9CQUFvQixDQUFDLHFCQUFxQixDQUFDO2dCQUN4RCxxQkFBcUIsR0FBRztvQkFDdEIsMEJBQTBCLEVBQUUsSUFBSSxDQUFDLDBCQUEwQixFQUFFLFFBQVEsRUFBRTtvQkFDdkUsb0JBQW9CLEVBQUUsSUFBSSxDQUFDLG9CQUFvQjtvQkFDL0MsbUNBQW1DLEVBQUUsSUFBSSxDQUFDLG1DQUFtQztvQkFDN0Usc0JBQXNCLEVBQUUsSUFBSSxDQUFDLHNCQUFzQixFQUFFLFFBQVEsRUFBRTtvQkFDL0QsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLGlCQUFpQjtvQkFDekMsWUFBWSxFQUFFLElBQUksQ0FBQyxZQUFZO2lCQUNoQyxDQUFDO2FBQ0g7WUFDRCxPQUFPO2dCQUNMLG9CQUFvQixFQUFFO29CQUNwQixxQkFBcUI7b0JBQ3JCLGNBQWMsRUFBRTt3QkFDZCwyQkFBMkIsRUFBRSxJQUFJLENBQUMscUNBQXFDLENBQUMsb0JBQW9CLENBQUMsY0FBYyxDQUFDO3dCQUM1RyxHQUFHLENBQUMsb0JBQW9CLENBQUMsdUJBQXVCLENBQUMsQ0FBQyxDQUFDOzRCQUNqRCxTQUFTLEVBQUUsb0JBQW9CLENBQUMsdUJBQXVCLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxFQUFFO2dDQUNyRSxJQUFJLFFBQVEsQ0FBQyxnQkFBZ0IsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLFFBQVEsQ0FBQyxnQkFBZ0IsRUFBRTtvQ0FDcEcsTUFBTSxJQUFJLEtBQUssQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO2lDQUM5QztnQ0FDRCxPQUFPO29DQUNMLFlBQVksRUFBRSxRQUFRLENBQUMsWUFBWSxDQUFDLFFBQVEsRUFBRTtvQ0FDOUMsMkJBQTJCLEVBQUUsUUFBUSxDQUFDLGNBQWM7d0NBQ2xELENBQUMsQ0FBQyxJQUFJLENBQUMscUNBQXFDLENBQUMsUUFBUSxDQUFDLGNBQWMsQ0FBQzt3Q0FDckUsQ0FBQyxDQUFDLFNBQVM7b0NBQ2IsZ0JBQWdCLEVBQUUsUUFBUSxDQUFDLGdCQUFnQixFQUFFLFFBQVEsRUFBRTtpQ0FDeEQsQ0FBQzs0QkFDSixDQUFDLENBQUM7eUJBQ0gsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO3FCQUNSO2lCQUNGO2FBQ0YsQ0FBQztTQUNIO1FBRUQsTUFBTSxJQUFJLEtBQUssQ0FBQyxvRkFBb0YsQ0FBQyxDQUFDO0tBQ3ZHO0lBRU8scUNBQXFDLENBQUMsY0FBbUM7UUFDL0UsSUFBSSxjQUFjLENBQUMsZ0JBQWdCLEVBQUU7WUFDbkMsT0FBTztnQkFDTCxnQkFBZ0IsRUFBRSxjQUFjLENBQUMsZ0JBQWdCO2dCQUNqRCxPQUFPLEVBQUUsY0FBYyxDQUFDLGFBQWE7YUFDdEMsQ0FBQztTQUNIO2FBQU07WUFDTCxPQUFPO2dCQUNMLGtCQUFrQixFQUFFLGNBQWMsQ0FBQyxrQkFBa0I7Z0JBQ3JELE9BQU8sRUFBRSxjQUFjLENBQUMsYUFBYTthQUN0QyxDQUFDO1NBQ0g7S0FDRjs7QUE3bkJILDRDQThuQkM7OztBQUVEOzs7O0dBSUc7QUFDSCxJQUFZLFVBaUJYO0FBakJELFdBQVksVUFBVTtJQUNwQjs7T0FFRztJQUNILDJCQUFhLENBQUE7SUFFYjs7OztPQUlHO0lBQ0gsMENBQTRCLENBQUE7SUFFNUI7O09BRUc7SUFDSCw4Q0FBZ0MsQ0FBQTtBQUNsQyxDQUFDLEVBakJXLFVBQVUsR0FBVixrQkFBVSxLQUFWLGtCQUFVLFFBaUJyQjtBQW1CRDs7R0FFRztBQUNILElBQVksWUF5Qlg7QUF6QkQsV0FBWSxZQUFZO0lBQ3RCOztPQUVHO0lBQ0gsbUVBQW1ELENBQUE7SUFFbkQ7O09BRUc7SUFDSCx5RUFBeUQsQ0FBQTtJQUV6RDs7T0FFRztJQUNILHFGQUFxRSxDQUFBO0lBRXJFOztPQUVHO0lBQ0gsK0VBQStELENBQUE7SUFFL0Q7O09BRUc7SUFDSCxtRUFBbUQsQ0FBQTtBQUNyRCxDQUFDLEVBekJXLFlBQVksR0FBWixvQkFBWSxLQUFaLG9CQUFZLFFBeUJ2QjtBQThFRDs7O0dBR0c7QUFDSCxNQUFhLGFBQWE7SUE4QnhCLFlBQVksR0FBRyxLQUFxQjs7Ozs7OytDQTlCekIsYUFBYTs7OztRQStCdEIsSUFBSSxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUM7S0FDckI7O0FBaENILHNDQWlDQzs7O0FBaENDOztHQUVHO0FBQ29CLG9CQUFNLEdBQUcsSUFBSSxhQUFhLENBQUMsWUFBWSxDQUFDLHFCQUFxQixFQUFFLFlBQVksQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0FBRTdIOztHQUVHO0FBQ29CLGlCQUFHLEdBQUcsSUFBSSxhQUFhLENBQUMsWUFBWSxDQUFDLGVBQWUsRUFDekUsWUFBWSxDQUFDLHFCQUFxQixFQUNsQyxZQUFZLENBQUMsa0JBQWtCLEVBQy9CLFlBQVksQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0FBRXpDOztHQUVHO0FBQ29CLDJCQUFhLEdBQUcsSUFBSSxhQUFhLENBQUMsWUFBWSxDQUFDLGVBQWUsRUFBRSxZQUFZLENBQUMscUJBQXFCLENBQUMsQ0FBQztBQUUzSDs7R0FFRztBQUNvQixnQ0FBa0IsR0FBRyxJQUFJLGFBQWEsQ0FBQyxZQUFZLENBQUMsa0JBQWtCLEVBQUUsWUFBWSxDQUFDLHdCQUF3QixDQUFDLENBQUM7QUFheEksSUFBWSxjQVNYO0FBVEQsV0FBWSxjQUFjO0lBQ3hCLG1DQUFpQixDQUFBO0lBQ2pCLHlDQUF1QixDQUFBO0lBQ3ZCLDhDQUE0QixDQUFBO0lBQzVCLHdEQUFzQyxDQUFBO0lBQ3RDLDhDQUE0QixDQUFBO0lBQzVCLDBEQUF3QyxDQUFBO0lBQ3hDLHdEQUFzQyxDQUFBO0lBQ3RDLDREQUEwQyxDQUFBO0FBQzVDLENBQUMsRUFUVyxjQUFjLEdBQWQsc0JBQWMsS0FBZCxzQkFBYyxRQVN6QjtBQUVELHNEQUFzRDtBQUN0RCw2RkFBNkY7QUFDN0YsTUFBTSx5QkFBeUIsR0FBRyxDQUFDLGNBQWMsQ0FBQyxZQUFZLEVBQUUsY0FBYyxDQUFDLGlCQUFpQixFQUFFLGNBQWMsQ0FBQyxZQUFZO0lBQzNILGNBQWMsQ0FBQyxrQkFBa0IsRUFBRSxjQUFjLENBQUMsaUJBQWlCLENBQUMsQ0FBQztBQTBCdkU7O0dBRUc7QUFDSCxNQUFhLFdBQVc7SUFvQnRCLFlBQW9DLElBQVksRUFBa0IsV0FBc0I7UUFBcEQsU0FBSSxHQUFKLElBQUksQ0FBUTtRQUFrQixnQkFBVyxHQUFYLFdBQVcsQ0FBVztLQUFLO0lBbkI3Rjs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLEdBQUcsQ0FBQyxVQUFpQyxFQUFFOzs7Ozs7Ozs7O1FBQ25ELE9BQU8sSUFBSSxXQUFXLENBQUMsZUFBZSxDQUFDLEdBQUcsRUFBRSxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUM7S0FDNUQ7SUFFRDs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyxHQUFHLENBQUMsT0FBOEI7Ozs7Ozs7Ozs7UUFDOUMsT0FBTyxJQUFJLFdBQVcsQ0FBQyxlQUFlLENBQUMsR0FBRyxFQUFFLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQztLQUM1RDs7QUFsQkgsa0NBcUJDOzs7QUFFRCxJQUFLLGVBR0o7QUFIRCxXQUFLLGVBQWU7SUFDbEIsOEJBQVcsQ0FBQTtJQUNYLDhCQUFXLENBQUE7QUFDYixDQUFDLEVBSEksZUFBZSxLQUFmLGVBQWUsUUFHbkI7QUFFRDs7R0FFRztBQUNILFNBQVMseUJBQXlCLENBQUMsU0FBcUMsRUFBRTtJQUN4RSxNQUFNLHFCQUFxQixHQUFHLE1BQU0sQ0FBQyw2QkFBNkIsS0FBSyxTQUFTLENBQUM7SUFDakYsTUFBTSxTQUFTLEdBQUcsTUFBTSxDQUFDLFNBQVMsSUFBSSxDQUFDLHFCQUFxQixDQUFDLENBQUMsQ0FBQyxlQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxlQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFFMUcsT0FBTztRQUNMLFlBQVksRUFBRSxNQUFNLENBQUMsWUFBWTtRQUNqQyxxQkFBcUIsRUFBRSxNQUFNLENBQUMscUJBQXFCO1FBQ25ELDZCQUE2QixFQUFFLGtCQUFrQixDQUFDLE1BQU0sQ0FBQyw2QkFBNkIsQ0FBQztRQUN2RixxQkFBcUI7UUFDckIsU0FBUyxFQUFFLFNBQVMsSUFBSSxTQUFTLENBQUMsV0FBVyxFQUFFO1FBQy9DLGdCQUFnQixFQUFFLE1BQU0sQ0FBQyxnQkFBZ0IsSUFBSSx5QkFBeUI7S0FDdkUsQ0FBQztBQUNKLENBQUM7QUFFRCxTQUFTLGtCQUFrQixDQUFDLENBQVU7SUFDcEMsSUFBSSxDQUFDLEtBQUssU0FBUyxJQUFJLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxDQUFDLEVBQUU7UUFBRSxPQUFPLENBQUMsQ0FBQztLQUFFO0lBQzFELE1BQU0sSUFBSSxLQUFLLENBQUMsdUNBQXVDLENBQUMsRUFBRSxDQUFDLENBQUM7QUFDOUQsQ0FBQztBQWtJRDs7Ozs7R0FLRztBQUNILFNBQVMsNkJBQTZCLENBQUMsU0FBb0IsRUFBRSxZQUEyQjtJQUN0RixPQUFPLFlBQVksQ0FBQyxHQUFHLENBQW9ELENBQUMsRUFBRSxVQUFVLEVBQUUsTUFBTSxFQUFFLGNBQWMsRUFBRSxFQUFFLEVBQUU7UUFDcEgsTUFBTSxFQUFFLFdBQVcsRUFBRSxTQUFTLEVBQUUsR0FBRyxFQUFFLEdBQUcsTUFBTSxDQUFDO1FBRS9DLElBQUksTUFBTSxLQUFLLDBCQUFpQixDQUFDLFVBQVUsSUFBSSxjQUFjLEtBQUssS0FBSyxFQUFFO1lBQ3ZFLE9BQU87Z0JBQ0wsVUFBVTtnQkFDVixRQUFRLEVBQUUsSUFBSTthQUNmLENBQUM7U0FDSDtRQUVELElBQUksR0FBRyxFQUFFO1lBQ1AsTUFBTSxFQUFFLElBQUksRUFBRSxVQUFVLEVBQUUsR0FBRyxHQUFHLENBQUM7WUFFakMsSUFBSSxDQUFDLElBQUksRUFBRTtnQkFDVCxJQUFJLFVBQVUsS0FBSyw0QkFBbUIsQ0FBQyxHQUFHLEVBQUU7b0JBQzFDLE1BQU0sSUFBSSxLQUFLLENBQUMsb0VBQW9FLENBQUMsQ0FBQztpQkFDdkY7YUFDRjtpQkFBTSxJQUFJLFVBQVUsS0FBSyw0QkFBbUIsQ0FBQyxHQUFHLEVBQUU7Z0JBQ2pELGtCQUFXLENBQUMsRUFBRSxDQUFDLFNBQVMsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxrRUFBa0UsQ0FBQyxDQUFDO2FBQzFHO1NBQ0Y7UUFFRCxPQUFPO1lBQ0wsVUFBVSxFQUFFLEdBQUcsRUFBRSxXQUFXO1NBQzdCLENBQUM7SUFDSixDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjbG91ZHdhdGNoIGZyb20gJy4uLy4uL2F3cy1jbG91ZHdhdGNoJztcbmltcG9ydCAqIGFzIGVjMiBmcm9tICcuLi8uLi9hd3MtZWMyJztcbmltcG9ydCAqIGFzIGVsYiBmcm9tICcuLi8uLi9hd3MtZWxhc3RpY2xvYWRiYWxhbmNpbmcnO1xuaW1wb3J0ICogYXMgZWxidjIgZnJvbSAnLi4vLi4vYXdzLWVsYXN0aWNsb2FkYmFsYW5jaW5ndjInO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJy4uLy4uL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgc25zIGZyb20gJy4uLy4uL2F3cy1zbnMnO1xuXG5pbXBvcnQge1xuICBBbm5vdGF0aW9ucyxcbiAgQXNwZWN0cyxcbiAgQXdzLFxuICBDZm5BdXRvU2NhbGluZ1JvbGxpbmdVcGRhdGUsIENmbkNyZWF0aW9uUG9saWN5LCBDZm5VcGRhdGVQb2xpY3ksXG4gIER1cmF0aW9uLCBGbiwgSVJlc291cmNlLCBMYXp5LCBQaHlzaWNhbE5hbWUsIFJlc291cmNlLCBTdGFjaywgVGFncyxcbiAgVG9rZW4sXG4gIFRva2VuaXphdGlvbiwgd2l0aFJlc29sdmVkLFxufSBmcm9tICcuLi8uLi9jb3JlJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgQXV0b1NjYWxpbmdHcm91cFJlcXVpcmVJbWRzdjJBc3BlY3QgfSBmcm9tICcuL2FzcGVjdHMnO1xuaW1wb3J0IHsgQ2ZuQXV0b1NjYWxpbmdHcm91cCwgQ2ZuQXV0b1NjYWxpbmdHcm91cFByb3BzLCBDZm5MYXVuY2hDb25maWd1cmF0aW9uIH0gZnJvbSAnLi9hdXRvc2NhbGluZy5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgQmFzaWNMaWZlY3ljbGVIb29rUHJvcHMsIExpZmVjeWNsZUhvb2sgfSBmcm9tICcuL2xpZmVjeWNsZS1ob29rJztcbmltcG9ydCB7IEJhc2ljU2NoZWR1bGVkQWN0aW9uUHJvcHMsIFNjaGVkdWxlZEFjdGlvbiB9IGZyb20gJy4vc2NoZWR1bGVkLWFjdGlvbic7XG5pbXBvcnQgeyBCYXNpY1N0ZXBTY2FsaW5nUG9saWN5UHJvcHMsIFN0ZXBTY2FsaW5nUG9saWN5IH0gZnJvbSAnLi9zdGVwLXNjYWxpbmctcG9saWN5JztcbmltcG9ydCB7IEJhc2VUYXJnZXRUcmFja2luZ1Byb3BzLCBQcmVkZWZpbmVkTWV0cmljLCBUYXJnZXRUcmFja2luZ1NjYWxpbmdQb2xpY3kgfSBmcm9tICcuL3RhcmdldC10cmFja2luZy1zY2FsaW5nLXBvbGljeSc7XG5pbXBvcnQgeyBUZXJtaW5hdGlvblBvbGljeSB9IGZyb20gJy4vdGVybWluYXRpb24tcG9saWN5JztcbmltcG9ydCB7IEJsb2NrRGV2aWNlLCBCbG9ja0RldmljZVZvbHVtZSwgRWJzRGV2aWNlVm9sdW1lVHlwZSB9IGZyb20gJy4vdm9sdW1lJztcbmltcG9ydCB7IFdhcm1Qb29sLCBXYXJtUG9vbE9wdGlvbnMgfSBmcm9tICcuL3dhcm0tcG9vbCc7XG5cbi8qKlxuICogTmFtZSB0YWcgY29uc3RhbnRcbiAqL1xuY29uc3QgTkFNRV9UQUc6IHN0cmluZyA9ICdOYW1lJztcblxuLyoqXG4gKiBUaGUgbW9uaXRvcmluZyBtb2RlIGZvciBpbnN0YW5jZXMgbGF1bmNoZWQgaW4gYW4gYXV0b3NjYWxpbmcgZ3JvdXBcbiAqL1xuZXhwb3J0IGVudW0gTW9uaXRvcmluZyB7XG4gIC8qKlxuICAgKiBHZW5lcmF0ZXMgbWV0cmljcyBldmVyeSA1IG1pbnV0ZXNcbiAgICovXG4gIEJBU0lDLFxuXG4gIC8qKlxuICAgKiBHZW5lcmF0ZXMgbWV0cmljcyBldmVyeSBtaW51dGVcbiAgICovXG4gIERFVEFJTEVELFxufVxuXG4vKipcbiAqIEJhc2ljIHByb3BlcnRpZXMgb2YgYW4gQXV0b1NjYWxpbmdHcm91cCwgZXhjZXB0IHRoZSBleGFjdCBtYWNoaW5lcyB0byBydW4gYW5kIHdoZXJlIHRoZXkgc2hvdWxkIHJ1blxuICpcbiAqIENvbnN0cnVjdHMgdGhhdCB3YW50IHRvIGNyZWF0ZSBBdXRvU2NhbGluZ0dyb3VwcyBjYW4gaW5oZXJpdFxuICogdGhpcyBpbnRlcmZhY2UgYW5kIHNwZWNpYWxpemUgdGhlIGVzc2VudGlhbCBwYXJ0cyBpbiB2YXJpb3VzIHdheXMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ29tbW9uQXV0b1NjYWxpbmdHcm91cFByb3BzIHtcbiAgLyoqXG4gICAqIE1pbmltdW0gbnVtYmVyIG9mIGluc3RhbmNlcyBpbiB0aGUgZmxlZXRcbiAgICpcbiAgICogQGRlZmF1bHQgMVxuICAgKi9cbiAgcmVhZG9ubHkgbWluQ2FwYWNpdHk/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIE1heGltdW0gbnVtYmVyIG9mIGluc3RhbmNlcyBpbiB0aGUgZmxlZXRcbiAgICpcbiAgICogQGRlZmF1bHQgZGVzaXJlZENhcGFjaXR5XG4gICAqL1xuICByZWFkb25seSBtYXhDYXBhY2l0eT86IG51bWJlcjtcblxuICAvKipcbiAgICogSW5pdGlhbCBhbW91bnQgb2YgaW5zdGFuY2VzIGluIHRoZSBmbGVldFxuICAgKlxuICAgKiBJZiB0aGlzIGlzIHNldCB0byBhIG51bWJlciwgZXZlcnkgZGVwbG95bWVudCB3aWxsIHJlc2V0IHRoZSBhbW91bnQgb2ZcbiAgICogaW5zdGFuY2VzIHRvIHRoaXMgbnVtYmVyLiBJdCBpcyByZWNvbW1lbmRlZCB0byBsZWF2ZSB0aGlzIHZhbHVlIGJsYW5rLlxuICAgKlxuICAgKiBAZGVmYXVsdCBtaW5DYXBhY2l0eSwgYW5kIGxlYXZlIHVuY2hhbmdlZCBkdXJpbmcgZGVwbG95bWVudFxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2F3cy1wcm9wZXJ0aWVzLWFzLWdyb3VwLmh0bWwjY2ZuLWFzLWdyb3VwLWRlc2lyZWRjYXBhY2l0eVxuICAgKi9cbiAgcmVhZG9ubHkgZGVzaXJlZENhcGFjaXR5PzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBOYW1lIG9mIFNTSCBrZXlwYWlyIHRvIGdyYW50IGFjY2VzcyB0byBpbnN0YW5jZXNcbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBTU0ggYWNjZXNzIHdpbGwgYmUgcG9zc2libGUuXG4gICAqL1xuICByZWFkb25seSBrZXlOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBXaGVyZSB0byBwbGFjZSBpbnN0YW5jZXMgd2l0aGluIHRoZSBWUENcbiAgICpcbiAgICogQGRlZmF1bHQgLSBBbGwgUHJpdmF0ZSBzdWJuZXRzLlxuICAgKi9cbiAgcmVhZG9ubHkgdnBjU3VibmV0cz86IGVjMi5TdWJuZXRTZWxlY3Rpb247XG5cbiAgLyoqXG4gICAqIFNOUyB0b3BpYyB0byBzZW5kIG5vdGlmaWNhdGlvbnMgYWJvdXQgZmxlZXQgY2hhbmdlc1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIGZsZWV0IGNoYW5nZSBub3RpZmljYXRpb25zIHdpbGwgYmUgc2VudC5cbiAgICogQGRlcHJlY2F0ZWQgdXNlIGBub3RpZmljYXRpb25zYFxuICAgKi9cbiAgcmVhZG9ubHkgbm90aWZpY2F0aW9uc1RvcGljPzogc25zLklUb3BpYztcblxuICAvKipcbiAgICogQ29uZmlndXJlIGF1dG9zY2FsaW5nIGdyb3VwIHRvIHNlbmQgbm90aWZpY2F0aW9ucyBhYm91dCBmbGVldCBjaGFuZ2VzIHRvIGFuIFNOUyB0b3BpYyhzKVxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2F3cy1wcm9wZXJ0aWVzLWFzLWdyb3VwLmh0bWwjY2ZuLWFzLWdyb3VwLW5vdGlmaWNhdGlvbmNvbmZpZ3VyYXRpb25zXG4gICAqIEBkZWZhdWx0IC0gTm8gZmxlZXQgY2hhbmdlIG5vdGlmaWNhdGlvbnMgd2lsbCBiZSBzZW50LlxuICAgKi9cbiAgcmVhZG9ubHkgbm90aWZpY2F0aW9ucz86IE5vdGlmaWNhdGlvbkNvbmZpZ3VyYXRpb25bXTtcblxuICAvKipcbiAgICogV2hldGhlciB0aGUgaW5zdGFuY2VzIGNhbiBpbml0aWF0ZSBjb25uZWN0aW9ucyB0byBhbnl3aGVyZSBieSBkZWZhdWx0XG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGFsbG93QWxsT3V0Ym91bmQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBXaGF0IHRvIGRvIHdoZW4gYW4gQXV0b1NjYWxpbmdHcm91cCdzIGluc3RhbmNlIGNvbmZpZ3VyYXRpb24gaXMgY2hhbmdlZFxuICAgKlxuICAgKiBUaGlzIGlzIGFwcGxpZWQgd2hlbiBhbnkgb2YgdGhlIHNldHRpbmdzIG9uIHRoZSBBU0cgYXJlIGNoYW5nZWQgdGhhdFxuICAgKiBhZmZlY3QgaG93IHRoZSBpbnN0YW5jZXMgc2hvdWxkIGJlIGNyZWF0ZWQgKFZQQywgaW5zdGFuY2UgdHlwZSwgc3RhcnR1cFxuICAgKiBzY3JpcHRzLCBldGMuKS4gSXQgaW5kaWNhdGVzIGhvdyB0aGUgZXhpc3RpbmcgaW5zdGFuY2VzIHNob3VsZCBiZVxuICAgKiByZXBsYWNlZCB3aXRoIG5ldyBpbnN0YW5jZXMgbWF0Y2hpbmcgdGhlIG5ldyBjb25maWcuIEJ5IGRlZmF1bHQsIG5vdGhpbmdcbiAgICogaXMgZG9uZSBhbmQgb25seSBuZXcgaW5zdGFuY2VzIGFyZSBsYXVuY2hlZCB3aXRoIHRoZSBuZXcgY29uZmlnLlxuICAgKlxuICAgKiBAZGVmYXVsdCBVcGRhdGVUeXBlLk5vbmVcbiAgICogQGRlcHJlY2F0ZWQgVXNlIGB1cGRhdGVQb2xpY3lgIGluc3RlYWRcbiAgICovXG4gIHJlYWRvbmx5IHVwZGF0ZVR5cGU/OiBVcGRhdGVUeXBlO1xuXG4gIC8qKlxuICAgKiBDb25maWd1cmF0aW9uIGZvciByb2xsaW5nIHVwZGF0ZXNcbiAgICpcbiAgICogT25seSB1c2VkIGlmIHVwZGF0ZVR5cGUgPT0gVXBkYXRlVHlwZS5Sb2xsaW5nVXBkYXRlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFJvbGxpbmdVcGRhdGVDb25maWd1cmF0aW9uIHdpdGggZGVmYXVsdHMuXG4gICAqIEBkZXByZWNhdGVkIFVzZSBgdXBkYXRlUG9saWN5YCBpbnN0ZWFkXG4gICAqL1xuICByZWFkb25seSByb2xsaW5nVXBkYXRlQ29uZmlndXJhdGlvbj86IFJvbGxpbmdVcGRhdGVDb25maWd1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBDb25maWd1cmF0aW9uIGZvciByZXBsYWNpbmcgdXBkYXRlcy5cbiAgICpcbiAgICogT25seSB1c2VkIGlmIHVwZGF0ZVR5cGUgPT0gVXBkYXRlVHlwZS5SZXBsYWNpbmdVcGRhdGUuIFNwZWNpZmllcyBob3dcbiAgICogbWFueSBpbnN0YW5jZXMgbXVzdCBzaWduYWwgc3VjY2VzcyBmb3IgdGhlIHVwZGF0ZSB0byBzdWNjZWVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBtaW5TdWNjZXNzZnVsSW5zdGFuY2VzUGVyY2VudFxuICAgKiBAZGVwcmVjYXRlZCBVc2UgYHNpZ25hbHNgIGluc3RlYWRcbiAgICovXG4gIHJlYWRvbmx5IHJlcGxhY2luZ1VwZGF0ZU1pblN1Y2Nlc3NmdWxJbnN0YW5jZXNQZXJjZW50PzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBJZiB0aGUgQVNHIGhhcyBzY2hlZHVsZWQgYWN0aW9ucywgZG9uJ3QgcmVzZXQgdW5jaGFuZ2VkIGdyb3VwIHNpemVzXG4gICAqXG4gICAqIE9ubHkgdXNlZCBpZiB0aGUgQVNHIGhhcyBzY2hlZHVsZWQgYWN0aW9ucyAod2hpY2ggbWF5IHNjYWxlIHlvdXIgQVNHIHVwXG4gICAqIG9yIGRvd24gcmVnYXJkbGVzcyBvZiBjZGsgZGVwbG95bWVudHMpLiBJZiB0cnVlLCB0aGUgc2l6ZSBvZiB0aGUgZ3JvdXBcbiAgICogd2lsbCBvbmx5IGJlIHJlc2V0IGlmIGl0IGhhcyBiZWVuIGNoYW5nZWQgaW4gdGhlIENESyBhcHAuIElmIGZhbHNlLCB0aGVcbiAgICogc2l6ZXMgd2lsbCBhbHdheXMgYmUgY2hhbmdlZCBiYWNrIHRvIHdoYXQgdGhleSB3ZXJlIGluIHRoZSBDREsgYXBwXG4gICAqIG9uIGRlcGxveW1lbnQuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGlnbm9yZVVubW9kaWZpZWRTaXplUHJvcGVydGllcz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEhvdyBtYW55IFJlc291cmNlU2lnbmFsIGNhbGxzIENsb3VkRm9ybWF0aW9uIGV4cGVjdHMgYmVmb3JlIHRoZSByZXNvdXJjZSBpcyBjb25zaWRlcmVkIGNyZWF0ZWRcbiAgICpcbiAgICogQGRlZmF1bHQgMSBpZiByZXNvdXJjZVNpZ25hbFRpbWVvdXQgaXMgc2V0LCAwIG90aGVyd2lzZVxuICAgKiBAZGVwcmVjYXRlZCBVc2UgYHNpZ25hbHNgIGluc3RlYWQuXG4gICAqL1xuICByZWFkb25seSByZXNvdXJjZVNpZ25hbENvdW50PzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgbGVuZ3RoIG9mIHRpbWUgdG8gd2FpdCBmb3IgdGhlIHJlc291cmNlU2lnbmFsQ291bnRcbiAgICpcbiAgICogVGhlIG1heGltdW0gdmFsdWUgaXMgNDMyMDAgKDEyIGhvdXJzKS5cbiAgICpcbiAgICogQGRlZmF1bHQgRHVyYXRpb24ubWludXRlcyg1KSBpZiByZXNvdXJjZVNpZ25hbENvdW50IGlzIHNldCwgTi9BIG90aGVyd2lzZVxuICAgKiBAZGVwcmVjYXRlZCBVc2UgYHNpZ25hbHNgIGluc3RlYWQuXG4gICAqL1xuICByZWFkb25seSByZXNvdXJjZVNpZ25hbFRpbWVvdXQ/OiBEdXJhdGlvbjtcblxuICAvKipcbiAgICogRGVmYXVsdCBzY2FsaW5nIGNvb2xkb3duIGZvciB0aGlzIEF1dG9TY2FsaW5nR3JvdXBcbiAgICpcbiAgICogQGRlZmF1bHQgRHVyYXRpb24ubWludXRlcyg1KVxuICAgKi9cbiAgcmVhZG9ubHkgY29vbGRvd24/OiBEdXJhdGlvbjtcblxuICAvKipcbiAgICogV2hldGhlciBpbnN0YW5jZXMgaW4gdGhlIEF1dG8gU2NhbGluZyBHcm91cCBzaG91bGQgaGF2ZSBwdWJsaWNcbiAgICogSVAgYWRkcmVzc2VzIGFzc29jaWF0ZWQgd2l0aCB0aGVtLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFVzZSBzdWJuZXQgc2V0dGluZy5cbiAgICovXG4gIHJlYWRvbmx5IGFzc29jaWF0ZVB1YmxpY0lwQWRkcmVzcz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBtYXhpbXVtIGhvdXJseSBwcmljZSAoaW4gVVNEKSB0byBiZSBwYWlkIGZvciBhbnkgU3BvdCBJbnN0YW5jZSBsYXVuY2hlZCB0byBmdWxmaWxsIHRoZSByZXF1ZXN0LiBTcG90IEluc3RhbmNlcyBhcmVcbiAgICogbGF1bmNoZWQgd2hlbiB0aGUgcHJpY2UgeW91IHNwZWNpZnkgZXhjZWVkcyB0aGUgY3VycmVudCBTcG90IG1hcmtldCBwcmljZS5cbiAgICpcbiAgICogQGRlZmF1bHQgbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgc3BvdFByaWNlPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBDb25maWd1cmF0aW9uIGZvciBoZWFsdGggY2hlY2tzXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gSGVhbHRoQ2hlY2suZWMyIHdpdGggbm8gZ3JhY2UgcGVyaW9kXG4gICAqL1xuICByZWFkb25seSBoZWFsdGhDaGVjaz86IEhlYWx0aENoZWNrO1xuXG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgaG93IGJsb2NrIGRldmljZXMgYXJlIGV4cG9zZWQgdG8gdGhlIGluc3RhbmNlLiBZb3UgY2FuIHNwZWNpZnkgdmlydHVhbCBkZXZpY2VzIGFuZCBFQlMgdm9sdW1lcy5cbiAgICpcbiAgICogRWFjaCBpbnN0YW5jZSB0aGF0IGlzIGxhdW5jaGVkIGhhcyBhbiBhc3NvY2lhdGVkIHJvb3QgZGV2aWNlIHZvbHVtZSxcbiAgICogZWl0aGVyIGFuIEFtYXpvbiBFQlMgdm9sdW1lIG9yIGFuIGluc3RhbmNlIHN0b3JlIHZvbHVtZS5cbiAgICogWW91IGNhbiB1c2UgYmxvY2sgZGV2aWNlIG1hcHBpbmdzIHRvIHNwZWNpZnkgYWRkaXRpb25hbCBFQlMgdm9sdW1lcyBvclxuICAgKiBpbnN0YW5jZSBzdG9yZSB2b2x1bWVzIHRvIGF0dGFjaCB0byBhbiBpbnN0YW5jZSB3aGVuIGl0IGlzIGxhdW5jaGVkLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NFQzIvbGF0ZXN0L1VzZXJHdWlkZS9ibG9jay1kZXZpY2UtbWFwcGluZy1jb25jZXB0cy5odG1sXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gVXNlcyB0aGUgYmxvY2sgZGV2aWNlIG1hcHBpbmcgb2YgdGhlIEFNSVxuICAgKi9cbiAgcmVhZG9ubHkgYmxvY2tEZXZpY2VzPzogQmxvY2tEZXZpY2VbXTtcblxuICAvKipcbiAgICogVGhlIG1heGltdW0gYW1vdW50IG9mIHRpbWUgdGhhdCBhbiBpbnN0YW5jZSBjYW4gYmUgaW4gc2VydmljZS4gVGhlIG1heGltdW0gZHVyYXRpb24gYXBwbGllc1xuICAgKiB0byBhbGwgY3VycmVudCBhbmQgZnV0dXJlIGluc3RhbmNlcyBpbiB0aGUgZ3JvdXAuIEFzIGFuIGluc3RhbmNlIGFwcHJvYWNoZXMgaXRzIG1heGltdW0gZHVyYXRpb24sXG4gICAqIGl0IGlzIHRlcm1pbmF0ZWQgYW5kIHJlcGxhY2VkLCBhbmQgY2Fubm90IGJlIHVzZWQgYWdhaW4uXG4gICAqXG4gICAqIFlvdSBtdXN0IHNwZWNpZnkgYSB2YWx1ZSBvZiBhdCBsZWFzdCA2MDQsODAwIHNlY29uZHMgKDcgZGF5cykuIFRvIGNsZWFyIGEgcHJldmlvdXNseSBzZXQgdmFsdWUsXG4gICAqIGxlYXZlIHRoaXMgcHJvcGVydHkgdW5kZWZpbmVkLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hdXRvc2NhbGluZy9lYzIvdXNlcmd1aWRlL2FzZy1tYXgtaW5zdGFuY2UtbGlmZXRpbWUuaHRtbFxuICAgKlxuICAgKiBAZGVmYXVsdCBub25lXG4gICAqL1xuICByZWFkb25seSBtYXhJbnN0YW5jZUxpZmV0aW1lPzogRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIENvbnRyb2xzIHdoZXRoZXIgaW5zdGFuY2VzIGluIHRoaXMgZ3JvdXAgYXJlIGxhdW5jaGVkIHdpdGggZGV0YWlsZWQgb3IgYmFzaWMgbW9uaXRvcmluZy5cbiAgICpcbiAgICogV2hlbiBkZXRhaWxlZCBtb25pdG9yaW5nIGlzIGVuYWJsZWQsIEFtYXpvbiBDbG91ZFdhdGNoIGdlbmVyYXRlcyBtZXRyaWNzIGV2ZXJ5IG1pbnV0ZSBhbmQgeW91ciBhY2NvdW50XG4gICAqIGlzIGNoYXJnZWQgYSBmZWUuIFdoZW4geW91IGRpc2FibGUgZGV0YWlsZWQgbW9uaXRvcmluZywgQ2xvdWRXYXRjaCBnZW5lcmF0ZXMgbWV0cmljcyBldmVyeSA1IG1pbnV0ZXMuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F1dG9zY2FsaW5nL2xhdGVzdC91c2VyZ3VpZGUvYXMtaW5zdGFuY2UtbW9uaXRvcmluZy5odG1sI2VuYWJsZS1hcy1pbnN0YW5jZS1tZXRyaWNzXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTW9uaXRvcmluZy5ERVRBSUxFRFxuICAgKi9cbiAgcmVhZG9ubHkgaW5zdGFuY2VNb25pdG9yaW5nPzogTW9uaXRvcmluZztcblxuICAvKipcbiAgICogRW5hYmxlIG1vbml0b3JpbmcgZm9yIGdyb3VwIG1ldHJpY3MsIHRoZXNlIG1ldHJpY3MgZGVzY3JpYmUgdGhlIGdyb3VwIHJhdGhlciB0aGFuIGFueSBvZiBpdHMgaW5zdGFuY2VzLlxuICAgKiBUbyByZXBvcnQgYWxsIGdyb3VwIG1ldHJpY3MgdXNlIGBHcm91cE1ldHJpY3MuYWxsKClgXG4gICAqIEdyb3VwIG1ldHJpY3MgYXJlIHJlcG9ydGVkIGluIGEgZ3JhbnVsYXJpdHkgb2YgMSBtaW51dGUgYXQgbm8gYWRkaXRpb25hbCBjaGFyZ2UuXG4gICAqIEBkZWZhdWx0IC0gbm8gZ3JvdXAgbWV0cmljcyB3aWxsIGJlIHJlcG9ydGVkXG4gICAqXG4gICAqL1xuICByZWFkb25seSBncm91cE1ldHJpY3M/OiBHcm91cE1ldHJpY3NbXTtcblxuICAvKipcbiAgICogQ29uZmlndXJlIHdhaXRpbmcgZm9yIHNpZ25hbHMgZHVyaW5nIGRlcGxveW1lbnRcbiAgICpcbiAgICogVXNlIHRoaXMgdG8gcGF1c2UgdGhlIENsb3VkRm9ybWF0aW9uIGRlcGxveW1lbnQgdG8gd2FpdCBmb3IgdGhlIGluc3RhbmNlc1xuICAgKiBpbiB0aGUgQXV0b1NjYWxpbmdHcm91cCB0byByZXBvcnQgc3VjY2Vzc2Z1bCBzdGFydHVwIGR1cmluZ1xuICAgKiBjcmVhdGlvbiBhbmQgdXBkYXRlcy4gVGhlIFVzZXJEYXRhIHNjcmlwdCBuZWVkcyB0byBpbnZva2UgYGNmbi1zaWduYWxgXG4gICAqIHdpdGggYSBzdWNjZXNzIG9yIGZhaWx1cmUgY29kZSBhZnRlciBpdCBpcyBkb25lIHNldHRpbmcgdXAgdGhlIGluc3RhbmNlLlxuICAgKlxuICAgKiBXaXRob3V0IHdhaXRpbmcgZm9yIHNpZ25hbHMsIHRoZSBDbG91ZEZvcm1hdGlvbiBkZXBsb3ltZW50IHdpbGwgcHJvY2VlZCBhc1xuICAgKiBzb29uIGFzIHRoZSBBdXRvU2NhbGluZ0dyb3VwIGhhcyBiZWVuIGNyZWF0ZWQgb3IgdXBkYXRlZCBidXQgYmVmb3JlIHRoZVxuICAgKiBpbnN0YW5jZXMgaW4gdGhlIGdyb3VwIGhhdmUgYmVlbiBzdGFydGVkLlxuICAgKlxuICAgKiBGb3IgZXhhbXBsZSwgdG8gaGF2ZSBpbnN0YW5jZXMgd2FpdCBmb3IgYW4gRWxhc3RpYyBMb2FkIEJhbGFuY2luZyBoZWFsdGggY2hlY2sgYmVmb3JlXG4gICAqIHRoZXkgc2lnbmFsIHN1Y2Nlc3MsIGFkZCBhIGhlYWx0aC1jaGVjayB2ZXJpZmljYXRpb24gYnkgdXNpbmcgdGhlXG4gICAqIGNmbi1pbml0IGhlbHBlciBzY3JpcHQuIEZvciBhbiBleGFtcGxlLCBzZWUgdGhlIHZlcmlmeV9pbnN0YW5jZV9oZWFsdGhcbiAgICogY29tbWFuZCBpbiB0aGUgQXV0byBTY2FsaW5nIHJvbGxpbmcgdXBkYXRlcyBzYW1wbGUgdGVtcGxhdGU6XG4gICAqXG4gICAqIGh0dHBzOi8vZ2l0aHViLmNvbS9hd3NsYWJzL2F3cy1jbG91ZGZvcm1hdGlvbi10ZW1wbGF0ZXMvYmxvYi9tYXN0ZXIvYXdzL3NlcnZpY2VzL0F1dG9TY2FsaW5nL0F1dG9TY2FsaW5nUm9sbGluZ1VwZGF0ZXMueWFtbFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIERvIG5vdCB3YWl0IGZvciBzaWduYWxzXG4gICAqL1xuICByZWFkb25seSBzaWduYWxzPzogU2lnbmFscztcblxuICAvKipcbiAgICogV2hhdCB0byBkbyB3aGVuIGFuIEF1dG9TY2FsaW5nR3JvdXAncyBpbnN0YW5jZSBjb25maWd1cmF0aW9uIGlzIGNoYW5nZWRcbiAgICpcbiAgICogVGhpcyBpcyBhcHBsaWVkIHdoZW4gYW55IG9mIHRoZSBzZXR0aW5ncyBvbiB0aGUgQVNHIGFyZSBjaGFuZ2VkIHRoYXRcbiAgICogYWZmZWN0IGhvdyB0aGUgaW5zdGFuY2VzIHNob3VsZCBiZSBjcmVhdGVkIChWUEMsIGluc3RhbmNlIHR5cGUsIHN0YXJ0dXBcbiAgICogc2NyaXB0cywgZXRjLikuIEl0IGluZGljYXRlcyBob3cgdGhlIGV4aXN0aW5nIGluc3RhbmNlcyBzaG91bGQgYmVcbiAgICogcmVwbGFjZWQgd2l0aCBuZXcgaW5zdGFuY2VzIG1hdGNoaW5nIHRoZSBuZXcgY29uZmlnLiBCeSBkZWZhdWx0LCBub3RoaW5nXG4gICAqIGlzIGRvbmUgYW5kIG9ubHkgbmV3IGluc3RhbmNlcyBhcmUgbGF1bmNoZWQgd2l0aCB0aGUgbmV3IGNvbmZpZy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBgVXBkYXRlUG9saWN5LnJvbGxpbmdVcGRhdGUoKWAgaWYgdXNpbmcgYGluaXRgLCBgVXBkYXRlUG9saWN5Lm5vbmUoKWAgb3RoZXJ3aXNlXG4gICAqL1xuICByZWFkb25seSB1cGRhdGVQb2xpY3k/OiBVcGRhdGVQb2xpY3k7XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgbmV3bHktbGF1bmNoZWQgaW5zdGFuY2VzIGFyZSBwcm90ZWN0ZWQgZnJvbSB0ZXJtaW5hdGlvbiBieSBBbWF6b25cbiAgICogRUMyIEF1dG8gU2NhbGluZyB3aGVuIHNjYWxpbmcgaW4uXG4gICAqXG4gICAqIEJ5IGRlZmF1bHQsIEF1dG8gU2NhbGluZyBjYW4gdGVybWluYXRlIGFuIGluc3RhbmNlIGF0IGFueSB0aW1lIGFmdGVyIGxhdW5jaFxuICAgKiB3aGVuIHNjYWxpbmcgaW4gYW4gQXV0byBTY2FsaW5nIEdyb3VwLCBzdWJqZWN0IHRvIHRoZSBncm91cCdzIHRlcm1pbmF0aW9uXG4gICAqIHBvbGljeS4gSG93ZXZlciwgeW91IG1heSB3aXNoIHRvIHByb3RlY3QgbmV3bHktbGF1bmNoZWQgaW5zdGFuY2VzIGZyb21cbiAgICogYmVpbmcgc2NhbGVkIGluIGlmIHRoZXkgYXJlIGdvaW5nIHRvIHJ1biBjcml0aWNhbCBhcHBsaWNhdGlvbnMgdGhhdCBzaG91bGRcbiAgICogbm90IGJlIHByZW1hdHVyZWx5IHRlcm1pbmF0ZWQuXG4gICAqXG4gICAqIFRoaXMgZmxhZyBtdXN0IGJlIGVuYWJsZWQgaWYgdGhlIEF1dG8gU2NhbGluZyBHcm91cCB3aWxsIGJlIGFzc29jaWF0ZWQgd2l0aFxuICAgKiBhbiBFQ1MgQ2FwYWNpdHkgUHJvdmlkZXIgd2l0aCBtYW5hZ2VkIHRlcm1pbmF0aW9uIHByb3RlY3Rpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBuZXdJbnN0YW5jZXNQcm90ZWN0ZWRGcm9tU2NhbGVJbj86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBBdXRvIFNjYWxpbmcgZ3JvdXAuIFRoaXMgbmFtZSBtdXN0IGJlIHVuaXF1ZSBwZXIgUmVnaW9uIHBlciBhY2NvdW50LlxuICAgKiBAZGVmYXVsdCAtIEF1dG8gZ2VuZXJhdGVkIGJ5IENsb3VkRm9ybWF0aW9uXG4gICAqL1xuICByZWFkb25seSBhdXRvU2NhbGluZ0dyb3VwTmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogQSBwb2xpY3kgb3IgYSBsaXN0IG9mIHBvbGljaWVzIHRoYXQgYXJlIHVzZWQgdG8gc2VsZWN0IHRoZSBpbnN0YW5jZXMgdG9cbiAgICogdGVybWluYXRlLiBUaGUgcG9saWNpZXMgYXJlIGV4ZWN1dGVkIGluIHRoZSBvcmRlciB0aGF0IHlvdSBsaXN0IHRoZW0uXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F1dG9zY2FsaW5nL2VjMi91c2VyZ3VpZGUvYXMtaW5zdGFuY2UtdGVybWluYXRpb24uaHRtbFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGBUZXJtaW5hdGlvblBvbGljeS5ERUZBVUxUYFxuICAgKi9cbiAgcmVhZG9ubHkgdGVybWluYXRpb25Qb2xpY2llcz86IFRlcm1pbmF0aW9uUG9saWN5W107XG59XG5cbi8qKlxuICogTWl4ZWRJbnN0YW5jZXNQb2xpY3kgYWxsb3dzIHlvdSB0byBjb25maWd1cmUgYSBncm91cCB0aGF0IGRpdmVyc2lmaWVzIGFjcm9zcyBPbi1EZW1hbmQgSW5zdGFuY2VzXG4gKiBhbmQgU3BvdCBJbnN0YW5jZXMgb2YgbXVsdGlwbGUgaW5zdGFuY2UgdHlwZXMuIEZvciBtb3JlIGluZm9ybWF0aW9uLCBzZWUgQXV0byBTY2FsaW5nIGdyb3VwcyB3aXRoXG4gKiBtdWx0aXBsZSBpbnN0YW5jZSB0eXBlcyBhbmQgcHVyY2hhc2Ugb3B0aW9ucyBpbiB0aGUgQW1hem9uIEVDMiBBdXRvIFNjYWxpbmcgVXNlciBHdWlkZTpcbiAqXG4gKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXV0b3NjYWxpbmcvZWMyL3VzZXJndWlkZS9hc2ctcHVyY2hhc2Utb3B0aW9ucy5odG1sXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTWl4ZWRJbnN0YW5jZXNQb2xpY3kge1xuICAvKipcbiAgICogSW5zdGFuY2VzRGlzdHJpYnV0aW9uIHRvIHVzZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBUaGUgdmFsdWUgZm9yIGVhY2ggcHJvcGVydHkgaW4gaXQgdXNlcyBhIGRlZmF1bHQgdmFsdWUuXG4gICAqL1xuICByZWFkb25seSBpbnN0YW5jZXNEaXN0cmlidXRpb24/OiBJbnN0YW5jZXNEaXN0cmlidXRpb247XG5cbiAgLyoqXG4gICAqIExhdW5jaCB0ZW1wbGF0ZSB0byB1c2UuXG4gICAqL1xuICByZWFkb25seSBsYXVuY2hUZW1wbGF0ZTogZWMyLklMYXVuY2hUZW1wbGF0ZTtcblxuICAvKipcbiAgICogTGF1bmNoIHRlbXBsYXRlIG92ZXJyaWRlcy5cbiAgICpcbiAgICogVGhlIG1heGltdW0gbnVtYmVyIG9mIGluc3RhbmNlIHR5cGVzIHRoYXQgY2FuIGJlIGFzc29jaWF0ZWQgd2l0aCBhbiBBdXRvIFNjYWxpbmcgZ3JvdXAgaXMgNDAuXG4gICAqXG4gICAqIFRoZSBtYXhpbXVtIG51bWJlciBvZiBkaXN0aW5jdCBsYXVuY2ggdGVtcGxhdGVzIHlvdSBjYW4gZGVmaW5lIGZvciBhbiBBdXRvIFNjYWxpbmcgZ3JvdXAgaXMgMjAuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gRG8gbm90IHByb3ZpZGUgYW55IG92ZXJyaWRlc1xuICAgKi9cbiAgcmVhZG9ubHkgbGF1bmNoVGVtcGxhdGVPdmVycmlkZXM/OiBMYXVuY2hUZW1wbGF0ZU92ZXJyaWRlc1tdO1xufVxuXG4vKipcbiAqIEluZGljYXRlcyBob3cgdG8gYWxsb2NhdGUgaW5zdGFuY2UgdHlwZXMgdG8gZnVsZmlsbCBPbi1EZW1hbmQgY2FwYWNpdHkuXG4gKi9cbmV4cG9ydCBlbnVtIE9uRGVtYW5kQWxsb2NhdGlvblN0cmF0ZWd5IHtcbiAgLyoqXG4gICAqIFRoaXMgc3RyYXRlZ3kgdXNlcyB0aGUgb3JkZXIgb2YgaW5zdGFuY2UgdHlwZXMgaW4gdGhlIExhdW5jaFRlbXBsYXRlT3ZlcnJpZGVzIHRvIGRlZmluZSB0aGUgbGF1bmNoXG4gICAqIHByaW9yaXR5IG9mIGVhY2ggaW5zdGFuY2UgdHlwZS4gVGhlIGZpcnN0IGluc3RhbmNlIHR5cGUgaW4gdGhlIGFycmF5IGlzIHByaW9yaXRpemVkIGhpZ2hlciB0aGFuIHRoZVxuICAgKiBsYXN0LiBJZiBhbGwgeW91ciBPbi1EZW1hbmQgY2FwYWNpdHkgY2Fubm90IGJlIGZ1bGZpbGxlZCB1c2luZyB5b3VyIGhpZ2hlc3QgcHJpb3JpdHkgaW5zdGFuY2UsIHRoZW5cbiAgICogdGhlIEF1dG8gU2NhbGluZyBncm91cCBsYXVuY2hlcyB0aGUgcmVtYWluaW5nIGNhcGFjaXR5IHVzaW5nIHRoZSBzZWNvbmQgcHJpb3JpdHkgaW5zdGFuY2UgdHlwZSwgYW5kXG4gICAqIHNvIG9uLlxuICAgKi9cbiAgUFJJT1JJVElaRUQgPSAncHJpb3JpdGl6ZWQnLFxufVxuXG4vKipcbiAqIEluZGljYXRlcyBob3cgdG8gYWxsb2NhdGUgaW5zdGFuY2UgdHlwZXMgdG8gZnVsZmlsbCBTcG90IGNhcGFjaXR5LlxuICovXG5leHBvcnQgZW51bSBTcG90QWxsb2NhdGlvblN0cmF0ZWd5IHtcbiAgLyoqXG4gICAqIFRoZSBBdXRvIFNjYWxpbmcgZ3JvdXAgbGF1bmNoZXMgaW5zdGFuY2VzIHVzaW5nIHRoZSBTcG90IHBvb2xzIHdpdGggdGhlIGxvd2VzdCBwcmljZSwgYW5kIGV2ZW5seVxuICAgKiBhbGxvY2F0ZXMgeW91ciBpbnN0YW5jZXMgYWNyb3NzIHRoZSBudW1iZXIgb2YgU3BvdCBwb29scyB0aGF0IHlvdSBzcGVjaWZ5LlxuICAgKi9cbiAgTE9XRVNUX1BSSUNFID0gJ2xvd2VzdC1wcmljZScsXG5cbiAgLyoqXG4gICAqIFRoZSBBdXRvIFNjYWxpbmcgZ3JvdXAgbGF1bmNoZXMgaW5zdGFuY2VzIHVzaW5nIFNwb3QgcG9vbHMgdGhhdCBhcmUgb3B0aW1hbGx5IGNob3NlbiBiYXNlZCBvbiB0aGVcbiAgICogYXZhaWxhYmxlIFNwb3QgY2FwYWNpdHkuXG4gICAqXG4gICAqIFJlY29tbWVuZGVkLlxuICAgKi9cbiAgQ0FQQUNJVFlfT1BUSU1JWkVEID0gJ2NhcGFjaXR5LW9wdGltaXplZCcsXG5cbiAgLyoqXG4gICAqIFdoZW4geW91IHVzZSB0aGlzIHN0cmF0ZWd5LCB5b3UgbmVlZCB0byBzZXQgdGhlIG9yZGVyIG9mIGluc3RhbmNlIHR5cGVzIGluIHRoZSBsaXN0IG9mIGxhdW5jaCB0ZW1wbGF0ZVxuICAgKiBvdmVycmlkZXMgZnJvbSBoaWdoZXN0IHRvIGxvd2VzdCBwcmlvcml0eSAoZnJvbSBmaXJzdCB0byBsYXN0IGluIHRoZSBsaXN0KS4gQW1hem9uIEVDMiBBdXRvIFNjYWxpbmdcbiAgICogaG9ub3JzIHRoZSBpbnN0YW5jZSB0eXBlIHByaW9yaXRpZXMgb24gYSBiZXN0LWVmZm9ydCBiYXNpcyBidXQgb3B0aW1pemVzIGZvciBjYXBhY2l0eSBmaXJzdC5cbiAgICovXG4gIENBUEFDSVRZX09QVElNSVpFRF9QUklPUklUSVpFRCA9ICdjYXBhY2l0eS1vcHRpbWl6ZWQtcHJpb3JpdGl6ZWQnLFxufVxuXG4vKipcbiAqIEluc3RhbmNlc0Rpc3RyaWJ1dGlvbiBpcyBhIHN1YnByb3BlcnR5IG9mIE1peGVkSW5zdGFuY2VzUG9saWN5IHRoYXQgZGVzY3JpYmVzIGFuIGluc3RhbmNlcyBkaXN0cmlidXRpb25cbiAqIGZvciBhbiBBdXRvIFNjYWxpbmcgZ3JvdXAuIFRoZSBpbnN0YW5jZXMgZGlzdHJpYnV0aW9uIHNwZWNpZmllcyB0aGUgZGlzdHJpYnV0aW9uIG9mIE9uLURlbWFuZCBJbnN0YW5jZXNcbiAqIGFuZCBTcG90IEluc3RhbmNlcywgdGhlIG1heGltdW0gcHJpY2UgdG8gcGF5IGZvciBTcG90IEluc3RhbmNlcywgYW5kIGhvdyB0aGUgQXV0byBTY2FsaW5nIGdyb3VwIGFsbG9jYXRlc1xuICogaW5zdGFuY2UgdHlwZXMgdG8gZnVsZmlsbCBPbi1EZW1hbmQgYW5kIFNwb3QgY2FwYWNpdGllcy5cbiAqXG4gKiBGb3IgbW9yZSBpbmZvcm1hdGlvbiBhbmQgZXhhbXBsZSBjb25maWd1cmF0aW9ucywgc2VlIEF1dG8gU2NhbGluZyBncm91cHMgd2l0aCBtdWx0aXBsZSBpbnN0YW5jZSB0eXBlc1xuICogYW5kIHB1cmNoYXNlIG9wdGlvbnMgaW4gdGhlIEFtYXpvbiBFQzIgQXV0byBTY2FsaW5nIFVzZXIgR3VpZGU6XG4gKlxuICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F1dG9zY2FsaW5nL2VjMi91c2VyZ3VpZGUvYXNnLXB1cmNoYXNlLW9wdGlvbnMuaHRtbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIEluc3RhbmNlc0Rpc3RyaWJ1dGlvbiB7XG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgaG93IHRvIGFsbG9jYXRlIGluc3RhbmNlIHR5cGVzIHRvIGZ1bGZpbGwgT24tRGVtYW5kIGNhcGFjaXR5LiBUaGUgb25seSB2YWxpZCB2YWx1ZSBpcyBwcmlvcml0aXplZCxcbiAgICogd2hpY2ggaXMgYWxzbyB0aGUgZGVmYXVsdCB2YWx1ZS5cbiAgICpcbiAgICogQGRlZmF1bHQgT25EZW1hbmRBbGxvY2F0aW9uU3RyYXRlZ3kuUFJJT1JJVElaRURcbiAgICovXG4gIHJlYWRvbmx5IG9uRGVtYW5kQWxsb2NhdGlvblN0cmF0ZWd5PzogT25EZW1hbmRBbGxvY2F0aW9uU3RyYXRlZ3ksXG5cbiAgLyoqXG4gICAqIFRoZSBtaW5pbXVtIGFtb3VudCBvZiB0aGUgQXV0byBTY2FsaW5nIGdyb3VwJ3MgY2FwYWNpdHkgdGhhdCBtdXN0IGJlIGZ1bGZpbGxlZCBieSBPbi1EZW1hbmQgSW5zdGFuY2VzLiBUaGlzXG4gICAqIGJhc2UgcG9ydGlvbiBpcyBwcm92aXNpb25lZCBmaXJzdCBhcyB5b3VyIGdyb3VwIHNjYWxlcy4gRGVmYXVsdHMgdG8gMCBpZiBub3Qgc3BlY2lmaWVkLiBJZiB5b3Ugc3BlY2lmeSB3ZWlnaHRzXG4gICAqIGZvciB0aGUgaW5zdGFuY2UgdHlwZXMgaW4gdGhlIG92ZXJyaWRlcywgc2V0IHRoZSB2YWx1ZSBvZiBPbkRlbWFuZEJhc2VDYXBhY2l0eSBpbiB0ZXJtcyBvZiB0aGUgbnVtYmVyIG9mXG4gICAqIGNhcGFjaXR5IHVuaXRzLCBhbmQgbm90IHRoZSBudW1iZXIgb2YgaW5zdGFuY2VzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAwXG4gICAqL1xuICByZWFkb25seSBvbkRlbWFuZEJhc2VDYXBhY2l0eT86IG51bWJlcixcblxuICAvKipcbiAgICogQ29udHJvbHMgdGhlIHBlcmNlbnRhZ2VzIG9mIE9uLURlbWFuZCBJbnN0YW5jZXMgYW5kIFNwb3QgSW5zdGFuY2VzIGZvciB5b3VyIGFkZGl0aW9uYWwgY2FwYWNpdHkgYmV5b25kXG4gICAqIE9uRGVtYW5kQmFzZUNhcGFjaXR5LiBFeHByZXNzZWQgYXMgYSBudW1iZXIgKGZvciBleGFtcGxlLCAyMCBzcGVjaWZpZXMgMjAlIE9uLURlbWFuZCBJbnN0YW5jZXMsIDgwJSBTcG90IEluc3RhbmNlcykuXG4gICAqIERlZmF1bHRzIHRvIDEwMCBpZiBub3Qgc3BlY2lmaWVkLiBJZiBzZXQgdG8gMTAwLCBvbmx5IE9uLURlbWFuZCBJbnN0YW5jZXMgYXJlIHByb3Zpc2lvbmVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAxMDBcbiAgICovXG4gIHJlYWRvbmx5IG9uRGVtYW5kUGVyY2VudGFnZUFib3ZlQmFzZUNhcGFjaXR5PzogbnVtYmVyLFxuXG4gIC8qKlxuICAgKiBJZiB0aGUgYWxsb2NhdGlvbiBzdHJhdGVneSBpcyBsb3dlc3QtcHJpY2UsIHRoZSBBdXRvIFNjYWxpbmcgZ3JvdXAgbGF1bmNoZXMgaW5zdGFuY2VzIHVzaW5nIHRoZSBTcG90IHBvb2xzIHdpdGggdGhlXG4gICAqIGxvd2VzdCBwcmljZSwgYW5kIGV2ZW5seSBhbGxvY2F0ZXMgeW91ciBpbnN0YW5jZXMgYWNyb3NzIHRoZSBudW1iZXIgb2YgU3BvdCBwb29scyB0aGF0IHlvdSBzcGVjaWZ5LiBEZWZhdWx0cyB0b1xuICAgKiBsb3dlc3QtcHJpY2UgaWYgbm90IHNwZWNpZmllZC5cbiAgICpcbiAgICogSWYgdGhlIGFsbG9jYXRpb24gc3RyYXRlZ3kgaXMgY2FwYWNpdHktb3B0aW1pemVkIChyZWNvbW1lbmRlZCksIHRoZSBBdXRvIFNjYWxpbmcgZ3JvdXAgbGF1bmNoZXMgaW5zdGFuY2VzIHVzaW5nIFNwb3RcbiAgICogcG9vbHMgdGhhdCBhcmUgb3B0aW1hbGx5IGNob3NlbiBiYXNlZCBvbiB0aGUgYXZhaWxhYmxlIFNwb3QgY2FwYWNpdHkuIEFsdGVybmF0aXZlbHksIHlvdSBjYW4gdXNlIGNhcGFjaXR5LW9wdGltaXplZC1wcmlvcml0aXplZFxuICAgKiBhbmQgc2V0IHRoZSBvcmRlciBvZiBpbnN0YW5jZSB0eXBlcyBpbiB0aGUgbGlzdCBvZiBsYXVuY2ggdGVtcGxhdGUgb3ZlcnJpZGVzIGZyb20gaGlnaGVzdCB0byBsb3dlc3QgcHJpb3JpdHlcbiAgICogKGZyb20gZmlyc3QgdG8gbGFzdCBpbiB0aGUgbGlzdCkuIEFtYXpvbiBFQzIgQXV0byBTY2FsaW5nIGhvbm9ycyB0aGUgaW5zdGFuY2UgdHlwZSBwcmlvcml0aWVzIG9uIGEgYmVzdC1lZmZvcnQgYmFzaXMgYnV0XG4gICAqIG9wdGltaXplcyBmb3IgY2FwYWNpdHkgZmlyc3QuXG4gICAqXG4gICAqIEBkZWZhdWx0IFNwb3RBbGxvY2F0aW9uU3RyYXRlZ3kuTE9XRVNUX1BSSUNFXG4gICAqL1xuICByZWFkb25seSBzcG90QWxsb2NhdGlvblN0cmF0ZWd5PzogU3BvdEFsbG9jYXRpb25TdHJhdGVneSxcblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBTcG90IEluc3RhbmNlIHBvb2xzIHRvIHVzZSB0byBhbGxvY2F0ZSB5b3VyIFNwb3QgY2FwYWNpdHkuIFRoZSBTcG90IHBvb2xzIGFyZSBkZXRlcm1pbmVkIGZyb20gdGhlIGRpZmZlcmVudCBpbnN0YW5jZVxuICAgKiB0eXBlcyBpbiB0aGUgb3ZlcnJpZGVzLiBWYWxpZCBvbmx5IHdoZW4gdGhlIFNwb3QgYWxsb2NhdGlvbiBzdHJhdGVneSBpcyBsb3dlc3QtcHJpY2UuIFZhbHVlIG11c3QgYmUgaW4gdGhlIHJhbmdlIG9mIDEgdG8gMjAuXG4gICAqIERlZmF1bHRzIHRvIDIgaWYgbm90IHNwZWNpZmllZC5cbiAgICpcbiAgICogQGRlZmF1bHQgMlxuICAgKi9cbiAgcmVhZG9ubHkgc3BvdEluc3RhbmNlUG9vbHM/OiBudW1iZXIsXG5cbiAgLyoqXG4gICAqIFRoZSBtYXhpbXVtIHByaWNlIHBlciB1bml0IGhvdXIgdGhhdCB5b3UgYXJlIHdpbGxpbmcgdG8gcGF5IGZvciBhIFNwb3QgSW5zdGFuY2UuIElmIHlvdSBsZWF2ZSB0aGUgdmFsdWUgYXQgaXRzIGRlZmF1bHQgKGVtcHR5KSxcbiAgICogQW1hem9uIEVDMiBBdXRvIFNjYWxpbmcgdXNlcyB0aGUgT24tRGVtYW5kIHByaWNlIGFzIHRoZSBtYXhpbXVtIFNwb3QgcHJpY2UuIFRvIHJlbW92ZSBhIHZhbHVlIHRoYXQgeW91IHByZXZpb3VzbHkgc2V0LCBpbmNsdWRlXG4gICAqIHRoZSBwcm9wZXJ0eSBidXQgc3BlY2lmeSBhbiBlbXB0eSBzdHJpbmcgKFwiXCIpIGZvciB0aGUgdmFsdWUuXG4gICAqXG4gICAqIEBkZWZhdWx0IFwiXCIgLSBPbi1EZW1hbmQgcHJpY2VcbiAgICovXG4gIHJlYWRvbmx5IHNwb3RNYXhQcmljZT86IHN0cmluZ1xufVxuXG4vKipcbiAqIExhdW5jaFRlbXBsYXRlT3ZlcnJpZGVzIGlzIGEgc3VicHJvcGVydHkgb2YgTGF1bmNoVGVtcGxhdGUgdGhhdCBkZXNjcmliZXMgYW4gb3ZlcnJpZGUgZm9yIGEgbGF1bmNoIHRlbXBsYXRlLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIExhdW5jaFRlbXBsYXRlT3ZlcnJpZGVzIHtcbiAgLyoqXG4gICAqIFRoZSBpbnN0YW5jZSB0eXBlLCBzdWNoIGFzIG0zLnhsYXJnZS4gWW91IG11c3QgdXNlIGFuIGluc3RhbmNlIHR5cGUgdGhhdCBpcyBzdXBwb3J0ZWQgaW4geW91ciByZXF1ZXN0ZWQgUmVnaW9uXG4gICAqIGFuZCBBdmFpbGFiaWxpdHkgWm9uZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gRG8gbm90IG92ZXJyaWRlIGluc3RhbmNlIHR5cGVcbiAgICovXG4gIHJlYWRvbmx5IGluc3RhbmNlVHlwZTogZWMyLkluc3RhbmNlVHlwZSxcblxuICAvKipcbiAgICogUHJvdmlkZXMgdGhlIGxhdW5jaCB0ZW1wbGF0ZSB0byBiZSB1c2VkIHdoZW4gbGF1bmNoaW5nIHRoZSBpbnN0YW5jZSB0eXBlLiBGb3IgZXhhbXBsZSwgc29tZSBpbnN0YW5jZSB0eXBlcyBtaWdodFxuICAgKiByZXF1aXJlIGEgbGF1bmNoIHRlbXBsYXRlIHdpdGggYSBkaWZmZXJlbnQgQU1JLiBJZiBub3QgcHJvdmlkZWQsIEFtYXpvbiBFQzIgQXV0byBTY2FsaW5nIHVzZXMgdGhlIGxhdW5jaCB0ZW1wbGF0ZVxuICAgKiB0aGF0J3MgZGVmaW5lZCBmb3IgeW91ciBtaXhlZCBpbnN0YW5jZXMgcG9saWN5LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIERvIG5vdCBvdmVycmlkZSBsYXVuY2ggdGVtcGxhdGVcbiAgICovXG4gIHJlYWRvbmx5IGxhdW5jaFRlbXBsYXRlPzogZWMyLklMYXVuY2hUZW1wbGF0ZSxcblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBjYXBhY2l0eSB1bml0cyBwcm92aWRlZCBieSB0aGUgc3BlY2lmaWVkIGluc3RhbmNlIHR5cGUgaW4gdGVybXMgb2YgdmlydHVhbCBDUFVzLCBtZW1vcnksIHN0b3JhZ2UsXG4gICAqIHRocm91Z2hwdXQsIG9yIG90aGVyIHJlbGF0aXZlIHBlcmZvcm1hbmNlIGNoYXJhY3RlcmlzdGljLiBXaGVuIGEgU3BvdCBvciBPbi1EZW1hbmQgSW5zdGFuY2UgaXMgcHJvdmlzaW9uZWQsIHRoZVxuICAgKiBjYXBhY2l0eSB1bml0cyBjb3VudCB0b3dhcmQgdGhlIGRlc2lyZWQgY2FwYWNpdHkuIEFtYXpvbiBFQzIgQXV0byBTY2FsaW5nIHByb3Zpc2lvbnMgaW5zdGFuY2VzIHVudGlsIHRoZSBkZXNpcmVkXG4gICAqIGNhcGFjaXR5IGlzIHRvdGFsbHkgZnVsZmlsbGVkLCBldmVuIGlmIHRoaXMgcmVzdWx0cyBpbiBhbiBvdmVyYWdlLiBWYWx1ZSBtdXN0IGJlIGluIHRoZSByYW5nZSBvZiAxIHRvIDk5OS5cbiAgICpcbiAgICogRm9yIGV4YW1wbGUsIElmIHRoZXJlIGFyZSAyIHVuaXRzIHJlbWFpbmluZyB0byBmdWxmaWxsIGNhcGFjaXR5LCBhbmQgQW1hem9uIEVDMiBBdXRvIFNjYWxpbmcgY2FuIG9ubHkgcHJvdmlzaW9uXG4gICAqIGFuIGluc3RhbmNlIHdpdGggYSBXZWlnaHRlZENhcGFjaXR5IG9mIDUgdW5pdHMsIHRoZSBpbnN0YW5jZSBpcyBwcm92aXNpb25lZCwgYW5kIHRoZSBkZXNpcmVkIGNhcGFjaXR5IGlzIGV4Y2VlZGVkXG4gICAqIGJ5IDMgdW5pdHMuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F1dG9zY2FsaW5nL2VjMi91c2VyZ3VpZGUvYXNnLWluc3RhbmNlLXdlaWdodGluZy5odG1sXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gRG8gbm90IHByb3ZpZGUgd2VpZ2h0XG4gICAqL1xuICByZWFkb25seSB3ZWlnaHRlZENhcGFjaXR5PzogbnVtYmVyXG59XG5cbi8qKlxuICogUHJvcGVydGllcyBvZiBhIEZsZWV0XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXV0b1NjYWxpbmdHcm91cFByb3BzIGV4dGVuZHMgQ29tbW9uQXV0b1NjYWxpbmdHcm91cFByb3BzIHtcbiAgLyoqXG4gICAqIFZQQyB0byBsYXVuY2ggdGhlc2UgaW5zdGFuY2VzIGluLlxuICAgKi9cbiAgcmVhZG9ubHkgdnBjOiBlYzIuSVZwYztcblxuICAvKipcbiAgICogTGF1bmNoIHRlbXBsYXRlIHRvIHVzZS5cbiAgICpcbiAgICogTGF1bmNoIGNvbmZpZ3VyYXRpb24gcmVsYXRlZCBzZXR0aW5ncyBhbmQgTWl4ZWRJbnN0YW5jZXNQb2xpY3kgbXVzdCBub3QgYmUgc3BlY2lmaWVkIHdoZW4gYVxuICAgKiBsYXVuY2ggdGVtcGxhdGUgaXMgc3BlY2lmaWVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIERvIG5vdCBwcm92aWRlIGFueSBsYXVuY2ggdGVtcGxhdGVcbiAgICovXG4gIHJlYWRvbmx5IGxhdW5jaFRlbXBsYXRlPzogZWMyLklMYXVuY2hUZW1wbGF0ZTtcblxuICAvKipcbiAgICogTWl4ZWQgSW5zdGFuY2VzIFBvbGljeSB0byB1c2UuXG4gICAqXG4gICAqIExhdW5jaCBjb25maWd1cmF0aW9uIHJlbGF0ZWQgc2V0dGluZ3MgYW5kIExhdW5jaCBUZW1wbGF0ZSAgbXVzdCBub3QgYmUgc3BlY2lmaWVkIHdoZW4gYVxuICAgKiBNaXhlZEluc3RhbmNlc1BvbGljeSBpcyBzcGVjaWZpZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gRG8gbm90IHByb3ZpZGUgYW55IE1peGVkSW5zdGFuY2VzUG9saWN5XG4gICAqL1xuICByZWFkb25seSBtaXhlZEluc3RhbmNlc1BvbGljeT86IE1peGVkSW5zdGFuY2VzUG9saWN5O1xuXG4gIC8qKlxuICAgKiBUeXBlIG9mIGluc3RhbmNlIHRvIGxhdW5jaFxuICAgKlxuICAgKiBgbGF1bmNoVGVtcGxhdGVgIG11c3Qgbm90IGJlIHNwZWNpZmllZCB3aGVuIHRoaXMgcHJvcGVydHkgaXMgc3BlY2lmaWVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIERvIG5vdCBwcm92aWRlIGFueSBpbnN0YW5jZSB0eXBlXG4gICAqL1xuICByZWFkb25seSBpbnN0YW5jZVR5cGU/OiBlYzIuSW5zdGFuY2VUeXBlO1xuXG4gIC8qKlxuICAgKiBBTUkgdG8gbGF1bmNoXG4gICAqXG4gICAqIGBsYXVuY2hUZW1wbGF0ZWAgbXVzdCBub3QgYmUgc3BlY2lmaWVkIHdoZW4gdGhpcyBwcm9wZXJ0eSBpcyBzcGVjaWZpZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gRG8gbm90IHByb3ZpZGUgYW55IG1hY2hpbmUgaW1hZ2VcbiAgICovXG4gIHJlYWRvbmx5IG1hY2hpbmVJbWFnZT86IGVjMi5JTWFjaGluZUltYWdlO1xuXG4gIC8qKlxuICAgKiBTZWN1cml0eSBncm91cCB0byBsYXVuY2ggdGhlIGluc3RhbmNlcyBpbi5cbiAgICpcbiAgICogYGxhdW5jaFRlbXBsYXRlYCBtdXN0IG5vdCBiZSBzcGVjaWZpZWQgd2hlbiB0aGlzIHByb3BlcnR5IGlzIHNwZWNpZmllZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBIFNlY3VyaXR5R3JvdXAgd2lsbCBiZSBjcmVhdGVkIGlmIG5vbmUgaXMgc3BlY2lmaWVkLlxuICAgKi9cbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cD86IGVjMi5JU2VjdXJpdHlHcm91cDtcblxuICAvKipcbiAgICogU3BlY2lmaWMgVXNlckRhdGEgdG8gdXNlXG4gICAqXG4gICAqIFRoZSBVc2VyRGF0YSBtYXkgc3RpbGwgYmUgbXV0YXRlZCBhZnRlciBjcmVhdGlvbi5cbiAgICpcbiAgICogYGxhdW5jaFRlbXBsYXRlYCBtdXN0IG5vdCBiZSBzcGVjaWZpZWQgd2hlbiB0aGlzIHByb3BlcnR5IGlzIHNwZWNpZmllZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBIFVzZXJEYXRhIG9iamVjdCBhcHByb3ByaWF0ZSBmb3IgdGhlIE1hY2hpbmVJbWFnZSdzXG4gICAqIE9wZXJhdGluZyBTeXN0ZW0gaXMgY3JlYXRlZC5cbiAgICovXG4gIHJlYWRvbmx5IHVzZXJEYXRhPzogZWMyLlVzZXJEYXRhO1xuXG4gIC8qKlxuICAgKiBBbiBJQU0gcm9sZSB0byBhc3NvY2lhdGUgd2l0aCB0aGUgaW5zdGFuY2UgcHJvZmlsZSBhc3NpZ25lZCB0byB0aGlzIEF1dG8gU2NhbGluZyBHcm91cC5cbiAgICpcbiAgICogVGhlIHJvbGUgbXVzdCBiZSBhc3N1bWFibGUgYnkgdGhlIHNlcnZpY2UgcHJpbmNpcGFsIGBlYzIuYW1hem9uYXdzLmNvbWA6XG4gICAqXG4gICAqIGBsYXVuY2hUZW1wbGF0ZWAgbXVzdCBub3QgYmUgc3BlY2lmaWVkIHdoZW4gdGhpcyBwcm9wZXJ0eSBpcyBzcGVjaWZpZWQuXG4gICAqXG4gICAqIEBleGFtcGxlXG4gICAqXG4gICAqICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ015Um9sZScsIHtcbiAgICogICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWMyLmFtYXpvbmF3cy5jb20nKVxuICAgKiAgICB9KTtcbiAgICpcbiAgICogQGRlZmF1bHQgQSByb2xlIHdpbGwgYXV0b21hdGljYWxseSBiZSBjcmVhdGVkLCBpdCBjYW4gYmUgYWNjZXNzZWQgdmlhIHRoZSBgcm9sZWAgcHJvcGVydHlcbiAgICovXG4gIHJlYWRvbmx5IHJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgLyoqXG4gICAqIEFwcGx5IHRoZSBnaXZlbiBDbG91ZEZvcm1hdGlvbiBJbml0IGNvbmZpZ3VyYXRpb24gdG8gdGhlIGluc3RhbmNlcyBpbiB0aGUgQXV0b1NjYWxpbmdHcm91cCBhdCBzdGFydHVwXG4gICAqXG4gICAqIElmIHlvdSBzcGVjaWZ5IGBpbml0YCwgeW91IG11c3QgYWxzbyBzcGVjaWZ5IGBzaWduYWxzYCB0byBjb25maWd1cmVcbiAgICogdGhlIG51bWJlciBvZiBpbnN0YW5jZXMgdG8gd2FpdCBmb3IgYW5kIHRoZSB0aW1lb3V0IGZvciB3YWl0aW5nIGZvciB0aGVcbiAgICogaW5pdCBwcm9jZXNzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIENsb3VkRm9ybWF0aW9uIGluaXRcbiAgICovXG4gIHJlYWRvbmx5IGluaXQ/OiBlYzIuQ2xvdWRGb3JtYXRpb25Jbml0O1xuXG4gIC8qKlxuICAgKiBVc2UgdGhlIGdpdmVuIG9wdGlvbnMgZm9yIGFwcGx5aW5nIENsb3VkRm9ybWF0aW9uIEluaXRcbiAgICpcbiAgICogRGVzY3JpYmVzIHRoZSBjb25maWdzZXRzIHRvIHVzZSBhbmQgdGhlIHRpbWVvdXQgdG8gd2FpdFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGRlZmF1bHQgb3B0aW9uc1xuICAgKi9cbiAgcmVhZG9ubHkgaW5pdE9wdGlvbnM/OiBBcHBseUNsb3VkRm9ybWF0aW9uSW5pdE9wdGlvbnM7XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgSU1EU3YyIHNob3VsZCBiZSByZXF1aXJlZCBvbiBsYXVuY2hlZCBpbnN0YW5jZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSByZXF1aXJlSW1kc3YyPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBDb25maWd1cmUgd2hldGhlciB0aGUgQXV0b1NjYWxpbmdHcm91cCB3YWl0cyBmb3Igc2lnbmFsc1xuICpcbiAqIElmIHlvdSBkbyBjb25maWd1cmUgd2FpdGluZyBmb3Igc2lnbmFscywgeW91IHNob3VsZCBtYWtlIHN1cmUgdGhlIGluc3RhbmNlc1xuICogaW52b2tlIGBjZm4tc2lnbmFsYCBzb21ld2hlcmUgaW4gdGhlaXIgVXNlckRhdGEgdG8gc2lnbmFsIHRoYXQgdGhleSBoYXZlXG4gKiBzdGFydGVkIHVwIChlaXRoZXIgc3VjY2Vzc2Z1bGx5IG9yIHVuc3VjY2Vzc2Z1bGx5KS5cbiAqXG4gKiBTaWduYWxzIGFyZSB1c2VkIGJvdGggZHVyaW5nIGludGlhbCBjcmVhdGlvbiBhbmQgc3Vic2VxdWVudCB1cGRhdGVzLlxuICovXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgU2lnbmFscyB7XG4gIC8qKlxuICAgKiBXYWl0IGZvciB0aGUgZGVzaXJlZENhcGFjaXR5IG9mIHRoZSBBdXRvU2NhbGluZ0dyb3VwIGFtb3VudCBvZiBzaWduYWxzIHRvIGhhdmUgYmVlbiByZWNlaXZlZFxuICAgKlxuICAgKiBJZiBubyBkZXNpcmVkQ2FwYWNpdHkgaGFzIGJlZW4gY29uZmlndXJlZCwgd2FpdCBmb3IgbWluQ2FwYWNpdHkgc2lnbmFscyBpbnRlYWQuXG4gICAqXG4gICAqIFRoaXMgbnVtYmVyIGlzIHVzZWQgZHVyaW5nIGluaXRpYWwgY3JlYXRpb24gYW5kIGR1cmluZyByZXBsYWNpbmcgdXBkYXRlcy5cbiAgICogRHVyaW5nIHJvbGxpbmcgdXBkYXRlcywgYWxsIHVwZGF0ZWQgaW5zdGFuY2VzIG11c3Qgc2VuZCBhIHNpZ25hbC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgd2FpdEZvckFsbChvcHRpb25zOiBTaWduYWxzT3B0aW9ucyA9IHt9KTogU2lnbmFscyB7XG4gICAgdmFsaWRhdGVQZXJjZW50YWdlKG9wdGlvbnMubWluU3VjY2Vzc1BlcmNlbnRhZ2UpO1xuICAgIHJldHVybiBuZXcgY2xhc3MgZXh0ZW5kcyBTaWduYWxzIHtcbiAgICAgIHB1YmxpYyByZW5kZXJDcmVhdGlvblBvbGljeShyZW5kZXJPcHRpb25zOiBSZW5kZXJTaWduYWxzT3B0aW9ucyk6IENmbkNyZWF0aW9uUG9saWN5IHtcbiAgICAgICAgcmV0dXJuIHRoaXMuZG9SZW5kZXIob3B0aW9ucywgcmVuZGVyT3B0aW9ucy5kZXNpcmVkQ2FwYWNpdHkgPz8gcmVuZGVyT3B0aW9ucy5taW5DYXBhY2l0eSk7XG4gICAgICB9XG4gICAgfSgpO1xuICB9XG5cbiAgLyoqXG4gICAqIFdhaXQgZm9yIHRoZSBtaW5DYXBhY2l0eSBvZiB0aGUgQXV0b1NjYWxpbmdHcm91cCBhbW91bnQgb2Ygc2lnbmFscyB0byBoYXZlIGJlZW4gcmVjZWl2ZWRcbiAgICpcbiAgICogVGhpcyBudW1iZXIgaXMgdXNlZCBkdXJpbmcgaW5pdGlhbCBjcmVhdGlvbiBhbmQgZHVyaW5nIHJlcGxhY2luZyB1cGRhdGVzLlxuICAgKiBEdXJpbmcgcm9sbGluZyB1cGRhdGVzLCBhbGwgdXBkYXRlZCBpbnN0YW5jZXMgbXVzdCBzZW5kIGEgc2lnbmFsLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyB3YWl0Rm9yTWluQ2FwYWNpdHkob3B0aW9uczogU2lnbmFsc09wdGlvbnMgPSB7fSk6IFNpZ25hbHMge1xuICAgIHZhbGlkYXRlUGVyY2VudGFnZShvcHRpb25zLm1pblN1Y2Nlc3NQZXJjZW50YWdlKTtcbiAgICByZXR1cm4gbmV3IGNsYXNzIGV4dGVuZHMgU2lnbmFscyB7XG4gICAgICBwdWJsaWMgcmVuZGVyQ3JlYXRpb25Qb2xpY3kocmVuZGVyT3B0aW9uczogUmVuZGVyU2lnbmFsc09wdGlvbnMpOiBDZm5DcmVhdGlvblBvbGljeSB7XG4gICAgICAgIHJldHVybiB0aGlzLmRvUmVuZGVyKG9wdGlvbnMsIHJlbmRlck9wdGlvbnMubWluQ2FwYWNpdHkpO1xuICAgICAgfVxuICAgIH0oKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBXYWl0IGZvciBhIHNwZWNpZmljIGFtb3VudCBvZiBzaWduYWxzIHRvIGhhdmUgYmVlbiByZWNlaXZlZFxuICAgKlxuICAgKiBZb3Ugc2hvdWxkIHNlbmQgb25lIHNpZ25hbCBwZXIgaW5zdGFuY2UsIHNvIHRoaXMgcmVwcmVzZW50cyB0aGUgbnVtYmVyIG9mXG4gICAqIGluc3RhbmNlcyB0byB3YWl0IGZvci5cbiAgICpcbiAgICogVGhpcyBudW1iZXIgaXMgdXNlZCBkdXJpbmcgaW5pdGlhbCBjcmVhdGlvbiBhbmQgZHVyaW5nIHJlcGxhY2luZyB1cGRhdGVzLlxuICAgKiBEdXJpbmcgcm9sbGluZyB1cGRhdGVzLCBhbGwgdXBkYXRlZCBpbnN0YW5jZXMgbXVzdCBzZW5kIGEgc2lnbmFsLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyB3YWl0Rm9yQ291bnQoY291bnQ6IG51bWJlciwgb3B0aW9uczogU2lnbmFsc09wdGlvbnMgPSB7fSk6IFNpZ25hbHMge1xuICAgIHZhbGlkYXRlUGVyY2VudGFnZShvcHRpb25zLm1pblN1Y2Nlc3NQZXJjZW50YWdlKTtcbiAgICByZXR1cm4gbmV3IGNsYXNzIGV4dGVuZHMgU2lnbmFscyB7XG4gICAgICBwdWJsaWMgcmVuZGVyQ3JlYXRpb25Qb2xpY3koKTogQ2ZuQ3JlYXRpb25Qb2xpY3kge1xuICAgICAgICByZXR1cm4gdGhpcy5kb1JlbmRlcihvcHRpb25zLCBjb3VudCk7XG4gICAgICB9XG4gICAgfSgpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlbmRlciB0aGUgQVNHJ3MgQ3JlYXRpb25Qb2xpY3lcbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZW5kZXJDcmVhdGlvblBvbGljeShyZW5kZXJPcHRpb25zOiBSZW5kZXJTaWduYWxzT3B0aW9ucyk6IENmbkNyZWF0aW9uUG9saWN5O1xuXG4gIC8qKlxuICAgKiBIZWxwZXIgdG8gcmVuZGVyIHRoZSBhY3R1YWwgY3JlYXRpb24gcG9saWN5LCBhcyB0aGUgbG9naWMgYmV0d2VlbiB0aGVtIGlzIHF1aXRlIHNpbWlsYXJcbiAgICovXG4gIHByb3RlY3RlZCBkb1JlbmRlcihvcHRpb25zOiBTaWduYWxzT3B0aW9ucywgY291bnQ/OiBudW1iZXIpOiBDZm5DcmVhdGlvblBvbGljeSB7XG4gICAgY29uc3QgbWluU3VjY2Vzc2Z1bEluc3RhbmNlc1BlcmNlbnQgPSB2YWxpZGF0ZVBlcmNlbnRhZ2Uob3B0aW9ucy5taW5TdWNjZXNzUGVyY2VudGFnZSk7XG4gICAgcmV0dXJuIHtcbiAgICAgIC4uLm9wdGlvbnMubWluU3VjY2Vzc1BlcmNlbnRhZ2UgIT09IHVuZGVmaW5lZCA/IHsgYXV0b1NjYWxpbmdDcmVhdGlvblBvbGljeTogeyBtaW5TdWNjZXNzZnVsSW5zdGFuY2VzUGVyY2VudCB9IH0gOiB7IH0sXG4gICAgICByZXNvdXJjZVNpZ25hbDoge1xuICAgICAgICBjb3VudCxcbiAgICAgICAgdGltZW91dDogb3B0aW9ucy50aW1lb3V0Py50b0lzb1N0cmluZygpLFxuICAgICAgfSxcbiAgICB9O1xuICB9XG5cbn1cblxuLyoqXG4gKiBJbnB1dCBmb3IgU2lnbmFscy5yZW5kZXJDcmVhdGlvblBvbGljeVxuICovXG5leHBvcnQgaW50ZXJmYWNlIFJlbmRlclNpZ25hbHNPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBkZXNpcmVkQ2FwYWNpdHkgb2YgdGhlIEFTR1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIGRlc2lyZWQgY2FwYWNpdHkgbm90IGNvbmZpZ3VyZWRcbiAgICovXG4gIHJlYWRvbmx5IGRlc2lyZWRDYXBhY2l0eT86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIG1pblNpemUgb2YgdGhlIEFTR1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIG1pbkNhcGFjaXR5IG5vdCBjb25maWd1cmVkXG4gICAqL1xuICByZWFkb25seSBtaW5DYXBhY2l0eT86IG51bWJlcjtcbn1cblxuLyoqXG4gKiBDdXN0b21pemF0aW9uIG9wdGlvbnMgZm9yIFNpZ25hbCBoYW5kbGluZ1xuICovXG5leHBvcnQgaW50ZXJmYWNlIFNpZ25hbHNPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBwZXJjZW50YWdlIG9mIHNpZ25hbHMgdGhhdCBuZWVkIHRvIGJlIHN1Y2Nlc3NmdWxcbiAgICpcbiAgICogSWYgdGhpcyBudW1iZXIgaXMgbGVzcyB0aGFuIDEwMCwgYSBwZXJjZW50YWdlIG9mIHNpZ25hbHMgbWF5IGJlIGZhaWx1cmVcbiAgICogc2lnbmFscyB3aGlsZSBzdGlsbCBzdWNjZWVkaW5nIHRoZSBjcmVhdGlvbiBvciB1cGRhdGUgaW4gQ2xvdWRGb3JtYXRpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IDEwMFxuICAgKi9cbiAgcmVhZG9ubHkgbWluU3VjY2Vzc1BlcmNlbnRhZ2U/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIEhvdyBsb25nIHRvIHdhaXQgZm9yIHRoZSBzaWduYWxzIHRvIGJlIHNlbnRcbiAgICpcbiAgICogVGhpcyBzaG91bGQgcmVmbGVjdCBob3cgbG9uZyBpdCB0YWtlcyB5b3VyIGluc3RhbmNlcyB0byBzdGFydCB1cFxuICAgKiAoaW5jbHVkaW5nIGluc3RhbmNlIHN0YXJ0IHRpbWUgYW5kIGluc3RhbmNlIGluaXRpYWxpemF0aW9uIHRpbWUpLlxuICAgKlxuICAgKiBAZGVmYXVsdCBEdXJhdGlvbi5taW51dGVzKDUpXG4gICAqL1xuICByZWFkb25seSB0aW1lb3V0PzogRHVyYXRpb247XG59XG5cbi8qKlxuICogSG93IGV4aXN0aW5nIGluc3RhbmNlcyBzaG91bGQgYmUgdXBkYXRlZFxuICovXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgVXBkYXRlUG9saWN5IHtcbiAgLyoqXG4gICAqIENyZWF0ZSBhIG5ldyBBdXRvU2NhbGluZ0dyb3VwIGFuZCBzd2l0Y2ggb3ZlciB0byBpdFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZXBsYWNpbmdVcGRhdGUoKTogVXBkYXRlUG9saWN5IHtcbiAgICByZXR1cm4gbmV3IGNsYXNzIGV4dGVuZHMgVXBkYXRlUG9saWN5IHtcbiAgICAgIHB1YmxpYyBfcmVuZGVyVXBkYXRlUG9saWN5KCk6IENmblVwZGF0ZVBvbGljeSB7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgYXV0b1NjYWxpbmdSZXBsYWNpbmdVcGRhdGU6IHsgd2lsbFJlcGxhY2U6IHRydWUgfSxcbiAgICAgICAgfTtcbiAgICAgIH1cbiAgICB9KCk7XG4gIH1cblxuICAvKipcbiAgICogUmVwbGFjZSB0aGUgaW5zdGFuY2VzIGluIHRoZSBBdXRvU2NhbGluZ0dyb3VwIG9uZSBieSBvbmUsIG9yIGluIGJhdGNoZXNcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcm9sbGluZ1VwZGF0ZShvcHRpb25zOiBSb2xsaW5nVXBkYXRlT3B0aW9ucyA9IHt9KTogVXBkYXRlUG9saWN5IHtcbiAgICBjb25zdCBtaW5TdWNjZXNzUGVyY2VudGFnZSA9IHZhbGlkYXRlUGVyY2VudGFnZShvcHRpb25zLm1pblN1Y2Nlc3NQZXJjZW50YWdlKTtcblxuICAgIHJldHVybiBuZXcgY2xhc3MgZXh0ZW5kcyBVcGRhdGVQb2xpY3kge1xuICAgICAgcHVibGljIF9yZW5kZXJVcGRhdGVQb2xpY3kocmVuZGVyT3B0aW9uczogUmVuZGVyVXBkYXRlT3B0aW9ucyk6IENmblVwZGF0ZVBvbGljeSB7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgYXV0b1NjYWxpbmdSb2xsaW5nVXBkYXRlOiB7XG4gICAgICAgICAgICBtYXhCYXRjaFNpemU6IG9wdGlvbnMubWF4QmF0Y2hTaXplLFxuICAgICAgICAgICAgbWluSW5zdGFuY2VzSW5TZXJ2aWNlOiBvcHRpb25zLm1pbkluc3RhbmNlc0luU2VydmljZSxcbiAgICAgICAgICAgIHN1c3BlbmRQcm9jZXNzZXM6IG9wdGlvbnMuc3VzcGVuZFByb2Nlc3NlcyA/PyBERUZBVUxUX1NVU1BFTkRfUFJPQ0VTU0VTLFxuICAgICAgICAgICAgbWluU3VjY2Vzc2Z1bEluc3RhbmNlc1BlcmNlbnQ6XG4gICAgICAgICAgICAgIG1pblN1Y2Nlc3NQZXJjZW50YWdlID8/IHJlbmRlck9wdGlvbnMuY3JlYXRpb25Qb2xpY3k/LmF1dG9TY2FsaW5nQ3JlYXRpb25Qb2xpY3k/Lm1pblN1Y2Nlc3NmdWxJbnN0YW5jZXNQZXJjZW50LFxuICAgICAgICAgICAgd2FpdE9uUmVzb3VyY2VTaWduYWxzOiBvcHRpb25zLndhaXRPblJlc291cmNlU2lnbmFscyA/PyByZW5kZXJPcHRpb25zLmNyZWF0aW9uUG9saWN5Py5yZXNvdXJjZVNpZ25hbCAhPT0gdW5kZWZpbmVkID8gdHJ1ZSA6IHVuZGVmaW5lZCxcbiAgICAgICAgICAgIHBhdXNlVGltZTogb3B0aW9ucy5wYXVzZVRpbWU/LnRvSXNvU3RyaW5nKCkgPz8gcmVuZGVyT3B0aW9ucy5jcmVhdGlvblBvbGljeT8ucmVzb3VyY2VTaWduYWw/LnRpbWVvdXQsXG4gICAgICAgICAgfSxcbiAgICAgICAgfTtcbiAgICAgIH1cbiAgICB9KCk7XG4gIH1cblxuICAvKipcbiAgICogUmVuZGVyIHRoZSBBU0cncyBDcmVhdGlvblBvbGljeVxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCBfcmVuZGVyVXBkYXRlUG9saWN5KHJlbmRlck9wdGlvbnM6IFJlbmRlclVwZGF0ZU9wdGlvbnMpOiBDZm5VcGRhdGVQb2xpY3k7XG59XG5cbi8qKlxuICogT3B0aW9ucyBmb3IgcmVuZGVyaW5nIFVwZGF0ZVBvbGljeVxuICovXG5pbnRlcmZhY2UgUmVuZGVyVXBkYXRlT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgQ3JlYXRpb24gUG9saWN5IGFscmVhZHkgY3JlYXRlZFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIENyZWF0aW9uUG9saWN5IGNvbmZpZ3VyZWRcbiAgICovXG4gIHJlYWRvbmx5IGNyZWF0aW9uUG9saWN5PzogQ2ZuQ3JlYXRpb25Qb2xpY3k7XG59XG5cbi8qKlxuICogT3B0aW9ucyBmb3IgY3VzdG9taXppbmcgdGhlIHJvbGxpbmcgdXBkYXRlXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUm9sbGluZ1VwZGF0ZU9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIG1heGltdW0gbnVtYmVyIG9mIGluc3RhbmNlcyB0aGF0IEFXUyBDbG91ZEZvcm1hdGlvbiB1cGRhdGVzIGF0IG9uY2UuXG4gICAqXG4gICAqIFRoaXMgbnVtYmVyIGFmZmVjdHMgdGhlIHNwZWVkIG9mIHRoZSByZXBsYWNlbWVudC5cbiAgICpcbiAgICogQGRlZmF1bHQgMVxuICAgKi9cbiAgcmVhZG9ubHkgbWF4QmF0Y2hTaXplPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgbWluaW11bSBudW1iZXIgb2YgaW5zdGFuY2VzIHRoYXQgbXVzdCBiZSBpbiBzZXJ2aWNlIGJlZm9yZSBtb3JlIGluc3RhbmNlcyBhcmUgcmVwbGFjZWQuXG4gICAqXG4gICAqIFRoaXMgbnVtYmVyIGFmZmVjdHMgdGhlIHNwZWVkIG9mIHRoZSByZXBsYWNlbWVudC5cbiAgICpcbiAgICogQGRlZmF1bHQgMFxuICAgKi9cbiAgcmVhZG9ubHkgbWluSW5zdGFuY2VzSW5TZXJ2aWNlPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgdGhlIEF1dG8gU2NhbGluZyBwcm9jZXNzZXMgdG8gc3VzcGVuZCBkdXJpbmcgYSBzdGFjayB1cGRhdGUuXG4gICAqXG4gICAqIFN1c3BlbmRpbmcgcHJvY2Vzc2VzIHByZXZlbnRzIEF1dG8gU2NhbGluZyBmcm9tIGludGVyZmVyaW5nIHdpdGggYSBzdGFja1xuICAgKiB1cGRhdGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IEhlYWx0aENoZWNrLCBSZXBsYWNlVW5oZWFsdGh5LCBBWlJlYmFsYW5jZSwgQWxhcm1Ob3RpZmljYXRpb24sIFNjaGVkdWxlZEFjdGlvbnMuXG4gICAqL1xuICByZWFkb25seSBzdXNwZW5kUHJvY2Vzc2VzPzogU2NhbGluZ1Byb2Nlc3NbXTtcblxuICAvKipcbiAgICogU3BlY2lmaWVzIHdoZXRoZXIgdGhlIEF1dG8gU2NhbGluZyBncm91cCB3YWl0cyBvbiBzaWduYWxzIGZyb20gbmV3IGluc3RhbmNlcyBkdXJpbmcgYW4gdXBkYXRlLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlIGlmIHlvdSBjb25maWd1cmVkIGBzaWduYWxzYCBvbiB0aGUgQXV0b1NjYWxpbmdHcm91cCwgZmFsc2Ugb3RoZXJ3aXNlXG4gICAqL1xuICByZWFkb25seSB3YWl0T25SZXNvdXJjZVNpZ25hbHM/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgcGF1c2UgdGltZSBhZnRlciBtYWtpbmcgYSBjaGFuZ2UgdG8gYSBiYXRjaCBvZiBpbnN0YW5jZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gVGhlIGB0aW1lb3V0YCBjb25maWd1cmVkIGZvciBgc2lnbmFsc2Agb24gdGhlIEF1dG9TY2FsaW5nR3JvdXBcbiAgICovXG4gIHJlYWRvbmx5IHBhdXNlVGltZT86IER1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBUaGUgcGVyY2VudGFnZSBvZiBpbnN0YW5jZXMgdGhhdCBtdXN0IHNpZ25hbCBzdWNjZXNzIGZvciB0aGUgdXBkYXRlIHRvIHN1Y2NlZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gVGhlIGBtaW5TdWNjZXNzUGVyY2VudGFnZWAgY29uZmlndXJlZCBmb3IgYHNpZ25hbHNgIG9uIHRoZSBBdXRvU2NhbGluZ0dyb3VwXG4gICAqL1xuICByZWFkb25seSBtaW5TdWNjZXNzUGVyY2VudGFnZT86IG51bWJlcjtcbn1cblxuLyoqXG4gKiBBIHNldCBvZiBncm91cCBtZXRyaWNzXG4gKi9cbmV4cG9ydCBjbGFzcyBHcm91cE1ldHJpY3Mge1xuXG4gIC8qKlxuICAgKiBSZXBvcnQgYWxsIGdyb3VwIG1ldHJpY3MuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGFsbCgpOiBHcm91cE1ldHJpY3Mge1xuICAgIHJldHVybiBuZXcgR3JvdXBNZXRyaWNzKCk7XG4gIH1cblxuICAvKipcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX21ldHJpY3MgPSBuZXcgU2V0PEdyb3VwTWV0cmljPigpO1xuXG4gIGNvbnN0cnVjdG9yKC4uLm1ldHJpY3M6IEdyb3VwTWV0cmljW10pIHtcbiAgICBtZXRyaWNzPy5mb3JFYWNoKG1ldHJpYyA9PiB0aGlzLl9tZXRyaWNzLmFkZChtZXRyaWMpKTtcbiAgfVxufVxuXG4vKipcbiAqIEdyb3VwIG1ldHJpY3MgdGhhdCBhbiBBdXRvIFNjYWxpbmcgZ3JvdXAgc2VuZHMgdG8gQW1hem9uIENsb3VkV2F0Y2guXG4gKi9cbmV4cG9ydCBjbGFzcyBHcm91cE1ldHJpYyB7XG5cbiAgLyoqXG4gICAqIFRoZSBtaW5pbXVtIHNpemUgb2YgdGhlIEF1dG8gU2NhbGluZyBncm91cFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBNSU5fU0laRSA9IG5ldyBHcm91cE1ldHJpYygnR3JvdXBNaW5TaXplJyk7XG5cbiAgLyoqXG4gICAqIFRoZSBtYXhpbXVtIHNpemUgb2YgdGhlIEF1dG8gU2NhbGluZyBncm91cFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBNQVhfU0laRSA9IG5ldyBHcm91cE1ldHJpYygnR3JvdXBNYXhTaXplJyk7XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgaW5zdGFuY2VzIHRoYXQgdGhlIEF1dG8gU2NhbGluZyBncm91cCBhdHRlbXB0cyB0byBtYWludGFpblxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBERVNJUkVEX0NBUEFDSVRZID0gbmV3IEdyb3VwTWV0cmljKCdHcm91cERlc2lyZWRDYXBhY2l0eScpO1xuXG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIGluc3RhbmNlcyB0aGF0IGFyZSBydW5uaW5nIGFzIHBhcnQgb2YgdGhlIEF1dG8gU2NhbGluZyBncm91cFxuICAgKiBUaGlzIG1ldHJpYyBkb2VzIG5vdCBpbmNsdWRlIGluc3RhbmNlcyB0aGF0IGFyZSBwZW5kaW5nIG9yIHRlcm1pbmF0aW5nXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IElOX1NFUlZJQ0VfSU5TVEFOQ0VTID0gbmV3IEdyb3VwTWV0cmljKCdHcm91cEluU2VydmljZUluc3RhbmNlcycpO1xuXG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIGluc3RhbmNlcyB0aGF0IGFyZSBwZW5kaW5nXG4gICAqIEEgcGVuZGluZyBpbnN0YW5jZSBpcyBub3QgeWV0IGluIHNlcnZpY2UsIHRoaXMgbWV0cmljIGRvZXMgbm90IGluY2x1ZGUgaW5zdGFuY2VzIHRoYXQgYXJlIGluIHNlcnZpY2Ugb3IgdGVybWluYXRpbmdcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgUEVORElOR19JTlNUQU5DRVMgPSBuZXcgR3JvdXBNZXRyaWMoJ0dyb3VwUGVuZGluZ0luc3RhbmNlcycpO1xuXG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIGluc3RhbmNlcyB0aGF0IGFyZSBpbiBhIFN0YW5kYnkgc3RhdGVcbiAgICogSW5zdGFuY2VzIGluIHRoaXMgc3RhdGUgYXJlIHN0aWxsIHJ1bm5pbmcgYnV0IGFyZSBub3QgYWN0aXZlbHkgaW4gc2VydmljZVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBTVEFOREJZX0lOU1RBTkNFUyA9IG5ldyBHcm91cE1ldHJpYygnR3JvdXBTdGFuZGJ5SW5zdGFuY2VzJyk7XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgaW5zdGFuY2VzIHRoYXQgYXJlIGluIHRoZSBwcm9jZXNzIG9mIHRlcm1pbmF0aW5nXG4gICAqIFRoaXMgbWV0cmljIGRvZXMgbm90IGluY2x1ZGUgaW5zdGFuY2VzIHRoYXQgYXJlIGluIHNlcnZpY2Ugb3IgcGVuZGluZ1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBURVJNSU5BVElOR19JTlNUQU5DRVMgPSBuZXcgR3JvdXBNZXRyaWMoJ0dyb3VwVGVybWluYXRpbmdJbnN0YW5jZXMnKTtcblxuICAvKipcbiAgICogVGhlIHRvdGFsIG51bWJlciBvZiBpbnN0YW5jZXMgaW4gdGhlIEF1dG8gU2NhbGluZyBncm91cFxuICAgKiBUaGlzIG1ldHJpYyBpZGVudGlmaWVzIHRoZSBudW1iZXIgb2YgaW5zdGFuY2VzIHRoYXQgYXJlIGluIHNlcnZpY2UsIHBlbmRpbmcsIGFuZCB0ZXJtaW5hdGluZ1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBUT1RBTF9JTlNUQU5DRVMgPSBuZXcgR3JvdXBNZXRyaWMoJ0dyb3VwVG90YWxJbnN0YW5jZXMnKTtcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGdyb3VwIG1ldHJpY1xuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcblxuICBjb25zdHJ1Y3RvcihuYW1lOiBzdHJpbmcpIHtcbiAgICB0aGlzLm5hbWUgPSBuYW1lO1xuICB9XG59XG5cbmFic3RyYWN0IGNsYXNzIEF1dG9TY2FsaW5nR3JvdXBCYXNlIGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJQXV0b1NjYWxpbmdHcm91cCB7XG5cbiAgcHVibGljIGFic3RyYWN0IGF1dG9TY2FsaW5nR3JvdXBOYW1lOiBzdHJpbmc7XG4gIHB1YmxpYyBhYnN0cmFjdCBhdXRvU2NhbGluZ0dyb3VwQXJuOiBzdHJpbmc7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBvc1R5cGU6IGVjMi5PcGVyYXRpbmdTeXN0ZW1UeXBlO1xuICBwcm90ZWN0ZWQgYWxiVGFyZ2V0R3JvdXA/OiBlbGJ2Mi5BcHBsaWNhdGlvblRhcmdldEdyb3VwO1xuICBwdWJsaWMgcmVhZG9ubHkgZ3JhbnRQcmluY2lwYWw6IGlhbS5JUHJpbmNpcGFsID0gbmV3IGlhbS5Vbmtub3duUHJpbmNpcGFsKHsgcmVzb3VyY2U6IHRoaXMgfSk7XG5cbiAgLyoqXG4gICAqIFNlbmQgYSBtZXNzYWdlIHRvIGVpdGhlciBhbiBTUVMgcXVldWUgb3IgU05TIHRvcGljIHdoZW4gaW5zdGFuY2VzIGxhdW5jaCBvciB0ZXJtaW5hdGVcbiAgICovXG4gIHB1YmxpYyBhZGRMaWZlY3ljbGVIb29rKGlkOiBzdHJpbmcsIHByb3BzOiBCYXNpY0xpZmVjeWNsZUhvb2tQcm9wcyk6IExpZmVjeWNsZUhvb2sge1xuICAgIHJldHVybiBuZXcgTGlmZWN5Y2xlSG9vayh0aGlzLCBgTGlmZWN5Y2xlSG9vayR7aWR9YCwge1xuICAgICAgYXV0b1NjYWxpbmdHcm91cDogdGhpcyxcbiAgICAgIC4uLnByb3BzLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhIHBvb2wgb2YgcHJlLWluaXRpYWxpemVkIEVDMiBpbnN0YW5jZXMgdGhhdCBzaXRzIGFsb25nc2lkZSBhbiBBdXRvIFNjYWxpbmcgZ3JvdXBcbiAgICovXG4gIHB1YmxpYyBhZGRXYXJtUG9vbChvcHRpb25zPzogV2FybVBvb2xPcHRpb25zKTogV2FybVBvb2wge1xuICAgIHJldHVybiBuZXcgV2FybVBvb2wodGhpcywgJ1dhcm1Qb29sJywge1xuICAgICAgYXV0b1NjYWxpbmdHcm91cDogdGhpcyxcbiAgICAgIC4uLm9wdGlvbnMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogU2NhbGUgb3V0IG9yIGluIGJhc2VkIG9uIHRpbWVcbiAgICovXG4gIHB1YmxpYyBzY2FsZU9uU2NoZWR1bGUoaWQ6IHN0cmluZywgcHJvcHM6IEJhc2ljU2NoZWR1bGVkQWN0aW9uUHJvcHMpOiBTY2hlZHVsZWRBY3Rpb24ge1xuICAgIHJldHVybiBuZXcgU2NoZWR1bGVkQWN0aW9uKHRoaXMsIGBTY2hlZHVsZWRBY3Rpb24ke2lkfWAsIHtcbiAgICAgIGF1dG9TY2FsaW5nR3JvdXA6IHRoaXMsXG4gICAgICAuLi5wcm9wcyxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTY2FsZSBvdXQgb3IgaW4gdG8gYWNoaWV2ZSBhIHRhcmdldCBDUFUgdXRpbGl6YXRpb25cbiAgICovXG4gIHB1YmxpYyBzY2FsZU9uQ3B1VXRpbGl6YXRpb24oaWQ6IHN0cmluZywgcHJvcHM6IENwdVV0aWxpemF0aW9uU2NhbGluZ1Byb3BzKTogVGFyZ2V0VHJhY2tpbmdTY2FsaW5nUG9saWN5IHtcbiAgICByZXR1cm4gbmV3IFRhcmdldFRyYWNraW5nU2NhbGluZ1BvbGljeSh0aGlzLCBgU2NhbGluZ1BvbGljeSR7aWR9YCwge1xuICAgICAgYXV0b1NjYWxpbmdHcm91cDogdGhpcyxcbiAgICAgIHByZWRlZmluZWRNZXRyaWM6IFByZWRlZmluZWRNZXRyaWMuQVNHX0FWRVJBR0VfQ1BVX1VUSUxJWkFUSU9OLFxuICAgICAgdGFyZ2V0VmFsdWU6IHByb3BzLnRhcmdldFV0aWxpemF0aW9uUGVyY2VudCxcbiAgICAgIC4uLnByb3BzLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFNjYWxlIG91dCBvciBpbiB0byBhY2hpZXZlIGEgdGFyZ2V0IG5ldHdvcmsgaW5ncmVzcyByYXRlXG4gICAqL1xuICBwdWJsaWMgc2NhbGVPbkluY29taW5nQnl0ZXMoaWQ6IHN0cmluZywgcHJvcHM6IE5ldHdvcmtVdGlsaXphdGlvblNjYWxpbmdQcm9wcyk6IFRhcmdldFRyYWNraW5nU2NhbGluZ1BvbGljeSB7XG4gICAgcmV0dXJuIG5ldyBUYXJnZXRUcmFja2luZ1NjYWxpbmdQb2xpY3kodGhpcywgYFNjYWxpbmdQb2xpY3kke2lkfWAsIHtcbiAgICAgIGF1dG9TY2FsaW5nR3JvdXA6IHRoaXMsXG4gICAgICBwcmVkZWZpbmVkTWV0cmljOiBQcmVkZWZpbmVkTWV0cmljLkFTR19BVkVSQUdFX05FVFdPUktfSU4sXG4gICAgICB0YXJnZXRWYWx1ZTogcHJvcHMudGFyZ2V0Qnl0ZXNQZXJTZWNvbmQsXG4gICAgICAuLi5wcm9wcyxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTY2FsZSBvdXQgb3IgaW4gdG8gYWNoaWV2ZSBhIHRhcmdldCBuZXR3b3JrIGVncmVzcyByYXRlXG4gICAqL1xuICBwdWJsaWMgc2NhbGVPbk91dGdvaW5nQnl0ZXMoaWQ6IHN0cmluZywgcHJvcHM6IE5ldHdvcmtVdGlsaXphdGlvblNjYWxpbmdQcm9wcyk6IFRhcmdldFRyYWNraW5nU2NhbGluZ1BvbGljeSB7XG4gICAgcmV0dXJuIG5ldyBUYXJnZXRUcmFja2luZ1NjYWxpbmdQb2xpY3kodGhpcywgYFNjYWxpbmdQb2xpY3kke2lkfWAsIHtcbiAgICAgIGF1dG9TY2FsaW5nR3JvdXA6IHRoaXMsXG4gICAgICBwcmVkZWZpbmVkTWV0cmljOiBQcmVkZWZpbmVkTWV0cmljLkFTR19BVkVSQUdFX05FVFdPUktfT1VULFxuICAgICAgdGFyZ2V0VmFsdWU6IHByb3BzLnRhcmdldEJ5dGVzUGVyU2Vjb25kLFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogU2NhbGUgb3V0IG9yIGluIHRvIGFjaGlldmUgYSB0YXJnZXQgcmVxdWVzdCBoYW5kbGluZyByYXRlXG4gICAqXG4gICAqIFRoZSBBdXRvU2NhbGluZ0dyb3VwIG11c3QgaGF2ZSBiZWVuIGF0dGFjaGVkIHRvIGFuIEFwcGxpY2F0aW9uIExvYWQgQmFsYW5jZXJcbiAgICogaW4gb3JkZXIgdG8gYmUgYWJsZSB0byBjYWxsIHRoaXMuXG4gICAqL1xuICBwdWJsaWMgc2NhbGVPblJlcXVlc3RDb3VudChpZDogc3RyaW5nLCBwcm9wczogUmVxdWVzdENvdW50U2NhbGluZ1Byb3BzKTogVGFyZ2V0VHJhY2tpbmdTY2FsaW5nUG9saWN5IHtcbiAgICBpZiAodGhpcy5hbGJUYXJnZXRHcm91cCA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0F0dGFjaCB0aGUgQXV0b1NjYWxpbmdHcm91cCB0byBhIG5vbi1pbXBvcnRlZCBBcHBsaWNhdGlvbiBMb2FkIEJhbGFuY2VyIGJlZm9yZSBjYWxsaW5nIHNjYWxlT25SZXF1ZXN0Q291bnQoKScpO1xuICAgIH1cblxuICAgIGNvbnN0IHJlc291cmNlTGFiZWwgPSBgJHt0aGlzLmFsYlRhcmdldEdyb3VwLmZpcnN0TG9hZEJhbGFuY2VyRnVsbE5hbWV9LyR7dGhpcy5hbGJUYXJnZXRHcm91cC50YXJnZXRHcm91cEZ1bGxOYW1lfWA7XG5cbiAgICBpZiAoKHByb3BzLnRhcmdldFJlcXVlc3RzUGVyTWludXRlID09PSB1bmRlZmluZWQpID09PSAocHJvcHMudGFyZ2V0UmVxdWVzdHNQZXJTZWNvbmQgPT09IHVuZGVmaW5lZCkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignU3BlY2lmeSBleGFjdGx5IG9uZSBvZiBcXCd0YXJnZXRSZXF1ZXN0c1Blck1pbnV0ZVxcJyBvciBcXCd0YXJnZXRSZXF1ZXN0c1BlclNlY29uZFxcJycpO1xuICAgIH1cblxuICAgIGxldCBycG06IG51bWJlcjtcbiAgICBpZiAocHJvcHMudGFyZ2V0UmVxdWVzdHNQZXJTZWNvbmQgIT09IHVuZGVmaW5lZCkge1xuICAgICAgaWYgKFRva2VuLmlzVW5yZXNvbHZlZChwcm9wcy50YXJnZXRSZXF1ZXN0c1BlclNlY29uZCkpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdcXCd0YXJnZXRSZXF1ZXN0c1BlclNlY29uZFxcJyBjYW5ub3QgYmUgYW4gdW5yZXNvbHZlZCB2YWx1ZTsgdXNlIFxcJ3RhcmdldFJlcXVlc3RzUGVyTWludXRlXFwnIGluc3RlYWQuJyk7XG4gICAgICB9XG4gICAgICBycG0gPSBwcm9wcy50YXJnZXRSZXF1ZXN0c1BlclNlY29uZCAqIDYwO1xuICAgIH0gZWxzZSB7XG4gICAgICBycG0gPSBwcm9wcy50YXJnZXRSZXF1ZXN0c1Blck1pbnV0ZSE7XG4gICAgfVxuXG4gICAgY29uc3QgcG9saWN5ID0gbmV3IFRhcmdldFRyYWNraW5nU2NhbGluZ1BvbGljeSh0aGlzLCBgU2NhbGluZ1BvbGljeSR7aWR9YCwge1xuICAgICAgYXV0b1NjYWxpbmdHcm91cDogdGhpcyxcbiAgICAgIHByZWRlZmluZWRNZXRyaWM6IFByZWRlZmluZWRNZXRyaWMuQUxCX1JFUVVFU1RfQ09VTlRfUEVSX1RBUkdFVCxcbiAgICAgIHRhcmdldFZhbHVlOiBycG0sXG4gICAgICByZXNvdXJjZUxhYmVsLFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG5cbiAgICBwb2xpY3kubm9kZS5hZGREZXBlbmRlbmN5KHRoaXMuYWxiVGFyZ2V0R3JvdXAubG9hZEJhbGFuY2VyQXR0YWNoZWQpO1xuICAgIHJldHVybiBwb2xpY3k7XG4gIH1cblxuICAvKipcbiAgICogU2NhbGUgb3V0IG9yIGluIGluIG9yZGVyIHRvIGtlZXAgYSBtZXRyaWMgYXJvdW5kIGEgdGFyZ2V0IHZhbHVlXG4gICAqL1xuICBwdWJsaWMgc2NhbGVUb1RyYWNrTWV0cmljKGlkOiBzdHJpbmcsIHByb3BzOiBNZXRyaWNUYXJnZXRUcmFja2luZ1Byb3BzKTogVGFyZ2V0VHJhY2tpbmdTY2FsaW5nUG9saWN5IHtcbiAgICByZXR1cm4gbmV3IFRhcmdldFRyYWNraW5nU2NhbGluZ1BvbGljeSh0aGlzLCBgU2NhbGluZ1BvbGljeSR7aWR9YCwge1xuICAgICAgYXV0b1NjYWxpbmdHcm91cDogdGhpcyxcbiAgICAgIGN1c3RvbU1ldHJpYzogcHJvcHMubWV0cmljLFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogU2NhbGUgb3V0IG9yIGluLCBpbiByZXNwb25zZSB0byBhIG1ldHJpY1xuICAgKi9cbiAgcHVibGljIHNjYWxlT25NZXRyaWMoaWQ6IHN0cmluZywgcHJvcHM6IEJhc2ljU3RlcFNjYWxpbmdQb2xpY3lQcm9wcyk6IFN0ZXBTY2FsaW5nUG9saWN5IHtcbiAgICByZXR1cm4gbmV3IFN0ZXBTY2FsaW5nUG9saWN5KHRoaXMsIGlkLCB7IC4uLnByb3BzLCBhdXRvU2NhbGluZ0dyb3VwOiB0aGlzIH0pO1xuICB9XG5cbiAgcHVibGljIGFkZFVzZXJEYXRhKC4uLl9jb21tYW5kczogc3RyaW5nW10pOiB2b2lkIHtcbiAgICAvLyBkbyBub3RoaW5nXG4gIH1cbn1cblxuLyoqXG4gKiBBIEZsZWV0IHJlcHJlc2VudHMgYSBtYW5hZ2VkIHNldCBvZiBFQzIgaW5zdGFuY2VzXG4gKlxuICogVGhlIEZsZWV0IG1vZGVscyBhIG51bWJlciBvZiBBdXRvU2NhbGluZ0dyb3VwcywgYSBsYXVuY2ggY29uZmlndXJhdGlvbiwgYVxuICogc2VjdXJpdHkgZ3JvdXAgYW5kIGFuIGluc3RhbmNlIHJvbGUuXG4gKlxuICogSXQgYWxsb3dzIGFkZGluZyBhcmJpdHJhcnkgY29tbWFuZHMgdG8gdGhlIHN0YXJ0dXAgc2NyaXB0cyBvZiB0aGUgaW5zdGFuY2VzXG4gKiBpbiB0aGUgZmxlZXQuXG4gKlxuICogVGhlIEFTRyBzcGFucyB0aGUgYXZhaWxhYmlsaXR5IHpvbmVzIHNwZWNpZmllZCBieSB2cGNTdWJuZXRzLCBmYWxsaW5nIGJhY2sgdG9cbiAqIHRoZSBWcGMgZGVmYXVsdCBzdHJhdGVneSBpZiBub3Qgc3BlY2lmaWVkLlxuICovXG5leHBvcnQgY2xhc3MgQXV0b1NjYWxpbmdHcm91cCBleHRlbmRzIEF1dG9TY2FsaW5nR3JvdXBCYXNlIGltcGxlbWVudHNcbiAgZWxiLklMb2FkQmFsYW5jZXJUYXJnZXQsXG4gIGVjMi5JQ29ubmVjdGFibGUsXG4gIGVsYnYyLklBcHBsaWNhdGlvbkxvYWRCYWxhbmNlclRhcmdldCxcbiAgZWxidjIuSU5ldHdvcmtMb2FkQmFsYW5jZXJUYXJnZXQge1xuXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUF1dG9TY2FsaW5nR3JvdXBOYW1lKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGF1dG9TY2FsaW5nR3JvdXBOYW1lOiBzdHJpbmcpOiBJQXV0b1NjYWxpbmdHcm91cCB7XG4gICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgQXV0b1NjYWxpbmdHcm91cEJhc2Uge1xuICAgICAgcHVibGljIGF1dG9TY2FsaW5nR3JvdXBOYW1lID0gYXV0b1NjYWxpbmdHcm91cE5hbWU7XG4gICAgICBwdWJsaWMgYXV0b1NjYWxpbmdHcm91cEFybiA9IFN0YWNrLm9mKHRoaXMpLmZvcm1hdEFybih7XG4gICAgICAgIHNlcnZpY2U6ICdhdXRvc2NhbGluZycsXG4gICAgICAgIHJlc291cmNlOiAnYXV0b1NjYWxpbmdHcm91cDoqOmF1dG9TY2FsaW5nR3JvdXBOYW1lJyxcbiAgICAgICAgcmVzb3VyY2VOYW1lOiB0aGlzLmF1dG9TY2FsaW5nR3JvdXBOYW1lLFxuICAgICAgfSk7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgb3NUeXBlID0gZWMyLk9wZXJhdGluZ1N5c3RlbVR5cGUuVU5LTk9XTjtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSB0eXBlIG9mIE9TIGluc3RhbmNlcyBvZiB0aGlzIGZsZWV0IGFyZSBydW5uaW5nLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IG9zVHlwZTogZWMyLk9wZXJhdGluZ1N5c3RlbVR5cGU7XG5cbiAgLyoqXG4gICAqIFRoZSBwcmluY2lwYWwgdG8gZ3JhbnQgcGVybWlzc2lvbnMgdG9cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBncmFudFByaW5jaXBhbDogaWFtLklQcmluY2lwYWw7XG5cbiAgLyoqXG4gICAqIE5hbWUgb2YgdGhlIEF1dG9TY2FsaW5nR3JvdXBcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhdXRvU2NhbGluZ0dyb3VwTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBcm4gb2YgdGhlIEF1dG9TY2FsaW5nR3JvdXBcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhdXRvU2NhbGluZ0dyb3VwQXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBtYXhpbXVtIHNwb3QgcHJpY2UgY29uZmlndXJlZCBmb3IgdGhlIGF1dG9zY2FsaW5nIGdyb3VwLiBgdW5kZWZpbmVkYFxuICAgKiBpbmRpY2F0ZXMgdGhhdCB0aGlzIGdyb3VwIHVzZXMgb24tZGVtYW5kIGNhcGFjaXR5LlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHNwb3RQcmljZT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG1heGltdW0gYW1vdW50IG9mIHRpbWUgdGhhdCBhbiBpbnN0YW5jZSBjYW4gYmUgaW4gc2VydmljZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBtYXhJbnN0YW5jZUxpZmV0aW1lPzogRHVyYXRpb247XG5cbiAgcHJpdmF0ZSByZWFkb25seSBhdXRvU2NhbGluZ0dyb3VwOiBDZm5BdXRvU2NhbGluZ0dyb3VwO1xuICBwcml2YXRlIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXA/OiBlYzIuSVNlY3VyaXR5R3JvdXA7XG4gIHByaXZhdGUgcmVhZG9ubHkgc2VjdXJpdHlHcm91cHM/OiBlYzIuSVNlY3VyaXR5R3JvdXBbXTtcbiAgcHJpdmF0ZSByZWFkb25seSBsb2FkQmFsYW5jZXJOYW1lczogc3RyaW5nW10gPSBbXTtcbiAgcHJpdmF0ZSByZWFkb25seSB0YXJnZXRHcm91cEFybnM6IHN0cmluZ1tdID0gW107XG4gIHByaXZhdGUgcmVhZG9ubHkgZ3JvdXBNZXRyaWNzOiBHcm91cE1ldHJpY3NbXSA9IFtdO1xuICBwcml2YXRlIHJlYWRvbmx5IG5vdGlmaWNhdGlvbnM6IE5vdGlmaWNhdGlvbkNvbmZpZ3VyYXRpb25bXSA9IFtdO1xuICBwcml2YXRlIHJlYWRvbmx5IGxhdW5jaFRlbXBsYXRlPzogZWMyLkxhdW5jaFRlbXBsYXRlO1xuICBwcml2YXRlIHJlYWRvbmx5IF9jb25uZWN0aW9ucz86IGVjMi5Db25uZWN0aW9ucztcbiAgcHJpdmF0ZSByZWFkb25seSBfdXNlckRhdGE/OiBlYzIuVXNlckRhdGE7XG4gIHByaXZhdGUgcmVhZG9ubHkgX3JvbGU/OiBpYW0uSVJvbGU7XG5cbiAgcHJvdGVjdGVkIG5ld0luc3RhbmNlc1Byb3RlY3RlZEZyb21TY2FsZUluPzogYm9vbGVhbjtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQXV0b1NjYWxpbmdHcm91cFByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICBwaHlzaWNhbE5hbWU6IHByb3BzLmF1dG9TY2FsaW5nR3JvdXBOYW1lLFxuICAgIH0pO1xuXG4gICAgdGhpcy5uZXdJbnN0YW5jZXNQcm90ZWN0ZWRGcm9tU2NhbGVJbiA9IHByb3BzLm5ld0luc3RhbmNlc1Byb3RlY3RlZEZyb21TY2FsZUluO1xuXG4gICAgaWYgKHByb3BzLmluaXRPcHRpb25zICYmICFwcm9wcy5pbml0KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1NldHRpbmcgXFwnaW5pdE9wdGlvbnNcXCcgcmVxdWlyZXMgdGhhdCBcXCdpbml0XFwnIGlzIGFsc28gc2V0Jyk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmdyb3VwTWV0cmljcykge1xuICAgICAgdGhpcy5ncm91cE1ldHJpY3MucHVzaCguLi5wcm9wcy5ncm91cE1ldHJpY3MpO1xuICAgIH1cblxuICAgIGxldCBsYXVuY2hDb25maWc6IENmbkxhdW5jaENvbmZpZ3VyYXRpb24gfCB1bmRlZmluZWQgPSB1bmRlZmluZWQ7XG4gICAgaWYgKHByb3BzLmxhdW5jaFRlbXBsYXRlIHx8IHByb3BzLm1peGVkSW5zdGFuY2VzUG9saWN5KSB7XG4gICAgICB0aGlzLnZlcmlmeU5vTGF1bmNoQ29uZmlnUHJvcElzR2l2ZW4ocHJvcHMpO1xuXG4gICAgICBjb25zdCBiYXJlTGF1bmNoVGVtcGxhdGUgPSBwcm9wcy5sYXVuY2hUZW1wbGF0ZTtcbiAgICAgIGNvbnN0IG1peGVkSW5zdGFuY2VzUG9saWN5ID0gcHJvcHMubWl4ZWRJbnN0YW5jZXNQb2xpY3k7XG5cbiAgICAgIGlmIChiYXJlTGF1bmNoVGVtcGxhdGUgJiYgbWl4ZWRJbnN0YW5jZXNQb2xpY3kpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdTZXR0aW5nIFxcJ21peGVkSW5zdGFuY2VzUG9saWN5XFwnIG11c3Qgbm90IGJlIHNldCB3aGVuIFxcJ2xhdW5jaFRlbXBsYXRlXFwnIGlzIHNldCcpO1xuICAgICAgfVxuXG4gICAgICBpZiAoYmFyZUxhdW5jaFRlbXBsYXRlICYmIGJhcmVMYXVuY2hUZW1wbGF0ZSBpbnN0YW5jZW9mIGVjMi5MYXVuY2hUZW1wbGF0ZSkge1xuICAgICAgICBpZiAoIWJhcmVMYXVuY2hUZW1wbGF0ZS5pbnN0YW5jZVR5cGUpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1NldHRpbmcgXFwnbGF1bmNoVGVtcGxhdGVcXCcgcmVxdWlyZXMgaXRzIFxcJ2luc3RhbmNlVHlwZVxcJyB0byBiZSBzZXQnKTtcbiAgICAgICAgfVxuXG4gICAgICAgIGlmICghYmFyZUxhdW5jaFRlbXBsYXRlLmltYWdlSWQpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1NldHRpbmcgXFwnbGF1bmNoVGVtcGxhdGVcXCcgcmVxdWlyZXMgaXRzIFxcJ21hY2hpbmVJbWFnZVxcJyB0byBiZSBzZXQnKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMubGF1bmNoVGVtcGxhdGUgPSBiYXJlTGF1bmNoVGVtcGxhdGU7XG4gICAgICB9XG5cbiAgICAgIGlmIChtaXhlZEluc3RhbmNlc1BvbGljeSAmJiBtaXhlZEluc3RhbmNlc1BvbGljeS5sYXVuY2hUZW1wbGF0ZSBpbnN0YW5jZW9mIGVjMi5MYXVuY2hUZW1wbGF0ZSkge1xuICAgICAgICBpZiAoIW1peGVkSW5zdGFuY2VzUG9saWN5LmxhdW5jaFRlbXBsYXRlLmltYWdlSWQpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1NldHRpbmcgXFwnbWl4ZWRJbnN0YW5jZXNQb2xpY3kubGF1bmNoVGVtcGxhdGVcXCcgcmVxdWlyZXMgaXRzIFxcJ21hY2hpbmVJbWFnZVxcJyB0byBiZSBzZXQnKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMubGF1bmNoVGVtcGxhdGUgPSBtaXhlZEluc3RhbmNlc1BvbGljeS5sYXVuY2hUZW1wbGF0ZTtcbiAgICAgIH1cblxuICAgICAgdGhpcy5fcm9sZSA9IHRoaXMubGF1bmNoVGVtcGxhdGU/LnJvbGU7XG4gICAgICB0aGlzLmdyYW50UHJpbmNpcGFsID0gdGhpcy5fcm9sZSB8fCBuZXcgaWFtLlVua25vd25QcmluY2lwYWwoeyByZXNvdXJjZTogdGhpcyB9KTtcblxuICAgICAgdGhpcy5vc1R5cGUgPSB0aGlzLmxhdW5jaFRlbXBsYXRlPy5vc1R5cGUgfHwgZWMyLk9wZXJhdGluZ1N5c3RlbVR5cGUuVU5LTk9XTjtcbiAgICB9IGVsc2Uge1xuICAgICAgaWYgKCFwcm9wcy5tYWNoaW5lSW1hZ2UpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdTZXR0aW5nIFxcJ21hY2hpbmVJbWFnZVxcJyBpcyByZXF1aXJlZCB3aGVuIFxcJ2xhdW5jaFRlbXBsYXRlXFwnIGFuZCBcXCdtaXhlZEluc3RhbmNlc1BvbGljeVxcJyBpcyBub3Qgc2V0Jyk7XG4gICAgICB9XG4gICAgICBpZiAoIXByb3BzLmluc3RhbmNlVHlwZSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1NldHRpbmcgXFwnaW5zdGFuY2VUeXBlXFwnIGlzIHJlcXVpcmVkIHdoZW4gXFwnbGF1bmNoVGVtcGxhdGVcXCcgYW5kIFxcJ21peGVkSW5zdGFuY2VzUG9saWN5XFwnIGlzIG5vdCBzZXQnKTtcbiAgICAgIH1cblxuICAgICAgdGhpcy5zZWN1cml0eUdyb3VwID0gcHJvcHMuc2VjdXJpdHlHcm91cCB8fCBuZXcgZWMyLlNlY3VyaXR5R3JvdXAodGhpcywgJ0luc3RhbmNlU2VjdXJpdHlHcm91cCcsIHtcbiAgICAgICAgdnBjOiBwcm9wcy52cGMsXG4gICAgICAgIGFsbG93QWxsT3V0Ym91bmQ6IHByb3BzLmFsbG93QWxsT3V0Ym91bmQgIT09IGZhbHNlLFxuICAgICAgfSk7XG4gICAgICB0aGlzLl9jb25uZWN0aW9ucyA9IG5ldyBlYzIuQ29ubmVjdGlvbnMoeyBzZWN1cml0eUdyb3VwczogW3RoaXMuc2VjdXJpdHlHcm91cF0gfSk7XG4gICAgICB0aGlzLnNlY3VyaXR5R3JvdXBzID0gW3RoaXMuc2VjdXJpdHlHcm91cF07XG4gICAgICBUYWdzLm9mKHRoaXMpLmFkZChOQU1FX1RBRywgdGhpcy5ub2RlLnBhdGgpO1xuXG4gICAgICB0aGlzLl9yb2xlID0gcHJvcHMucm9sZSB8fCBuZXcgaWFtLlJvbGUodGhpcywgJ0luc3RhbmNlUm9sZScsIHtcbiAgICAgICAgcm9sZU5hbWU6IFBoeXNpY2FsTmFtZS5HRU5FUkFURV9JRl9ORUVERUQsXG4gICAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdlYzIuYW1hem9uYXdzLmNvbScpLFxuICAgICAgfSk7XG5cbiAgICAgIHRoaXMuZ3JhbnRQcmluY2lwYWwgPSB0aGlzLl9yb2xlO1xuXG4gICAgICBjb25zdCBpYW1Qcm9maWxlID0gbmV3IGlhbS5DZm5JbnN0YW5jZVByb2ZpbGUodGhpcywgJ0luc3RhbmNlUHJvZmlsZScsIHtcbiAgICAgICAgcm9sZXM6IFt0aGlzLnJvbGUucm9sZU5hbWVdLFxuICAgICAgfSk7XG5cbiAgICAgIC8vIHVzZSBkZWxheWVkIGV2YWx1YXRpb25cbiAgICAgIGNvbnN0IGltYWdlQ29uZmlnID0gcHJvcHMubWFjaGluZUltYWdlLmdldEltYWdlKHRoaXMpO1xuICAgICAgdGhpcy5fdXNlckRhdGEgPSBwcm9wcy51c2VyRGF0YSA/PyBpbWFnZUNvbmZpZy51c2VyRGF0YTtcbiAgICAgIGNvbnN0IHVzZXJEYXRhVG9rZW4gPSBMYXp5LnN0cmluZyh7IHByb2R1Y2U6ICgpID0+IEZuLmJhc2U2NCh0aGlzLnVzZXJEYXRhIS5yZW5kZXIoKSkgfSk7XG4gICAgICBjb25zdCBzZWN1cml0eUdyb3Vwc1Rva2VuID0gTGF6eS5saXN0KHsgcHJvZHVjZTogKCkgPT4gdGhpcy5zZWN1cml0eUdyb3VwcyEubWFwKHNnID0+IHNnLnNlY3VyaXR5R3JvdXBJZCkgfSk7XG5cbiAgICAgIGxhdW5jaENvbmZpZyA9IG5ldyBDZm5MYXVuY2hDb25maWd1cmF0aW9uKHRoaXMsICdMYXVuY2hDb25maWcnLCB7XG4gICAgICAgIGltYWdlSWQ6IGltYWdlQ29uZmlnLmltYWdlSWQsXG4gICAgICAgIGtleU5hbWU6IHByb3BzLmtleU5hbWUsXG4gICAgICAgIGluc3RhbmNlVHlwZTogcHJvcHMuaW5zdGFuY2VUeXBlLnRvU3RyaW5nKCksXG4gICAgICAgIGluc3RhbmNlTW9uaXRvcmluZzogKHByb3BzLmluc3RhbmNlTW9uaXRvcmluZyAhPT0gdW5kZWZpbmVkID8gKHByb3BzLmluc3RhbmNlTW9uaXRvcmluZyA9PT0gTW9uaXRvcmluZy5ERVRBSUxFRCkgOiB1bmRlZmluZWQpLFxuICAgICAgICBzZWN1cml0eUdyb3Vwczogc2VjdXJpdHlHcm91cHNUb2tlbixcbiAgICAgICAgaWFtSW5zdGFuY2VQcm9maWxlOiBpYW1Qcm9maWxlLnJlZixcbiAgICAgICAgdXNlckRhdGE6IHVzZXJEYXRhVG9rZW4sXG4gICAgICAgIGFzc29jaWF0ZVB1YmxpY0lwQWRkcmVzczogcHJvcHMuYXNzb2NpYXRlUHVibGljSXBBZGRyZXNzLFxuICAgICAgICBzcG90UHJpY2U6IHByb3BzLnNwb3RQcmljZSxcbiAgICAgICAgYmxvY2tEZXZpY2VNYXBwaW5nczogKHByb3BzLmJsb2NrRGV2aWNlcyAhPT0gdW5kZWZpbmVkID9cbiAgICAgICAgICBzeW50aGVzaXplQmxvY2tEZXZpY2VNYXBwaW5ncyh0aGlzLCBwcm9wcy5ibG9ja0RldmljZXMpIDogdW5kZWZpbmVkKSxcbiAgICAgIH0pO1xuXG4gICAgICBsYXVuY2hDb25maWcubm9kZS5hZGREZXBlbmRlbmN5KHRoaXMucm9sZSk7XG4gICAgICB0aGlzLm9zVHlwZSA9IGltYWdlQ29uZmlnLm9zVHlwZTtcbiAgICB9XG5cbiAgICAvLyBkZXNpcmVkQ2FwYWNpdHkganVzdCByZWZsZWN0cyB3aGF0IHRoZSB1c2VyIGhhcyBzdXBwbGllZC5cbiAgICBjb25zdCBkZXNpcmVkQ2FwYWNpdHkgPSBwcm9wcy5kZXNpcmVkQ2FwYWNpdHk7XG4gICAgY29uc3QgbWluQ2FwYWNpdHkgPSBwcm9wcy5taW5DYXBhY2l0eSA/PyAxO1xuICAgIGNvbnN0IG1heENhcGFjaXR5ID0gcHJvcHMubWF4Q2FwYWNpdHkgPz8gZGVzaXJlZENhcGFjaXR5ID8/IE1hdGgubWF4KG1pbkNhcGFjaXR5LCAxKTtcblxuICAgIHdpdGhSZXNvbHZlZChtaW5DYXBhY2l0eSwgbWF4Q2FwYWNpdHksIChtaW4sIG1heCkgPT4ge1xuICAgICAgaWYgKG1pbiA+IG1heCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYG1pbkNhcGFjaXR5ICgke21pbn0pIHNob3VsZCBiZSA8PSBtYXhDYXBhY2l0eSAoJHttYXh9KWApO1xuICAgICAgfVxuICAgIH0pO1xuICAgIHdpdGhSZXNvbHZlZChkZXNpcmVkQ2FwYWNpdHksIG1pbkNhcGFjaXR5LCAoZGVzaXJlZCwgbWluKSA9PiB7XG4gICAgICBpZiAoZGVzaXJlZCA9PT0gdW5kZWZpbmVkKSB7IHJldHVybjsgfVxuICAgICAgaWYgKGRlc2lyZWQgPCBtaW4pIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBTaG91bGQgaGF2ZSBtaW5DYXBhY2l0eSAoJHttaW59KSA8PSBkZXNpcmVkQ2FwYWNpdHkgKCR7ZGVzaXJlZH0pYCk7XG4gICAgICB9XG4gICAgfSk7XG4gICAgd2l0aFJlc29sdmVkKGRlc2lyZWRDYXBhY2l0eSwgbWF4Q2FwYWNpdHksIChkZXNpcmVkLCBtYXgpID0+IHtcbiAgICAgIGlmIChkZXNpcmVkID09PSB1bmRlZmluZWQpIHsgcmV0dXJuOyB9XG4gICAgICBpZiAobWF4IDwgZGVzaXJlZCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFNob3VsZCBoYXZlIGRlc2lyZWRDYXBhY2l0eSAoJHtkZXNpcmVkfSkgPD0gbWF4Q2FwYWNpdHkgKCR7bWF4fSlgKTtcbiAgICAgIH1cbiAgICB9KTtcblxuICAgIGlmIChkZXNpcmVkQ2FwYWNpdHkgIT09IHVuZGVmaW5lZCkge1xuICAgICAgQW5ub3RhdGlvbnMub2YodGhpcykuYWRkV2FybmluZygnZGVzaXJlZENhcGFjaXR5IGhhcyBiZWVuIGNvbmZpZ3VyZWQuIEJlIGF3YXJlIHRoaXMgd2lsbCByZXNldCB0aGUgc2l6ZSBvZiB5b3VyIEF1dG9TY2FsaW5nR3JvdXAgb24gZXZlcnkgZGVwbG95bWVudC4gU2VlIGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MvYXdzLWNkay9pc3N1ZXMvNTIxNScpO1xuICAgIH1cblxuICAgIHRoaXMubWF4SW5zdGFuY2VMaWZldGltZSA9IHByb3BzLm1heEluc3RhbmNlTGlmZXRpbWU7XG4gICAgLy8gU2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hdXRvc2NhbGluZy9lYzIvdXNlcmd1aWRlL2FzZy1tYXgtaW5zdGFuY2UtbGlmZXRpbWUuaHRtbCBmb3IgZGV0YWlscyBvbiBtYXggaW5zdGFuY2UgbGlmZXRpbWUuXG4gICAgaWYgKHRoaXMubWF4SW5zdGFuY2VMaWZldGltZSAmJiAhdGhpcy5tYXhJbnN0YW5jZUxpZmV0aW1lLmlzVW5yZXNvbHZlZCgpICYmXG4gICAgICAodGhpcy5tYXhJbnN0YW5jZUxpZmV0aW1lLnRvU2Vjb25kcygpICE9PSAwKSAmJlxuICAgICAgKHRoaXMubWF4SW5zdGFuY2VMaWZldGltZS50b1NlY29uZHMoKSA8IDg2NDAwIHx8IHRoaXMubWF4SW5zdGFuY2VMaWZldGltZS50b1NlY29uZHMoKSA+IDMxNTM2MDAwKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdtYXhJbnN0YW5jZUxpZmV0aW1lIG11c3QgYmUgYmV0d2VlbiAxIGFuZCAzNjUgZGF5cyAoaW5jbHVzaXZlKScpO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5ub3RpZmljYXRpb25zVG9waWMgJiYgcHJvcHMubm90aWZpY2F0aW9ucykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3Qgc2V0IFxcJ25vdGlmaWNhdGlvbnNUb3BpY1xcJyBhbmQgXFwnbm90aWZpY2F0aW9uc1xcJywgXFwnbm90aWZpY2F0aW9uc1RvcGljXFwnIGlzIGRlcHJlY2F0ZWQgdXNlIFxcJ25vdGlmaWNhdGlvbnNcXCcgaW5zdGVhZCcpO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5ub3RpZmljYXRpb25zVG9waWMpIHtcbiAgICAgIHRoaXMubm90aWZpY2F0aW9ucyA9IFt7XG4gICAgICAgIHRvcGljOiBwcm9wcy5ub3RpZmljYXRpb25zVG9waWMsXG4gICAgICB9XTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMubm90aWZpY2F0aW9ucykge1xuICAgICAgdGhpcy5ub3RpZmljYXRpb25zID0gcHJvcHMubm90aWZpY2F0aW9ucy5tYXAobmMgPT4gKHtcbiAgICAgICAgdG9waWM6IG5jLnRvcGljLFxuICAgICAgICBzY2FsaW5nRXZlbnRzOiBuYy5zY2FsaW5nRXZlbnRzID8/IFNjYWxpbmdFdmVudHMuQUxMLFxuICAgICAgfSkpO1xuICAgIH1cblxuICAgIGNvbnN0IHsgc3VibmV0SWRzLCBoYXNQdWJsaWMgfSA9IHByb3BzLnZwYy5zZWxlY3RTdWJuZXRzKHByb3BzLnZwY1N1Ym5ldHMpO1xuICAgIGNvbnN0IGFzZ1Byb3BzOiBDZm5BdXRvU2NhbGluZ0dyb3VwUHJvcHMgPSB7XG4gICAgICBhdXRvU2NhbGluZ0dyb3VwTmFtZTogdGhpcy5waHlzaWNhbE5hbWUsXG4gICAgICBjb29sZG93bjogcHJvcHMuY29vbGRvd24/LnRvU2Vjb25kcygpLnRvU3RyaW5nKCksXG4gICAgICBtaW5TaXplOiBUb2tlbml6YXRpb24uc3RyaW5naWZ5TnVtYmVyKG1pbkNhcGFjaXR5KSxcbiAgICAgIG1heFNpemU6IFRva2VuaXphdGlvbi5zdHJpbmdpZnlOdW1iZXIobWF4Q2FwYWNpdHkpLFxuICAgICAgZGVzaXJlZENhcGFjaXR5OiBkZXNpcmVkQ2FwYWNpdHkgIT09IHVuZGVmaW5lZCA/IFRva2VuaXphdGlvbi5zdHJpbmdpZnlOdW1iZXIoZGVzaXJlZENhcGFjaXR5KSA6IHVuZGVmaW5lZCxcbiAgICAgIGxvYWRCYWxhbmNlck5hbWVzOiBMYXp5Lmxpc3QoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLmxvYWRCYWxhbmNlck5hbWVzIH0sIHsgb21pdEVtcHR5OiB0cnVlIH0pLFxuICAgICAgdGFyZ2V0R3JvdXBBcm5zOiBMYXp5Lmxpc3QoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLnRhcmdldEdyb3VwQXJucyB9LCB7IG9taXRFbXB0eTogdHJ1ZSB9KSxcbiAgICAgIG5vdGlmaWNhdGlvbkNvbmZpZ3VyYXRpb25zOiB0aGlzLnJlbmRlck5vdGlmaWNhdGlvbkNvbmZpZ3VyYXRpb24oKSxcbiAgICAgIG1ldHJpY3NDb2xsZWN0aW9uOiBMYXp5LmFueSh7IHByb2R1Y2U6ICgpID0+IHRoaXMucmVuZGVyTWV0cmljc0NvbGxlY3Rpb24oKSB9KSxcbiAgICAgIHZwY1pvbmVJZGVudGlmaWVyOiBzdWJuZXRJZHMsXG4gICAgICBoZWFsdGhDaGVja1R5cGU6IHByb3BzLmhlYWx0aENoZWNrICYmIHByb3BzLmhlYWx0aENoZWNrLnR5cGUsXG4gICAgICBoZWFsdGhDaGVja0dyYWNlUGVyaW9kOiBwcm9wcy5oZWFsdGhDaGVjayAmJiBwcm9wcy5oZWFsdGhDaGVjay5ncmFjZVBlcmlvZCAmJiBwcm9wcy5oZWFsdGhDaGVjay5ncmFjZVBlcmlvZC50b1NlY29uZHMoKSxcbiAgICAgIG1heEluc3RhbmNlTGlmZXRpbWU6IHRoaXMubWF4SW5zdGFuY2VMaWZldGltZSA/IHRoaXMubWF4SW5zdGFuY2VMaWZldGltZS50b1NlY29uZHMoKSA6IHVuZGVmaW5lZCxcbiAgICAgIG5ld0luc3RhbmNlc1Byb3RlY3RlZEZyb21TY2FsZUluOiBMYXp5LmFueSh7IHByb2R1Y2U6ICgpID0+IHRoaXMubmV3SW5zdGFuY2VzUHJvdGVjdGVkRnJvbVNjYWxlSW4gfSksXG4gICAgICB0ZXJtaW5hdGlvblBvbGljaWVzOiBwcm9wcy50ZXJtaW5hdGlvblBvbGljaWVzLFxuICAgICAgLi4udGhpcy5nZXRMYXVuY2hTZXR0aW5ncyhsYXVuY2hDb25maWcsIHByb3BzLmxhdW5jaFRlbXBsYXRlLCBwcm9wcy5taXhlZEluc3RhbmNlc1BvbGljeSksXG4gICAgfTtcblxuICAgIGlmICghaGFzUHVibGljICYmIHByb3BzLmFzc29jaWF0ZVB1YmxpY0lwQWRkcmVzcykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFwiVG8gc2V0ICdhc3NvY2lhdGVQdWJsaWNJcEFkZHJlc3M6IHRydWUnIHlvdSBtdXN0IHNlbGVjdCBQdWJsaWMgc3VibmV0cyAodnBjU3VibmV0czogeyBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBVQkxJQyB9KVwiKTtcbiAgICB9XG5cbiAgICB0aGlzLmF1dG9TY2FsaW5nR3JvdXAgPSBuZXcgQ2ZuQXV0b1NjYWxpbmdHcm91cCh0aGlzLCAnQVNHJywgYXNnUHJvcHMpO1xuICAgIHRoaXMuYXV0b1NjYWxpbmdHcm91cE5hbWUgPSB0aGlzLmdldFJlc291cmNlTmFtZUF0dHJpYnV0ZSh0aGlzLmF1dG9TY2FsaW5nR3JvdXAucmVmKSxcbiAgICB0aGlzLmF1dG9TY2FsaW5nR3JvdXBBcm4gPSBTdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oe1xuICAgICAgc2VydmljZTogJ2F1dG9zY2FsaW5nJyxcbiAgICAgIHJlc291cmNlOiAnYXV0b1NjYWxpbmdHcm91cDoqOmF1dG9TY2FsaW5nR3JvdXBOYW1lJyxcbiAgICAgIHJlc291cmNlTmFtZTogdGhpcy5hdXRvU2NhbGluZ0dyb3VwTmFtZSxcbiAgICB9KTtcbiAgICB0aGlzLm5vZGUuZGVmYXVsdENoaWxkID0gdGhpcy5hdXRvU2NhbGluZ0dyb3VwO1xuXG4gICAgdGhpcy5hcHBseVVwZGF0ZVBvbGljaWVzKHByb3BzLCB7IGRlc2lyZWRDYXBhY2l0eSwgbWluQ2FwYWNpdHkgfSk7XG4gICAgaWYgKHByb3BzLmluaXQpIHtcbiAgICAgIHRoaXMuYXBwbHlDbG91ZEZvcm1hdGlvbkluaXQocHJvcHMuaW5pdCwgcHJvcHMuaW5pdE9wdGlvbnMpO1xuICAgIH1cblxuICAgIHRoaXMuc3BvdFByaWNlID0gcHJvcHMuc3BvdFByaWNlO1xuXG4gICAgaWYgKHByb3BzLnJlcXVpcmVJbWRzdjIpIHtcbiAgICAgIEFzcGVjdHMub2YodGhpcykuYWRkKG5ldyBBdXRvU2NhbGluZ0dyb3VwUmVxdWlyZUltZHN2MkFzcGVjdCgpKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQWRkIHRoZSBzZWN1cml0eSBncm91cCB0byBhbGwgaW5zdGFuY2VzIHZpYSB0aGUgbGF1bmNoIGNvbmZpZ3VyYXRpb25cbiAgICogc2VjdXJpdHkgZ3JvdXBzIGFycmF5LlxuICAgKlxuICAgKiBAcGFyYW0gc2VjdXJpdHlHcm91cDogVGhlIHNlY3VyaXR5IGdyb3VwIHRvIGFkZFxuICAgKi9cbiAgcHVibGljIGFkZFNlY3VyaXR5R3JvdXAoc2VjdXJpdHlHcm91cDogZWMyLklTZWN1cml0eUdyb3VwKTogdm9pZCB7XG4gICAgaWYgKCF0aGlzLnNlY3VyaXR5R3JvdXBzKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1lvdSBjYW5ub3QgYWRkIHNlY3VyaXR5IGdyb3VwcyB3aGVuIHRoZSBBdXRvIFNjYWxpbmcgR3JvdXAgaXMgY3JlYXRlZCBmcm9tIGEgTGF1bmNoIFRlbXBsYXRlLicpO1xuICAgIH1cblxuICAgIHRoaXMuc2VjdXJpdHlHcm91cHMucHVzaChzZWN1cml0eUdyb3VwKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBdHRhY2ggdG8gYSBjbGFzc2ljIGxvYWQgYmFsYW5jZXJcbiAgICovXG4gIHB1YmxpYyBhdHRhY2hUb0NsYXNzaWNMQihsb2FkQmFsYW5jZXI6IGVsYi5Mb2FkQmFsYW5jZXIpOiB2b2lkIHtcbiAgICB0aGlzLmxvYWRCYWxhbmNlck5hbWVzLnB1c2gobG9hZEJhbGFuY2VyLmxvYWRCYWxhbmNlck5hbWUpO1xuICB9XG5cbiAgLyoqXG4gICAqIEF0dGFjaCB0byBFTEJ2MiBBcHBsaWNhdGlvbiBUYXJnZXQgR3JvdXBcbiAgICovXG4gIHB1YmxpYyBhdHRhY2hUb0FwcGxpY2F0aW9uVGFyZ2V0R3JvdXAodGFyZ2V0R3JvdXA6IGVsYnYyLklBcHBsaWNhdGlvblRhcmdldEdyb3VwKTogZWxidjIuTG9hZEJhbGFuY2VyVGFyZ2V0UHJvcHMge1xuICAgIGlmICh0aGlzLmFsYlRhcmdldEdyb3VwICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IGFkZCBBdXRvU2NhbGluZ0dyb3VwIHRvIDJuZCBUYXJnZXQgR3JvdXAnKTtcbiAgICB9XG5cbiAgICB0aGlzLnRhcmdldEdyb3VwQXJucy5wdXNoKHRhcmdldEdyb3VwLnRhcmdldEdyb3VwQXJuKTtcbiAgICBpZiAodGFyZ2V0R3JvdXAgaW5zdGFuY2VvZiBlbGJ2Mi5BcHBsaWNhdGlvblRhcmdldEdyb3VwKSB7XG4gICAgICAvLyBDb3B5IG9udG8gc2VsZiBpZiBpdCdzIGEgY29uY3JldGUgdHlwZS4gV2UgbmVlZCB0aGlzIGZvciBhdXRvc2NhbGluZ1xuICAgICAgLy8gYmFzZWQgb24gcmVxdWVzdCBjb3VudCwgd2hpY2ggd2UgY2Fubm90IGRvIHdpdGggYW4gaW1wb3J0ZWQgVGFyZ2V0R3JvdXAuXG4gICAgICB0aGlzLmFsYlRhcmdldEdyb3VwID0gdGFyZ2V0R3JvdXA7XG4gICAgfVxuXG4gICAgdGFyZ2V0R3JvdXAucmVnaXN0ZXJDb25uZWN0YWJsZSh0aGlzKTtcbiAgICByZXR1cm4geyB0YXJnZXRUeXBlOiBlbGJ2Mi5UYXJnZXRUeXBlLklOU1RBTkNFIH07XG4gIH1cblxuICAvKipcbiAgICogQXR0YWNoIHRvIEVMQnYyIEFwcGxpY2F0aW9uIFRhcmdldCBHcm91cFxuICAgKi9cbiAgcHVibGljIGF0dGFjaFRvTmV0d29ya1RhcmdldEdyb3VwKHRhcmdldEdyb3VwOiBlbGJ2Mi5JTmV0d29ya1RhcmdldEdyb3VwKTogZWxidjIuTG9hZEJhbGFuY2VyVGFyZ2V0UHJvcHMge1xuICAgIHRoaXMudGFyZ2V0R3JvdXBBcm5zLnB1c2godGFyZ2V0R3JvdXAudGFyZ2V0R3JvdXBBcm4pO1xuICAgIHJldHVybiB7IHRhcmdldFR5cGU6IGVsYnYyLlRhcmdldFR5cGUuSU5TVEFOQ0UgfTtcbiAgfVxuXG4gIHB1YmxpYyBhZGRVc2VyRGF0YSguLi5jb21tYW5kczogc3RyaW5nW10pOiB2b2lkIHtcbiAgICB0aGlzLnVzZXJEYXRhLmFkZENvbW1hbmRzKC4uLmNvbW1hbmRzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgc3RhdGVtZW50IHRvIHRoZSBJQU0gcm9sZSBhc3N1bWVkIGJ5IGluc3RhbmNlcyBvZiB0aGlzIGZsZWV0LlxuICAgKi9cbiAgcHVibGljIGFkZFRvUm9sZVBvbGljeShzdGF0ZW1lbnQ6IGlhbS5Qb2xpY3lTdGF0ZW1lbnQpIHtcbiAgICB0aGlzLnJvbGUuYWRkVG9QcmluY2lwYWxQb2xpY3koc3RhdGVtZW50KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBVc2UgYSBDbG91ZEZvcm1hdGlvbiBJbml0IGNvbmZpZ3VyYXRpb24gYXQgaW5zdGFuY2Ugc3RhcnR1cFxuICAgKlxuICAgKiBUaGlzIGRvZXMgdGhlIGZvbGxvd2luZzpcbiAgICpcbiAgICogLSBBdHRhY2hlcyB0aGUgQ2xvdWRGb3JtYXRpb24gSW5pdCBtZXRhZGF0YSB0byB0aGUgQXV0b1NjYWxpbmdHcm91cCByZXNvdXJjZS5cbiAgICogLSBBZGQgY29tbWFuZHMgdG8gdGhlIFVzZXJEYXRhIHRvIHJ1biBgY2ZuLWluaXRgIGFuZCBgY2ZuLXNpZ25hbGAuXG4gICAqIC0gVXBkYXRlIHRoZSBpbnN0YW5jZSdzIENyZWF0aW9uUG9saWN5IHRvIHdhaXQgZm9yIGBjZm4taW5pdGAgdG8gZmluaXNoXG4gICAqICAgYmVmb3JlIHJlcG9ydGluZyBzdWNjZXNzLlxuICAgKi9cbiAgcHVibGljIGFwcGx5Q2xvdWRGb3JtYXRpb25Jbml0KGluaXQ6IGVjMi5DbG91ZEZvcm1hdGlvbkluaXQsIG9wdGlvbnM6IEFwcGx5Q2xvdWRGb3JtYXRpb25Jbml0T3B0aW9ucyA9IHt9KSB7XG4gICAgaWYgKCF0aGlzLmF1dG9TY2FsaW5nR3JvdXAuY2ZuT3B0aW9ucy5jcmVhdGlvblBvbGljeT8ucmVzb3VyY2VTaWduYWwpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignV2hlbiBhcHBseWluZyBDbG91ZEZvcm1hdGlvbkluaXQsIHlvdSBtdXN0IGFsc28gY29uZmlndXJlIHNpZ25hbHMgYnkgc3VwcGx5aW5nIFxcJ3NpZ25hbHNcXCcgYXQgaW5zdGFudGlhdGlvbiB0aW1lLicpO1xuICAgIH1cblxuICAgIGluaXQuYXR0YWNoKHRoaXMuYXV0b1NjYWxpbmdHcm91cCwge1xuICAgICAgcGxhdGZvcm06IHRoaXMub3NUeXBlLFxuICAgICAgaW5zdGFuY2VSb2xlOiB0aGlzLnJvbGUsXG4gICAgICB1c2VyRGF0YTogdGhpcy51c2VyRGF0YSxcbiAgICAgIGNvbmZpZ1NldHM6IG9wdGlvbnMuY29uZmlnU2V0cyxcbiAgICAgIGVtYmVkRmluZ2VycHJpbnQ6IG9wdGlvbnMuZW1iZWRGaW5nZXJwcmludCxcbiAgICAgIHByaW50TG9nOiBvcHRpb25zLnByaW50TG9nLFxuICAgICAgaWdub3JlRmFpbHVyZXM6IG9wdGlvbnMuaWdub3JlRmFpbHVyZXMsXG4gICAgICBpbmNsdWRlUm9sZTogb3B0aW9ucy5pbmNsdWRlUm9sZSxcbiAgICAgIGluY2x1ZGVVcmw6IG9wdGlvbnMuaW5jbHVkZVVybCxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBFbnN1cmVzIG5ld2x5LWxhdW5jaGVkIGluc3RhbmNlcyBhcmUgcHJvdGVjdGVkIGZyb20gc2NhbGUtaW4uXG4gICAqL1xuICBwdWJsaWMgcHJvdGVjdE5ld0luc3RhbmNlc0Zyb21TY2FsZUluKCkge1xuICAgIHRoaXMubmV3SW5zdGFuY2VzUHJvdGVjdGVkRnJvbVNjYWxlSW4gPSB0cnVlO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYHRydWVgIGlmIG5ld2x5LWxhdW5jaGVkIGluc3RhbmNlcyBhcmUgcHJvdGVjdGVkIGZyb20gc2NhbGUtaW4uXG4gICAqL1xuICBwdWJsaWMgYXJlTmV3SW5zdGFuY2VzUHJvdGVjdGVkRnJvbVNjYWxlSW4oKTogYm9vbGVhbiB7XG4gICAgcmV0dXJuIHRoaXMubmV3SW5zdGFuY2VzUHJvdGVjdGVkRnJvbVNjYWxlSW4gPT09IHRydWU7XG4gIH1cblxuICAvKipcbiAgICogVGhlIG5ldHdvcmsgY29ubmVjdGlvbnMgYXNzb2NpYXRlZCB3aXRoIHRoaXMgcmVzb3VyY2UuXG4gICAqL1xuICBwdWJsaWMgZ2V0IGNvbm5lY3Rpb25zKCk6IGVjMi5Db25uZWN0aW9ucyB7XG4gICAgaWYgKHRoaXMuX2Nvbm5lY3Rpb25zKSB7XG4gICAgICByZXR1cm4gdGhpcy5fY29ubmVjdGlvbnM7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMubGF1bmNoVGVtcGxhdGUpIHtcbiAgICAgIHJldHVybiB0aGlzLmxhdW5jaFRlbXBsYXRlLmNvbm5lY3Rpb25zO1xuICAgIH1cblxuICAgIHRocm93IG5ldyBFcnJvcignQXV0b1NjYWxpbmdHcm91cCBjYW4gb25seSBiZSB1c2VkIGFzIElDb25uZWN0YWJsZSBpZiBpdCBpcyBub3QgY3JlYXRlZCBmcm9tIGFuIGltcG9ydGVkIExhdW5jaCBUZW1wbGF0ZS4nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgQmFzZTY0LWVuY29kZWQgdXNlciBkYXRhIHRvIG1ha2UgYXZhaWxhYmxlIHRvIHRoZSBsYXVuY2hlZCBFQzIgaW5zdGFuY2VzLlxuICAgKlxuICAgKiBAdGhyb3dzIGFuIGVycm9yIGlmIGEgbGF1bmNoIHRlbXBsYXRlIGlzIGdpdmVuIGFuZCBpdCBkb2VzIG5vdCBwcm92aWRlIGEgbm9uLW51bGwgYHVzZXJEYXRhYFxuICAgKi9cbiAgcHVibGljIGdldCB1c2VyRGF0YSgpOiBlYzIuVXNlckRhdGEge1xuICAgIGlmICh0aGlzLl91c2VyRGF0YSkge1xuICAgICAgcmV0dXJuIHRoaXMuX3VzZXJEYXRhO1xuICAgIH1cblxuICAgIGlmICh0aGlzLmxhdW5jaFRlbXBsYXRlPy51c2VyRGF0YSkge1xuICAgICAgcmV0dXJuIHRoaXMubGF1bmNoVGVtcGxhdGUudXNlckRhdGE7XG4gICAgfVxuXG4gICAgdGhyb3cgbmV3IEVycm9yKCdUaGUgcHJvdmlkZWQgbGF1bmNoIHRlbXBsYXRlIGRvZXMgbm90IGV4cG9zZSBpdHMgdXNlciBkYXRhLicpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBJQU0gUm9sZSBpbiB0aGUgaW5zdGFuY2UgcHJvZmlsZVxuICAgKlxuICAgKiBAdGhyb3dzIGFuIGVycm9yIGlmIGEgbGF1bmNoIHRlbXBsYXRlIGlzIGdpdmVuXG4gICAqL1xuICBwdWJsaWMgZ2V0IHJvbGUoKTogaWFtLklSb2xlIHtcbiAgICBpZiAodGhpcy5fcm9sZSkge1xuICAgICAgcmV0dXJuIHRoaXMuX3JvbGU7XG4gICAgfVxuXG4gICAgdGhyb3cgbmV3IEVycm9yKCdUaGUgcHJvdmlkZWQgbGF1bmNoIHRlbXBsYXRlIGRvZXMgbm90IGV4cG9zZSBvciBkb2VzIG5vdCBkZWZpbmUgaXRzIHJvbGUuJyk7XG4gIH1cblxuICBwcml2YXRlIHZlcmlmeU5vTGF1bmNoQ29uZmlnUHJvcElzR2l2ZW4ocHJvcHM6IEF1dG9TY2FsaW5nR3JvdXBQcm9wcykge1xuICAgIGlmIChwcm9wcy5tYWNoaW5lSW1hZ2UpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignU2V0dGluZyBcXCdtYWNoaW5lSW1hZ2VcXCcgbXVzdCBub3QgYmUgc2V0IHdoZW4gXFwnbGF1bmNoVGVtcGxhdGVcXCcgb3IgXFwnbWl4ZWRJbnN0YW5jZXNQb2xpY3lcXCcgaXMgc2V0Jyk7XG4gICAgfVxuICAgIGlmIChwcm9wcy5pbnN0YW5jZVR5cGUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignU2V0dGluZyBcXCdpbnN0YW5jZVR5cGVcXCcgbXVzdCBub3QgYmUgc2V0IHdoZW4gXFwnbGF1bmNoVGVtcGxhdGVcXCcgb3IgXFwnbWl4ZWRJbnN0YW5jZXNQb2xpY3lcXCcgaXMgc2V0Jyk7XG4gICAgfVxuICAgIGlmIChwcm9wcy5yb2xlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1NldHRpbmcgXFwncm9sZVxcJyBtdXN0IG5vdCBiZSBzZXQgd2hlbiBcXCdsYXVuY2hUZW1wbGF0ZVxcJyBvciBcXCdtaXhlZEluc3RhbmNlc1BvbGljeVxcJyBpcyBzZXQnKTtcbiAgICB9XG4gICAgaWYgKHByb3BzLnVzZXJEYXRhKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1NldHRpbmcgXFwndXNlckRhdGFcXCcgbXVzdCBub3QgYmUgc2V0IHdoZW4gXFwnbGF1bmNoVGVtcGxhdGVcXCcgb3IgXFwnbWl4ZWRJbnN0YW5jZXNQb2xpY3lcXCcgaXMgc2V0Jyk7XG4gICAgfVxuICAgIGlmIChwcm9wcy5zZWN1cml0eUdyb3VwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1NldHRpbmcgXFwnc2VjdXJpdHlHcm91cFxcJyBtdXN0IG5vdCBiZSBzZXQgd2hlbiBcXCdsYXVuY2hUZW1wbGF0ZVxcJyBvciBcXCdtaXhlZEluc3RhbmNlc1BvbGljeVxcJyBpcyBzZXQnKTtcbiAgICB9XG4gICAgaWYgKHByb3BzLmtleU5hbWUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignU2V0dGluZyBcXCdrZXlOYW1lXFwnIG11c3Qgbm90IGJlIHNldCB3aGVuIFxcJ2xhdW5jaFRlbXBsYXRlXFwnIG9yIFxcJ21peGVkSW5zdGFuY2VzUG9saWN5XFwnIGlzIHNldCcpO1xuICAgIH1cbiAgICBpZiAocHJvcHMuaW5zdGFuY2VNb25pdG9yaW5nKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1NldHRpbmcgXFwnaW5zdGFuY2VNb25pdG9yaW5nXFwnIG11c3Qgbm90IGJlIHNldCB3aGVuIFxcJ2xhdW5jaFRlbXBsYXRlXFwnIG9yIFxcJ21peGVkSW5zdGFuY2VzUG9saWN5XFwnIGlzIHNldCcpO1xuICAgIH1cbiAgICBpZiAocHJvcHMuYXNzb2NpYXRlUHVibGljSXBBZGRyZXNzKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1NldHRpbmcgXFwnYXNzb2NpYXRlUHVibGljSXBBZGRyZXNzXFwnIG11c3Qgbm90IGJlIHNldCB3aGVuIFxcJ2xhdW5jaFRlbXBsYXRlXFwnIG9yIFxcJ21peGVkSW5zdGFuY2VzUG9saWN5XFwnIGlzIHNldCcpO1xuICAgIH1cbiAgICBpZiAocHJvcHMuc3BvdFByaWNlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1NldHRpbmcgXFwnc3BvdFByaWNlXFwnIG11c3Qgbm90IGJlIHNldCB3aGVuIFxcJ2xhdW5jaFRlbXBsYXRlXFwnIG9yIFxcJ21peGVkSW5zdGFuY2VzUG9saWN5XFwnIGlzIHNldCcpO1xuICAgIH1cbiAgICBpZiAocHJvcHMuYmxvY2tEZXZpY2VzKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1NldHRpbmcgXFwnYmxvY2tEZXZpY2VzXFwnIG11c3Qgbm90IGJlIHNldCB3aGVuIFxcJ2xhdW5jaFRlbXBsYXRlXFwnIG9yIFxcJ21peGVkSW5zdGFuY2VzUG9saWN5XFwnIGlzIHNldCcpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBBcHBseSBDbG91ZEZvcm1hdGlvbiB1cGRhdGUgcG9saWNpZXMgZm9yIHRoZSBBdXRvU2NhbGluZ0dyb3VwXG4gICAqL1xuICBwcml2YXRlIGFwcGx5VXBkYXRlUG9saWNpZXMocHJvcHM6IEF1dG9TY2FsaW5nR3JvdXBQcm9wcywgc2lnbmFsT3B0aW9uczogUmVuZGVyU2lnbmFsc09wdGlvbnMpIHtcbiAgICAvLyBNYWtlIHN1cmUgcGVvcGxlIGFyZSBub3QgdXNpbmcgdGhlIG9sZCBhbmQgbmV3IHByb3BlcnRpZXMgdG9nZXRoZXJcbiAgICBjb25zdCBvbGRQcm9wczogQXJyYXk8a2V5b2YgQXV0b1NjYWxpbmdHcm91cFByb3BzPiA9IFtcbiAgICAgICd1cGRhdGVUeXBlJyxcbiAgICAgICdyb2xsaW5nVXBkYXRlQ29uZmlndXJhdGlvbicsXG4gICAgICAncmVzb3VyY2VTaWduYWxDb3VudCcsXG4gICAgICAncmVzb3VyY2VTaWduYWxUaW1lb3V0JyxcbiAgICAgICdyZXBsYWNpbmdVcGRhdGVNaW5TdWNjZXNzZnVsSW5zdGFuY2VzUGVyY2VudCcsXG4gICAgXTtcbiAgICBmb3IgKGNvbnN0IHByb3Agb2Ygb2xkUHJvcHMpIHtcbiAgICAgIGlmICgocHJvcHMuc2lnbmFscyB8fCBwcm9wcy51cGRhdGVQb2xpY3kpICYmIHByb3BzW3Byb3BdICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBDYW5ub3Qgc2V0ICdzaWduYWxzJy8ndXBkYXRlUG9saWN5JyBhbmQgJyR7cHJvcH0nIHRvZ2V0aGVyLiBQcmVmZXIgJ3NpZ25hbHMnLyd1cGRhdGVQb2xpY3knYCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgLy8gUmVpZnkgdXBkYXRlUG9saWN5IHRvIGByb2xsaW5nVXBkYXRlYCBkZWZhdWx0IGluIGNhc2UgaXQgaXMgY29tYmluZWQgd2l0aCBgaW5pdGBcbiAgICBwcm9wcyA9IHtcbiAgICAgIC4uLnByb3BzLFxuICAgICAgdXBkYXRlUG9saWN5OiBwcm9wcy51cGRhdGVQb2xpY3kgPz8gKHByb3BzLmluaXQgPyBVcGRhdGVQb2xpY3kucm9sbGluZ1VwZGF0ZSgpIDogdW5kZWZpbmVkKSxcbiAgICB9O1xuXG4gICAgaWYgKHByb3BzLnNpZ25hbHMgfHwgcHJvcHMudXBkYXRlUG9saWN5KSB7XG4gICAgICB0aGlzLmFwcGx5TmV3U2lnbmFsVXBkYXRlUG9saWNpZXMocHJvcHMsIHNpZ25hbE9wdGlvbnMpO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLmFwcGx5TGVnYWN5U2lnbmFsVXBkYXRlUG9saWNpZXMocHJvcHMpO1xuICAgIH1cblxuICAgIC8vIFRoZSBmb2xsb3dpbmcgaXMgdGVjaG5pY2FsbHkgcGFydCBvZiB0aGUgXCJ1cGRhdGUgcG9saWN5XCIgYnV0IGl0J3MgYWxzbyBhIGNvbXBsZXRlbHlcbiAgICAvLyBzZXBhcmF0ZSBhc3BlY3Qgb2Ygcm9sbGluZy9yZXBsYWNpbmcgdXBkYXRlLCBzbyBpdCdzIGp1c3QgaXRzIG93biB0b3AtbGV2ZWwgcHJvcGVydHkuXG4gICAgLy8gRGVmYXVsdCBpcyAndHJ1ZScgYmVjYXVzZSB0aGF0J3Mgd2hhdCB5b3UncmUgbW9zdCBsaWtlbHkgdG8gd2FudFxuICAgIGlmIChwcm9wcy5pZ25vcmVVbm1vZGlmaWVkU2l6ZVByb3BlcnRpZXMgIT09IGZhbHNlKSB7XG4gICAgICB0aGlzLmF1dG9TY2FsaW5nR3JvdXAuY2ZuT3B0aW9ucy51cGRhdGVQb2xpY3kgPSB7XG4gICAgICAgIC4uLnRoaXMuYXV0b1NjYWxpbmdHcm91cC5jZm5PcHRpb25zLnVwZGF0ZVBvbGljeSxcbiAgICAgICAgYXV0b1NjYWxpbmdTY2hlZHVsZWRBY3Rpb246IHsgaWdub3JlVW5tb2RpZmllZEdyb3VwU2l6ZVByb3BlcnRpZXM6IHRydWUgfSxcbiAgICAgIH07XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLnNpZ25hbHMgJiYgIXByb3BzLmluaXQpIHtcbiAgICAgIC8vIFRvIGJlIGFibGUgdG8gc2VuZCBhIHNpZ25hbCB1c2luZyBgY2ZuLWluaXRgLCB0aGUgZXhlY3V0aW9uIHJvbGUgbmVlZHNcbiAgICAgIC8vIGBjbG91ZGZvcm1hdGlvbjpTaWduYWxSZXNvdXJjZWAuIE5vcm1hbGx5IHRoZSBiaW5kaW5nIG9mIENmbkluaXQgd291bGRcbiAgICAgIC8vIGdyYW50IHRoYXQgcGVybWlzc2lvbnMgYW5kIGFub3RoZXIgb25lLCBidXQgaWYgdGhlIHVzZXIgd2FudHMgdG8gdXNlXG4gICAgICAvLyBgc2lnbmFsc2Agd2l0aG91dCBgaW5pdGAsIGFkZCB0aGUgcGVybWlzc2lvbnMgaGVyZS5cbiAgICAgIC8vXG4gICAgICAvLyBJZiB0aGV5IGNhbGwgYGFwcGx5Q2xvdWRGb3JtYXRpb25Jbml0KClgIGFmdGVyIGNvbnN0cnVjdGlvbiwgbm90aGluZyBiYWRcbiAgICAgIC8vIGhhcHBlbnMgZWl0aGVyLCB3ZSdsbCBqdXN0IGhhdmUgYSBkdXBsaWNhdGUgc3RhdGVtZW50IHdoaWNoIGRvZXNuJ3QgaHVydC5cbiAgICAgIHRoaXMuYWRkVG9Sb2xlUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgYWN0aW9uczogWydjbG91ZGZvcm1hdGlvbjpTaWduYWxSZXNvdXJjZSddLFxuICAgICAgICByZXNvdXJjZXM6IFtBd3MuU1RBQ0tfSURdLFxuICAgICAgfSkpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBVc2UgJ3NpZ25hbHMnIGFuZCAndXBkYXRlUG9saWN5JyB0byBkZXRlcm1pbmUgdGhlIGNyZWF0aW9uIGFuZCB1cGRhdGUgcG9saWNpZXNcbiAgICovXG4gIHByaXZhdGUgYXBwbHlOZXdTaWduYWxVcGRhdGVQb2xpY2llcyhwcm9wczogQXV0b1NjYWxpbmdHcm91cFByb3BzLCBzaWduYWxPcHRpb25zOiBSZW5kZXJTaWduYWxzT3B0aW9ucykge1xuICAgIHRoaXMuYXV0b1NjYWxpbmdHcm91cC5jZm5PcHRpb25zLmNyZWF0aW9uUG9saWN5ID0gcHJvcHMuc2lnbmFscz8ucmVuZGVyQ3JlYXRpb25Qb2xpY3koc2lnbmFsT3B0aW9ucyk7XG4gICAgdGhpcy5hdXRvU2NhbGluZ0dyb3VwLmNmbk9wdGlvbnMudXBkYXRlUG9saWN5ID0gcHJvcHMudXBkYXRlUG9saWN5Py5fcmVuZGVyVXBkYXRlUG9saWN5KHtcbiAgICAgIGNyZWF0aW9uUG9saWN5OiB0aGlzLmF1dG9TY2FsaW5nR3JvdXAuY2ZuT3B0aW9ucy5jcmVhdGlvblBvbGljeSxcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgYXBwbHlMZWdhY3lTaWduYWxVcGRhdGVQb2xpY2llcyhwcm9wczogQXV0b1NjYWxpbmdHcm91cFByb3BzKSB7XG4gICAgaWYgKHByb3BzLnVwZGF0ZVR5cGUgPT09IFVwZGF0ZVR5cGUuUkVQTEFDSU5HX1VQREFURSkge1xuICAgICAgdGhpcy5hdXRvU2NhbGluZ0dyb3VwLmNmbk9wdGlvbnMudXBkYXRlUG9saWN5ID0ge1xuICAgICAgICAuLi50aGlzLmF1dG9TY2FsaW5nR3JvdXAuY2ZuT3B0aW9ucy51cGRhdGVQb2xpY3ksXG4gICAgICAgIGF1dG9TY2FsaW5nUmVwbGFjaW5nVXBkYXRlOiB7XG4gICAgICAgICAgd2lsbFJlcGxhY2U6IHRydWUsXG4gICAgICAgIH0sXG4gICAgICB9O1xuXG4gICAgICBpZiAocHJvcHMucmVwbGFjaW5nVXBkYXRlTWluU3VjY2Vzc2Z1bEluc3RhbmNlc1BlcmNlbnQgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICAvLyBZZXMsIHRoaXMgZ29lcyBvbiBDcmVhdGlvblBvbGljeSwgbm90IGFzIGEgcHJvY2VzcyBwYXJhbWV0ZXIgdG8gUmVwbGFjaW5nVXBkYXRlLlxuICAgICAgICAvLyBJdCdzIGEgbGl0dGxlIGNvbmZ1c2luZywgYnV0IHRoZSBkb2NzIHNlZW0gdG8gZXhwbGljaXRseSBzdGF0ZSBpdCB3aWxsIG9ubHkgYmUgdXNlZFxuICAgICAgICAvLyBkdXJpbmcgdGhlIHVwZGF0ZT9cbiAgICAgICAgLy9cbiAgICAgICAgLy8gaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLWF0dHJpYnV0ZS1jcmVhdGlvbnBvbGljeS5odG1sXG4gICAgICAgIHRoaXMuYXV0b1NjYWxpbmdHcm91cC5jZm5PcHRpb25zLmNyZWF0aW9uUG9saWN5ID0ge1xuICAgICAgICAgIC4uLnRoaXMuYXV0b1NjYWxpbmdHcm91cC5jZm5PcHRpb25zLmNyZWF0aW9uUG9saWN5LFxuICAgICAgICAgIGF1dG9TY2FsaW5nQ3JlYXRpb25Qb2xpY3k6IHtcbiAgICAgICAgICAgIG1pblN1Y2Nlc3NmdWxJbnN0YW5jZXNQZXJjZW50OiB2YWxpZGF0ZVBlcmNlbnRhZ2UocHJvcHMucmVwbGFjaW5nVXBkYXRlTWluU3VjY2Vzc2Z1bEluc3RhbmNlc1BlcmNlbnQpLFxuICAgICAgICAgIH0sXG4gICAgICAgIH07XG4gICAgICB9XG4gICAgfSBlbHNlIGlmIChwcm9wcy51cGRhdGVUeXBlID09PSBVcGRhdGVUeXBlLlJPTExJTkdfVVBEQVRFKSB7XG4gICAgICB0aGlzLmF1dG9TY2FsaW5nR3JvdXAuY2ZuT3B0aW9ucy51cGRhdGVQb2xpY3kgPSB7XG4gICAgICAgIC4uLnRoaXMuYXV0b1NjYWxpbmdHcm91cC5jZm5PcHRpb25zLnVwZGF0ZVBvbGljeSxcbiAgICAgICAgYXV0b1NjYWxpbmdSb2xsaW5nVXBkYXRlOiByZW5kZXJSb2xsaW5nVXBkYXRlQ29uZmlnKHByb3BzLnJvbGxpbmdVcGRhdGVDb25maWd1cmF0aW9uKSxcbiAgICAgIH07XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLnJlc291cmNlU2lnbmFsQ291bnQgIT09IHVuZGVmaW5lZCB8fCBwcm9wcy5yZXNvdXJjZVNpZ25hbFRpbWVvdXQgIT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhpcy5hdXRvU2NhbGluZ0dyb3VwLmNmbk9wdGlvbnMuY3JlYXRpb25Qb2xpY3kgPSB7XG4gICAgICAgIC4uLnRoaXMuYXV0b1NjYWxpbmdHcm91cC5jZm5PcHRpb25zLmNyZWF0aW9uUG9saWN5LFxuICAgICAgICByZXNvdXJjZVNpZ25hbDoge1xuICAgICAgICAgIGNvdW50OiBwcm9wcy5yZXNvdXJjZVNpZ25hbENvdW50LFxuICAgICAgICAgIHRpbWVvdXQ6IHByb3BzLnJlc291cmNlU2lnbmFsVGltZW91dCAmJiBwcm9wcy5yZXNvdXJjZVNpZ25hbFRpbWVvdXQudG9Jc29TdHJpbmcoKSxcbiAgICAgICAgfSxcbiAgICAgIH07XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSByZW5kZXJOb3RpZmljYXRpb25Db25maWd1cmF0aW9uKCk6IENmbkF1dG9TY2FsaW5nR3JvdXAuTm90aWZpY2F0aW9uQ29uZmlndXJhdGlvblByb3BlcnR5W10gfCB1bmRlZmluZWQge1xuICAgIGlmICh0aGlzLm5vdGlmaWNhdGlvbnMubGVuZ3RoID09PSAwKSB7XG4gICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cblxuICAgIHJldHVybiB0aGlzLm5vdGlmaWNhdGlvbnMubWFwKG5vdGlmaWNhdGlvbiA9PiAoe1xuICAgICAgdG9waWNBcm46IG5vdGlmaWNhdGlvbi50b3BpYy50b3BpY0FybixcbiAgICAgIG5vdGlmaWNhdGlvblR5cGVzOiBub3RpZmljYXRpb24uc2NhbGluZ0V2ZW50cyA/IG5vdGlmaWNhdGlvbi5zY2FsaW5nRXZlbnRzLl90eXBlcyA6IFNjYWxpbmdFdmVudHMuQUxMLl90eXBlcyxcbiAgICB9KSk7XG4gIH1cblxuICBwcml2YXRlIHJlbmRlck1ldHJpY3NDb2xsZWN0aW9uKCk6IENmbkF1dG9TY2FsaW5nR3JvdXAuTWV0cmljc0NvbGxlY3Rpb25Qcm9wZXJ0eVtdIHwgdW5kZWZpbmVkIHtcbiAgICBpZiAodGhpcy5ncm91cE1ldHJpY3MubGVuZ3RoID09PSAwKSB7XG4gICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cblxuICAgIHJldHVybiB0aGlzLmdyb3VwTWV0cmljcy5tYXAoZ3JvdXAgPT4gKHtcbiAgICAgIGdyYW51bGFyaXR5OiAnMU1pbnV0ZScsXG4gICAgICBtZXRyaWNzOiBncm91cC5fbWV0cmljcz8uc2l6ZSAhPT0gMCA/IFsuLi5ncm91cC5fbWV0cmljc10ubWFwKG0gPT4gbS5uYW1lKSA6IHVuZGVmaW5lZCxcbiAgICB9KSk7XG4gIH1cblxuICBwcml2YXRlIGdldExhdW5jaFNldHRpbmdzKGxhdW5jaENvbmZpZz86IENmbkxhdW5jaENvbmZpZ3VyYXRpb24sIGxhdW5jaFRlbXBsYXRlPzogZWMyLklMYXVuY2hUZW1wbGF0ZSwgbWl4ZWRJbnN0YW5jZXNQb2xpY3k/OiBNaXhlZEluc3RhbmNlc1BvbGljeSlcbiAgICA6IFBpY2s8Q2ZuQXV0b1NjYWxpbmdHcm91cFByb3BzLCAnbGF1bmNoQ29uZmlndXJhdGlvbk5hbWUnPlxuICAgIHwgUGljazxDZm5BdXRvU2NhbGluZ0dyb3VwUHJvcHMsICdsYXVuY2hUZW1wbGF0ZSc+XG4gICAgfCBQaWNrPENmbkF1dG9TY2FsaW5nR3JvdXBQcm9wcywgJ21peGVkSW5zdGFuY2VzUG9saWN5Jz4ge1xuICAgIGlmIChsYXVuY2hDb25maWcpIHtcbiAgICAgIHJldHVybiB7XG4gICAgICAgIGxhdW5jaENvbmZpZ3VyYXRpb25OYW1lOiBsYXVuY2hDb25maWcucmVmLFxuICAgICAgfTtcbiAgICB9XG5cbiAgICBpZiAobGF1bmNoVGVtcGxhdGUpIHtcbiAgICAgIHJldHVybiB7XG4gICAgICAgIGxhdW5jaFRlbXBsYXRlOiB0aGlzLmNvbnZlcnRJTGF1bmNoVGVtcGxhdGVUb1NwZWNpZmljYXRpb24obGF1bmNoVGVtcGxhdGUpLFxuICAgICAgfTtcbiAgICB9XG5cbiAgICBpZiAobWl4ZWRJbnN0YW5jZXNQb2xpY3kpIHtcbiAgICAgIGxldCBpbnN0YW5jZXNEaXN0cmlidXRpb246IENmbkF1dG9TY2FsaW5nR3JvdXAuSW5zdGFuY2VzRGlzdHJpYnV0aW9uUHJvcGVydHkgfCB1bmRlZmluZWQgPSB1bmRlZmluZWQ7XG4gICAgICBpZiAobWl4ZWRJbnN0YW5jZXNQb2xpY3kuaW5zdGFuY2VzRGlzdHJpYnV0aW9uKSB7XG4gICAgICAgIGNvbnN0IGRpc3QgPSBtaXhlZEluc3RhbmNlc1BvbGljeS5pbnN0YW5jZXNEaXN0cmlidXRpb247XG4gICAgICAgIGluc3RhbmNlc0Rpc3RyaWJ1dGlvbiA9IHtcbiAgICAgICAgICBvbkRlbWFuZEFsbG9jYXRpb25TdHJhdGVneTogZGlzdC5vbkRlbWFuZEFsbG9jYXRpb25TdHJhdGVneT8udG9TdHJpbmcoKSxcbiAgICAgICAgICBvbkRlbWFuZEJhc2VDYXBhY2l0eTogZGlzdC5vbkRlbWFuZEJhc2VDYXBhY2l0eSxcbiAgICAgICAgICBvbkRlbWFuZFBlcmNlbnRhZ2VBYm92ZUJhc2VDYXBhY2l0eTogZGlzdC5vbkRlbWFuZFBlcmNlbnRhZ2VBYm92ZUJhc2VDYXBhY2l0eSxcbiAgICAgICAgICBzcG90QWxsb2NhdGlvblN0cmF0ZWd5OiBkaXN0LnNwb3RBbGxvY2F0aW9uU3RyYXRlZ3k/LnRvU3RyaW5nKCksXG4gICAgICAgICAgc3BvdEluc3RhbmNlUG9vbHM6IGRpc3Quc3BvdEluc3RhbmNlUG9vbHMsXG4gICAgICAgICAgc3BvdE1heFByaWNlOiBkaXN0LnNwb3RNYXhQcmljZSxcbiAgICAgICAgfTtcbiAgICAgIH1cbiAgICAgIHJldHVybiB7XG4gICAgICAgIG1peGVkSW5zdGFuY2VzUG9saWN5OiB7XG4gICAgICAgICAgaW5zdGFuY2VzRGlzdHJpYnV0aW9uLFxuICAgICAgICAgIGxhdW5jaFRlbXBsYXRlOiB7XG4gICAgICAgICAgICBsYXVuY2hUZW1wbGF0ZVNwZWNpZmljYXRpb246IHRoaXMuY29udmVydElMYXVuY2hUZW1wbGF0ZVRvU3BlY2lmaWNhdGlvbihtaXhlZEluc3RhbmNlc1BvbGljeS5sYXVuY2hUZW1wbGF0ZSksXG4gICAgICAgICAgICAuLi4obWl4ZWRJbnN0YW5jZXNQb2xpY3kubGF1bmNoVGVtcGxhdGVPdmVycmlkZXMgPyB7XG4gICAgICAgICAgICAgIG92ZXJyaWRlczogbWl4ZWRJbnN0YW5jZXNQb2xpY3kubGF1bmNoVGVtcGxhdGVPdmVycmlkZXMubWFwKG92ZXJyaWRlID0+IHtcbiAgICAgICAgICAgICAgICBpZiAob3ZlcnJpZGUud2VpZ2h0ZWRDYXBhY2l0eSAmJiBNYXRoLmZsb29yKG92ZXJyaWRlLndlaWdodGVkQ2FwYWNpdHkpICE9PSBvdmVycmlkZS53ZWlnaHRlZENhcGFjaXR5KSB7XG4gICAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1dlaWdodCBtdXN0IGJlIGFuIGludGVnZXInKTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgICAgICAgIGluc3RhbmNlVHlwZTogb3ZlcnJpZGUuaW5zdGFuY2VUeXBlLnRvU3RyaW5nKCksXG4gICAgICAgICAgICAgICAgICBsYXVuY2hUZW1wbGF0ZVNwZWNpZmljYXRpb246IG92ZXJyaWRlLmxhdW5jaFRlbXBsYXRlXG4gICAgICAgICAgICAgICAgICAgID8gdGhpcy5jb252ZXJ0SUxhdW5jaFRlbXBsYXRlVG9TcGVjaWZpY2F0aW9uKG92ZXJyaWRlLmxhdW5jaFRlbXBsYXRlKVxuICAgICAgICAgICAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgICAgICAgICAgICAgIHdlaWdodGVkQ2FwYWNpdHk6IG92ZXJyaWRlLndlaWdodGVkQ2FwYWNpdHk/LnRvU3RyaW5nKCksXG4gICAgICAgICAgICAgICAgfTtcbiAgICAgICAgICAgICAgfSksXG4gICAgICAgICAgICB9IDoge30pLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9O1xuICAgIH1cblxuICAgIHRocm93IG5ldyBFcnJvcignRWl0aGVyIGxhdW5jaENvbmZpZywgbGF1bmNoVGVtcGxhdGUgb3IgbWl4ZWRJbnN0YW5jZXNQb2xpY3kgbmVlZHMgdG8gYmUgc3BlY2lmaWVkLicpO1xuICB9XG5cbiAgcHJpdmF0ZSBjb252ZXJ0SUxhdW5jaFRlbXBsYXRlVG9TcGVjaWZpY2F0aW9uKGxhdW5jaFRlbXBsYXRlOiBlYzIuSUxhdW5jaFRlbXBsYXRlKTogQ2ZuQXV0b1NjYWxpbmdHcm91cC5MYXVuY2hUZW1wbGF0ZVNwZWNpZmljYXRpb25Qcm9wZXJ0eSB7XG4gICAgaWYgKGxhdW5jaFRlbXBsYXRlLmxhdW5jaFRlbXBsYXRlSWQpIHtcbiAgICAgIHJldHVybiB7XG4gICAgICAgIGxhdW5jaFRlbXBsYXRlSWQ6IGxhdW5jaFRlbXBsYXRlLmxhdW5jaFRlbXBsYXRlSWQsXG4gICAgICAgIHZlcnNpb246IGxhdW5jaFRlbXBsYXRlLnZlcnNpb25OdW1iZXIsXG4gICAgICB9O1xuICAgIH0gZWxzZSB7XG4gICAgICByZXR1cm4ge1xuICAgICAgICBsYXVuY2hUZW1wbGF0ZU5hbWU6IGxhdW5jaFRlbXBsYXRlLmxhdW5jaFRlbXBsYXRlTmFtZSxcbiAgICAgICAgdmVyc2lvbjogbGF1bmNoVGVtcGxhdGUudmVyc2lvbk51bWJlcixcbiAgICAgIH07XG4gICAgfVxuICB9XG59XG5cbi8qKlxuICogVGhlIHR5cGUgb2YgdXBkYXRlIHRvIHBlcmZvcm0gb24gaW5zdGFuY2VzIGluIHRoaXMgQXV0b1NjYWxpbmdHcm91cFxuICpcbiAqIEBkZXByZWNhdGVkIFVzZSBVcGRhdGVQb2xpY3kgaW5zdGVhZFxuICovXG5leHBvcnQgZW51bSBVcGRhdGVUeXBlIHtcbiAgLyoqXG4gICAqIERvbid0IGRvIGFueXRoaW5nXG4gICAqL1xuICBOT05FID0gJ05vbmUnLFxuXG4gIC8qKlxuICAgKiBSZXBsYWNlIHRoZSBlbnRpcmUgQXV0b1NjYWxpbmdHcm91cFxuICAgKlxuICAgKiBCdWlsZHMgYSBuZXcgQXV0b1NjYWxpbmdHcm91cCBmaXJzdCwgdGhlbiBkZWxldGUgdGhlIG9sZCBvbmUuXG4gICAqL1xuICBSRVBMQUNJTkdfVVBEQVRFID0gJ1JlcGxhY2UnLFxuXG4gIC8qKlxuICAgKiBSZXBsYWNlIHRoZSBpbnN0YW5jZXMgaW4gdGhlIEF1dG9TY2FsaW5nR3JvdXAuXG4gICAqL1xuICBST0xMSU5HX1VQREFURSA9ICdSb2xsaW5nVXBkYXRlJyxcbn1cblxuLyoqXG4gKiBBdXRvU2NhbGluZ0dyb3VwIGZsZWV0IGNoYW5nZSBub3RpZmljYXRpb25zIGNvbmZpZ3VyYXRpb25zLlxuICogWW91IGNhbiBjb25maWd1cmUgQXV0b1NjYWxpbmcgdG8gc2VuZCBhbiBTTlMgbm90aWZpY2F0aW9uIHdoZW5ldmVyIHlvdXIgQXV0byBTY2FsaW5nIGdyb3VwIHNjYWxlcy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBOb3RpZmljYXRpb25Db25maWd1cmF0aW9uIHtcbiAgLyoqXG4gICAqIFNOUyB0b3BpYyB0byBzZW5kIG5vdGlmaWNhdGlvbnMgYWJvdXQgZmxlZXQgc2NhbGluZyBldmVudHNcbiAgICovXG4gIHJlYWRvbmx5IHRvcGljOiBzbnMuSVRvcGljO1xuXG4gIC8qKlxuICAgKiBXaGljaCBmbGVldCBzY2FsaW5nIGV2ZW50cyB0cmlnZ2VycyBhIG5vdGlmaWNhdGlvblxuICAgKiBAZGVmYXVsdCBTY2FsaW5nRXZlbnRzLkFMTFxuICAgKi9cbiAgcmVhZG9ubHkgc2NhbGluZ0V2ZW50cz86IFNjYWxpbmdFdmVudHM7XG59XG5cbi8qKlxuICogRmxlZXQgc2NhbGluZyBldmVudHNcbiAqL1xuZXhwb3J0IGVudW0gU2NhbGluZ0V2ZW50IHtcbiAgLyoqXG4gICAqIE5vdGlmeSB3aGVuIGFuIGluc3RhbmNlIHdhcyBsYXVuY2hlZFxuICAgKi9cbiAgSU5TVEFOQ0VfTEFVTkNIID0gJ2F1dG9zY2FsaW5nOkVDMl9JTlNUQU5DRV9MQVVOQ0gnLFxuXG4gIC8qKlxuICAgKiBOb3RpZnkgd2hlbiBhbiBpbnN0YW5jZSB3YXMgdGVybWluYXRlZFxuICAgKi9cbiAgSU5TVEFOQ0VfVEVSTUlOQVRFID0gJ2F1dG9zY2FsaW5nOkVDMl9JTlNUQU5DRV9URVJNSU5BVEUnLFxuXG4gIC8qKlxuICAgKiBOb3RpZnkgd2hlbiBhbiBpbnN0YW5jZSBmYWlsZWQgdG8gdGVybWluYXRlXG4gICAqL1xuICBJTlNUQU5DRV9URVJNSU5BVEVfRVJST1IgPSAnYXV0b3NjYWxpbmc6RUMyX0lOU1RBTkNFX1RFUk1JTkFURV9FUlJPUicsXG5cbiAgLyoqXG4gICAqIE5vdGlmeSB3aGVuIGFuIGluc3RhbmNlIGZhaWxlZCB0byBsYXVuY2hcbiAgICovXG4gIElOU1RBTkNFX0xBVU5DSF9FUlJPUiA9ICdhdXRvc2NhbGluZzpFQzJfSU5TVEFOQ0VfTEFVTkNIX0VSUk9SJyxcblxuICAvKipcbiAgICogU2VuZCBhIHRlc3Qgbm90aWZpY2F0aW9uIHRvIHRoZSB0b3BpY1xuICAgKi9cbiAgVEVTVF9OT1RJRklDQVRJT04gPSAnYXV0b3NjYWxpbmc6VEVTVF9OT1RJRklDQVRJT04nXG59XG5cbi8qKlxuICogQWRkaXRpb25hbCBzZXR0aW5ncyB3aGVuIGEgcm9sbGluZyB1cGRhdGUgaXMgc2VsZWN0ZWRcbiAqIEBkZXByZWNhdGVkIHVzZSBgVXBkYXRlUG9saWN5LnJvbGxpbmdVcGRhdGUoKWBcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBSb2xsaW5nVXBkYXRlQ29uZmlndXJhdGlvbiB7XG4gIC8qKlxuICAgKiBUaGUgbWF4aW11bSBudW1iZXIgb2YgaW5zdGFuY2VzIHRoYXQgQVdTIENsb3VkRm9ybWF0aW9uIHVwZGF0ZXMgYXQgb25jZS5cbiAgICpcbiAgICogQGRlZmF1bHQgMVxuICAgKi9cbiAgcmVhZG9ubHkgbWF4QmF0Y2hTaXplPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgbWluaW11bSBudW1iZXIgb2YgaW5zdGFuY2VzIHRoYXQgbXVzdCBiZSBpbiBzZXJ2aWNlIGJlZm9yZSBtb3JlIGluc3RhbmNlcyBhcmUgcmVwbGFjZWQuXG4gICAqXG4gICAqIFRoaXMgbnVtYmVyIGFmZmVjdHMgdGhlIHNwZWVkIG9mIHRoZSByZXBsYWNlbWVudC5cbiAgICpcbiAgICogQGRlZmF1bHQgMFxuICAgKi9cbiAgcmVhZG9ubHkgbWluSW5zdGFuY2VzSW5TZXJ2aWNlPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgcGVyY2VudGFnZSBvZiBpbnN0YW5jZXMgdGhhdCBtdXN0IHNpZ25hbCBzdWNjZXNzIGZvciBhbiB1cGRhdGUgdG8gc3VjY2VlZC5cbiAgICpcbiAgICogSWYgYW4gaW5zdGFuY2UgZG9lc24ndCBzZW5kIGEgc2lnbmFsIHdpdGhpbiB0aGUgdGltZSBzcGVjaWZpZWQgaW4gdGhlXG4gICAqIHBhdXNlVGltZSBwcm9wZXJ0eSwgQVdTIENsb3VkRm9ybWF0aW9uIGFzc3VtZXMgdGhhdCB0aGUgaW5zdGFuY2Ugd2Fzbid0XG4gICAqIHVwZGF0ZWQuXG4gICAqXG4gICAqIFRoaXMgbnVtYmVyIGFmZmVjdHMgdGhlIHN1Y2Nlc3Mgb2YgdGhlIHJlcGxhY2VtZW50LlxuICAgKlxuICAgKiBJZiB5b3Ugc3BlY2lmeSB0aGlzIHByb3BlcnR5LCB5b3UgbXVzdCBhbHNvIGVuYWJsZSB0aGVcbiAgICogd2FpdE9uUmVzb3VyY2VTaWduYWxzIGFuZCBwYXVzZVRpbWUgcHJvcGVydGllcy5cbiAgICpcbiAgICogQGRlZmF1bHQgMTAwXG4gICAqL1xuICByZWFkb25seSBtaW5TdWNjZXNzZnVsSW5zdGFuY2VzUGVyY2VudD86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIHBhdXNlIHRpbWUgYWZ0ZXIgbWFraW5nIGEgY2hhbmdlIHRvIGEgYmF0Y2ggb2YgaW5zdGFuY2VzLlxuICAgKlxuICAgKiBUaGlzIGlzIGludGVuZGVkIHRvIGdpdmUgdGhvc2UgaW5zdGFuY2VzIHRpbWUgdG8gc3RhcnQgc29mdHdhcmUgYXBwbGljYXRpb25zLlxuICAgKlxuICAgKiBTcGVjaWZ5IFBhdXNlVGltZSBpbiB0aGUgSVNPODYwMSBkdXJhdGlvbiBmb3JtYXQgKGluIHRoZSBmb3JtYXRcbiAgICogUFQjSCNNI1MsIHdoZXJlIGVhY2ggIyBpcyB0aGUgbnVtYmVyIG9mIGhvdXJzLCBtaW51dGVzLCBhbmQgc2Vjb25kcyxcbiAgICogcmVzcGVjdGl2ZWx5KS4gVGhlIG1heGltdW0gUGF1c2VUaW1lIGlzIG9uZSBob3VyIChQVDFIKS5cbiAgICpcbiAgICogQGRlZmF1bHQgRHVyYXRpb24ubWludXRlcyg1KSBpZiB0aGUgd2FpdE9uUmVzb3VyY2VTaWduYWxzIHByb3BlcnR5IGlzIHRydWUsIG90aGVyd2lzZSAwXG4gICAqL1xuICByZWFkb25seSBwYXVzZVRpbWU/OiBEdXJhdGlvbjtcblxuICAvKipcbiAgICogU3BlY2lmaWVzIHdoZXRoZXIgdGhlIEF1dG8gU2NhbGluZyBncm91cCB3YWl0cyBvbiBzaWduYWxzIGZyb20gbmV3IGluc3RhbmNlcyBkdXJpbmcgYW4gdXBkYXRlLlxuICAgKlxuICAgKiBBV1MgQ2xvdWRGb3JtYXRpb24gbXVzdCByZWNlaXZlIGEgc2lnbmFsIGZyb20gZWFjaCBuZXcgaW5zdGFuY2Ugd2l0aGluXG4gICAqIHRoZSBzcGVjaWZpZWQgUGF1c2VUaW1lIGJlZm9yZSBjb250aW51aW5nIHRoZSB1cGRhdGUuXG4gICAqXG4gICAqIFRvIGhhdmUgaW5zdGFuY2VzIHdhaXQgZm9yIGFuIEVsYXN0aWMgTG9hZCBCYWxhbmNpbmcgaGVhbHRoIGNoZWNrIGJlZm9yZVxuICAgKiB0aGV5IHNpZ25hbCBzdWNjZXNzLCBhZGQgYSBoZWFsdGgtY2hlY2sgdmVyaWZpY2F0aW9uIGJ5IHVzaW5nIHRoZVxuICAgKiBjZm4taW5pdCBoZWxwZXIgc2NyaXB0LiBGb3IgYW4gZXhhbXBsZSwgc2VlIHRoZSB2ZXJpZnlfaW5zdGFuY2VfaGVhbHRoXG4gICAqIGNvbW1hbmQgaW4gdGhlIEF1dG8gU2NhbGluZyByb2xsaW5nIHVwZGF0ZXMgc2FtcGxlIHRlbXBsYXRlLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlIGlmIHlvdSBzcGVjaWZpZWQgdGhlIG1pblN1Y2Nlc3NmdWxJbnN0YW5jZXNQZXJjZW50IHByb3BlcnR5LCBmYWxzZSBvdGhlcndpc2VcbiAgICovXG4gIHJlYWRvbmx5IHdhaXRPblJlc291cmNlU2lnbmFscz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFNwZWNpZmllcyB0aGUgQXV0byBTY2FsaW5nIHByb2Nlc3NlcyB0byBzdXNwZW5kIGR1cmluZyBhIHN0YWNrIHVwZGF0ZS5cbiAgICpcbiAgICogU3VzcGVuZGluZyBwcm9jZXNzZXMgcHJldmVudHMgQXV0byBTY2FsaW5nIGZyb20gaW50ZXJmZXJpbmcgd2l0aCBhIHN0YWNrXG4gICAqIHVwZGF0ZS5cbiAgICpcbiAgICogQGRlZmF1bHQgSGVhbHRoQ2hlY2ssIFJlcGxhY2VVbmhlYWx0aHksIEFaUmViYWxhbmNlLCBBbGFybU5vdGlmaWNhdGlvbiwgU2NoZWR1bGVkQWN0aW9ucy5cbiAgICovXG4gIHJlYWRvbmx5IHN1c3BlbmRQcm9jZXNzZXM/OiBTY2FsaW5nUHJvY2Vzc1tdO1xufVxuXG4vKipcbiAqIEEgbGlzdCBvZiBTY2FsaW5nRXZlbnRzLCB5b3UgY2FuIHVzZSBvbmUgb2YgdGhlIHByZWRlZmluZWQgbGlzdHMsIHN1Y2ggYXMgU2NhbGluZ0V2ZW50cy5FUlJPUlNcbiAqIG9yIGNyZWF0ZSBhIGN1c3RvbSBncm91cCBieSBpbnN0YW50aWF0aW5nIGEgYE5vdGlmaWNhdGlvblR5cGVzYCBvYmplY3QsIGUuZzogYG5ldyBOb3RpZmljYXRpb25UeXBlcyhgTm90aWZpY2F0aW9uVHlwZS5JTlNUQU5DRV9MQVVOQ0hgKWAuXG4gKi9cbmV4cG9ydCBjbGFzcyBTY2FsaW5nRXZlbnRzIHtcbiAgLyoqXG4gICAqIEZsZWV0IHNjYWxpbmcgZXJyb3JzXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IEVSUk9SUyA9IG5ldyBTY2FsaW5nRXZlbnRzKFNjYWxpbmdFdmVudC5JTlNUQU5DRV9MQVVOQ0hfRVJST1IsIFNjYWxpbmdFdmVudC5JTlNUQU5DRV9URVJNSU5BVEVfRVJST1IpO1xuXG4gIC8qKlxuICAgKiBBbGwgZmxlZXQgc2NhbGluZyBldmVudHNcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQUxMID0gbmV3IFNjYWxpbmdFdmVudHMoU2NhbGluZ0V2ZW50LklOU1RBTkNFX0xBVU5DSCxcbiAgICBTY2FsaW5nRXZlbnQuSU5TVEFOQ0VfTEFVTkNIX0VSUk9SLFxuICAgIFNjYWxpbmdFdmVudC5JTlNUQU5DRV9URVJNSU5BVEUsXG4gICAgU2NhbGluZ0V2ZW50LklOU1RBTkNFX1RFUk1JTkFURV9FUlJPUik7XG5cbiAgLyoqXG4gICAqIEZsZWV0IHNjYWxpbmcgbGF1bmNoIGV2ZW50c1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBMQVVOQ0hfRVZFTlRTID0gbmV3IFNjYWxpbmdFdmVudHMoU2NhbGluZ0V2ZW50LklOU1RBTkNFX0xBVU5DSCwgU2NhbGluZ0V2ZW50LklOU1RBTkNFX0xBVU5DSF9FUlJPUik7XG5cbiAgLyoqXG4gICAqIEZsZWV0IHRlcm1pbmF0aW9uIGxhdW5jaCBldmVudHNcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVEVSTUlOQVRJT05fRVZFTlRTID0gbmV3IFNjYWxpbmdFdmVudHMoU2NhbGluZ0V2ZW50LklOU1RBTkNFX1RFUk1JTkFURSwgU2NhbGluZ0V2ZW50LklOU1RBTkNFX1RFUk1JTkFURV9FUlJPUik7XG5cblxuICAvKipcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgX3R5cGVzOiBTY2FsaW5nRXZlbnRbXTtcblxuICBjb25zdHJ1Y3RvciguLi50eXBlczogU2NhbGluZ0V2ZW50W10pIHtcbiAgICB0aGlzLl90eXBlcyA9IHR5cGVzO1xuICB9XG59XG5cbmV4cG9ydCBlbnVtIFNjYWxpbmdQcm9jZXNzIHtcbiAgTEFVTkNIID0gJ0xhdW5jaCcsXG4gIFRFUk1JTkFURSA9ICdUZXJtaW5hdGUnLFxuICBIRUFMVEhfQ0hFQ0sgPSAnSGVhbHRoQ2hlY2snLFxuICBSRVBMQUNFX1VOSEVBTFRIWSA9ICdSZXBsYWNlVW5oZWFsdGh5JyxcbiAgQVpfUkVCQUxBTkNFID0gJ0FaUmViYWxhbmNlJyxcbiAgQUxBUk1fTk9USUZJQ0FUSU9OID0gJ0FsYXJtTm90aWZpY2F0aW9uJyxcbiAgU0NIRURVTEVEX0FDVElPTlMgPSAnU2NoZWR1bGVkQWN0aW9ucycsXG4gIEFERF9UT19MT0FEX0JBTEFOQ0VSID0gJ0FkZFRvTG9hZEJhbGFuY2VyJ1xufVxuXG4vLyBSZWNvbW1lbmRlZCBsaXN0IG9mIHByb2Nlc3NlcyB0byBzdXNwZW5kIGZyb20gaGVyZTpcbi8vIGh0dHBzOi8vYXdzLmFtYXpvbi5jb20vcHJlbWl1bXN1cHBvcnQva25vd2xlZGdlLWNlbnRlci9hdXRvLXNjYWxpbmctZ3JvdXAtcm9sbGluZy11cGRhdGVzL1xuY29uc3QgREVGQVVMVF9TVVNQRU5EX1BST0NFU1NFUyA9IFtTY2FsaW5nUHJvY2Vzcy5IRUFMVEhfQ0hFQ0ssIFNjYWxpbmdQcm9jZXNzLlJFUExBQ0VfVU5IRUFMVEhZLCBTY2FsaW5nUHJvY2Vzcy5BWl9SRUJBTEFOQ0UsXG4gIFNjYWxpbmdQcm9jZXNzLkFMQVJNX05PVElGSUNBVElPTiwgU2NhbGluZ1Byb2Nlc3MuU0NIRURVTEVEX0FDVElPTlNdO1xuXG4vKipcbiAqIEVDMiBIZWF0aCBjaGVjayBvcHRpb25zXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRWMySGVhbHRoQ2hlY2tPcHRpb25zIHtcbiAgLyoqXG4gICAqIFNwZWNpZmllZCB0aGUgdGltZSBBdXRvIFNjYWxpbmcgd2FpdHMgYmVmb3JlIGNoZWNraW5nIHRoZSBoZWFsdGggc3RhdHVzIG9mIGFuIEVDMiBpbnN0YW5jZSB0aGF0IGhhcyBjb21lIGludG8gc2VydmljZVxuICAgKlxuICAgKiBAZGVmYXVsdCBEdXJhdGlvbi5zZWNvbmRzKDApXG4gICAqL1xuICByZWFkb25seSBncmFjZT86IER1cmF0aW9uO1xufVxuXG4vKipcbiAqIEVMQiBIZWF0aCBjaGVjayBvcHRpb25zXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRWxiSGVhbHRoQ2hlY2tPcHRpb25zIHtcbiAgLyoqXG4gICAqIFNwZWNpZmllZCB0aGUgdGltZSBBdXRvIFNjYWxpbmcgd2FpdHMgYmVmb3JlIGNoZWNraW5nIHRoZSBoZWFsdGggc3RhdHVzIG9mIGFuIEVDMiBpbnN0YW5jZSB0aGF0IGhhcyBjb21lIGludG8gc2VydmljZVxuICAgKlxuICAgKiBUaGlzIG9wdGlvbiBpcyByZXF1aXJlZCBmb3IgRUxCIGhlYWx0aCBjaGVja3MuXG4gICAqL1xuICByZWFkb25seSBncmFjZTogRHVyYXRpb247XG59XG5cbi8qKlxuICogSGVhbHRoIGNoZWNrIHNldHRpbmdzXG4gKi9cbmV4cG9ydCBjbGFzcyBIZWFsdGhDaGVjayB7XG4gIC8qKlxuICAgKiBVc2UgRUMyIGZvciBoZWFsdGggY2hlY2tzXG4gICAqXG4gICAqIEBwYXJhbSBvcHRpb25zIEVDMiBoZWFsdGggY2hlY2sgb3B0aW9uc1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyBlYzIob3B0aW9uczogRWMySGVhbHRoQ2hlY2tPcHRpb25zID0ge30pOiBIZWFsdGhDaGVjayB7XG4gICAgcmV0dXJuIG5ldyBIZWFsdGhDaGVjayhIZWFsdGhDaGVja1R5cGUuRUMyLCBvcHRpb25zLmdyYWNlKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBVc2UgRUxCIGZvciBoZWFsdGggY2hlY2tzLlxuICAgKiBJdCBjb25zaWRlcnMgdGhlIGluc3RhbmNlIHVuaGVhbHRoeSBpZiBpdCBmYWlscyBlaXRoZXIgdGhlIEVDMiBzdGF0dXMgY2hlY2tzIG9yIHRoZSBsb2FkIGJhbGFuY2VyIGhlYWx0aCBjaGVja3MuXG4gICAqXG4gICAqIEBwYXJhbSBvcHRpb25zIEVMQiBoZWFsdGggY2hlY2sgb3B0aW9uc1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyBlbGIob3B0aW9uczogRWxiSGVhbHRoQ2hlY2tPcHRpb25zKTogSGVhbHRoQ2hlY2sge1xuICAgIHJldHVybiBuZXcgSGVhbHRoQ2hlY2soSGVhbHRoQ2hlY2tUeXBlLkVMQiwgb3B0aW9ucy5ncmFjZSk7XG4gIH1cblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKHB1YmxpYyByZWFkb25seSB0eXBlOiBzdHJpbmcsIHB1YmxpYyByZWFkb25seSBncmFjZVBlcmlvZD86IER1cmF0aW9uKSB7IH1cbn1cblxuZW51bSBIZWFsdGhDaGVja1R5cGUge1xuICBFQzIgPSAnRUMyJyxcbiAgRUxCID0gJ0VMQicsXG59XG5cbi8qKlxuICogUmVuZGVyIHRoZSByb2xsaW5nIHVwZGF0ZSBjb25maWd1cmF0aW9uIGludG8gdGhlIGFwcHJvcHJpYXRlIG9iamVjdFxuICovXG5mdW5jdGlvbiByZW5kZXJSb2xsaW5nVXBkYXRlQ29uZmlnKGNvbmZpZzogUm9sbGluZ1VwZGF0ZUNvbmZpZ3VyYXRpb24gPSB7fSk6IENmbkF1dG9TY2FsaW5nUm9sbGluZ1VwZGF0ZSB7XG4gIGNvbnN0IHdhaXRPblJlc291cmNlU2lnbmFscyA9IGNvbmZpZy5taW5TdWNjZXNzZnVsSW5zdGFuY2VzUGVyY2VudCAhPT0gdW5kZWZpbmVkO1xuICBjb25zdCBwYXVzZVRpbWUgPSBjb25maWcucGF1c2VUaW1lIHx8ICh3YWl0T25SZXNvdXJjZVNpZ25hbHMgPyBEdXJhdGlvbi5taW51dGVzKDUpIDogRHVyYXRpb24uc2Vjb25kcygwKSk7XG5cbiAgcmV0dXJuIHtcbiAgICBtYXhCYXRjaFNpemU6IGNvbmZpZy5tYXhCYXRjaFNpemUsXG4gICAgbWluSW5zdGFuY2VzSW5TZXJ2aWNlOiBjb25maWcubWluSW5zdGFuY2VzSW5TZXJ2aWNlLFxuICAgIG1pblN1Y2Nlc3NmdWxJbnN0YW5jZXNQZXJjZW50OiB2YWxpZGF0ZVBlcmNlbnRhZ2UoY29uZmlnLm1pblN1Y2Nlc3NmdWxJbnN0YW5jZXNQZXJjZW50KSxcbiAgICB3YWl0T25SZXNvdXJjZVNpZ25hbHMsXG4gICAgcGF1c2VUaW1lOiBwYXVzZVRpbWUgJiYgcGF1c2VUaW1lLnRvSXNvU3RyaW5nKCksXG4gICAgc3VzcGVuZFByb2Nlc3NlczogY29uZmlnLnN1c3BlbmRQcm9jZXNzZXMgPz8gREVGQVVMVF9TVVNQRU5EX1BST0NFU1NFUyxcbiAgfTtcbn1cblxuZnVuY3Rpb24gdmFsaWRhdGVQZXJjZW50YWdlKHg/OiBudW1iZXIpOiBudW1iZXIgfCB1bmRlZmluZWQge1xuICBpZiAoeCA9PT0gdW5kZWZpbmVkIHx8ICgwIDw9IHggJiYgeCA8PSAxMDApKSB7IHJldHVybiB4OyB9XG4gIHRocm93IG5ldyBFcnJvcihgRXhwZWN0ZWQ6IGEgcGVyY2VudGFnZSAwLi4xMDAsIGdvdDogJHt4fWApO1xufVxuXG4vKipcbiAqIEFuIEF1dG9TY2FsaW5nR3JvdXBcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJQXV0b1NjYWxpbmdHcm91cCBleHRlbmRzIElSZXNvdXJjZSwgaWFtLklHcmFudGFibGUge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIEF1dG9TY2FsaW5nR3JvdXBcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgYXV0b1NjYWxpbmdHcm91cE5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGFybiBvZiB0aGUgQXV0b1NjYWxpbmdHcm91cFxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSBhdXRvU2NhbGluZ0dyb3VwQXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBvcGVyYXRpbmcgc3lzdGVtIGZhbWlseSB0aGF0IHRoZSBpbnN0YW5jZXMgaW4gdGhpcyBhdXRvLXNjYWxpbmcgZ3JvdXAgYmVsb25nIHRvLlxuICAgKiBJcyAnVU5LTk9XTicgZm9yIGltcG9ydGVkIEFTR3MuXG4gICAqL1xuICByZWFkb25seSBvc1R5cGU6IGVjMi5PcGVyYXRpbmdTeXN0ZW1UeXBlO1xuXG4gIC8qKlxuICAgKiBBZGQgY29tbWFuZCB0byB0aGUgc3RhcnR1cCBzY3JpcHQgb2YgZmxlZXQgaW5zdGFuY2VzLlxuICAgKiBUaGUgY29tbWFuZCBtdXN0IGJlIGluIHRoZSBzY3JpcHRpbmcgbGFuZ3VhZ2Ugc3VwcG9ydGVkIGJ5IHRoZSBmbGVldCdzIE9TIChpLmUuIExpbnV4L1dpbmRvd3MpLlxuICAgKiBEb2VzIG5vdGhpbmcgZm9yIGltcG9ydGVkIEFTR3MuXG4gICAqL1xuICBhZGRVc2VyRGF0YSguLi5jb21tYW5kczogc3RyaW5nW10pOiB2b2lkO1xuXG4gIC8qKlxuICAgKiBTZW5kIGEgbWVzc2FnZSB0byBlaXRoZXIgYW4gU1FTIHF1ZXVlIG9yIFNOUyB0b3BpYyB3aGVuIGluc3RhbmNlcyBsYXVuY2ggb3IgdGVybWluYXRlXG4gICAqL1xuICBhZGRMaWZlY3ljbGVIb29rKGlkOiBzdHJpbmcsIHByb3BzOiBCYXNpY0xpZmVjeWNsZUhvb2tQcm9wcyk6IExpZmVjeWNsZUhvb2s7XG5cbiAgLyoqXG4gICAqIEFkZCBhIHBvb2wgb2YgcHJlLWluaXRpYWxpemVkIEVDMiBpbnN0YW5jZXMgdGhhdCBzaXRzIGFsb25nc2lkZSBhbiBBdXRvIFNjYWxpbmcgZ3JvdXBcbiAgICovXG4gIGFkZFdhcm1Qb29sKG9wdGlvbnM/OiBXYXJtUG9vbE9wdGlvbnMpOiBXYXJtUG9vbDtcblxuICAvKipcbiAgICogU2NhbGUgb3V0IG9yIGluIGJhc2VkIG9uIHRpbWVcbiAgICovXG4gIHNjYWxlT25TY2hlZHVsZShpZDogc3RyaW5nLCBwcm9wczogQmFzaWNTY2hlZHVsZWRBY3Rpb25Qcm9wcyk6IFNjaGVkdWxlZEFjdGlvbjtcblxuICAvKipcbiAgICogU2NhbGUgb3V0IG9yIGluIHRvIGFjaGlldmUgYSB0YXJnZXQgQ1BVIHV0aWxpemF0aW9uXG4gICAqL1xuICBzY2FsZU9uQ3B1VXRpbGl6YXRpb24oaWQ6IHN0cmluZywgcHJvcHM6IENwdVV0aWxpemF0aW9uU2NhbGluZ1Byb3BzKTogVGFyZ2V0VHJhY2tpbmdTY2FsaW5nUG9saWN5O1xuXG4gIC8qKlxuICAgKiBTY2FsZSBvdXQgb3IgaW4gdG8gYWNoaWV2ZSBhIHRhcmdldCBuZXR3b3JrIGluZ3Jlc3MgcmF0ZVxuICAgKi9cbiAgc2NhbGVPbkluY29taW5nQnl0ZXMoaWQ6IHN0cmluZywgcHJvcHM6IE5ldHdvcmtVdGlsaXphdGlvblNjYWxpbmdQcm9wcyk6IFRhcmdldFRyYWNraW5nU2NhbGluZ1BvbGljeTtcblxuICAvKipcbiAgICogU2NhbGUgb3V0IG9yIGluIHRvIGFjaGlldmUgYSB0YXJnZXQgbmV0d29yayBlZ3Jlc3MgcmF0ZVxuICAgKi9cbiAgc2NhbGVPbk91dGdvaW5nQnl0ZXMoaWQ6IHN0cmluZywgcHJvcHM6IE5ldHdvcmtVdGlsaXphdGlvblNjYWxpbmdQcm9wcyk6IFRhcmdldFRyYWNraW5nU2NhbGluZ1BvbGljeTtcblxuICAvKipcbiAgICogU2NhbGUgb3V0IG9yIGluIGluIG9yZGVyIHRvIGtlZXAgYSBtZXRyaWMgYXJvdW5kIGEgdGFyZ2V0IHZhbHVlXG4gICAqL1xuICBzY2FsZVRvVHJhY2tNZXRyaWMoaWQ6IHN0cmluZywgcHJvcHM6IE1ldHJpY1RhcmdldFRyYWNraW5nUHJvcHMpOiBUYXJnZXRUcmFja2luZ1NjYWxpbmdQb2xpY3k7XG5cbiAgLyoqXG4gICAqIFNjYWxlIG91dCBvciBpbiwgaW4gcmVzcG9uc2UgdG8gYSBtZXRyaWNcbiAgICovXG4gIHNjYWxlT25NZXRyaWMoaWQ6IHN0cmluZywgcHJvcHM6IEJhc2ljU3RlcFNjYWxpbmdQb2xpY3lQcm9wcyk6IFN0ZXBTY2FsaW5nUG9saWN5O1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGVuYWJsaW5nIHNjYWxpbmcgYmFzZWQgb24gQ1BVIHV0aWxpemF0aW9uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ3B1VXRpbGl6YXRpb25TY2FsaW5nUHJvcHMgZXh0ZW5kcyBCYXNlVGFyZ2V0VHJhY2tpbmdQcm9wcyB7XG4gIC8qKlxuICAgKiBUYXJnZXQgYXZlcmFnZSBDUFUgdXRpbGl6YXRpb24gYWNyb3NzIHRoZSB0YXNrXG4gICAqL1xuICByZWFkb25seSB0YXJnZXRVdGlsaXphdGlvblBlcmNlbnQ6IG51bWJlcjtcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBlbmFibGluZyBzY2FsaW5nIGJhc2VkIG9uIG5ldHdvcmsgdXRpbGl6YXRpb25cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBOZXR3b3JrVXRpbGl6YXRpb25TY2FsaW5nUHJvcHMgZXh0ZW5kcyBCYXNlVGFyZ2V0VHJhY2tpbmdQcm9wcyB7XG4gIC8qKlxuICAgKiBUYXJnZXQgYXZlcmFnZSBieXRlcy9zZWNvbmRzIG9uIGVhY2ggaW5zdGFuY2VcbiAgICovXG4gIHJlYWRvbmx5IHRhcmdldEJ5dGVzUGVyU2Vjb25kOiBudW1iZXI7XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgZW5hYmxpbmcgc2NhbGluZyBiYXNlZCBvbiByZXF1ZXN0L3NlY29uZFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFJlcXVlc3RDb3VudFNjYWxpbmdQcm9wcyBleHRlbmRzIEJhc2VUYXJnZXRUcmFja2luZ1Byb3BzIHtcbiAgLyoqXG4gICAqIFRhcmdldCBhdmVyYWdlIHJlcXVlc3RzL3NlY29uZHMgb24gZWFjaCBpbnN0YW5jZVxuICAgKlxuICAgKiBAZGVwcmVjYXRlZCBVc2UgJ3RhcmdldFJlcXVlc3RzUGVyTWludXRlJyBpbnN0ZWFkXG4gICAqIEBkZWZhdWx0IC0gU3BlY2lmeSBleGFjdGx5IG9uZSBvZiAndGFyZ2V0UmVxdWVzdHNQZXJNaW51dGUnIGFuZCAndGFyZ2V0UmVxdWVzdHNQZXJTZWNvbmQnXG4gICAqL1xuICByZWFkb25seSB0YXJnZXRSZXF1ZXN0c1BlclNlY29uZD86IG51bWJlcjtcblxuICAvKipcbiAgICogVGFyZ2V0IGF2ZXJhZ2UgcmVxdWVzdHMvbWludXRlIG9uIGVhY2ggaW5zdGFuY2VcbiAgICogQGRlZmF1bHQgLSBTcGVjaWZ5IGV4YWN0bHkgb25lIG9mICd0YXJnZXRSZXF1ZXN0c1Blck1pbnV0ZScgYW5kICd0YXJnZXRSZXF1ZXN0c1BlclNlY29uZCdcbiAgICovXG4gIHJlYWRvbmx5IHRhcmdldFJlcXVlc3RzUGVyTWludXRlPzogbnVtYmVyO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGVuYWJsaW5nIHRyYWNraW5nIG9mIGFuIGFyYml0cmFyeSBtZXRyaWNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBNZXRyaWNUYXJnZXRUcmFja2luZ1Byb3BzIGV4dGVuZHMgQmFzZVRhcmdldFRyYWNraW5nUHJvcHMge1xuICAvKipcbiAgICogTWV0cmljIHRvIHRyYWNrXG4gICAqXG4gICAqIFRoZSBtZXRyaWMgbXVzdCByZXByZXNlbnQgYSB1dGlsaXphdGlvbiwgc28gdGhhdCBpZiBpdCdzIGhpZ2hlciB0aGFuIHRoZVxuICAgKiB0YXJnZXQgdmFsdWUsIHlvdXIgQVNHIHNob3VsZCBzY2FsZSBvdXQsIGFuZCBpZiBpdCdzIGxvd2VyIGl0IHNob3VsZFxuICAgKiBzY2FsZSBpbi5cbiAgICovXG4gIHJlYWRvbmx5IG1ldHJpYzogY2xvdWR3YXRjaC5JTWV0cmljO1xuXG4gIC8qKlxuICAgKiBWYWx1ZSB0byBrZWVwIHRoZSBtZXRyaWMgYXJvdW5kXG4gICAqL1xuICByZWFkb25seSB0YXJnZXRWYWx1ZTogbnVtYmVyO1xufVxuXG4vKipcbiAqIFN5bnRoZXNpemUgYW4gYXJyYXkgb2YgYmxvY2sgZGV2aWNlIG1hcHBpbmdzIGZyb20gYSBsaXN0IG9mIGJsb2NrIGRldmljZVxuICpcbiAqIEBwYXJhbSBjb25zdHJ1Y3QgdGhlIGluc3RhbmNlL2FzZyBjb25zdHJ1Y3QsIHVzZWQgdG8gaG9zdCBhbnkgd2FybmluZ1xuICogQHBhcmFtIGJsb2NrRGV2aWNlcyBsaXN0IG9mIGJsb2NrIGRldmljZXNcbiAqL1xuZnVuY3Rpb24gc3ludGhlc2l6ZUJsb2NrRGV2aWNlTWFwcGluZ3MoY29uc3RydWN0OiBDb25zdHJ1Y3QsIGJsb2NrRGV2aWNlczogQmxvY2tEZXZpY2VbXSk6IENmbkxhdW5jaENvbmZpZ3VyYXRpb24uQmxvY2tEZXZpY2VNYXBwaW5nUHJvcGVydHlbXSB7XG4gIHJldHVybiBibG9ja0RldmljZXMubWFwPENmbkxhdW5jaENvbmZpZ3VyYXRpb24uQmxvY2tEZXZpY2VNYXBwaW5nUHJvcGVydHk+KCh7IGRldmljZU5hbWUsIHZvbHVtZSwgbWFwcGluZ0VuYWJsZWQgfSkgPT4ge1xuICAgIGNvbnN0IHsgdmlydHVhbE5hbWUsIGVic0RldmljZTogZWJzIH0gPSB2b2x1bWU7XG5cbiAgICBpZiAodm9sdW1lID09PSBCbG9ja0RldmljZVZvbHVtZS5fTk9fREVWSUNFIHx8IG1hcHBpbmdFbmFibGVkID09PSBmYWxzZSkge1xuICAgICAgcmV0dXJuIHtcbiAgICAgICAgZGV2aWNlTmFtZSxcbiAgICAgICAgbm9EZXZpY2U6IHRydWUsXG4gICAgICB9O1xuICAgIH1cblxuICAgIGlmIChlYnMpIHtcbiAgICAgIGNvbnN0IHsgaW9wcywgdm9sdW1lVHlwZSB9ID0gZWJzO1xuXG4gICAgICBpZiAoIWlvcHMpIHtcbiAgICAgICAgaWYgKHZvbHVtZVR5cGUgPT09IEVic0RldmljZVZvbHVtZVR5cGUuSU8xKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdpb3BzIHByb3BlcnR5IGlzIHJlcXVpcmVkIHdpdGggdm9sdW1lVHlwZTogRWJzRGV2aWNlVm9sdW1lVHlwZS5JTzEnKTtcbiAgICAgICAgfVxuICAgICAgfSBlbHNlIGlmICh2b2x1bWVUeXBlICE9PSBFYnNEZXZpY2VWb2x1bWVUeXBlLklPMSkge1xuICAgICAgICBBbm5vdGF0aW9ucy5vZihjb25zdHJ1Y3QpLmFkZFdhcm5pbmcoJ2lvcHMgd2lsbCBiZSBpZ25vcmVkIHdpdGhvdXQgdm9sdW1lVHlwZTogRWJzRGV2aWNlVm9sdW1lVHlwZS5JTzEnKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICByZXR1cm4ge1xuICAgICAgZGV2aWNlTmFtZSwgZWJzLCB2aXJ0dWFsTmFtZSxcbiAgICB9O1xuICB9KTtcbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBhcHBseWluZyBDbG91ZEZvcm1hdGlvbiBpbml0IHRvIGFuIGluc3RhbmNlIG9yIGluc3RhbmNlIGdyb3VwXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXBwbHlDbG91ZEZvcm1hdGlvbkluaXRPcHRpb25zIHtcbiAgLyoqXG4gICAqIENvbmZpZ1NldCB0byBhY3RpdmF0ZVxuICAgKlxuICAgKiBAZGVmYXVsdCBbJ2RlZmF1bHQnXVxuICAgKi9cbiAgcmVhZG9ubHkgY29uZmlnU2V0cz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBGb3JjZSBpbnN0YW5jZSByZXBsYWNlbWVudCBieSBlbWJlZGRpbmcgYSBjb25maWcgZmluZ2VycHJpbnRcbiAgICpcbiAgICogSWYgYHRydWVgICh0aGUgZGVmYXVsdCksIGEgaGFzaCBvZiB0aGUgY29uZmlnIHdpbGwgYmUgZW1iZWRkZWQgaW50byB0aGVcbiAgICogVXNlckRhdGEsIHNvIHRoYXQgaWYgdGhlIGNvbmZpZyBjaGFuZ2VzLCB0aGUgVXNlckRhdGEgY2hhbmdlcyBhbmRcbiAgICogaW5zdGFuY2VzIHdpbGwgYmUgcmVwbGFjZWQgKGdpdmVuIGFuIFVwZGF0ZVBvbGljeSBoYXMgYmVlbiBjb25maWd1cmVkIG9uXG4gICAqIHRoZSBBdXRvU2NhbGluZ0dyb3VwKS5cbiAgICpcbiAgICogSWYgYGZhbHNlYCwgbm8gc3VjaCBoYXNoIHdpbGwgYmUgZW1iZWRkZWQsIGFuZCBpZiB0aGUgQ2xvdWRGb3JtYXRpb24gSW5pdFxuICAgKiBjb25maWcgY2hhbmdlcyBub3RoaW5nIHdpbGwgaGFwcGVuIHRvIHRoZSBydW5uaW5nIGluc3RhbmNlcy4gSWYgYVxuICAgKiBjb25maWcgdXBkYXRlIGludHJvZHVjZXMgZXJyb3JzLCB5b3Ugd2lsbCBub3Qgbm90aWNlIHVudGlsIGFmdGVyIHRoZVxuICAgKiBDbG91ZEZvcm1hdGlvbiBkZXBsb3ltZW50IHN1Y2Nlc3NmdWxseSBmaW5pc2hlcyBhbmQgdGhlIG5leHQgaW5zdGFuY2VcbiAgICogZmFpbHMgdG8gbGF1bmNoLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBlbWJlZEZpbmdlcnByaW50PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogUHJpbnQgdGhlIHJlc3VsdHMgb2YgcnVubmluZyBjZm4taW5pdCB0byB0aGUgSW5zdGFuY2UgU3lzdGVtIExvZ1xuICAgKlxuICAgKiBCeSBkZWZhdWx0LCB0aGUgb3V0cHV0IG9mIHJ1bm5pbmcgY2ZuLWluaXQgaXMgd3JpdHRlbiB0byBhIGxvZyBmaWxlXG4gICAqIG9uIHRoZSBpbnN0YW5jZS4gU2V0IHRoaXMgdG8gYHRydWVgIHRvIHByaW50IGl0IHRvIHRoZSBTeXN0ZW0gTG9nXG4gICAqICh2aXNpYmxlIGZyb20gdGhlIEVDMiBDb25zb2xlKSwgYGZhbHNlYCB0byBub3QgcHJpbnQgaXQuXG4gICAqXG4gICAqIChCZSBhd2FyZSB0aGF0IHRoZSBzeXN0ZW0gbG9nIGlzIHJlZnJlc2hlZCBhdCBjZXJ0YWluIHBvaW50cyBpblxuICAgKiB0aW1lIG9mIHRoZSBpbnN0YW5jZSBsaWZlIGN5Y2xlLCBhbmQgc3VjY2Vzc2Z1bCBleGVjdXRpb24gbWF5XG4gICAqIG5vdCBhbHdheXMgc2hvdyB1cCkuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IHByaW50TG9nPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogRG9uJ3QgZmFpbCB0aGUgaW5zdGFuY2UgY3JlYXRpb24gd2hlbiBjZm4taW5pdCBmYWlsc1xuICAgKlxuICAgKiBZb3UgY2FuIHVzZSB0aGlzIHRvIHByZXZlbnQgQ2xvdWRGb3JtYXRpb24gZnJvbSByb2xsaW5nIGJhY2sgd2hlblxuICAgKiBpbnN0YW5jZXMgZmFpbCB0byBzdGFydCB1cCwgdG8gaGVscCBpbiBkZWJ1Z2dpbmcuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBpZ25vcmVGYWlsdXJlcz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEluY2x1ZGUgLS11cmwgYXJndW1lbnQgd2hlbiBydW5uaW5nIGNmbi1pbml0IGFuZCBjZm4tc2lnbmFsIGNvbW1hbmRzXG4gICAqXG4gICAqIFRoaXMgd2lsbCBiZSB0aGUgY2xvdWRmb3JtYXRpb24gZW5kcG9pbnQgaW4gdGhlIGRlcGxveWVkIHJlZ2lvblxuICAgKiBlLmcuIGh0dHBzOi8vY2xvdWRmb3JtYXRpb24udXMtZWFzdC0xLmFtYXpvbmF3cy5jb21cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGluY2x1ZGVVcmw/OiBib29sZWFuO1xuXG4gIC8qKlxuICAqIEluY2x1ZGUgLS1yb2xlIGFyZ3VtZW50IHdoZW4gcnVubmluZyBjZm4taW5pdCBhbmQgY2ZuLXNpZ25hbCBjb21tYW5kc1xuICAqXG4gICogVGhpcyB3aWxsIGJlIHRoZSBJQU0gaW5zdGFuY2UgcHJvZmlsZSBhdHRhY2hlZCB0byB0aGUgRUMyIGluc3RhbmNlXG4gICpcbiAgKiBAZGVmYXVsdCBmYWxzZVxuICAqL1xuICByZWFkb25seSBpbmNsdWRlUm9sZT86IGJvb2xlYW47XG59XG4iXX0=