"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackupPlanRule = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const events = require("../../aws-events");
const core_1 = require("../../core");
/**
 * A backup plan rule
 */
class BackupPlanRule {
    /** @param props Rule properties */
    constructor(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_backup_BackupPlanRuleProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, BackupPlanRule);
            }
            throw error;
        }
        if (props.deleteAfter && props.moveToColdStorageAfter &&
            props.deleteAfter.toDays() < props.moveToColdStorageAfter.toDays()) {
            throw new Error('`deleteAfter` must be greater than `moveToColdStorageAfter`');
        }
        if (props.scheduleExpression && !/^cron/.test(props.scheduleExpression.expressionString)) {
            throw new Error('`scheduleExpression` must be of type `cron`');
        }
        const deleteAfter = (props.enableContinuousBackup && !props.deleteAfter) ? core_1.Duration.days(35) : props.deleteAfter;
        if (props.enableContinuousBackup && props.moveToColdStorageAfter) {
            throw new Error('`moveToColdStorageAfter` must not be specified if `enableContinuousBackup` is enabled');
        }
        if (props.enableContinuousBackup && props.deleteAfter &&
            (props.deleteAfter?.toDays() < 1 || props.deleteAfter?.toDays() > 35)) {
            throw new Error(`'deleteAfter' must be between 1 and 35 days if 'enableContinuousBackup' is enabled, but got ${props.deleteAfter.toHumanString()}`);
        }
        this.props = {
            ...props,
            deleteAfter,
        };
    }
    /**
     * Daily with 35 days retention
     */
    static daily(backupVault) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_backup_IBackupVault(backupVault);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.daily);
            }
            throw error;
        }
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Daily',
            scheduleExpression: events.Schedule.cron({
                hour: '5',
                minute: '0',
            }),
            deleteAfter: core_1.Duration.days(35),
        });
    }
    /**
     * Weekly with 3 months retention
     */
    static weekly(backupVault) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_backup_IBackupVault(backupVault);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.weekly);
            }
            throw error;
        }
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Weekly',
            scheduleExpression: events.Schedule.cron({
                hour: '5',
                minute: '0',
                weekDay: 'SAT',
            }),
            deleteAfter: core_1.Duration.days(30 * 3),
        });
    }
    /**
     * Monthly 1 year retention, move to cold storage after 1 month
     */
    static monthly1Year(backupVault) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_backup_IBackupVault(backupVault);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.monthly1Year);
            }
            throw error;
        }
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Monthly1Year',
            scheduleExpression: events.Schedule.cron({
                day: '1',
                hour: '5',
                minute: '0',
            }),
            moveToColdStorageAfter: core_1.Duration.days(30),
            deleteAfter: core_1.Duration.days(365),
        });
    }
    /**
     * Monthly 5 year retention, move to cold storage after 3 months
     */
    static monthly5Year(backupVault) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_backup_IBackupVault(backupVault);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.monthly5Year);
            }
            throw error;
        }
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Monthly5Year',
            scheduleExpression: events.Schedule.cron({
                day: '1',
                hour: '5',
                minute: '0',
            }),
            moveToColdStorageAfter: core_1.Duration.days(30 * 3),
            deleteAfter: core_1.Duration.days(365 * 5),
        });
    }
    /**
     * Monthly 7 year retention, move to cold storage after 3 months
     */
    static monthly7Year(backupVault) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_backup_IBackupVault(backupVault);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.monthly7Year);
            }
            throw error;
        }
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Monthly7Year',
            scheduleExpression: events.Schedule.cron({
                day: '1',
                hour: '5',
                minute: '0',
            }),
            moveToColdStorageAfter: core_1.Duration.days(30 * 3),
            deleteAfter: core_1.Duration.days(365 * 7),
        });
    }
}
exports.BackupPlanRule = BackupPlanRule;
_a = JSII_RTTI_SYMBOL_1;
BackupPlanRule[_a] = { fqn: "monocdk.aws_backup.BackupPlanRule", version: "1.191.0" };
//# sourceMappingURL=data:application/json;base64,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