import * as iam from '../../aws-iam';
import * as kms from '../../aws-kms';
import * as sns from '../../aws-sns';
import { IResource, RemovalPolicy, Resource } from '../../core';
import { Construct } from 'constructs';
/**
 * A backup vault
 */
export interface IBackupVault extends IResource {
    /**
     * The name of a logical container where backups are stored.
     *
     * @attribute
     */
    readonly backupVaultName: string;
    /**
     * The ARN of the backup vault.
     *
     * @attribute
     */
    readonly backupVaultArn: string;
    /**
     * Grant the actions defined in actions to the given grantee
     * on this backup vault.
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
}
/**
 * Properties for a BackupVault
 */
export interface BackupVaultProps {
    /**
     * The name of a logical container where backups are stored. Backup vaults
     * are identified by names that are unique to the account used to create
     * them and the AWS Region where they are created.
     *
     * @default - A CDK generated name
     */
    readonly backupVaultName?: string;
    /**
     * A resource-based policy that is used to manage access permissions on the
     * backup vault.
     *
     * @default - access is not restricted
     */
    readonly accessPolicy?: iam.PolicyDocument;
    /**
     * The server-side encryption key to use to protect your backups.
     *
     * @default - an Amazon managed KMS key
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * A SNS topic to send vault events to.
     *
     * @see https://docs.aws.amazon.com/aws-backup/latest/devguide/sns-notifications.html
     *
     * @default - no notifications
     */
    readonly notificationTopic?: sns.ITopic;
    /**
     * The vault events to send.
     *
     * @see https://docs.aws.amazon.com/aws-backup/latest/devguide/sns-notifications.html
     *
     * @default - all vault events if `notificationTopic` is defined
     */
    readonly notificationEvents?: BackupVaultEvents[];
    /**
     * The removal policy to apply to the vault. Note that removing a vault
     * that contains recovery points will fail.
     *
     * @default RemovalPolicy.RETAIN
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * Whether to add statements to the vault access policy that prevents anyone
     * from deleting a recovery point.
     *
     * @default false
     */
    readonly blockRecoveryPointDeletion?: boolean;
}
/**
 * Backup vault events
 */
export declare enum BackupVaultEvents {
    /** BACKUP_JOB_STARTED */
    BACKUP_JOB_STARTED = "BACKUP_JOB_STARTED",
    /** BACKUP_JOB_COMPLETED */
    BACKUP_JOB_COMPLETED = "BACKUP_JOB_COMPLETED",
    /** BACKUP_JOB_SUCCESSFUL */
    BACKUP_JOB_SUCCESSFUL = "BACKUP_JOB_SUCCESSFUL",
    /** BACKUP_JOB_FAILED */
    BACKUP_JOB_FAILED = "BACKUP_JOB_FAILED",
    /** BACKUP_JOB_EXPIRED */
    BACKUP_JOB_EXPIRED = "BACKUP_JOB_EXPIRED",
    /** RESTORE_JOB_STARTED */
    RESTORE_JOB_STARTED = "RESTORE_JOB_STARTED",
    /** RESTORE_JOB_COMPLETED */
    RESTORE_JOB_COMPLETED = "RESTORE_JOB_COMPLETED",
    /** RESTORE_JOB_SUCCESSFUL */
    RESTORE_JOB_SUCCESSFUL = "RESTORE_JOB_SUCCESSFUL",
    /** RESTORE_JOB_FAILED */
    RESTORE_JOB_FAILED = "RESTORE_JOB_FAILED",
    /** COPY_JOB_STARTED */
    COPY_JOB_STARTED = "COPY_JOB_STARTED",
    /** COPY_JOB_SUCCESSFUL */
    COPY_JOB_SUCCESSFUL = "COPY_JOB_SUCCESSFUL",
    /** COPY_JOB_FAILED */
    COPY_JOB_FAILED = "COPY_JOB_FAILED",
    /** RECOVERY_POINT_MODIFIED */
    RECOVERY_POINT_MODIFIED = "RECOVERY_POINT_MODIFIED",
    /** BACKUP_PLAN_CREATED */
    BACKUP_PLAN_CREATED = "BACKUP_PLAN_CREATED",
    /** BACKUP_PLAN_MODIFIED */
    BACKUP_PLAN_MODIFIED = "BACKUP_PLAN_MODIFIED"
}
declare abstract class BackupVaultBase extends Resource implements IBackupVault {
    abstract readonly backupVaultName: string;
    abstract readonly backupVaultArn: string;
    /**
     * Grant the actions defined in actions to the given grantee
     * on this Backup Vault resource.
     *
     * @param grantee Principal to grant right to
     * @param actions The actions to grant
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
}
/**
 * A backup vault
 */
export declare class BackupVault extends BackupVaultBase {
    /**
     * Import an existing backup vault by name
     */
    static fromBackupVaultName(scope: Construct, id: string, backupVaultName: string): IBackupVault;
    /**
     * Import an existing backup vault by arn
     */
    static fromBackupVaultArn(scope: Construct, id: string, backupVaultArn: string): IBackupVault;
    readonly backupVaultName: string;
    readonly backupVaultArn: string;
    private readonly accessPolicy;
    constructor(scope: Construct, id: string, props?: BackupVaultProps);
    /**
     * Adds a statement to the vault access policy
     */
    addToAccessPolicy(statement: iam.PolicyStatement): void;
    /**
     * Adds a statement to the vault access policy that prevents anyone
     * from deleting a recovery point.
     */
    blockRecoveryPointDeletion(): void;
    private uniqueVaultName;
}
export {};
