"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ComputeEnvironment = exports.AllocationStrategy = exports.ComputeResourceType = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const batch_generated_1 = require("./batch.generated");
/**
 * Property to specify if the compute environment
 * uses On-Demand, SpotFleet, Fargate, or Fargate Spot compute resources.
 */
var ComputeResourceType;
(function (ComputeResourceType) {
    /**
     * Resources will be EC2 On-Demand resources.
     */
    ComputeResourceType["ON_DEMAND"] = "EC2";
    /**
     * Resources will be EC2 SpotFleet resources.
     */
    ComputeResourceType["SPOT"] = "SPOT";
    /**
     * Resources will be Fargate resources.
     */
    ComputeResourceType["FARGATE"] = "FARGATE";
    /**
     * Resources will be Fargate Spot resources.
     *
     * Fargate Spot uses spare capacity in the AWS cloud to run your fault-tolerant,
     * time-flexible jobs at up to a 70% discount. If AWS needs the resources back,
     * jobs running on Fargate Spot will be interrupted with two minutes of notification.
     */
    ComputeResourceType["FARGATE_SPOT"] = "FARGATE_SPOT";
})(ComputeResourceType = exports.ComputeResourceType || (exports.ComputeResourceType = {}));
/**
 * Properties for how to prepare compute resources
 * that are provisioned for a compute environment.
 */
var AllocationStrategy;
(function (AllocationStrategy) {
    /**
     * Batch will use the best fitting instance type will be used
     * when assigning a batch job in this compute environment.
     */
    AllocationStrategy["BEST_FIT"] = "BEST_FIT";
    /**
     * Batch will select additional instance types that are large enough to
     * meet the requirements of the jobs in the queue, with a preference for
     * instance types with a lower cost per unit vCPU.
     */
    AllocationStrategy["BEST_FIT_PROGRESSIVE"] = "BEST_FIT_PROGRESSIVE";
    /**
     * This is only available for Spot Instance compute resources and will select
     * additional instance types that are large enough to meet the requirements of
     * the jobs in the queue, with a preference for instance types that are less
     * likely to be interrupted.
     */
    AllocationStrategy["SPOT_CAPACITY_OPTIMIZED"] = "SPOT_CAPACITY_OPTIMIZED";
})(AllocationStrategy = exports.AllocationStrategy || (exports.AllocationStrategy = {}));
/**
 * Batch Compute Environment.
 *
 * Defines a batch compute environment to run batch jobs on.
 */
class ComputeEnvironment extends core_1.Resource {
    constructor(scope, id, props = { enabled: true, managed: true }) {
        super(scope, id, {
            physicalName: props.computeEnvironmentName,
        });
        try {
            jsiiDeprecationWarnings.monocdk_aws_batch_ComputeEnvironmentProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ComputeEnvironment);
            }
            throw error;
        }
        const isFargate = ComputeResourceType.FARGATE === props.computeResources?.type
            || ComputeResourceType.FARGATE_SPOT === props.computeResources?.type;
        ;
        this.validateProps(props, isFargate);
        const spotFleetRole = this.getSpotFleetRole(props);
        let computeResources;
        // Only allow compute resources to be set when using MANAGED type
        if (props.computeResources && this.isManaged(props)) {
            computeResources = {
                bidPercentage: props.computeResources.bidPercentage,
                desiredvCpus: props.computeResources.desiredvCpus,
                ec2KeyPair: props.computeResources.ec2KeyPair,
                imageId: props.computeResources.image && props.computeResources.image.getImage(this).imageId,
                launchTemplate: props.computeResources.launchTemplate,
                maxvCpus: props.computeResources.maxvCpus || 256,
                placementGroup: props.computeResources.placementGroup,
                securityGroupIds: this.buildSecurityGroupIds(props.computeResources.vpc, props.computeResources.securityGroups),
                spotIamFleetRole: spotFleetRole?.roleArn,
                subnets: props.computeResources.vpc.selectSubnets(props.computeResources.vpcSubnets).subnetIds,
                tags: props.computeResources.computeResourcesTags,
                type: props.computeResources.type || ComputeResourceType.ON_DEMAND,
                ...(!isFargate ? {
                    allocationStrategy: props.computeResources.allocationStrategy
                        || (props.computeResources.type === ComputeResourceType.SPOT
                            ? AllocationStrategy.SPOT_CAPACITY_OPTIMIZED
                            : AllocationStrategy.BEST_FIT),
                    instanceRole: props.computeResources.instanceRole
                        ? props.computeResources.instanceRole
                        : new iam.CfnInstanceProfile(this, 'Instance-Profile', {
                            roles: [new iam.Role(this, 'Ecs-Instance-Role', {
                                    assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
                                    managedPolicies: [
                                        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2ContainerServiceforEC2Role'),
                                    ],
                                }).roleName],
                        }).attrArn,
                    instanceTypes: this.buildInstanceTypes(props.computeResources.instanceTypes),
                    minvCpus: props.computeResources.minvCpus || 0,
                } : {}),
            };
        }
        const computeEnvironment = new batch_generated_1.CfnComputeEnvironment(this, 'Resource', {
            computeEnvironmentName: this.physicalName,
            computeResources,
            serviceRole: props.serviceRole?.roleArn
                ?? new iam.Role(this, 'Resource-Service-Instance-Role', {
                    managedPolicies: [
                        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSBatchServiceRole'),
                    ],
                    assumedBy: new iam.ServicePrincipal('batch.amazonaws.com'),
                }).roleArn,
            state: props.enabled === undefined ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            type: this.isManaged(props) ? 'MANAGED' : 'UNMANAGED',
        });
        if (props.computeResources && props.computeResources.vpc) {
            this.node.addDependency(props.computeResources.vpc);
        }
        this.computeEnvironmentArn = this.getResourceArnAttribute(computeEnvironment.ref, {
            service: 'batch',
            resource: 'compute-environment',
            resourceName: this.physicalName,
        });
        this.computeEnvironmentName = this.getResourceNameAttribute(computeEnvironment.ref);
    }
    /**
     * Fetches an existing batch compute environment by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param computeEnvironmentArn
     */
    static fromComputeEnvironmentArn(scope, id, computeEnvironmentArn) {
        const stack = core_1.Stack.of(scope);
        const computeEnvironmentName = stack.splitArn(computeEnvironmentArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.computeEnvironmentArn = computeEnvironmentArn;
                this.computeEnvironmentName = computeEnvironmentName;
            }
        }
        return new Import(scope, id);
    }
    isManaged(props) {
        return props.managed ?? true;
    }
    /**
     * Validates the properties provided for a new batch compute environment.
     */
    validateProps(props, isFargate) {
        if (props === undefined) {
            return;
        }
        if (!this.isManaged(props) && props.computeResources !== undefined) {
            throw new Error('It is not allowed to set computeResources on an AWS unmanaged compute environment');
        }
        if (this.isManaged(props) && props.computeResources === undefined) {
            throw new Error('computeResources is missing but required on a managed compute environment');
        }
        if (props.computeResources) {
            if (isFargate) {
                // VALIDATE FOR FARGATE
                // Bid percentage cannot be set for Fargate evnvironments
                if (props.computeResources.bidPercentage !== undefined) {
                    throw new Error('Bid percentage must not be set for Fargate compute environments');
                }
                // Allocation strategy cannot be set for Fargate evnvironments
                if (props.computeResources.allocationStrategy !== undefined) {
                    throw new Error('Allocation strategy must not be set for Fargate compute environments');
                }
                // Desired vCPUs cannot be set for Fargate evnvironments
                if (props.computeResources.desiredvCpus !== undefined) {
                    throw new Error('Desired vCPUs must not be set for Fargate compute environments');
                }
                // Image ID cannot be set for Fargate evnvironments
                if (props.computeResources.image !== undefined) {
                    throw new Error('Image must not be set for Fargate compute environments');
                }
                // Instance types cannot be set for Fargate evnvironments
                if (props.computeResources.instanceTypes !== undefined) {
                    throw new Error('Instance types must not be set for Fargate compute environments');
                }
                // EC2 key pair cannot be set for Fargate evnvironments
                if (props.computeResources.ec2KeyPair !== undefined) {
                    throw new Error('EC2 key pair must not be set for Fargate compute environments');
                }
                // Instance role cannot be set for Fargate evnvironments
                if (props.computeResources.instanceRole !== undefined) {
                    throw new Error('Instance role must not be set for Fargate compute environments');
                }
                // Launch template cannot be set for Fargate evnvironments
                if (props.computeResources.launchTemplate !== undefined) {
                    throw new Error('Launch template must not be set for Fargate compute environments');
                }
                // Min vCPUs cannot be set for Fargate evnvironments
                if (props.computeResources.minvCpus !== undefined) {
                    throw new Error('Min vCPUs must not be set for Fargate compute environments');
                }
                // Placement group cannot be set for Fargate evnvironments
                if (props.computeResources.placementGroup !== undefined) {
                    throw new Error('Placement group must not be set for Fargate compute environments');
                }
                // Spot fleet role cannot be set for Fargate evnvironments
                if (props.computeResources.spotFleetRole !== undefined) {
                    throw new Error('Spot fleet role must not be set for Fargate compute environments');
                }
            }
            else {
                // VALIDATE FOR ON_DEMAND AND SPOT
                if (props.computeResources.minvCpus) {
                    // minvCpus cannot be less than 0
                    if (props.computeResources.minvCpus < 0) {
                        throw new Error('Minimum vCpus for a batch compute environment cannot be less than 0');
                    }
                    // minvCpus cannot exceed max vCpus
                    if (props.computeResources.maxvCpus &&
                        props.computeResources.minvCpus > props.computeResources.maxvCpus) {
                        throw new Error('Minimum vCpus cannot be greater than the maximum vCpus');
                    }
                }
                // Setting a bid percentage is only allowed on SPOT resources +
                // Cannot use SPOT_CAPACITY_OPTIMIZED when using ON_DEMAND
                if (props.computeResources.type === ComputeResourceType.ON_DEMAND) {
                    // VALIDATE FOR ON_DEMAND
                    // Bid percentage is not allowed
                    if (props.computeResources.bidPercentage !== undefined) {
                        throw new Error('Setting the bid percentage is only allowed for SPOT type resources on a batch compute environment');
                    }
                    // SPOT_CAPACITY_OPTIMIZED allocation is not allowed
                    if (props.computeResources.allocationStrategy && props.computeResources.allocationStrategy === AllocationStrategy.SPOT_CAPACITY_OPTIMIZED) {
                        throw new Error('The SPOT_CAPACITY_OPTIMIZED allocation strategy is only allowed if the environment is a SPOT type compute environment');
                    }
                }
                else if (props.computeResources.type === ComputeResourceType.SPOT) {
                    // VALIDATE FOR SPOT
                    // Bid percentage must be from 0 - 100
                    if (props.computeResources.bidPercentage !== undefined &&
                        (props.computeResources.bidPercentage < 0 || props.computeResources.bidPercentage > 100)) {
                        throw new Error('Bid percentage can only be a value between 0 and 100');
                    }
                }
            }
        }
    }
    buildInstanceTypes(instanceTypes) {
        if (instanceTypes === undefined) {
            return [
                'optimal',
            ];
        }
        return instanceTypes.map((type) => type.toString());
    }
    buildSecurityGroupIds(vpc, securityGroups) {
        if (securityGroups === undefined) {
            return [
                new ec2.SecurityGroup(this, 'Resource-Security-Group', { vpc }).securityGroupId,
            ];
        }
        return securityGroups.map((group) => group.securityGroupId);
    }
    /**
     * Generates an AWS IAM role for provisioning spotfleet resources
     * if the allocation strategy is set to BEST_FIT or not defined.
     *
     * @param props - the compute environment construct properties
     */
    getSpotFleetRole(props) {
        if (props.computeResources?.allocationStrategy && props.computeResources.allocationStrategy !== AllocationStrategy.BEST_FIT) {
            return undefined;
        }
        if (props.computeResources) {
            if (props.computeResources.spotFleetRole) {
                return props.computeResources.spotFleetRole;
            }
            else if (props.computeResources.type === ComputeResourceType.SPOT) {
                return iam.Role.fromRoleArn(this, 'Resource-SpotFleet-Role', `arn:${this.stack.partition}:iam::${this.env.account}:role/aws-service-role/spotfleet.amazonaws.com/AWSServiceRoleForEC2SpotFleet`);
            }
        }
        return undefined;
    }
}
exports.ComputeEnvironment = ComputeEnvironment;
_a = JSII_RTTI_SYMBOL_1;
ComputeEnvironment[_a] = { fqn: "monocdk.aws_batch.ComputeEnvironment", version: "1.191.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29tcHV0ZS1lbnZpcm9ubWVudC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImNvbXB1dGUtZW52aXJvbm1lbnQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBQUEscUNBQXFDO0FBQ3JDLHFDQUFxQztBQUNyQyxxQ0FBbUU7QUFFbkUsdURBQTBEO0FBRTFEOzs7R0FHRztBQUNILElBQVksbUJBd0JYO0FBeEJELFdBQVksbUJBQW1CO0lBQzdCOztPQUVHO0lBQ0gsd0NBQWlCLENBQUE7SUFFakI7O09BRUc7SUFDSCxvQ0FBYSxDQUFBO0lBRWI7O09BRUc7SUFDSCwwQ0FBbUIsQ0FBQTtJQUVuQjs7Ozs7O09BTUc7SUFDSCxvREFBNkIsQ0FBQTtBQUMvQixDQUFDLEVBeEJXLG1CQUFtQixHQUFuQiwyQkFBbUIsS0FBbkIsMkJBQW1CLFFBd0I5QjtBQUVEOzs7R0FHRztBQUNILElBQVksa0JBcUJYO0FBckJELFdBQVksa0JBQWtCO0lBQzVCOzs7T0FHRztJQUNILDJDQUFxQixDQUFBO0lBRXJCOzs7O09BSUc7SUFDSCxtRUFBNkMsQ0FBQTtJQUU3Qzs7Ozs7T0FLRztJQUNILHlFQUFtRCxDQUFBO0FBQ3JELENBQUMsRUFyQlcsa0JBQWtCLEdBQWxCLDBCQUFrQixLQUFsQiwwQkFBa0IsUUFxQjdCO0FBMlBEOzs7O0dBSUc7QUFDSCxNQUFhLGtCQUFtQixTQUFRLGVBQVE7SUFrQzlDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBaUMsRUFBRSxPQUFPLEVBQUUsSUFBSSxFQUFFLE9BQU8sRUFBRSxJQUFJLEVBQUU7UUFDekcsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixZQUFZLEVBQUUsS0FBSyxDQUFDLHNCQUFzQjtTQUMzQyxDQUFDLENBQUM7Ozs7OzsrQ0FyQ00sa0JBQWtCOzs7O1FBdUMzQixNQUFNLFNBQVMsR0FBRyxtQkFBbUIsQ0FBQyxPQUFPLEtBQUssS0FBSyxDQUFDLGdCQUFnQixFQUFFLElBQUk7ZUFDekUsbUJBQW1CLENBQUMsWUFBWSxLQUFLLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRSxJQUFJLENBQUM7UUFBQSxDQUFDO1FBRXhFLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBRXJDLE1BQU0sYUFBYSxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNuRCxJQUFJLGdCQUE0RSxDQUFDO1FBRWpGLGlFQUFpRTtRQUNqRSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsSUFBSSxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxFQUFFO1lBQ25ELGdCQUFnQixHQUFHO2dCQUNqQixhQUFhLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGFBQWE7Z0JBQ25ELFlBQVksRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsWUFBWTtnQkFDakQsVUFBVSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVO2dCQUM3QyxPQUFPLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLEtBQUssSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPO2dCQUM1RixjQUFjLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGNBQWM7Z0JBQ3JELFFBQVEsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxJQUFJLEdBQUc7Z0JBQ2hELGNBQWMsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsY0FBYztnQkFDckQsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLHFCQUFxQixDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGNBQWMsQ0FBQztnQkFDL0csZ0JBQWdCLEVBQUUsYUFBYSxFQUFFLE9BQU87Z0JBQ3hDLE9BQU8sRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsVUFBVSxDQUFDLENBQUMsU0FBUztnQkFDOUYsSUFBSSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxvQkFBb0I7Z0JBQ2pELElBQUksRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxJQUFJLG1CQUFtQixDQUFDLFNBQVM7Z0JBQ2xFLEdBQUcsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUM7b0JBQ2Ysa0JBQWtCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGtCQUFrQjsyQkFDeEQsQ0FDRCxLQUFLLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxLQUFLLG1CQUFtQixDQUFDLElBQUk7NEJBQ3RELENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyx1QkFBdUI7NEJBQzVDLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxRQUFRLENBQ2hDO29CQUNILFlBQVksRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsWUFBWTt3QkFDL0MsQ0FBQyxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxZQUFZO3dCQUNyQyxDQUFDLENBQUMsSUFBSSxHQUFHLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFOzRCQUNyRCxLQUFLLEVBQUUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLG1CQUFtQixFQUFFO29DQUM5QyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsbUJBQW1CLENBQUM7b0NBQ3hELGVBQWUsRUFBRTt3Q0FDZixHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLGtEQUFrRCxDQUFDO3FDQUMvRjtpQ0FDRixDQUFDLENBQUMsUUFBUSxDQUFDO3lCQUNiLENBQUMsQ0FBQyxPQUFPO29CQUNaLGFBQWEsRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsQ0FBQztvQkFDNUUsUUFBUSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRLElBQUksQ0FBQztpQkFDL0MsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO2FBQ1IsQ0FBQztTQUNIO1FBRUQsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLHVDQUFxQixDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDckUsc0JBQXNCLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDekMsZ0JBQWdCO1lBQ2hCLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVyxFQUFFLE9BQU87bUJBQ2xDLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsZ0NBQWdDLEVBQUU7b0JBQ3RELGVBQWUsRUFBRTt3QkFDZixHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLGtDQUFrQyxDQUFDO3FCQUMvRTtvQkFDRCxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMscUJBQXFCLENBQUM7aUJBQzNELENBQUMsQ0FBQyxPQUFPO1lBQ1osS0FBSyxFQUFFLEtBQUssQ0FBQyxPQUFPLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUM7WUFDekYsSUFBSSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsV0FBVztTQUN0RCxDQUFDLENBQUM7UUFFSCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxFQUFFO1lBQ3hELElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUNyRDtRQUVELElBQUksQ0FBQyxxQkFBcUIsR0FBRyxJQUFJLENBQUMsdUJBQXVCLENBQUMsa0JBQWtCLENBQUMsR0FBRyxFQUFFO1lBQ2hGLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLFFBQVEsRUFBRSxxQkFBcUI7WUFDL0IsWUFBWSxFQUFFLElBQUksQ0FBQyxZQUFZO1NBQ2hDLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxzQkFBc0IsR0FBRyxJQUFJLENBQUMsd0JBQXdCLENBQUMsa0JBQWtCLENBQUMsR0FBRyxDQUFDLENBQUM7S0FDckY7SUE1R0Q7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLHlCQUF5QixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLHFCQUE2QjtRQUNqRyxNQUFNLEtBQUssR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzlCLE1BQU0sc0JBQXNCLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQyxxQkFBcUIsRUFBRSxnQkFBUyxDQUFDLG1CQUFtQixDQUFDLENBQUMsWUFBYSxDQUFDO1FBRWxILE1BQU0sTUFBTyxTQUFRLGVBQVE7WUFBN0I7O2dCQUNrQiwwQkFBcUIsR0FBRyxxQkFBcUIsQ0FBQztnQkFDOUMsMkJBQXNCLEdBQUcsc0JBQXNCLENBQUM7WUFDbEUsQ0FBQztTQUFBO1FBRUQsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7S0FDOUI7SUE2Rk8sU0FBUyxDQUFDLEtBQThCO1FBQzlDLE9BQU8sS0FBSyxDQUFDLE9BQU8sSUFBSSxJQUFJLENBQUM7S0FDOUI7SUFFRDs7T0FFRztJQUNLLGFBQWEsQ0FBQyxLQUE4QixFQUFFLFNBQWtCO1FBQ3RFLElBQUksS0FBSyxLQUFLLFNBQVMsRUFBRTtZQUN2QixPQUFPO1NBQ1I7UUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEtBQUssU0FBUyxFQUFFO1lBQ2xFLE1BQU0sSUFBSSxLQUFLLENBQUMsbUZBQW1GLENBQUMsQ0FBQztTQUN0RztRQUVELElBQUksSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEtBQUssU0FBUyxFQUFFO1lBQ2pFLE1BQU0sSUFBSSxLQUFLLENBQUMsMkVBQTJFLENBQUMsQ0FBQztTQUM5RjtRQUVELElBQUksS0FBSyxDQUFDLGdCQUFnQixFQUFFO1lBQzFCLElBQUksU0FBUyxFQUFFO2dCQUNiLHVCQUF1QjtnQkFFdkIseURBQXlEO2dCQUN6RCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhLEtBQUssU0FBUyxFQUFFO29CQUN0RCxNQUFNLElBQUksS0FBSyxDQUFDLGlFQUFpRSxDQUFDLENBQUM7aUJBQ3BGO2dCQUVELDhEQUE4RDtnQkFDOUQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsa0JBQWtCLEtBQUssU0FBUyxFQUFFO29CQUMzRCxNQUFNLElBQUksS0FBSyxDQUFDLHNFQUFzRSxDQUFDLENBQUM7aUJBQ3pGO2dCQUVELHdEQUF3RDtnQkFDeEQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsWUFBWSxLQUFLLFNBQVMsRUFBRTtvQkFDckQsTUFBTSxJQUFJLEtBQUssQ0FBQyxnRUFBZ0UsQ0FBQyxDQUFDO2lCQUNuRjtnQkFFRCxtREFBbUQ7Z0JBQ25ELElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLEtBQUssS0FBSyxTQUFTLEVBQUU7b0JBQzlDLE1BQU0sSUFBSSxLQUFLLENBQUMsd0RBQXdELENBQUMsQ0FBQztpQkFDM0U7Z0JBRUQseURBQXlEO2dCQUN6RCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhLEtBQUssU0FBUyxFQUFFO29CQUN0RCxNQUFNLElBQUksS0FBSyxDQUFDLGlFQUFpRSxDQUFDLENBQUM7aUJBQ3BGO2dCQUVELHVEQUF1RDtnQkFDdkQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsVUFBVSxLQUFLLFNBQVMsRUFBRTtvQkFDbkQsTUFBTSxJQUFJLEtBQUssQ0FBQywrREFBK0QsQ0FBQyxDQUFDO2lCQUNsRjtnQkFFRCx3REFBd0Q7Z0JBQ3hELElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLFlBQVksS0FBSyxTQUFTLEVBQUU7b0JBQ3JELE1BQU0sSUFBSSxLQUFLLENBQUMsZ0VBQWdFLENBQUMsQ0FBQztpQkFDbkY7Z0JBRUQsMERBQTBEO2dCQUMxRCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxjQUFjLEtBQUssU0FBUyxFQUFFO29CQUN2RCxNQUFNLElBQUksS0FBSyxDQUFDLGtFQUFrRSxDQUFDLENBQUM7aUJBQ3JGO2dCQUVELG9EQUFvRDtnQkFDcEQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxLQUFLLFNBQVMsRUFBRTtvQkFDakQsTUFBTSxJQUFJLEtBQUssQ0FBQyw0REFBNEQsQ0FBQyxDQUFDO2lCQUMvRTtnQkFFRCwwREFBMEQ7Z0JBQzFELElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLGNBQWMsS0FBSyxTQUFTLEVBQUU7b0JBQ3ZELE1BQU0sSUFBSSxLQUFLLENBQUMsa0VBQWtFLENBQUMsQ0FBQztpQkFDckY7Z0JBRUQsMERBQTBEO2dCQUMxRCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhLEtBQUssU0FBUyxFQUFFO29CQUN0RCxNQUFNLElBQUksS0FBSyxDQUFDLGtFQUFrRSxDQUFDLENBQUM7aUJBQ3JGO2FBQ0Y7aUJBQU07Z0JBQ0wsa0NBQWtDO2dCQUNsQyxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRLEVBQUU7b0JBQ25DLGlDQUFpQztvQkFDakMsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxHQUFHLENBQUMsRUFBRTt3QkFDdkMsTUFBTSxJQUFJLEtBQUssQ0FBQyxxRUFBcUUsQ0FBQyxDQUFDO3FCQUN4RjtvQkFFRCxtQ0FBbUM7b0JBQ25DLElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLFFBQVE7d0JBQ2pDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRLEdBQUcsS0FBSyxDQUFDLGdCQUFnQixDQUFDLFFBQVEsRUFBRTt3QkFDbkUsTUFBTSxJQUFJLEtBQUssQ0FBQyx3REFBd0QsQ0FBQyxDQUFDO3FCQUMzRTtpQkFDRjtnQkFDRCwrREFBK0Q7Z0JBQy9ELDBEQUEwRDtnQkFDMUQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxLQUFLLG1CQUFtQixDQUFDLFNBQVMsRUFBRTtvQkFDakUseUJBQXlCO29CQUV6QixnQ0FBZ0M7b0JBQ2hDLElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsS0FBSyxTQUFTLEVBQUU7d0JBQ3RELE1BQU0sSUFBSSxLQUFLLENBQUMsbUdBQW1HLENBQUMsQ0FBQztxQkFDdEg7b0JBRUQsb0RBQW9EO29CQUNwRCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxrQkFBa0IsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsa0JBQWtCLEtBQUssa0JBQWtCLENBQUMsdUJBQXVCLEVBQUU7d0JBQ3pJLE1BQU0sSUFBSSxLQUFLLENBQUMsdUhBQXVILENBQUMsQ0FBQztxQkFDMUk7aUJBQ0Y7cUJBQU0sSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxLQUFLLG1CQUFtQixDQUFDLElBQUksRUFBRTtvQkFDbkUsb0JBQW9CO29CQUVwQixzQ0FBc0M7b0JBQ3RDLElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsS0FBSyxTQUFTO3dCQUNwRCxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhLEdBQUcsQ0FBQyxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhLEdBQUcsR0FBRyxDQUFDLEVBQUU7d0JBQzFGLE1BQU0sSUFBSSxLQUFLLENBQUMsc0RBQXNELENBQUMsQ0FBQztxQkFDekU7aUJBQ0Y7YUFDRjtTQUNGO0tBQ0Y7SUFFTyxrQkFBa0IsQ0FBQyxhQUFrQztRQUMzRCxJQUFJLGFBQWEsS0FBSyxTQUFTLEVBQUU7WUFDL0IsT0FBTztnQkFDTCxTQUFTO2FBQ1YsQ0FBQztTQUNIO1FBRUQsT0FBTyxhQUFhLENBQUMsR0FBRyxDQUFDLENBQUMsSUFBc0IsRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUM7S0FDdkU7SUFFTyxxQkFBcUIsQ0FBQyxHQUFhLEVBQUUsY0FBcUM7UUFDaEYsSUFBSSxjQUFjLEtBQUssU0FBUyxFQUFFO1lBQ2hDLE9BQU87Z0JBQ0wsSUFBSSxHQUFHLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSx5QkFBeUIsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUMsZUFBZTthQUNoRixDQUFDO1NBQ0g7UUFFRCxPQUFPLGNBQWMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxLQUF5QixFQUFFLEVBQUUsQ0FBQyxLQUFLLENBQUMsZUFBZSxDQUFDLENBQUM7S0FDakY7SUFFRDs7Ozs7T0FLRztJQUNLLGdCQUFnQixDQUFDLEtBQThCO1FBQ3JELElBQUksS0FBSyxDQUFDLGdCQUFnQixFQUFFLGtCQUFrQixJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxrQkFBa0IsS0FBSyxrQkFBa0IsQ0FBQyxRQUFRLEVBQUU7WUFDM0gsT0FBTyxTQUFTLENBQUM7U0FDbEI7UUFFRCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUMxQixJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhLEVBQUU7Z0JBQ3hDLE9BQU8sS0FBSyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsQ0FBQzthQUM3QztpQkFBTSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLEtBQUssbUJBQW1CLENBQUMsSUFBSSxFQUFFO2dCQUNuRSxPQUFPLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSx5QkFBeUIsRUFDekQsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsU0FBUyxJQUFJLENBQUMsR0FBRyxDQUFDLE9BQU8sOEVBQThFLENBQUMsQ0FBQzthQUN2STtTQUNGO1FBRUQsT0FBTyxTQUFTLENBQUM7S0FDbEI7O0FBL1FILGdEQWdSQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGVjMiBmcm9tICcuLi8uLi9hd3MtZWMyJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICcuLi8uLi9hd3MtaWFtJztcbmltcG9ydCB7IEFybkZvcm1hdCwgSVJlc291cmNlLCBSZXNvdXJjZSwgU3RhY2sgfSBmcm9tICcuLi8uLi9jb3JlJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgQ2ZuQ29tcHV0ZUVudmlyb25tZW50IH0gZnJvbSAnLi9iYXRjaC5nZW5lcmF0ZWQnO1xuXG4vKipcbiAqIFByb3BlcnR5IHRvIHNwZWNpZnkgaWYgdGhlIGNvbXB1dGUgZW52aXJvbm1lbnRcbiAqIHVzZXMgT24tRGVtYW5kLCBTcG90RmxlZXQsIEZhcmdhdGUsIG9yIEZhcmdhdGUgU3BvdCBjb21wdXRlIHJlc291cmNlcy5cbiAqL1xuZXhwb3J0IGVudW0gQ29tcHV0ZVJlc291cmNlVHlwZSB7XG4gIC8qKlxuICAgKiBSZXNvdXJjZXMgd2lsbCBiZSBFQzIgT24tRGVtYW5kIHJlc291cmNlcy5cbiAgICovXG4gIE9OX0RFTUFORCA9ICdFQzInLFxuXG4gIC8qKlxuICAgKiBSZXNvdXJjZXMgd2lsbCBiZSBFQzIgU3BvdEZsZWV0IHJlc291cmNlcy5cbiAgICovXG4gIFNQT1QgPSAnU1BPVCcsXG5cbiAgLyoqXG4gICAqIFJlc291cmNlcyB3aWxsIGJlIEZhcmdhdGUgcmVzb3VyY2VzLlxuICAgKi9cbiAgRkFSR0FURSA9ICdGQVJHQVRFJyxcblxuICAvKipcbiAgICogUmVzb3VyY2VzIHdpbGwgYmUgRmFyZ2F0ZSBTcG90IHJlc291cmNlcy5cbiAgICpcbiAgICogRmFyZ2F0ZSBTcG90IHVzZXMgc3BhcmUgY2FwYWNpdHkgaW4gdGhlIEFXUyBjbG91ZCB0byBydW4geW91ciBmYXVsdC10b2xlcmFudCxcbiAgICogdGltZS1mbGV4aWJsZSBqb2JzIGF0IHVwIHRvIGEgNzAlIGRpc2NvdW50LiBJZiBBV1MgbmVlZHMgdGhlIHJlc291cmNlcyBiYWNrLFxuICAgKiBqb2JzIHJ1bm5pbmcgb24gRmFyZ2F0ZSBTcG90IHdpbGwgYmUgaW50ZXJydXB0ZWQgd2l0aCB0d28gbWludXRlcyBvZiBub3RpZmljYXRpb24uXG4gICAqL1xuICBGQVJHQVRFX1NQT1QgPSAnRkFSR0FURV9TUE9UJyxcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBob3cgdG8gcHJlcGFyZSBjb21wdXRlIHJlc291cmNlc1xuICogdGhhdCBhcmUgcHJvdmlzaW9uZWQgZm9yIGEgY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAqL1xuZXhwb3J0IGVudW0gQWxsb2NhdGlvblN0cmF0ZWd5IHtcbiAgLyoqXG4gICAqIEJhdGNoIHdpbGwgdXNlIHRoZSBiZXN0IGZpdHRpbmcgaW5zdGFuY2UgdHlwZSB3aWxsIGJlIHVzZWRcbiAgICogd2hlbiBhc3NpZ25pbmcgYSBiYXRjaCBqb2IgaW4gdGhpcyBjb21wdXRlIGVudmlyb25tZW50LlxuICAgKi9cbiAgQkVTVF9GSVQgPSAnQkVTVF9GSVQnLFxuXG4gIC8qKlxuICAgKiBCYXRjaCB3aWxsIHNlbGVjdCBhZGRpdGlvbmFsIGluc3RhbmNlIHR5cGVzIHRoYXQgYXJlIGxhcmdlIGVub3VnaCB0b1xuICAgKiBtZWV0IHRoZSByZXF1aXJlbWVudHMgb2YgdGhlIGpvYnMgaW4gdGhlIHF1ZXVlLCB3aXRoIGEgcHJlZmVyZW5jZSBmb3JcbiAgICogaW5zdGFuY2UgdHlwZXMgd2l0aCBhIGxvd2VyIGNvc3QgcGVyIHVuaXQgdkNQVS5cbiAgICovXG4gIEJFU1RfRklUX1BST0dSRVNTSVZFID0gJ0JFU1RfRklUX1BST0dSRVNTSVZFJyxcblxuICAvKipcbiAgICogVGhpcyBpcyBvbmx5IGF2YWlsYWJsZSBmb3IgU3BvdCBJbnN0YW5jZSBjb21wdXRlIHJlc291cmNlcyBhbmQgd2lsbCBzZWxlY3RcbiAgICogYWRkaXRpb25hbCBpbnN0YW5jZSB0eXBlcyB0aGF0IGFyZSBsYXJnZSBlbm91Z2ggdG8gbWVldCB0aGUgcmVxdWlyZW1lbnRzIG9mXG4gICAqIHRoZSBqb2JzIGluIHRoZSBxdWV1ZSwgd2l0aCBhIHByZWZlcmVuY2UgZm9yIGluc3RhbmNlIHR5cGVzIHRoYXQgYXJlIGxlc3NcbiAgICogbGlrZWx5IHRvIGJlIGludGVycnVwdGVkLlxuICAgKi9cbiAgU1BPVF9DQVBBQ0lUWV9PUFRJTUlaRUQgPSAnU1BPVF9DQVBBQ0lUWV9PUFRJTUlaRUQnLFxufVxuXG4vKipcbiAqIExhdW5jaCB0ZW1wbGF0ZSBwcm9wZXJ0eSBzcGVjaWZpY2F0aW9uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTGF1bmNoVGVtcGxhdGVTcGVjaWZpY2F0aW9uIHtcbiAgLyoqXG4gICAqIFRoZSBMYXVuY2ggdGVtcGxhdGUgbmFtZVxuICAgKi9cbiAgcmVhZG9ubHkgbGF1bmNoVGVtcGxhdGVOYW1lOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgbGF1bmNoIHRlbXBsYXRlIHZlcnNpb24gdG8gYmUgdXNlZCAob3B0aW9uYWwpLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHRoZSBkZWZhdWx0IHZlcnNpb24gb2YgdGhlIGxhdW5jaCB0ZW1wbGF0ZVxuICAgKi9cbiAgcmVhZG9ubHkgdmVyc2lvbj86IHN0cmluZztcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBkZWZpbmluZyB0aGUgc3RydWN0dXJlIG9mIHRoZSBiYXRjaCBjb21wdXRlIGNsdXN0ZXIuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ29tcHV0ZVJlc291cmNlcyB7XG4gIC8qKlxuICAgKiBUaGUgYWxsb2NhdGlvbiBzdHJhdGVneSB0byB1c2UgZm9yIHRoZSBjb21wdXRlIHJlc291cmNlIGluIGNhc2Ugbm90IGVub3VnaCBpbnN0YW5jZXMgb2YgdGhlIGJlc3RcbiAgICogZml0dGluZyBpbnN0YW5jZSB0eXBlIGNhbiBiZSBhbGxvY2F0ZWQuIFRoaXMgY291bGQgYmUgZHVlIHRvIGF2YWlsYWJpbGl0eSBvZiB0aGUgaW5zdGFuY2UgdHlwZSBpblxuICAgKiB0aGUgcmVnaW9uIG9yIEFtYXpvbiBFQzIgc2VydmljZSBsaW1pdHMuIElmIHRoaXMgaXMgbm90IHNwZWNpZmllZCwgdGhlIGRlZmF1bHQgZm9yIHRoZSBFQzJcbiAgICogQ29tcHV0ZVJlc291cmNlVHlwZSBpcyBCRVNUX0ZJVCwgd2hpY2ggd2lsbCB1c2Ugb25seSB0aGUgYmVzdCBmaXR0aW5nIGluc3RhbmNlIHR5cGUsIHdhaXRpbmcgZm9yXG4gICAqIGFkZGl0aW9uYWwgY2FwYWNpdHkgaWYgaXQncyBub3QgYXZhaWxhYmxlLiBUaGlzIGFsbG9jYXRpb24gc3RyYXRlZ3kga2VlcHMgY29zdHMgbG93ZXIgYnV0IGNhbiBsaW1pdFxuICAgKiBzY2FsaW5nLiBJZiB5b3UgYXJlIHVzaW5nIFNwb3QgRmxlZXRzIHdpdGggQkVTVF9GSVQgdGhlbiB0aGUgU3BvdCBGbGVldCBJQU0gUm9sZSBtdXN0IGJlIHNwZWNpZmllZC5cbiAgICogQkVTVF9GSVRfUFJPR1JFU1NJVkUgd2lsbCBzZWxlY3QgYW4gYWRkaXRpb25hbCBpbnN0YW5jZSB0eXBlIHRoYXQgaXMgbGFyZ2UgZW5vdWdoIHRvIG1lZXQgdGhlXG4gICAqIHJlcXVpcmVtZW50cyBvZiB0aGUgam9icyBpbiB0aGUgcXVldWUsIHdpdGggYSBwcmVmZXJlbmNlIGZvciBhbiBpbnN0YW5jZSB0eXBlIHdpdGggYSBsb3dlciBjb3N0LlxuICAgKiBUaGUgZGVmYXVsdCB2YWx1ZSBmb3IgdGhlIFNQT1QgaW5zdGFuY2UgdHlwZSBpcyBTUE9UX0NBUEFDSVRZX09QVElNSVpFRCwgd2hpY2ggaXMgb25seSBhdmFpbGFibGUgZm9yXG4gICAqIGZvciB0aGlzIHR5cGUgb2YgY29tcHV0ZSByZXNvdXJjZXMgYW5kIHdpbGwgc2VsZWN0IGFuIGFkZGl0aW9uYWwgaW5zdGFuY2UgdHlwZSB0aGF0IGlzIGxhcmdlIGVub3VnaFxuICAgKiB0byBtZWV0IHRoZSByZXF1aXJlbWVudHMgb2YgdGhlIGpvYnMgaW4gdGhlIHF1ZXVlLCB3aXRoIGEgcHJlZmVyZW5jZSBmb3IgYW4gaW5zdGFuY2UgdHlwZSB0aGF0IGlzXG4gICAqIGxlc3MgbGlrZWx5IHRvIGJlIGludGVycnVwdGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBBbGxvY2F0aW9uU3RyYXRlZ3kuQkVTVF9GSVRcbiAgICovXG4gIHJlYWRvbmx5IGFsbG9jYXRpb25TdHJhdGVneT86IEFsbG9jYXRpb25TdHJhdGVneTtcblxuICAvKipcbiAgICogVGhlIEFtYXpvbiBFQ1MgaW5zdGFuY2UgcHJvZmlsZSBhcHBsaWVkIHRvIEFtYXpvbiBFQzIgaW5zdGFuY2VzIGluIGEgY29tcHV0ZSBlbnZpcm9ubWVudC4gWW91IGNhbiBzcGVjaWZ5XG4gICAqIHRoZSBzaG9ydCBuYW1lIG9yIGZ1bGwgQW1hem9uIFJlc291cmNlIE5hbWUgKEFSTikgb2YgYW4gaW5zdGFuY2UgcHJvZmlsZS4gRm9yIGV4YW1wbGUsIGVjc0luc3RhbmNlUm9sZSBvclxuICAgKiBhcm46YXdzOmlhbTo6PGF3c19hY2NvdW50X2lkPjppbnN0YW5jZS1wcm9maWxlL2Vjc0luc3RhbmNlUm9sZSAuIEZvciBtb3JlIGluZm9ybWF0aW9uLCBzZWUgQW1hem9uIEVDU1xuICAgKiBJbnN0YW5jZSBSb2xlIGluIHRoZSBBV1MgQmF0Y2ggVXNlciBHdWlkZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBhIG5ldyByb2xlIHdpbGwgYmUgY3JlYXRlZC5cbiAgICogQGxpbmsgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC91c2VyZ3VpZGUvaW5zdGFuY2VfSUFNX3JvbGUuaHRtbFxuICAgKi9cbiAgcmVhZG9ubHkgaW5zdGFuY2VSb2xlPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBbiBvcHRpb25hbCBsYXVuY2ggdGVtcGxhdGUgdG8gYXNzb2NpYXRlIHdpdGggeW91ciBjb21wdXRlIHJlc291cmNlcy5cbiAgICogRm9yIG1vcmUgaW5mb3JtYXRpb24sIHNlZSBSRUFETUUgZmlsZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBjdXN0b20gbGF1bmNoIHRlbXBsYXRlIHdpbGwgYmUgdXNlZFxuICAgKiBAbGluayBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L3VzZXJndWlkZS9sYXVuY2gtdGVtcGxhdGVzLmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IGxhdW5jaFRlbXBsYXRlPzogTGF1bmNoVGVtcGxhdGVTcGVjaWZpY2F0aW9uO1xuXG4gIC8qKlxuICAgKiBUaGUgdHlwZXMgb2YgRUMyIGluc3RhbmNlcyB0aGF0IG1heSBiZSBsYXVuY2hlZCBpbiB0aGUgY29tcHV0ZSBlbnZpcm9ubWVudC4gWW91IGNhbiBzcGVjaWZ5IGluc3RhbmNlXG4gICAqIGZhbWlsaWVzIHRvIGxhdW5jaCBhbnkgaW5zdGFuY2UgdHlwZSB3aXRoaW4gdGhvc2UgZmFtaWxpZXMgKGZvciBleGFtcGxlLCBjNCBvciBwMyksIG9yIHlvdSBjYW4gc3BlY2lmeVxuICAgKiBzcGVjaWZpYyBzaXplcyB3aXRoaW4gYSBmYW1pbHkgKHN1Y2ggYXMgYzQuOHhsYXJnZSkuIFlvdSBjYW4gYWxzbyBjaG9vc2Ugb3B0aW1hbCB0byBwaWNrIGluc3RhbmNlIHR5cGVzXG4gICAqIChmcm9tIHRoZSBDLCBNLCBhbmQgUiBpbnN0YW5jZSBmYW1pbGllcykgb24gdGhlIGZseSB0aGF0IG1hdGNoIHRoZSBkZW1hbmQgb2YgeW91ciBqb2IgcXVldWVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBvcHRpbWFsXG4gICAqL1xuICByZWFkb25seSBpbnN0YW5jZVR5cGVzPzogZWMyLkluc3RhbmNlVHlwZVtdO1xuXG4gIC8qKlxuICAgKiBUaGUgRUMyIHNlY3VyaXR5IGdyb3VwKHMpIGFzc29jaWF0ZWQgd2l0aCBpbnN0YW5jZXMgbGF1bmNoZWQgaW4gdGhlIGNvbXB1dGUgZW52aXJvbm1lbnQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQVdTIGRlZmF1bHQgc2VjdXJpdHkgZ3JvdXAuXG4gICAqL1xuICByZWFkb25seSBzZWN1cml0eUdyb3Vwcz86IGVjMi5JU2VjdXJpdHlHcm91cFtdO1xuXG4gIC8qKlxuICAgKiBUaGUgVlBDIG5ldHdvcmsgdGhhdCBhbGwgY29tcHV0ZSByZXNvdXJjZXMgd2lsbCBiZSBjb25uZWN0ZWQgdG8uXG4gICAqL1xuICByZWFkb25seSB2cGM6IGVjMi5JVnBjO1xuXG4gIC8qKlxuICAgKiBUaGUgVlBDIHN1Ym5ldHMgaW50byB3aGljaCB0aGUgY29tcHV0ZSByZXNvdXJjZXMgYXJlIGxhdW5jaGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHByaXZhdGUgc3VibmV0cyBvZiB0aGUgc3VwcGxpZWQgVlBDLlxuICAgKi9cbiAgcmVhZG9ubHkgdnBjU3VibmV0cz86IGVjMi5TdWJuZXRTZWxlY3Rpb247XG5cbiAgLyoqXG4gICAqIFRoZSB0eXBlIG9mIGNvbXB1dGUgZW52aXJvbm1lbnQ6IE9OX0RFTUFORCwgU1BPVCwgRkFSR0FURSwgb3IgRkFSR0FURV9TUE9ULlxuICAgKlxuICAgKiBAZGVmYXVsdCBPTl9ERU1BTkRcbiAgICovXG4gIHJlYWRvbmx5IHR5cGU/OiBDb21wdXRlUmVzb3VyY2VUeXBlO1xuXG4gIC8qKlxuICAgKiBUaGlzIHByb3BlcnR5IHdpbGwgYmUgaWdub3JlZCBpZiB5b3Ugc2V0IHRoZSBlbnZpcm9ubWVudCB0eXBlIHRvIE9OX0RFTUFORC5cbiAgICpcbiAgICogVGhlIG1heGltdW0gcGVyY2VudGFnZSB0aGF0IGEgU3BvdCBJbnN0YW5jZSBwcmljZSBjYW4gYmUgd2hlbiBjb21wYXJlZCB3aXRoIHRoZSBPbi1EZW1hbmQgcHJpY2UgZm9yXG4gICAqIHRoYXQgaW5zdGFuY2UgdHlwZSBiZWZvcmUgaW5zdGFuY2VzIGFyZSBsYXVuY2hlZC4gRm9yIGV4YW1wbGUsIGlmIHlvdXIgbWF4aW11bSBwZXJjZW50YWdlIGlzIDIwJSxcbiAgICogdGhlbiB0aGUgU3BvdCBwcmljZSBtdXN0IGJlIGJlbG93IDIwJSBvZiB0aGUgY3VycmVudCBPbi1EZW1hbmQgcHJpY2UgZm9yIHRoYXQgRUMyIGluc3RhbmNlLiBZb3UgYWx3YXlzXG4gICAqIHBheSB0aGUgbG93ZXN0IChtYXJrZXQpIHByaWNlIGFuZCBuZXZlciBtb3JlIHRoYW4geW91ciBtYXhpbXVtIHBlcmNlbnRhZ2UuIElmIHlvdSBsZWF2ZSB0aGlzIGZpZWxkIGVtcHR5LFxuICAgKiB0aGUgZGVmYXVsdCB2YWx1ZSBpcyAxMDAlIG9mIHRoZSBPbi1EZW1hbmQgcHJpY2UuXG4gICAqXG4gICAqIEBkZWZhdWx0IDEwMFxuICAgKi9cbiAgcmVhZG9ubHkgYmlkUGVyY2VudGFnZT86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIGRlc2lyZWQgbnVtYmVyIG9mIEVDMiB2Q1BVUyBpbiB0aGUgY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBkZXNpcmVkIHZjcHUgdmFsdWUgd2lsbCBiZSB1c2VkLlxuICAgKi9cbiAgcmVhZG9ubHkgZGVzaXJlZHZDcHVzPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgbWF4aW11bSBudW1iZXIgb2YgRUMyIHZDUFVzIHRoYXQgYW4gZW52aXJvbm1lbnQgY2FuIHJlYWNoLiBFYWNoIHZDUFUgaXMgZXF1aXZhbGVudCB0b1xuICAgKiAxLDAyNCBDUFUgc2hhcmVzLiBZb3UgbXVzdCBzcGVjaWZ5IGF0IGxlYXN0IG9uZSB2Q1BVLlxuICAgKlxuICAgKiBAZGVmYXVsdCAyNTZcbiAgICovXG4gIHJlYWRvbmx5IG1heHZDcHVzPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgbWluaW11bSBudW1iZXIgb2YgRUMyIHZDUFVzIHRoYXQgYW4gZW52aXJvbm1lbnQgc2hvdWxkIG1haW50YWluIChldmVuIGlmIHRoZSBjb21wdXRlIGVudmlyb25tZW50IHN0YXRlIGlzIERJU0FCTEVEKS5cbiAgICogRWFjaCB2Q1BVIGlzIGVxdWl2YWxlbnQgdG8gMSwwMjQgQ1BVIHNoYXJlcy4gQnkga2VlcGluZyB0aGlzIHNldCB0byAwIHlvdSB3aWxsIG5vdCBoYXZlIGluc3RhbmNlIHRpbWUgd2FzdGVkIHdoZW5cbiAgICogdGhlcmUgaXMgbm8gd29yayB0byBiZSBydW4uIElmIHlvdSBzZXQgdGhpcyBhYm92ZSB6ZXJvIHlvdSB3aWxsIG1haW50YWluIHRoYXQgbnVtYmVyIG9mIHZDUFVzIGF0IGFsbCB0aW1lcy5cbiAgICpcbiAgICogQGRlZmF1bHQgMFxuICAgKi9cbiAgcmVhZG9ubHkgbWludkNwdXM/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBBbWF6b24gRUMyIHBsYWNlbWVudCBncm91cCB0byBhc3NvY2lhdGUgd2l0aCB5b3VyIGNvbXB1dGUgcmVzb3VyY2VzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIHBsYWNlbWVudCBncm91cCB3aWxsIGJlIHVzZWQuXG4gICAqL1xuICByZWFkb25seSBwbGFjZW1lbnRHcm91cD86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIEVDMiBrZXkgcGFpciB0aGF0IGlzIHVzZWQgZm9yIGluc3RhbmNlcyBsYXVuY2hlZCBpbiB0aGUgY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICogSWYgbm8ga2V5IGlzIGRlZmluZWQsIHRoZW4gU1NIIGFjY2VzcyBpcyBub3QgYWxsb3dlZCB0byBwcm92aXNpb25lZCBjb21wdXRlIHJlc291cmNlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBTU0ggYWNjZXNzIHdpbGwgYmUgcG9zc2libGUuXG4gICAqL1xuICByZWFkb25seSBlYzJLZXlQYWlyPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgQW1hem9uIE1hY2hpbmUgSW1hZ2UgKEFNSSkgSUQgdXNlZCBmb3IgaW5zdGFuY2VzIGxhdW5jaGVkIGluIHRoZSBjb21wdXRlIGVudmlyb25tZW50LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGltYWdlIHdpbGwgYmUgdXNlZC5cbiAgICovXG4gIHJlYWRvbmx5IGltYWdlPzogZWMyLklNYWNoaW5lSW1hZ2U7XG5cbiAgLyoqXG4gICAqIFRoaXMgcHJvcGVydHkgd2lsbCBiZSBpZ25vcmVkIGlmIHlvdSBzZXQgdGhlIGVudmlyb25tZW50IHR5cGUgdG8gT05fREVNQU5ELlxuICAgKlxuICAgKiBUaGUgQW1hem9uIFJlc291cmNlIE5hbWUgKEFSTikgb2YgdGhlIEFtYXpvbiBFQzIgU3BvdCBGbGVldCBJQU0gcm9sZSBhcHBsaWVkIHRvIGEgU1BPVCBjb21wdXRlIGVudmlyb25tZW50LlxuICAgKiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgc2VlIEFtYXpvbiBFQzIgU3BvdCBGbGVldCBSb2xlIGluIHRoZSBBV1MgQmF0Y2ggVXNlciBHdWlkZS5cbiAgICpcbiAgICogQGxpbmsgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC91c2VyZ3VpZGUvc3BvdF9mbGVldF9JQU1fcm9sZS5odG1sXG4gICAqIEBkZWZhdWx0IC0gbm8gZmxlZXQgcm9sZSB3aWxsIGJlIHVzZWQuXG4gICAqL1xuICByZWFkb25seSBzcG90RmxlZXRSb2xlPzogaWFtLklSb2xlO1xuXG4gIC8qKlxuICAgKiBLZXktdmFsdWUgcGFpciB0YWdzIHRvIGJlIGFwcGxpZWQgdG8gcmVzb3VyY2VzIHRoYXQgYXJlIGxhdW5jaGVkIGluIHRoZSBjb21wdXRlIGVudmlyb25tZW50LlxuICAgKiBGb3IgQVdTIEJhdGNoLCB0aGVzZSB0YWtlIHRoZSBmb3JtIG9mIFwiU3RyaW5nMVwiOiBcIlN0cmluZzJcIiwgd2hlcmUgU3RyaW5nMSBpcyB0aGUgdGFnIGtleSBhbmRcbiAgICogU3RyaW5nMiBpcyB0aGUgdGFnIHZhbHVl4oCUZm9yIGV4YW1wbGUsIHsgXCJOYW1lXCI6IFwiQVdTIEJhdGNoIEluc3RhbmNlIC0gQzRPbkRlbWFuZFwiIH0uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gdGFncyB3aWxsIGJlIGFzc2lnbmVkIG9uIGNvbXB1dGUgcmVzb3VyY2VzLlxuICAgKi9cbiAgcmVhZG9ubHkgY29tcHV0ZVJlc291cmNlc1RhZ3M/OiB7XG4gICAgW2tleTogc3RyaW5nXTogc3RyaW5nXG4gIH07XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgY3JlYXRpbmcgYSBuZXcgQ29tcHV0ZSBFbnZpcm9ubWVudFxuICovXG5leHBvcnQgaW50ZXJmYWNlIENvbXB1dGVFbnZpcm9ubWVudFByb3BzIHtcbiAgLyoqXG4gICAqIEEgbmFtZSBmb3IgdGhlIGNvbXB1dGUgZW52aXJvbm1lbnQuXG4gICAqXG4gICAqIFVwIHRvIDEyOCBsZXR0ZXJzICh1cHBlcmNhc2UgYW5kIGxvd2VyY2FzZSksIG51bWJlcnMsIGh5cGhlbnMsIGFuZCB1bmRlcnNjb3JlcyBhcmUgYWxsb3dlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBDbG91ZEZvcm1hdGlvbi1nZW5lcmF0ZWQgbmFtZVxuICAgKi9cbiAgcmVhZG9ubHkgY29tcHV0ZUVudmlyb25tZW50TmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGRldGFpbHMgb2YgdGhlIHJlcXVpcmVkIGNvbXB1dGUgcmVzb3VyY2VzIGZvciB0aGUgbWFuYWdlZCBjb21wdXRlIGVudmlyb25tZW50LlxuICAgKlxuICAgKiBJZiBzcGVjaWZpZWQsIGFuZCB0aGlzIGlzIGFuIHVubWFuYWdlZCBjb21wdXRlIGVudmlyb25tZW50LCB3aWxsIHRocm93IGFuIGVycm9yLlxuICAgKlxuICAgKiBCeSBkZWZhdWx0LCBBV1MgQmF0Y2ggbWFuYWdlZCBjb21wdXRlIGVudmlyb25tZW50cyB1c2UgYSByZWNlbnQsIGFwcHJvdmVkIHZlcnNpb24gb2YgdGhlXG4gICAqIEFtYXpvbiBFQ1Mtb3B0aW1pemVkIEFNSSBmb3IgY29tcHV0ZSByZXNvdXJjZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQ2xvdWRGb3JtYXRpb24gZGVmYXVsdHNcbiAgICogQGxpbmsgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXByb3BlcnRpZXMtYmF0Y2gtY29tcHV0ZWVudmlyb25tZW50LWNvbXB1dGVyZXNvdXJjZXMuaHRtbFxuICAgKi9cbiAgcmVhZG9ubHkgY29tcHV0ZVJlc291cmNlcz86IENvbXB1dGVSZXNvdXJjZXM7XG5cbiAgLyoqXG4gICAqIFRoZSBzdGF0ZSBvZiB0aGUgY29tcHV0ZSBlbnZpcm9ubWVudC4gSWYgdGhlIHN0YXRlIGlzIHNldCB0byB0cnVlLCB0aGVuIHRoZSBjb21wdXRlXG4gICAqIGVudmlyb25tZW50IGFjY2VwdHMgam9icyBmcm9tIGEgcXVldWUgYW5kIGNhbiBzY2FsZSBvdXQgYXV0b21hdGljYWxseSBiYXNlZCBvbiBxdWV1ZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGVuYWJsZWQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgSUFNIHJvbGUgdXNlZCBieSBCYXRjaCB0byBtYWtlIGNhbGxzIHRvIG90aGVyIEFXUyBzZXJ2aWNlcyBvbiB5b3VyIGJlaGFsZiBmb3IgbWFuYWdpbmdcbiAgICogdGhlIHJlc291cmNlcyB0aGF0IHlvdSB1c2Ugd2l0aCB0aGUgc2VydmljZS4gQnkgZGVmYXVsdCwgdGhpcyByb2xlIGlzIGNyZWF0ZWQgZm9yIHlvdSB1c2luZ1xuICAgKiB0aGUgQVdTIG1hbmFnZWQgc2VydmljZSBwb2xpY3kgZm9yIEJhdGNoLlxuICAgKlxuICAgKiBAbGluayBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L3VzZXJndWlkZS9zZXJ2aWNlX0lBTV9yb2xlLmh0bWxcbiAgICpcbiAgICogQGRlZmF1bHQgLSBSb2xlIHVzaW5nIHRoZSAnc2VydmljZS1yb2xlL0FXU0JhdGNoU2VydmljZVJvbGUnIHBvbGljeS5cbiAgICovXG4gIHJlYWRvbmx5IHNlcnZpY2VSb2xlPzogaWFtLklSb2xlLFxuXG4gIC8qKlxuICAgKiBEZXRlcm1pbmVzIGlmIEFXUyBzaG91bGQgbWFuYWdlIHRoZSBhbGxvY2F0aW9uIG9mIGNvbXB1dGUgcmVzb3VyY2VzIGZvciBwcm9jZXNzaW5nIGpvYnMuXG4gICAqIElmIHNldCB0byBmYWxzZSwgdGhlbiB5b3UgYXJlIGluIGNoYXJnZSBvZiBwcm92aWRpbmcgdGhlIGNvbXB1dGUgcmVzb3VyY2UgZGV0YWlscy5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgbWFuYWdlZD86IGJvb2xlYW47XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBvZiBhIGNvbXB1dGUgZW52aXJvbm1lbnQuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSUNvbXB1dGVFbnZpcm9ubWVudCBleHRlbmRzIElSZXNvdXJjZSB7XG4gIC8qKlxuICAgKiBUaGUgQVJOIG9mIHRoaXMgY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgY29tcHV0ZUVudmlyb25tZW50QXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoaXMgY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgY29tcHV0ZUVudmlyb25tZW50TmFtZTogc3RyaW5nO1xufVxuXG4vKipcbiAqIEJhdGNoIENvbXB1dGUgRW52aXJvbm1lbnQuXG4gKlxuICogRGVmaW5lcyBhIGJhdGNoIGNvbXB1dGUgZW52aXJvbm1lbnQgdG8gcnVuIGJhdGNoIGpvYnMgb24uXG4gKi9cbmV4cG9ydCBjbGFzcyBDb21wdXRlRW52aXJvbm1lbnQgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElDb21wdXRlRW52aXJvbm1lbnQge1xuICAvKipcbiAgICogRmV0Y2hlcyBhbiBleGlzdGluZyBiYXRjaCBjb21wdXRlIGVudmlyb25tZW50IGJ5IGl0cyBhbWF6b24gcmVzb3VyY2UgbmFtZS5cbiAgICpcbiAgICogQHBhcmFtIHNjb3BlXG4gICAqIEBwYXJhbSBpZFxuICAgKiBAcGFyYW0gY29tcHV0ZUVudmlyb25tZW50QXJuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21Db21wdXRlRW52aXJvbm1lbnRBcm4oc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgY29tcHV0ZUVudmlyb25tZW50QXJuOiBzdHJpbmcpOiBJQ29tcHV0ZUVudmlyb25tZW50IHtcbiAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHNjb3BlKTtcbiAgICBjb25zdCBjb21wdXRlRW52aXJvbm1lbnROYW1lID0gc3RhY2suc3BsaXRBcm4oY29tcHV0ZUVudmlyb25tZW50QXJuLCBBcm5Gb3JtYXQuU0xBU0hfUkVTT1VSQ0VfTkFNRSkucmVzb3VyY2VOYW1lITtcblxuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSUNvbXB1dGVFbnZpcm9ubWVudCB7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgY29tcHV0ZUVudmlyb25tZW50QXJuID0gY29tcHV0ZUVudmlyb25tZW50QXJuO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGNvbXB1dGVFbnZpcm9ubWVudE5hbWUgPSBjb21wdXRlRW52aXJvbm1lbnROYW1lO1xuICAgIH1cblxuICAgIHJldHVybiBuZXcgSW1wb3J0KHNjb3BlLCBpZCk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIEFSTiBvZiB0aGlzIGNvbXB1dGUgZW52aXJvbm1lbnQuXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBjb21wdXRlRW52aXJvbm1lbnRBcm46IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhpcyBjb21wdXRlIGVudmlyb25tZW50LlxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgY29tcHV0ZUVudmlyb25tZW50TmFtZTogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDb21wdXRlRW52aXJvbm1lbnRQcm9wcyA9IHsgZW5hYmxlZDogdHJ1ZSwgbWFuYWdlZDogdHJ1ZSB9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICBwaHlzaWNhbE5hbWU6IHByb3BzLmNvbXB1dGVFbnZpcm9ubWVudE5hbWUsXG4gICAgfSk7XG5cbiAgICBjb25zdCBpc0ZhcmdhdGUgPSBDb21wdXRlUmVzb3VyY2VUeXBlLkZBUkdBVEUgPT09IHByb3BzLmNvbXB1dGVSZXNvdXJjZXM/LnR5cGVcbiAgICAgIHx8IENvbXB1dGVSZXNvdXJjZVR5cGUuRkFSR0FURV9TUE9UID09PSBwcm9wcy5jb21wdXRlUmVzb3VyY2VzPy50eXBlOztcblxuICAgIHRoaXMudmFsaWRhdGVQcm9wcyhwcm9wcywgaXNGYXJnYXRlKTtcblxuICAgIGNvbnN0IHNwb3RGbGVldFJvbGUgPSB0aGlzLmdldFNwb3RGbGVldFJvbGUocHJvcHMpO1xuICAgIGxldCBjb21wdXRlUmVzb3VyY2VzOiBDZm5Db21wdXRlRW52aXJvbm1lbnQuQ29tcHV0ZVJlc291cmNlc1Byb3BlcnR5IHwgdW5kZWZpbmVkO1xuXG4gICAgLy8gT25seSBhbGxvdyBjb21wdXRlIHJlc291cmNlcyB0byBiZSBzZXQgd2hlbiB1c2luZyBNQU5BR0VEIHR5cGVcbiAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcyAmJiB0aGlzLmlzTWFuYWdlZChwcm9wcykpIHtcbiAgICAgIGNvbXB1dGVSZXNvdXJjZXMgPSB7XG4gICAgICAgIGJpZFBlcmNlbnRhZ2U6IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuYmlkUGVyY2VudGFnZSxcbiAgICAgICAgZGVzaXJlZHZDcHVzOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmRlc2lyZWR2Q3B1cyxcbiAgICAgICAgZWMyS2V5UGFpcjogcHJvcHMuY29tcHV0ZVJlc291cmNlcy5lYzJLZXlQYWlyLFxuICAgICAgICBpbWFnZUlkOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmltYWdlICYmIHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuaW1hZ2UuZ2V0SW1hZ2UodGhpcykuaW1hZ2VJZCxcbiAgICAgICAgbGF1bmNoVGVtcGxhdGU6IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMubGF1bmNoVGVtcGxhdGUsXG4gICAgICAgIG1heHZDcHVzOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLm1heHZDcHVzIHx8IDI1NixcbiAgICAgICAgcGxhY2VtZW50R3JvdXA6IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMucGxhY2VtZW50R3JvdXAsXG4gICAgICAgIHNlY3VyaXR5R3JvdXBJZHM6IHRoaXMuYnVpbGRTZWN1cml0eUdyb3VwSWRzKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMudnBjLCBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLnNlY3VyaXR5R3JvdXBzKSxcbiAgICAgICAgc3BvdElhbUZsZWV0Um9sZTogc3BvdEZsZWV0Um9sZT8ucm9sZUFybixcbiAgICAgICAgc3VibmV0czogcHJvcHMuY29tcHV0ZVJlc291cmNlcy52cGMuc2VsZWN0U3VibmV0cyhwcm9wcy5jb21wdXRlUmVzb3VyY2VzLnZwY1N1Ym5ldHMpLnN1Ym5ldElkcyxcbiAgICAgICAgdGFnczogcHJvcHMuY29tcHV0ZVJlc291cmNlcy5jb21wdXRlUmVzb3VyY2VzVGFncyxcbiAgICAgICAgdHlwZTogcHJvcHMuY29tcHV0ZVJlc291cmNlcy50eXBlIHx8IENvbXB1dGVSZXNvdXJjZVR5cGUuT05fREVNQU5ELFxuICAgICAgICAuLi4oIWlzRmFyZ2F0ZSA/IHtcbiAgICAgICAgICBhbGxvY2F0aW9uU3RyYXRlZ3k6IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuYWxsb2NhdGlvblN0cmF0ZWd5XG4gICAgICAgICAgICB8fCAoXG4gICAgICAgICAgICAgIHByb3BzLmNvbXB1dGVSZXNvdXJjZXMudHlwZSA9PT0gQ29tcHV0ZVJlc291cmNlVHlwZS5TUE9UXG4gICAgICAgICAgICAgICAgPyBBbGxvY2F0aW9uU3RyYXRlZ3kuU1BPVF9DQVBBQ0lUWV9PUFRJTUlaRURcbiAgICAgICAgICAgICAgICA6IEFsbG9jYXRpb25TdHJhdGVneS5CRVNUX0ZJVFxuICAgICAgICAgICAgKSxcbiAgICAgICAgICBpbnN0YW5jZVJvbGU6IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuaW5zdGFuY2VSb2xlXG4gICAgICAgICAgICA/IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuaW5zdGFuY2VSb2xlXG4gICAgICAgICAgICA6IG5ldyBpYW0uQ2ZuSW5zdGFuY2VQcm9maWxlKHRoaXMsICdJbnN0YW5jZS1Qcm9maWxlJywge1xuICAgICAgICAgICAgICByb2xlczogW25ldyBpYW0uUm9sZSh0aGlzLCAnRWNzLUluc3RhbmNlLVJvbGUnLCB7XG4gICAgICAgICAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2VjMi5hbWF6b25hd3MuY29tJyksXG4gICAgICAgICAgICAgICAgbWFuYWdlZFBvbGljaWVzOiBbXG4gICAgICAgICAgICAgICAgICBpYW0uTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ3NlcnZpY2Utcm9sZS9BbWF6b25FQzJDb250YWluZXJTZXJ2aWNlZm9yRUMyUm9sZScpLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0pLnJvbGVOYW1lXSxcbiAgICAgICAgICAgIH0pLmF0dHJBcm4sXG4gICAgICAgICAgaW5zdGFuY2VUeXBlczogdGhpcy5idWlsZEluc3RhbmNlVHlwZXMocHJvcHMuY29tcHV0ZVJlc291cmNlcy5pbnN0YW5jZVR5cGVzKSxcbiAgICAgICAgICBtaW52Q3B1czogcHJvcHMuY29tcHV0ZVJlc291cmNlcy5taW52Q3B1cyB8fCAwLFxuICAgICAgICB9IDoge30pLFxuICAgICAgfTtcbiAgICB9XG5cbiAgICBjb25zdCBjb21wdXRlRW52aXJvbm1lbnQgPSBuZXcgQ2ZuQ29tcHV0ZUVudmlyb25tZW50KHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIGNvbXB1dGVFbnZpcm9ubWVudE5hbWU6IHRoaXMucGh5c2ljYWxOYW1lLFxuICAgICAgY29tcHV0ZVJlc291cmNlcyxcbiAgICAgIHNlcnZpY2VSb2xlOiBwcm9wcy5zZXJ2aWNlUm9sZT8ucm9sZUFyblxuICAgICAgICA/PyBuZXcgaWFtLlJvbGUodGhpcywgJ1Jlc291cmNlLVNlcnZpY2UtSW5zdGFuY2UtUm9sZScsIHtcbiAgICAgICAgICBtYW5hZ2VkUG9saWNpZXM6IFtcbiAgICAgICAgICAgIGlhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnc2VydmljZS1yb2xlL0FXU0JhdGNoU2VydmljZVJvbGUnKSxcbiAgICAgICAgICBdLFxuICAgICAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdiYXRjaC5hbWF6b25hd3MuY29tJyksXG4gICAgICAgIH0pLnJvbGVBcm4sXG4gICAgICBzdGF0ZTogcHJvcHMuZW5hYmxlZCA9PT0gdW5kZWZpbmVkID8gJ0VOQUJMRUQnIDogKHByb3BzLmVuYWJsZWQgPyAnRU5BQkxFRCcgOiAnRElTQUJMRUQnKSxcbiAgICAgIHR5cGU6IHRoaXMuaXNNYW5hZ2VkKHByb3BzKSA/ICdNQU5BR0VEJyA6ICdVTk1BTkFHRUQnLFxuICAgIH0pO1xuXG4gICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMgJiYgcHJvcHMuY29tcHV0ZVJlc291cmNlcy52cGMpIHtcbiAgICAgIHRoaXMubm9kZS5hZGREZXBlbmRlbmN5KHByb3BzLmNvbXB1dGVSZXNvdXJjZXMudnBjKTtcbiAgICB9XG5cbiAgICB0aGlzLmNvbXB1dGVFbnZpcm9ubWVudEFybiA9IHRoaXMuZ2V0UmVzb3VyY2VBcm5BdHRyaWJ1dGUoY29tcHV0ZUVudmlyb25tZW50LnJlZiwge1xuICAgICAgc2VydmljZTogJ2JhdGNoJyxcbiAgICAgIHJlc291cmNlOiAnY29tcHV0ZS1lbnZpcm9ubWVudCcsXG4gICAgICByZXNvdXJjZU5hbWU6IHRoaXMucGh5c2ljYWxOYW1lLFxuICAgIH0pO1xuICAgIHRoaXMuY29tcHV0ZUVudmlyb25tZW50TmFtZSA9IHRoaXMuZ2V0UmVzb3VyY2VOYW1lQXR0cmlidXRlKGNvbXB1dGVFbnZpcm9ubWVudC5yZWYpO1xuICB9XG5cbiAgcHJpdmF0ZSBpc01hbmFnZWQocHJvcHM6IENvbXB1dGVFbnZpcm9ubWVudFByb3BzKTogYm9vbGVhbiB7XG4gICAgcmV0dXJuIHByb3BzLm1hbmFnZWQgPz8gdHJ1ZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBWYWxpZGF0ZXMgdGhlIHByb3BlcnRpZXMgcHJvdmlkZWQgZm9yIGEgbmV3IGJhdGNoIGNvbXB1dGUgZW52aXJvbm1lbnQuXG4gICAqL1xuICBwcml2YXRlIHZhbGlkYXRlUHJvcHMocHJvcHM6IENvbXB1dGVFbnZpcm9ubWVudFByb3BzLCBpc0ZhcmdhdGU6IGJvb2xlYW4pIHtcbiAgICBpZiAocHJvcHMgPT09IHVuZGVmaW5lZCkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGlmICghdGhpcy5pc01hbmFnZWQocHJvcHMpICYmIHByb3BzLmNvbXB1dGVSZXNvdXJjZXMgIT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdJdCBpcyBub3QgYWxsb3dlZCB0byBzZXQgY29tcHV0ZVJlc291cmNlcyBvbiBhbiBBV1MgdW5tYW5hZ2VkIGNvbXB1dGUgZW52aXJvbm1lbnQnKTtcbiAgICB9XG5cbiAgICBpZiAodGhpcy5pc01hbmFnZWQocHJvcHMpICYmIHByb3BzLmNvbXB1dGVSZXNvdXJjZXMgPT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdjb21wdXRlUmVzb3VyY2VzIGlzIG1pc3NpbmcgYnV0IHJlcXVpcmVkIG9uIGEgbWFuYWdlZCBjb21wdXRlIGVudmlyb25tZW50Jyk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMpIHtcbiAgICAgIGlmIChpc0ZhcmdhdGUpIHtcbiAgICAgICAgLy8gVkFMSURBVEUgRk9SIEZBUkdBVEVcblxuICAgICAgICAvLyBCaWQgcGVyY2VudGFnZSBjYW5ub3QgYmUgc2V0IGZvciBGYXJnYXRlIGV2bnZpcm9ubWVudHNcbiAgICAgICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuYmlkUGVyY2VudGFnZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdCaWQgcGVyY2VudGFnZSBtdXN0IG5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgY29tcHV0ZSBlbnZpcm9ubWVudHMnKTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIEFsbG9jYXRpb24gc3RyYXRlZ3kgY2Fubm90IGJlIHNldCBmb3IgRmFyZ2F0ZSBldm52aXJvbm1lbnRzXG4gICAgICAgIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmFsbG9jYXRpb25TdHJhdGVneSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdBbGxvY2F0aW9uIHN0cmF0ZWd5IG11c3Qgbm90IGJlIHNldCBmb3IgRmFyZ2F0ZSBjb21wdXRlIGVudmlyb25tZW50cycpO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gRGVzaXJlZCB2Q1BVcyBjYW5ub3QgYmUgc2V0IGZvciBGYXJnYXRlIGV2bnZpcm9ubWVudHNcbiAgICAgICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuZGVzaXJlZHZDcHVzICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0Rlc2lyZWQgdkNQVXMgbXVzdCBub3QgYmUgc2V0IGZvciBGYXJnYXRlIGNvbXB1dGUgZW52aXJvbm1lbnRzJyk7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBJbWFnZSBJRCBjYW5ub3QgYmUgc2V0IGZvciBGYXJnYXRlIGV2bnZpcm9ubWVudHNcbiAgICAgICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuaW1hZ2UgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcignSW1hZ2UgbXVzdCBub3QgYmUgc2V0IGZvciBGYXJnYXRlIGNvbXB1dGUgZW52aXJvbm1lbnRzJyk7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBJbnN0YW5jZSB0eXBlcyBjYW5ub3QgYmUgc2V0IGZvciBGYXJnYXRlIGV2bnZpcm9ubWVudHNcbiAgICAgICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuaW5zdGFuY2VUeXBlcyAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdJbnN0YW5jZSB0eXBlcyBtdXN0IG5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgY29tcHV0ZSBlbnZpcm9ubWVudHMnKTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIEVDMiBrZXkgcGFpciBjYW5ub3QgYmUgc2V0IGZvciBGYXJnYXRlIGV2bnZpcm9ubWVudHNcbiAgICAgICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuZWMyS2V5UGFpciAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdFQzIga2V5IHBhaXIgbXVzdCBub3QgYmUgc2V0IGZvciBGYXJnYXRlIGNvbXB1dGUgZW52aXJvbm1lbnRzJyk7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBJbnN0YW5jZSByb2xlIGNhbm5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgZXZudmlyb25tZW50c1xuICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5pbnN0YW5jZVJvbGUgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcignSW5zdGFuY2Ugcm9sZSBtdXN0IG5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgY29tcHV0ZSBlbnZpcm9ubWVudHMnKTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIExhdW5jaCB0ZW1wbGF0ZSBjYW5ub3QgYmUgc2V0IGZvciBGYXJnYXRlIGV2bnZpcm9ubWVudHNcbiAgICAgICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMubGF1bmNoVGVtcGxhdGUgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcignTGF1bmNoIHRlbXBsYXRlIG11c3Qgbm90IGJlIHNldCBmb3IgRmFyZ2F0ZSBjb21wdXRlIGVudmlyb25tZW50cycpO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gTWluIHZDUFVzIGNhbm5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgZXZudmlyb25tZW50c1xuICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5taW52Q3B1cyAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdNaW4gdkNQVXMgbXVzdCBub3QgYmUgc2V0IGZvciBGYXJnYXRlIGNvbXB1dGUgZW52aXJvbm1lbnRzJyk7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBQbGFjZW1lbnQgZ3JvdXAgY2Fubm90IGJlIHNldCBmb3IgRmFyZ2F0ZSBldm52aXJvbm1lbnRzXG4gICAgICAgIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzLnBsYWNlbWVudEdyb3VwICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1BsYWNlbWVudCBncm91cCBtdXN0IG5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgY29tcHV0ZSBlbnZpcm9ubWVudHMnKTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIFNwb3QgZmxlZXQgcm9sZSBjYW5ub3QgYmUgc2V0IGZvciBGYXJnYXRlIGV2bnZpcm9ubWVudHNcbiAgICAgICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuc3BvdEZsZWV0Um9sZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdTcG90IGZsZWV0IHJvbGUgbXVzdCBub3QgYmUgc2V0IGZvciBGYXJnYXRlIGNvbXB1dGUgZW52aXJvbm1lbnRzJyk7XG4gICAgICAgIH1cbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIC8vIFZBTElEQVRFIEZPUiBPTl9ERU1BTkQgQU5EIFNQT1RcbiAgICAgICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMubWludkNwdXMpIHtcbiAgICAgICAgICAvLyBtaW52Q3B1cyBjYW5ub3QgYmUgbGVzcyB0aGFuIDBcbiAgICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5taW52Q3B1cyA8IDApIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignTWluaW11bSB2Q3B1cyBmb3IgYSBiYXRjaCBjb21wdXRlIGVudmlyb25tZW50IGNhbm5vdCBiZSBsZXNzIHRoYW4gMCcpO1xuICAgICAgICAgIH1cblxuICAgICAgICAgIC8vIG1pbnZDcHVzIGNhbm5vdCBleGNlZWQgbWF4IHZDcHVzXG4gICAgICAgICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMubWF4dkNwdXMgJiZcbiAgICAgICAgICAgIHByb3BzLmNvbXB1dGVSZXNvdXJjZXMubWludkNwdXMgPiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLm1heHZDcHVzKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ01pbmltdW0gdkNwdXMgY2Fubm90IGJlIGdyZWF0ZXIgdGhhbiB0aGUgbWF4aW11bSB2Q3B1cycpO1xuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICAvLyBTZXR0aW5nIGEgYmlkIHBlcmNlbnRhZ2UgaXMgb25seSBhbGxvd2VkIG9uIFNQT1QgcmVzb3VyY2VzICtcbiAgICAgICAgLy8gQ2Fubm90IHVzZSBTUE9UX0NBUEFDSVRZX09QVElNSVpFRCB3aGVuIHVzaW5nIE9OX0RFTUFORFxuICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy50eXBlID09PSBDb21wdXRlUmVzb3VyY2VUeXBlLk9OX0RFTUFORCkge1xuICAgICAgICAgIC8vIFZBTElEQVRFIEZPUiBPTl9ERU1BTkRcblxuICAgICAgICAgIC8vIEJpZCBwZXJjZW50YWdlIGlzIG5vdCBhbGxvd2VkXG4gICAgICAgICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuYmlkUGVyY2VudGFnZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1NldHRpbmcgdGhlIGJpZCBwZXJjZW50YWdlIGlzIG9ubHkgYWxsb3dlZCBmb3IgU1BPVCB0eXBlIHJlc291cmNlcyBvbiBhIGJhdGNoIGNvbXB1dGUgZW52aXJvbm1lbnQnKTtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICAvLyBTUE9UX0NBUEFDSVRZX09QVElNSVpFRCBhbGxvY2F0aW9uIGlzIG5vdCBhbGxvd2VkXG4gICAgICAgICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuYWxsb2NhdGlvblN0cmF0ZWd5ICYmIHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuYWxsb2NhdGlvblN0cmF0ZWd5ID09PSBBbGxvY2F0aW9uU3RyYXRlZ3kuU1BPVF9DQVBBQ0lUWV9PUFRJTUlaRUQpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignVGhlIFNQT1RfQ0FQQUNJVFlfT1BUSU1JWkVEIGFsbG9jYXRpb24gc3RyYXRlZ3kgaXMgb25seSBhbGxvd2VkIGlmIHRoZSBlbnZpcm9ubWVudCBpcyBhIFNQT1QgdHlwZSBjb21wdXRlIGVudmlyb25tZW50Jyk7XG4gICAgICAgICAgfVxuICAgICAgICB9IGVsc2UgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMudHlwZSA9PT0gQ29tcHV0ZVJlc291cmNlVHlwZS5TUE9UKSB7XG4gICAgICAgICAgLy8gVkFMSURBVEUgRk9SIFNQT1RcblxuICAgICAgICAgIC8vIEJpZCBwZXJjZW50YWdlIG11c3QgYmUgZnJvbSAwIC0gMTAwXG4gICAgICAgICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuYmlkUGVyY2VudGFnZSAhPT0gdW5kZWZpbmVkICYmXG4gICAgICAgICAgICAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5iaWRQZXJjZW50YWdlIDwgMCB8fCBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmJpZFBlcmNlbnRhZ2UgPiAxMDApKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0JpZCBwZXJjZW50YWdlIGNhbiBvbmx5IGJlIGEgdmFsdWUgYmV0d2VlbiAwIGFuZCAxMDAnKTtcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICBwcml2YXRlIGJ1aWxkSW5zdGFuY2VUeXBlcyhpbnN0YW5jZVR5cGVzPzogZWMyLkluc3RhbmNlVHlwZVtdKTogc3RyaW5nW10ge1xuICAgIGlmIChpbnN0YW5jZVR5cGVzID09PSB1bmRlZmluZWQpIHtcbiAgICAgIHJldHVybiBbXG4gICAgICAgICdvcHRpbWFsJyxcbiAgICAgIF07XG4gICAgfVxuXG4gICAgcmV0dXJuIGluc3RhbmNlVHlwZXMubWFwKCh0eXBlOiBlYzIuSW5zdGFuY2VUeXBlKSA9PiB0eXBlLnRvU3RyaW5nKCkpO1xuICB9XG5cbiAgcHJpdmF0ZSBidWlsZFNlY3VyaXR5R3JvdXBJZHModnBjOiBlYzIuSVZwYywgc2VjdXJpdHlHcm91cHM/OiBlYzIuSVNlY3VyaXR5R3JvdXBbXSk6IHN0cmluZ1tdIHwgdW5kZWZpbmVkIHtcbiAgICBpZiAoc2VjdXJpdHlHcm91cHMgPT09IHVuZGVmaW5lZCkge1xuICAgICAgcmV0dXJuIFtcbiAgICAgICAgbmV3IGVjMi5TZWN1cml0eUdyb3VwKHRoaXMsICdSZXNvdXJjZS1TZWN1cml0eS1Hcm91cCcsIHsgdnBjIH0pLnNlY3VyaXR5R3JvdXBJZCxcbiAgICAgIF07XG4gICAgfVxuXG4gICAgcmV0dXJuIHNlY3VyaXR5R3JvdXBzLm1hcCgoZ3JvdXA6IGVjMi5JU2VjdXJpdHlHcm91cCkgPT4gZ3JvdXAuc2VjdXJpdHlHcm91cElkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHZW5lcmF0ZXMgYW4gQVdTIElBTSByb2xlIGZvciBwcm92aXNpb25pbmcgc3BvdGZsZWV0IHJlc291cmNlc1xuICAgKiBpZiB0aGUgYWxsb2NhdGlvbiBzdHJhdGVneSBpcyBzZXQgdG8gQkVTVF9GSVQgb3Igbm90IGRlZmluZWQuXG4gICAqXG4gICAqIEBwYXJhbSBwcm9wcyAtIHRoZSBjb21wdXRlIGVudmlyb25tZW50IGNvbnN0cnVjdCBwcm9wZXJ0aWVzXG4gICAqL1xuICBwcml2YXRlIGdldFNwb3RGbGVldFJvbGUocHJvcHM6IENvbXB1dGVFbnZpcm9ubWVudFByb3BzKTogaWFtLklSb2xlIHwgdW5kZWZpbmVkIHtcbiAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcz8uYWxsb2NhdGlvblN0cmF0ZWd5ICYmIHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuYWxsb2NhdGlvblN0cmF0ZWd5ICE9PSBBbGxvY2F0aW9uU3RyYXRlZ3kuQkVTVF9GSVQpIHtcbiAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMpIHtcbiAgICAgIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzLnNwb3RGbGVldFJvbGUpIHtcbiAgICAgICAgcmV0dXJuIHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuc3BvdEZsZWV0Um9sZTtcbiAgICAgIH0gZWxzZSBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy50eXBlID09PSBDb21wdXRlUmVzb3VyY2VUeXBlLlNQT1QpIHtcbiAgICAgICAgcmV0dXJuIGlhbS5Sb2xlLmZyb21Sb2xlQXJuKHRoaXMsICdSZXNvdXJjZS1TcG90RmxlZXQtUm9sZScsXG4gICAgICAgICAgYGFybjoke3RoaXMuc3RhY2sucGFydGl0aW9ufTppYW06OiR7dGhpcy5lbnYuYWNjb3VudH06cm9sZS9hd3Mtc2VydmljZS1yb2xlL3Nwb3RmbGVldC5hbWF6b25hd3MuY29tL0FXU1NlcnZpY2VSb2xlRm9yRUMyU3BvdEZsZWV0YCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxufVxuIl19