"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DnsValidatedCertificate = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const iam = require("../../aws-iam");
const lambda = require("../../aws-lambda");
const cdk = require("../../core");
const certificate_base_1 = require("./certificate-base");
/**
 * A certificate managed by AWS Certificate Manager.  Will be automatically
 * validated using DNS validation against the specified Route 53 hosted zone.
 *
 * @resource AWS::CertificateManager::Certificate
 */
class DnsValidatedCertificate extends certificate_base_1.CertificateBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.monocdk_aws_certificatemanager_DnsValidatedCertificateProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, DnsValidatedCertificate);
            }
            throw error;
        }
        this.region = props.region;
        this.domainName = props.domainName;
        this.normalizedZoneName = props.hostedZone.zoneName;
        // Remove trailing `.` from zone name
        if (this.normalizedZoneName.endsWith('.')) {
            this.normalizedZoneName = this.normalizedZoneName.substring(0, this.normalizedZoneName.length - 1);
        }
        // Remove any `/hostedzone/` prefix from the Hosted Zone ID
        this.hostedZoneId = props.hostedZone.hostedZoneId.replace(/^\/hostedzone\//, '');
        this.tags = new cdk.TagManager(cdk.TagType.MAP, 'AWS::CertificateManager::Certificate');
        const requestorFunction = new lambda.Function(this, 'CertificateRequestorFunction', {
            code: lambda.Code.fromAsset(path.resolve(__dirname, '..', 'lambda-packages', 'dns_validated_certificate_handler', 'lib')),
            handler: 'index.certificateRequestHandler',
            runtime: lambda.Runtime.NODEJS_14_X,
            timeout: cdk.Duration.minutes(15),
            role: props.customResourceRole,
        });
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['acm:RequestCertificate', 'acm:DescribeCertificate', 'acm:DeleteCertificate', 'acm:AddTagsToCertificate'],
            resources: ['*'],
        }));
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['route53:GetChange'],
            resources: ['*'],
        }));
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['route53:changeResourceRecordSets'],
            resources: [`arn:${cdk.Stack.of(requestorFunction).partition}:route53:::hostedzone/${this.hostedZoneId}`],
        }));
        const certificate = new cdk.CustomResource(this, 'CertificateRequestorResource', {
            serviceToken: requestorFunction.functionArn,
            properties: {
                DomainName: props.domainName,
                SubjectAlternativeNames: cdk.Lazy.list({ produce: () => props.subjectAlternativeNames }, { omitEmpty: true }),
                HostedZoneId: this.hostedZoneId,
                Region: props.region,
                Route53Endpoint: props.route53Endpoint,
                // Custom resources properties are always converted to strings; might as well be explict here.
                CleanupRecords: props.cleanupRoute53Records ? 'true' : undefined,
                Tags: cdk.Lazy.list({ produce: () => this.tags.renderTags() }),
            },
        });
        this.certificateArn = certificate.getAtt('Arn').toString();
    }
    validate() {
        const errors = [];
        // Ensure the zone name is a parent zone of the certificate domain name
        if (!cdk.Token.isUnresolved(this.normalizedZoneName) &&
            this.domainName !== this.normalizedZoneName &&
            !this.domainName.endsWith('.' + this.normalizedZoneName)) {
            errors.push(`DNS zone ${this.normalizedZoneName} is not authoritative for certificate domain name ${this.domainName}`);
        }
        return errors;
    }
}
exports.DnsValidatedCertificate = DnsValidatedCertificate;
_a = JSII_RTTI_SYMBOL_1;
DnsValidatedCertificate[_a] = { fqn: "monocdk.aws_certificatemanager.DnsValidatedCertificate", version: "1.191.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZG5zLXZhbGlkYXRlZC1jZXJ0aWZpY2F0ZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImRucy12YWxpZGF0ZWQtY2VydGlmaWNhdGUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBQUEsNkJBQTZCO0FBQzdCLHFDQUFxQztBQUNyQywyQ0FBMkM7QUFFM0Msa0NBQWtDO0FBR2xDLHlEQUFxRDtBQXVEckQ7Ozs7O0dBS0c7QUFDSCxNQUFhLHVCQUF3QixTQUFRLGtDQUFlO0lBYzFELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBbUM7UUFDM0UsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQzs7Ozs7OytDQWZSLHVCQUF1Qjs7OztRQWlCaEMsSUFBSSxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUMsTUFBTSxDQUFDO1FBRTNCLElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQztRQUNuQyxJQUFJLENBQUMsa0JBQWtCLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUM7UUFDcEQscUNBQXFDO1FBQ3JDLElBQUksSUFBSSxDQUFDLGtCQUFrQixDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsRUFBRTtZQUN6QyxJQUFJLENBQUMsa0JBQWtCLEdBQUcsSUFBSSxDQUFDLGtCQUFrQixDQUFDLFNBQVMsQ0FBQyxDQUFDLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQztTQUNwRztRQUVELDJEQUEyRDtRQUMzRCxJQUFJLENBQUMsWUFBWSxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUMsWUFBWSxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNqRixJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxzQ0FBc0MsQ0FBQyxDQUFDO1FBRXhGLE1BQU0saUJBQWlCLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSw4QkFBOEIsRUFBRTtZQUNsRixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLEVBQUUsSUFBSSxFQUFFLGlCQUFpQixFQUFFLG1DQUFtQyxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBQ3pILE9BQU8sRUFBRSxpQ0FBaUM7WUFDMUMsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztZQUNuQyxPQUFPLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1lBQ2pDLElBQUksRUFBRSxLQUFLLENBQUMsa0JBQWtCO1NBQy9CLENBQUMsQ0FBQztRQUNILGlCQUFpQixDQUFDLGVBQWUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDeEQsT0FBTyxFQUFFLENBQUMsd0JBQXdCLEVBQUUseUJBQXlCLEVBQUUsdUJBQXVCLEVBQUUsMEJBQTBCLENBQUM7WUFDbkgsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO1NBQ2pCLENBQUMsQ0FBQyxDQUFDO1FBQ0osaUJBQWlCLENBQUMsZUFBZSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUN4RCxPQUFPLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQztZQUM5QixTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7U0FDakIsQ0FBQyxDQUFDLENBQUM7UUFDSixpQkFBaUIsQ0FBQyxlQUFlLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3hELE9BQU8sRUFBRSxDQUFDLGtDQUFrQyxDQUFDO1lBQzdDLFNBQVMsRUFBRSxDQUFDLE9BQU8sR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxTQUFTLHlCQUF5QixJQUFJLENBQUMsWUFBWSxFQUFFLENBQUM7U0FDMUcsQ0FBQyxDQUFDLENBQUM7UUFFSixNQUFNLFdBQVcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLDhCQUE4QixFQUFFO1lBQy9FLFlBQVksRUFBRSxpQkFBaUIsQ0FBQyxXQUFXO1lBQzNDLFVBQVUsRUFBRTtnQkFDVixVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7Z0JBQzVCLHVCQUF1QixFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLEtBQUssQ0FBQyx1QkFBdUIsRUFBRSxFQUFFLEVBQUUsU0FBUyxFQUFFLElBQUksRUFBRSxDQUFDO2dCQUM3RyxZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVk7Z0JBQy9CLE1BQU0sRUFBRSxLQUFLLENBQUMsTUFBTTtnQkFDcEIsZUFBZSxFQUFFLEtBQUssQ0FBQyxlQUFlO2dCQUN0Qyw4RkFBOEY7Z0JBQzlGLGNBQWMsRUFBRSxLQUFLLENBQUMscUJBQXFCLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsU0FBUztnQkFDaEUsSUFBSSxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLEVBQUUsQ0FBQzthQUMvRDtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxjQUFjLEdBQUcsV0FBVyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxRQUFRLEVBQUUsQ0FBQztLQUM1RDtJQUVTLFFBQVE7UUFDaEIsTUFBTSxNQUFNLEdBQWEsRUFBRSxDQUFDO1FBQzVCLHVFQUF1RTtRQUN2RSxJQUFJLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLGtCQUFrQixDQUFDO1lBQ2xELElBQUksQ0FBQyxVQUFVLEtBQUssSUFBSSxDQUFDLGtCQUFrQjtZQUMzQyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLEdBQUcsR0FBRyxJQUFJLENBQUMsa0JBQWtCLENBQUMsRUFBRTtZQUMxRCxNQUFNLENBQUMsSUFBSSxDQUFDLFlBQVksSUFBSSxDQUFDLGtCQUFrQixxREFBcUQsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUM7U0FDeEg7UUFDRCxPQUFPLE1BQU0sQ0FBQztLQUNmOztBQTVFSCwwREE2RUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJy4uLy4uL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJy4uLy4uL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgcm91dGU1MyBmcm9tICcuLi8uLi9hd3Mtcm91dGU1Myc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnLi4vLi4vY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IENlcnRpZmljYXRlUHJvcHMsIElDZXJ0aWZpY2F0ZSB9IGZyb20gJy4vY2VydGlmaWNhdGUnO1xuaW1wb3J0IHsgQ2VydGlmaWNhdGVCYXNlIH0gZnJvbSAnLi9jZXJ0aWZpY2F0ZS1iYXNlJztcblxuLyoqXG4gKiBQcm9wZXJ0aWVzIHRvIGNyZWF0ZSBhIEROUyB2YWxpZGF0ZWQgY2VydGlmaWNhdGUgbWFuYWdlZCBieSBBV1MgQ2VydGlmaWNhdGUgTWFuYWdlclxuICpcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBEbnNWYWxpZGF0ZWRDZXJ0aWZpY2F0ZVByb3BzIGV4dGVuZHMgQ2VydGlmaWNhdGVQcm9wcyB7XG4gIC8qKlxuICAgKiBSb3V0ZSA1MyBIb3N0ZWQgWm9uZSB1c2VkIHRvIHBlcmZvcm0gRE5TIHZhbGlkYXRpb24gb2YgdGhlIHJlcXVlc3QuICBUaGUgem9uZVxuICAgKiBtdXN0IGJlIGF1dGhvcml0YXRpdmUgZm9yIHRoZSBkb21haW4gbmFtZSBzcGVjaWZpZWQgaW4gdGhlIENlcnRpZmljYXRlIFJlcXVlc3QuXG4gICAqL1xuICByZWFkb25seSBob3N0ZWRab25lOiByb3V0ZTUzLklIb3N0ZWRab25lO1xuICAvKipcbiAgICogQVdTIHJlZ2lvbiB0aGF0IHdpbGwgaG9zdCB0aGUgY2VydGlmaWNhdGUuIFRoaXMgaXMgbmVlZGVkIGVzcGVjaWFsbHlcbiAgICogZm9yIGNlcnRpZmljYXRlcyB1c2VkIGZvciBDbG91ZEZyb250IGRpc3RyaWJ1dGlvbnMsIHdoaWNoIHJlcXVpcmUgdGhlIHJlZ2lvblxuICAgKiB0byBiZSB1cy1lYXN0LTEuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRoZSByZWdpb24gdGhlIHN0YWNrIGlzIGRlcGxveWVkIGluLlxuICAgKi9cbiAgcmVhZG9ubHkgcmVnaW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBbiBlbmRwb2ludCBvZiBSb3V0ZTUzIHNlcnZpY2UsIHdoaWNoIGlzIG5vdCBuZWNlc3NhcnkgYXMgQVdTIFNESyBjb3VsZCBmaWd1cmVcbiAgICogb3V0IHRoZSByaWdodCBlbmRwb2ludHMgZm9yIG1vc3QgcmVnaW9ucywgYnV0IGZvciBzb21lIHJlZ2lvbnMgc3VjaCBhcyB0aG9zZSBpblxuICAgKiBhd3MtY24gcGFydGl0aW9uLCB0aGUgZGVmYXVsdCBlbmRwb2ludCBpcyBub3Qgd29ya2luZyBub3csIGhlbmNlIHRoZSByaWdodCBlbmRwb2ludFxuICAgKiBuZWVkIHRvIGJlIHNwZWNpZmllZCB0aHJvdWdoIHRoaXMgcHJvcC5cbiAgICpcbiAgICogUm91dGU1MyBpcyBub3QgYmVlbiBvZmZpY2lhbGx5IGxhdW5jaGVkIGluIENoaW5hLCBpdCBpcyBvbmx5IGF2YWlsYWJsZSBmb3IgQVdTXG4gICAqIGludGVybmFsIGFjY291bnRzIG5vdy4gVG8gbWFrZSBEbnNWYWxpZGF0ZWRDZXJ0aWZpY2F0ZSB3b3JrIGZvciBpbnRlcm5hbCBhY2NvdW50c1xuICAgKiBub3csIGEgc3BlY2lhbCBlbmRwb2ludCBuZWVkcyB0byBiZSBwcm92aWRlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBUaGUgQVdTIFNESyB3aWxsIGRldGVybWluZSB0aGUgUm91dGU1MyBlbmRwb2ludCB0byB1c2UgYmFzZWQgb24gcmVnaW9uXG4gICAqL1xuICByZWFkb25seSByb3V0ZTUzRW5kcG9pbnQ/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFJvbGUgdG8gdXNlIGZvciB0aGUgY3VzdG9tIHJlc291cmNlIHRoYXQgY3JlYXRlcyB0aGUgdmFsaWRhdGVkIGNlcnRpZmljYXRlXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQSBuZXcgcm9sZSB3aWxsIGJlIGNyZWF0ZWRcbiAgICovXG4gIHJlYWRvbmx5IGN1c3RvbVJlc291cmNlUm9sZT86IGlhbS5JUm9sZTtcblxuICAvKipcbiAgICogV2hlbiBzZXQgdG8gdHJ1ZSwgd2hlbiB0aGUgRG5zVmFsaWRhdGVkQ2VydGlmaWNhdGUgaXMgZGVsZXRlZCxcbiAgICogdGhlIGFzc29jaWF0ZWQgUm91dGU1MyB2YWxpZGF0aW9uIHJlY29yZHMgYXJlIHJlbW92ZWQuXG4gICAqXG4gICAqIENBVVRJT046IElmIG11bHRpcGxlIGNlcnRpZmljYXRlcyBzaGFyZSB0aGUgc2FtZSBkb21haW5zIChhbmQgc2FtZSB2YWxpZGF0aW9uIHJlY29yZHMpLFxuICAgKiB0aGlzIGNhbiBjYXVzZSB0aGUgb3RoZXIgY2VydGlmaWNhdGVzIHRvIGZhaWwgcmVuZXdhbCBhbmQvb3Igbm90IHZhbGlkYXRlLlxuICAgKiBOb3QgcmVjb21tZW5kZWQgZm9yIHByb2R1Y3Rpb24gdXNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgY2xlYW51cFJvdXRlNTNSZWNvcmRzPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBBIGNlcnRpZmljYXRlIG1hbmFnZWQgYnkgQVdTIENlcnRpZmljYXRlIE1hbmFnZXIuICBXaWxsIGJlIGF1dG9tYXRpY2FsbHlcbiAqIHZhbGlkYXRlZCB1c2luZyBETlMgdmFsaWRhdGlvbiBhZ2FpbnN0IHRoZSBzcGVjaWZpZWQgUm91dGUgNTMgaG9zdGVkIHpvbmUuXG4gKlxuICogQHJlc291cmNlIEFXUzo6Q2VydGlmaWNhdGVNYW5hZ2VyOjpDZXJ0aWZpY2F0ZVxuICovXG5leHBvcnQgY2xhc3MgRG5zVmFsaWRhdGVkQ2VydGlmaWNhdGUgZXh0ZW5kcyBDZXJ0aWZpY2F0ZUJhc2UgaW1wbGVtZW50cyBJQ2VydGlmaWNhdGUsIGNkay5JVGFnZ2FibGUge1xuICBwdWJsaWMgcmVhZG9ubHkgY2VydGlmaWNhdGVBcm46IHN0cmluZztcblxuICAvKipcbiAgKiBSZXNvdXJjZSBUYWdzLlxuICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXJlc291cmNlLWNlcnRpZmljYXRlbWFuYWdlci1jZXJ0aWZpY2F0ZS5odG1sI2Nmbi1jZXJ0aWZpY2F0ZW1hbmFnZXItY2VydGlmaWNhdGUtdGFnc1xuICAqL1xuXG4gIHB1YmxpYyByZWFkb25seSB0YWdzOiBjZGsuVGFnTWFuYWdlcjtcbiAgcHJvdGVjdGVkIHJlYWRvbmx5IHJlZ2lvbj86IHN0cmluZztcbiAgcHJpdmF0ZSBub3JtYWxpemVkWm9uZU5hbWU6IHN0cmluZztcbiAgcHJpdmF0ZSBob3N0ZWRab25lSWQ6IHN0cmluZztcbiAgcHJpdmF0ZSBkb21haW5OYW1lOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IERuc1ZhbGlkYXRlZENlcnRpZmljYXRlUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy5yZWdpb24gPSBwcm9wcy5yZWdpb247XG5cbiAgICB0aGlzLmRvbWFpbk5hbWUgPSBwcm9wcy5kb21haW5OYW1lO1xuICAgIHRoaXMubm9ybWFsaXplZFpvbmVOYW1lID0gcHJvcHMuaG9zdGVkWm9uZS56b25lTmFtZTtcbiAgICAvLyBSZW1vdmUgdHJhaWxpbmcgYC5gIGZyb20gem9uZSBuYW1lXG4gICAgaWYgKHRoaXMubm9ybWFsaXplZFpvbmVOYW1lLmVuZHNXaXRoKCcuJykpIHtcbiAgICAgIHRoaXMubm9ybWFsaXplZFpvbmVOYW1lID0gdGhpcy5ub3JtYWxpemVkWm9uZU5hbWUuc3Vic3RyaW5nKDAsIHRoaXMubm9ybWFsaXplZFpvbmVOYW1lLmxlbmd0aCAtIDEpO1xuICAgIH1cblxuICAgIC8vIFJlbW92ZSBhbnkgYC9ob3N0ZWR6b25lL2AgcHJlZml4IGZyb20gdGhlIEhvc3RlZCBab25lIElEXG4gICAgdGhpcy5ob3N0ZWRab25lSWQgPSBwcm9wcy5ob3N0ZWRab25lLmhvc3RlZFpvbmVJZC5yZXBsYWNlKC9eXFwvaG9zdGVkem9uZVxcLy8sICcnKTtcbiAgICB0aGlzLnRhZ3MgPSBuZXcgY2RrLlRhZ01hbmFnZXIoY2RrLlRhZ1R5cGUuTUFQLCAnQVdTOjpDZXJ0aWZpY2F0ZU1hbmFnZXI6OkNlcnRpZmljYXRlJyk7XG5cbiAgICBjb25zdCByZXF1ZXN0b3JGdW5jdGlvbiA9IG5ldyBsYW1iZGEuRnVuY3Rpb24odGhpcywgJ0NlcnRpZmljYXRlUmVxdWVzdG9yRnVuY3Rpb24nLCB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQocGF0aC5yZXNvbHZlKF9fZGlybmFtZSwgJy4uJywgJ2xhbWJkYS1wYWNrYWdlcycsICdkbnNfdmFsaWRhdGVkX2NlcnRpZmljYXRlX2hhbmRsZXInLCAnbGliJykpLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmNlcnRpZmljYXRlUmVxdWVzdEhhbmRsZXInLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE0X1gsXG4gICAgICB0aW1lb3V0OiBjZGsuRHVyYXRpb24ubWludXRlcygxNSksXG4gICAgICByb2xlOiBwcm9wcy5jdXN0b21SZXNvdXJjZVJvbGUsXG4gICAgfSk7XG4gICAgcmVxdWVzdG9yRnVuY3Rpb24uYWRkVG9Sb2xlUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFsnYWNtOlJlcXVlc3RDZXJ0aWZpY2F0ZScsICdhY206RGVzY3JpYmVDZXJ0aWZpY2F0ZScsICdhY206RGVsZXRlQ2VydGlmaWNhdGUnLCAnYWNtOkFkZFRhZ3NUb0NlcnRpZmljYXRlJ10sXG4gICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgIH0pKTtcbiAgICByZXF1ZXN0b3JGdW5jdGlvbi5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgYWN0aW9uczogWydyb3V0ZTUzOkdldENoYW5nZSddLFxuICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICB9KSk7XG4gICAgcmVxdWVzdG9yRnVuY3Rpb24uYWRkVG9Sb2xlUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFsncm91dGU1MzpjaGFuZ2VSZXNvdXJjZVJlY29yZFNldHMnXSxcbiAgICAgIHJlc291cmNlczogW2Bhcm46JHtjZGsuU3RhY2sub2YocmVxdWVzdG9yRnVuY3Rpb24pLnBhcnRpdGlvbn06cm91dGU1Mzo6Omhvc3RlZHpvbmUvJHt0aGlzLmhvc3RlZFpvbmVJZH1gXSxcbiAgICB9KSk7XG5cbiAgICBjb25zdCBjZXJ0aWZpY2F0ZSA9IG5ldyBjZGsuQ3VzdG9tUmVzb3VyY2UodGhpcywgJ0NlcnRpZmljYXRlUmVxdWVzdG9yUmVzb3VyY2UnLCB7XG4gICAgICBzZXJ2aWNlVG9rZW46IHJlcXVlc3RvckZ1bmN0aW9uLmZ1bmN0aW9uQXJuLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICBEb21haW5OYW1lOiBwcm9wcy5kb21haW5OYW1lLFxuICAgICAgICBTdWJqZWN0QWx0ZXJuYXRpdmVOYW1lczogY2RrLkxhenkubGlzdCh7IHByb2R1Y2U6ICgpID0+IHByb3BzLnN1YmplY3RBbHRlcm5hdGl2ZU5hbWVzIH0sIHsgb21pdEVtcHR5OiB0cnVlIH0pLFxuICAgICAgICBIb3N0ZWRab25lSWQ6IHRoaXMuaG9zdGVkWm9uZUlkLFxuICAgICAgICBSZWdpb246IHByb3BzLnJlZ2lvbixcbiAgICAgICAgUm91dGU1M0VuZHBvaW50OiBwcm9wcy5yb3V0ZTUzRW5kcG9pbnQsXG4gICAgICAgIC8vIEN1c3RvbSByZXNvdXJjZXMgcHJvcGVydGllcyBhcmUgYWx3YXlzIGNvbnZlcnRlZCB0byBzdHJpbmdzOyBtaWdodCBhcyB3ZWxsIGJlIGV4cGxpY3QgaGVyZS5cbiAgICAgICAgQ2xlYW51cFJlY29yZHM6IHByb3BzLmNsZWFudXBSb3V0ZTUzUmVjb3JkcyA/ICd0cnVlJyA6IHVuZGVmaW5lZCxcbiAgICAgICAgVGFnczogY2RrLkxhenkubGlzdCh7IHByb2R1Y2U6ICgpID0+IHRoaXMudGFncy5yZW5kZXJUYWdzKCkgfSksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGhpcy5jZXJ0aWZpY2F0ZUFybiA9IGNlcnRpZmljYXRlLmdldEF0dCgnQXJuJykudG9TdHJpbmcoKTtcbiAgfVxuXG4gIHByb3RlY3RlZCB2YWxpZGF0ZSgpOiBzdHJpbmdbXSB7XG4gICAgY29uc3QgZXJyb3JzOiBzdHJpbmdbXSA9IFtdO1xuICAgIC8vIEVuc3VyZSB0aGUgem9uZSBuYW1lIGlzIGEgcGFyZW50IHpvbmUgb2YgdGhlIGNlcnRpZmljYXRlIGRvbWFpbiBuYW1lXG4gICAgaWYgKCFjZGsuVG9rZW4uaXNVbnJlc29sdmVkKHRoaXMubm9ybWFsaXplZFpvbmVOYW1lKSAmJlxuICAgICAgdGhpcy5kb21haW5OYW1lICE9PSB0aGlzLm5vcm1hbGl6ZWRab25lTmFtZSAmJlxuICAgICAgIXRoaXMuZG9tYWluTmFtZS5lbmRzV2l0aCgnLicgKyB0aGlzLm5vcm1hbGl6ZWRab25lTmFtZSkpIHtcbiAgICAgIGVycm9ycy5wdXNoKGBETlMgem9uZSAke3RoaXMubm9ybWFsaXplZFpvbmVOYW1lfSBpcyBub3QgYXV0aG9yaXRhdGl2ZSBmb3IgY2VydGlmaWNhdGUgZG9tYWluIG5hbWUgJHt0aGlzLmRvbWFpbk5hbWV9YCk7XG4gICAgfVxuICAgIHJldHVybiBlcnJvcnM7XG4gIH1cbn1cbiJdfQ==