"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginBase = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
/**
 * Represents a distribution origin, that describes the Amazon S3 bucket, HTTP server (for example, a web server),
 * Amazon MediaStore, or other server from which CloudFront gets your files.
 */
class OriginBase {
    constructor(domainName, props = {}) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudfront_OriginProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, OriginBase);
            }
            throw error;
        }
        validateIntInRangeOrUndefined('connectionTimeout', 1, 10, props.connectionTimeout?.toSeconds());
        validateIntInRangeOrUndefined('connectionAttempts', 1, 3, props.connectionAttempts, false);
        validateCustomHeaders(props.customHeaders);
        this.domainName = domainName;
        this.originPath = this.validateOriginPath(props.originPath);
        this.connectionTimeout = props.connectionTimeout;
        this.connectionAttempts = props.connectionAttempts;
        this.customHeaders = props.customHeaders;
        this.originShieldRegion = props.originShieldRegion;
    }
    /**
     * Binds the origin to the associated Distribution. Can be used to grant permissions, create dependent resources, etc.
     */
    bind(_scope, options) {
        try {
            jsiiDeprecationWarnings.monocdk_Construct(_scope);
            jsiiDeprecationWarnings.monocdk_aws_cloudfront_OriginBindOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        const s3OriginConfig = this.renderS3OriginConfig();
        const customOriginConfig = this.renderCustomOriginConfig();
        if (!s3OriginConfig && !customOriginConfig) {
            throw new Error('Subclass must override and provide either s3OriginConfig or customOriginConfig');
        }
        return {
            originProperty: {
                domainName: this.domainName,
                id: options.originId,
                originPath: this.originPath,
                connectionAttempts: this.connectionAttempts,
                connectionTimeout: this.connectionTimeout?.toSeconds(),
                originCustomHeaders: this.renderCustomHeaders(),
                s3OriginConfig,
                customOriginConfig,
                originShield: this.renderOriginShield(this.originShieldRegion),
            },
        };
    }
    // Overridden by sub-classes to provide S3 origin config.
    renderS3OriginConfig() {
        return undefined;
    }
    // Overridden by sub-classes to provide custom origin config.
    renderCustomOriginConfig() {
        return undefined;
    }
    renderCustomHeaders() {
        if (!this.customHeaders || Object.entries(this.customHeaders).length === 0) {
            return undefined;
        }
        return Object.entries(this.customHeaders).map(([headerName, headerValue]) => {
            return { headerName, headerValue };
        });
    }
    /**
     * If the path is defined, it must start with a '/' and not end with a '/'.
     * This method takes in the originPath, and returns it back (if undefined) or adds/removes the '/' as appropriate.
     */
    validateOriginPath(originPath) {
        if (core_1.Token.isUnresolved(originPath)) {
            return originPath;
        }
        if (originPath === undefined) {
            return undefined;
        }
        let path = originPath;
        if (!path.startsWith('/')) {
            path = '/' + path;
        }
        if (path.endsWith('/')) {
            path = path.slice(0, -1);
        }
        return path;
    }
    /**
     * Takes origin shield region and converts to CfnDistribution.OriginShieldProperty
     */
    renderOriginShield(originShieldRegion) {
        return originShieldRegion
            ? { enabled: true, originShieldRegion }
            : undefined;
    }
}
exports.OriginBase = OriginBase;
_a = JSII_RTTI_SYMBOL_1;
OriginBase[_a] = { fqn: "monocdk.aws_cloudfront.OriginBase", version: "1.191.0" };
/**
 * Throws an error if a value is defined and not an integer or not in a range.
 */
function validateIntInRangeOrUndefined(name, min, max, value, isDuration = true) {
    if (value === undefined) {
        return;
    }
    if (!Number.isInteger(value) || value < min || value > max) {
        const seconds = isDuration ? ' seconds' : '';
        throw new Error(`${name}: Must be an int between ${min} and ${max}${seconds} (inclusive); received ${value}.`);
    }
}
/**
 * Throws an error if custom header assignment is prohibited by CloudFront.
 * @link: https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/add-origin-custom-headers.html#add-origin-custom-headers-denylist
 */
function validateCustomHeaders(customHeaders) {
    if (!customHeaders || Object.entries(customHeaders).length === 0) {
        return;
    }
    const customHeaderKeys = Object.keys(customHeaders);
    const prohibitedHeaderKeys = [
        'Cache-Control', 'Connection', 'Content-Length', 'Cookie', 'Host', 'If-Match', 'If-Modified-Since', 'If-None-Match', 'If-Range', 'If-Unmodified-Since',
        'Max-Forwards', 'Pragma', 'Proxy-Authorization', 'Proxy-Connection', 'Range', 'Request-Range', 'TE', 'Trailer', 'Transfer-Encoding', 'Upgrade', 'Via',
        'X-Real-Ip',
    ];
    const prohibitedHeaderKeyPrefixes = [
        'X-Amz-', 'X-Edge-',
    ];
    const prohibitedHeadersKeysMatches = customHeaderKeys.filter(customKey => {
        return prohibitedHeaderKeys.map((prohibitedKey) => prohibitedKey.toLowerCase()).includes(customKey.toLowerCase());
    });
    const prohibitedHeaderPrefixMatches = customHeaderKeys.filter(customKey => {
        return prohibitedHeaderKeyPrefixes.some(prohibitedKeyPrefix => customKey.toLowerCase().startsWith(prohibitedKeyPrefix.toLowerCase()));
    });
    if (prohibitedHeadersKeysMatches.length !== 0) {
        throw new Error(`The following headers cannot be configured as custom origin headers: ${prohibitedHeadersKeysMatches.join(', ')}`);
    }
    if (prohibitedHeaderPrefixMatches.length !== 0) {
        throw new Error(`The following headers cannot be used as prefixes for custom origin headers: ${prohibitedHeaderPrefixMatches.join(', ')}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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