"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Code = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const assets = require("../../aws-s3-assets");
/**
 * Represents the contents to initialize the repository with.
 */
class Code {
    /**
     * Code from directory.
     * @param directoryPath the path to the local directory containing the contents to initialize the repository with
     * @param branch the name of the branch to create in the repository. Default is "main"
     */
    static fromDirectory(directoryPath, branch) {
        const resolvedPath = path.resolve(directoryPath);
        const statResult = fs.statSync(resolvedPath);
        if (!statResult || !statResult.isDirectory()) {
            throw new Error(`'${directoryPath}' needs to be a path to a directory (resolved to: '${resolvedPath}')`);
        }
        return new PathResolvedCode(resolvedPath, branch);
    }
    /**
     * Code from preexisting ZIP file.
     * @param filePath the path to the local ZIP file containing the contents to initialize the repository with
     * @param branch the name of the branch to create in the repository. Default is "main"
     */
    static fromZipFile(filePath, branch) {
        const resolvedPath = path.resolve(filePath);
        const statResult = fs.statSync(resolvedPath);
        if (!statResult || !statResult.isFile()) {
            throw new Error(`'${filePath}' needs to be a path to a ZIP file (resolved to: '${resolvedPath}')`);
        }
        return new PathResolvedCode(resolvedPath, branch);
    }
    /**
     * Code from user-supplied asset.
     * @param asset pre-existing asset
     * @param branch the name of the branch to create in the repository. Default is "main"
     */
    static fromAsset(asset, branch) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_s3_assets_Asset(asset);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromAsset);
            }
            throw error;
        }
        return new AssetCode(asset, branch);
    }
}
exports.Code = Code;
_a = JSII_RTTI_SYMBOL_1;
Code[_a] = { fqn: "monocdk.aws_codecommit.Code", version: "1.191.0" };
class PathResolvedCode extends Code {
    constructor(resolvedPath, branch) {
        super();
        this.resolvedPath = resolvedPath;
        this.branch = branch;
    }
    bind(scope) {
        const asset = new assets.Asset(scope, 'PathResolvedCodeAsset', {
            path: this.resolvedPath,
        });
        return (new AssetCode(asset, this.branch)).bind(scope);
    }
}
class AssetCode extends Code {
    constructor(asset, branch) {
        super();
        this.asset = asset;
        this.branch = branch;
    }
    bind(_scope) {
        if (!this.asset.isZipArchive) {
            throw new Error('Asset must be a .zip file or a directory (resolved to: ' + this.asset.assetPath + ' )');
        }
        return {
            code: {
                branchName: this.branch,
                s3: {
                    bucket: this.asset.s3BucketName,
                    key: this.asset.s3ObjectKey,
                },
            },
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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