"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationDeleteStackAction = exports.CloudFormationCreateUpdateStackAction = exports.CloudFormationCreateReplaceChangeSetAction = exports.CloudFormationExecuteChangeSetAction = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudformation = require("../../../aws-cloudformation");
const codepipeline = require("../../../aws-codepipeline");
const iam = require("../../../aws-iam");
const cdk = require("../../../core");
const action_1 = require("../action");
const singleton_policy_1 = require("./private/singleton-policy");
/**
 * Base class for Actions that execute CloudFormation
 */
class CloudFormationAction extends action_1.Action {
    constructor(props, inputs) {
        super({
            ...props,
            provider: 'CloudFormation',
            category: codepipeline.ActionCategory.DEPLOY,
            artifactBounds: {
                minInputs: 0,
                maxInputs: 10,
                minOutputs: 0,
                maxOutputs: 1,
            },
            inputs,
            outputs: props.outputFileName
                ? [props.output || new codepipeline.Artifact(`${props.actionName}_${props.stackName}_Artifact`)]
                : undefined,
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        const singletonPolicy = singleton_policy_1.SingletonPolicy.forRole(options.role);
        if ((this.actionProperties.outputs || []).length > 0) {
            options.bucket.grantReadWrite(singletonPolicy);
        }
        else if ((this.actionProperties.inputs || []).length > 0) {
            options.bucket.grantRead(singletonPolicy);
        }
        return {
            configuration: {
                StackName: this.props.stackName,
                OutputFileName: this.props.outputFileName,
            },
        };
    }
}
/**
 * CodePipeline action to execute a prepared change set.
 */
class CloudFormationExecuteChangeSetAction extends CloudFormationAction {
    constructor(props) {
        super(props, undefined);
        try {
            jsiiDeprecationWarnings.monocdk_aws_codepipeline_actions_CloudFormationExecuteChangeSetActionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CloudFormationExecuteChangeSetAction);
            }
            throw error;
        }
        this.props2 = props;
    }
    bound(scope, stage, options) {
        try {
            jsiiDeprecationWarnings.monocdk_Construct(scope);
            jsiiDeprecationWarnings.monocdk_aws_codepipeline_IStage(stage);
            jsiiDeprecationWarnings.monocdk_aws_codepipeline_ActionBindOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bound);
            }
            throw error;
        }
        singleton_policy_1.SingletonPolicy.forRole(options.role).grantExecuteChangeSet(this.props2);
        const actionConfig = super.bound(scope, stage, options);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'CHANGE_SET_EXECUTE',
                ChangeSetName: this.props2.changeSetName,
            },
        };
    }
}
exports.CloudFormationExecuteChangeSetAction = CloudFormationExecuteChangeSetAction;
_a = JSII_RTTI_SYMBOL_1;
CloudFormationExecuteChangeSetAction[_a] = { fqn: "monocdk.aws_codepipeline_actions.CloudFormationExecuteChangeSetAction", version: "1.191.0" };
/**
 * Base class for all CloudFormation actions that execute or stage deployments.
 */
class CloudFormationDeployAction extends CloudFormationAction {
    constructor(props, inputs) {
        super(props, (props.extraInputs || []).concat(inputs || []));
        this.props2 = props;
    }
    /**
     * Add statement to the service role assumed by CloudFormation while executing this action.
     */
    addToDeploymentRolePolicy(statement) {
        return this.getDeploymentRole('method addToRolePolicy()').addToPolicy(statement);
    }
    get deploymentRole() {
        return this.getDeploymentRole('property role()');
    }
    bound(scope, stage, options) {
        if (this.props2.deploymentRole) {
            this._deploymentRole = this.props2.deploymentRole;
        }
        else {
            const roleStack = cdk.Stack.of(options.role);
            const pipelineStack = cdk.Stack.of(scope);
            if (roleStack.account !== pipelineStack.account) {
                // pass role is not allowed for cross-account access - so,
                // create the deployment Role in the other account!
                this._deploymentRole = new iam.Role(roleStack, `${cdk.Names.nodeUniqueId(stage.pipeline.node)}-${stage.stageName}-${this.actionProperties.actionName}-DeploymentRole`, {
                    assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
                    roleName: cdk.PhysicalName.GENERATE_IF_NEEDED,
                });
            }
            else {
                this._deploymentRole = new iam.Role(scope, 'Role', {
                    assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
                });
            }
            // the deployment role might need read access to the pipeline's bucket
            // (for example, if it's deploying a Lambda function),
            // and even if it has admin permissions, it won't be enough,
            // as it needs to be added to the key's resource policy
            // (and the bucket's, if the access is cross-account)
            options.bucket.grantRead(this._deploymentRole);
            if (this.props2.adminPermissions) {
                this._deploymentRole.addToPolicy(new iam.PolicyStatement({
                    actions: ['*'],
                    resources: ['*'],
                }));
            }
        }
        singleton_policy_1.SingletonPolicy.forRole(options.role).grantPassRole(this._deploymentRole);
        const providedCapabilities = this.props2.cfnCapabilities ??
            this.props2.capabilities?.map(c => {
                switch (c) {
                    case cloudformation.CloudFormationCapabilities.NONE: return cdk.CfnCapabilities.NONE;
                    case cloudformation.CloudFormationCapabilities.ANONYMOUS_IAM: return cdk.CfnCapabilities.ANONYMOUS_IAM;
                    case cloudformation.CloudFormationCapabilities.NAMED_IAM: return cdk.CfnCapabilities.NAMED_IAM;
                    case cloudformation.CloudFormationCapabilities.AUTO_EXPAND: return cdk.CfnCapabilities.AUTO_EXPAND;
                }
            });
        const capabilities = this.props2.adminPermissions && providedCapabilities === undefined
            ? [cdk.CfnCapabilities.NAMED_IAM]
            : providedCapabilities;
        const actionConfig = super.bound(scope, stage, options);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                // None evaluates to empty string which is falsey and results in undefined
                Capabilities: singleton_policy_1.parseCapabilities(capabilities),
                RoleArn: this.deploymentRole.roleArn,
                ParameterOverrides: cdk.Stack.of(scope).toJsonString(this.props2.parameterOverrides),
                TemplateConfiguration: this.props2.templateConfiguration
                    ? this.props2.templateConfiguration.location
                    : undefined,
                StackName: this.props2.stackName,
            },
        };
    }
    getDeploymentRole(member) {
        if (this._deploymentRole) {
            return this._deploymentRole;
        }
        else {
            throw new Error(`Cannot use the ${member} before the Action has been added to a Pipeline`);
        }
    }
}
/**
 * CodePipeline action to prepare a change set.
 *
 * Creates the change set if it doesn't exist based on the stack name and template that you submit.
 * If the change set exists, AWS CloudFormation deletes it, and then creates a new one.
 */
class CloudFormationCreateReplaceChangeSetAction extends CloudFormationDeployAction {
    constructor(props) {
        super(props, props.templateConfiguration
            ? [props.templatePath.artifact, props.templateConfiguration.artifact]
            : [props.templatePath.artifact]);
        try {
            jsiiDeprecationWarnings.monocdk_aws_codepipeline_actions_CloudFormationCreateReplaceChangeSetActionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CloudFormationCreateReplaceChangeSetAction);
            }
            throw error;
        }
        this.props3 = props;
    }
    bound(scope, stage, options) {
        try {
            jsiiDeprecationWarnings.monocdk_Construct(scope);
            jsiiDeprecationWarnings.monocdk_aws_codepipeline_IStage(stage);
            jsiiDeprecationWarnings.monocdk_aws_codepipeline_ActionBindOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bound);
            }
            throw error;
        }
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        singleton_policy_1.SingletonPolicy.forRole(options.role).grantCreateReplaceChangeSet(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'CHANGE_SET_REPLACE',
                ChangeSetName: this.props3.changeSetName,
                TemplatePath: this.props3.templatePath.location,
            },
        };
    }
}
exports.CloudFormationCreateReplaceChangeSetAction = CloudFormationCreateReplaceChangeSetAction;
_b = JSII_RTTI_SYMBOL_1;
CloudFormationCreateReplaceChangeSetAction[_b] = { fqn: "monocdk.aws_codepipeline_actions.CloudFormationCreateReplaceChangeSetAction", version: "1.191.0" };
/**
 * CodePipeline action to deploy a stack.
 *
 * Creates the stack if the specified stack doesn't exist. If the stack exists,
 * AWS CloudFormation updates the stack. Use this action to update existing
 * stacks.
 *
 * AWS CodePipeline won't replace the stack, and will fail deployment if the
 * stack is in a failed state. Use `ReplaceOnFailure` for an action that
 * will delete and recreate the stack to try and recover from failed states.
 *
 * Use this action to automatically replace failed stacks without recovering or
 * troubleshooting them. You would typically choose this mode for testing.
 */
class CloudFormationCreateUpdateStackAction extends CloudFormationDeployAction {
    constructor(props) {
        super(props, props.templateConfiguration
            ? [props.templatePath.artifact, props.templateConfiguration.artifact]
            : [props.templatePath.artifact]);
        try {
            jsiiDeprecationWarnings.monocdk_aws_codepipeline_actions_CloudFormationCreateUpdateStackActionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CloudFormationCreateUpdateStackAction);
            }
            throw error;
        }
        this.props3 = props;
    }
    bound(scope, stage, options) {
        try {
            jsiiDeprecationWarnings.monocdk_Construct(scope);
            jsiiDeprecationWarnings.monocdk_aws_codepipeline_IStage(stage);
            jsiiDeprecationWarnings.monocdk_aws_codepipeline_ActionBindOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bound);
            }
            throw error;
        }
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        singleton_policy_1.SingletonPolicy.forRole(options.role).grantCreateUpdateStack(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: this.props3.replaceOnFailure ? 'REPLACE_ON_FAILURE' : 'CREATE_UPDATE',
                TemplatePath: this.props3.templatePath.location,
            },
        };
    }
}
exports.CloudFormationCreateUpdateStackAction = CloudFormationCreateUpdateStackAction;
_c = JSII_RTTI_SYMBOL_1;
CloudFormationCreateUpdateStackAction[_c] = { fqn: "monocdk.aws_codepipeline_actions.CloudFormationCreateUpdateStackAction", version: "1.191.0" };
/**
 * CodePipeline action to delete a stack.
 *
 * Deletes a stack. If you specify a stack that doesn't exist, the action completes successfully
 * without deleting a stack.
 */
class CloudFormationDeleteStackAction extends CloudFormationDeployAction {
    constructor(props) {
        super(props, undefined);
        try {
            jsiiDeprecationWarnings.monocdk_aws_codepipeline_actions_CloudFormationDeleteStackActionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CloudFormationDeleteStackAction);
            }
            throw error;
        }
        this.props3 = props;
    }
    bound(scope, stage, options) {
        try {
            jsiiDeprecationWarnings.monocdk_Construct(scope);
            jsiiDeprecationWarnings.monocdk_aws_codepipeline_IStage(stage);
            jsiiDeprecationWarnings.monocdk_aws_codepipeline_ActionBindOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bound);
            }
            throw error;
        }
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        singleton_policy_1.SingletonPolicy.forRole(options.role).grantDeleteStack(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'DELETE_ONLY',
            },
        };
    }
}
exports.CloudFormationDeleteStackAction = CloudFormationDeleteStackAction;
_d = JSII_RTTI_SYMBOL_1;
CloudFormationDeleteStackAction[_d] = { fqn: "monocdk.aws_codepipeline_actions.CloudFormationDeleteStackAction", version: "1.191.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicGlwZWxpbmUtYWN0aW9ucy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInBpcGVsaW5lLWFjdGlvbnMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBQUEsOERBQThEO0FBQzlELDBEQUEwRDtBQUMxRCx3Q0FBd0M7QUFDeEMscUNBQXFDO0FBQ3JDLHNDQUFtQztBQUNuQyxpRUFBZ0Y7QUEwRGhGOztHQUVHO0FBQ0gsTUFBZSxvQkFBcUIsU0FBUSxlQUFNO0lBR2hELFlBQVksS0FBZ0MsRUFBRSxNQUEyQztRQUN2RixLQUFLLENBQUM7WUFDSixHQUFHLEtBQUs7WUFDUixRQUFRLEVBQUUsZ0JBQWdCO1lBQzFCLFFBQVEsRUFBRSxZQUFZLENBQUMsY0FBYyxDQUFDLE1BQU07WUFDNUMsY0FBYyxFQUFFO2dCQUNkLFNBQVMsRUFBRSxDQUFDO2dCQUNaLFNBQVMsRUFBRSxFQUFFO2dCQUNiLFVBQVUsRUFBRSxDQUFDO2dCQUNiLFVBQVUsRUFBRSxDQUFDO2FBQ2Q7WUFDRCxNQUFNO1lBQ04sT0FBTyxFQUFFLEtBQUssQ0FBQyxjQUFjO2dCQUMzQixDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsTUFBTSxJQUFJLElBQUksWUFBWSxDQUFDLFFBQVEsQ0FBQyxHQUFHLEtBQUssQ0FBQyxVQUFVLElBQUksS0FBSyxDQUFDLFNBQVMsV0FBVyxDQUFDLENBQUM7Z0JBQ2hHLENBQUMsQ0FBQyxTQUFTO1NBQ2QsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUM7S0FDcEI7SUFFUyxLQUFLLENBQUMsTUFBaUIsRUFBRSxNQUEyQixFQUFFLE9BQXVDO1FBRXJHLE1BQU0sZUFBZSxHQUFHLGtDQUFlLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUU5RCxJQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLE9BQU8sSUFBSSxFQUFFLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ3BELE9BQU8sQ0FBQyxNQUFNLENBQUMsY0FBYyxDQUFDLGVBQWUsQ0FBQyxDQUFDO1NBQ2hEO2FBQU0sSUFBSSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxNQUFNLElBQUksRUFBRSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUMxRCxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxlQUFlLENBQUMsQ0FBQztTQUMzQztRQUVELE9BQU87WUFDTCxhQUFhLEVBQUU7Z0JBQ2IsU0FBUyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUztnQkFDL0IsY0FBYyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYzthQUMxQztTQUNGLENBQUM7S0FDSDtDQUNGO0FBWUQ7O0dBRUc7QUFDSCxNQUFhLG9DQUFxQyxTQUFRLG9CQUFvQjtJQUc1RSxZQUFZLEtBQWdEO1FBQzFELEtBQUssQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7Ozs7OzsrQ0FKZixvQ0FBb0M7Ozs7UUFNN0MsSUFBSSxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUM7S0FDckI7SUFFUyxLQUFLLENBQUMsS0FBZ0IsRUFBRSxLQUEwQixFQUFFLE9BQXVDOzs7Ozs7Ozs7Ozs7UUFFbkcsa0NBQWUsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLHFCQUFxQixDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUV6RSxNQUFNLFlBQVksR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDeEQsT0FBTztZQUNMLEdBQUcsWUFBWTtZQUNmLGFBQWEsRUFBRTtnQkFDYixHQUFHLFlBQVksQ0FBQyxhQUFhO2dCQUM3QixVQUFVLEVBQUUsb0JBQW9CO2dCQUNoQyxhQUFhLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxhQUFhO2FBQ3pDO1NBQ0YsQ0FBQztLQUNIOztBQXRCSCxvRkF1QkM7OztBQWtIRDs7R0FFRztBQUNILE1BQWUsMEJBQTJCLFNBQVEsb0JBQW9CO0lBSXBFLFlBQVksS0FBc0MsRUFBRSxNQUEyQztRQUM3RixLQUFLLENBQUMsS0FBSyxFQUFFLENBQUMsS0FBSyxDQUFDLFdBQVcsSUFBSSxFQUFFLENBQUMsQ0FBQyxNQUFNLENBQUMsTUFBTSxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFFN0QsSUFBSSxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUM7S0FDckI7SUFFRDs7T0FFRztJQUNJLHlCQUF5QixDQUFDLFNBQThCO1FBQzdELE9BQU8sSUFBSSxDQUFDLGlCQUFpQixDQUFDLDBCQUEwQixDQUFDLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxDQUFDO0tBQ2xGO0lBRUQsSUFBVyxjQUFjO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLGlCQUFpQixDQUFDLGlCQUFpQixDQUFDLENBQUM7S0FDbEQ7SUFFUyxLQUFLLENBQUMsS0FBZ0IsRUFBRSxLQUEwQixFQUFFLE9BQXVDO1FBRW5HLElBQUksSUFBSSxDQUFDLE1BQU0sQ0FBQyxjQUFjLEVBQUU7WUFDOUIsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLGNBQWMsQ0FBQztTQUNuRDthQUFNO1lBQ0wsTUFBTSxTQUFTLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzdDLE1BQU0sYUFBYSxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQzFDLElBQUksU0FBUyxDQUFDLE9BQU8sS0FBSyxhQUFhLENBQUMsT0FBTyxFQUFFO2dCQUMvQywwREFBMEQ7Z0JBQzFELG1EQUFtRDtnQkFDbkQsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUMzQyxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxDQUFDLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWdCLENBQUMsVUFBVSxpQkFBaUIsRUFBRTtvQkFDdEgsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLDhCQUE4QixDQUFDO29CQUNuRSxRQUFRLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxrQkFBa0I7aUJBQzlDLENBQUMsQ0FBQzthQUNOO2lCQUFNO2dCQUNMLElBQUksQ0FBQyxlQUFlLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7b0JBQ2pELFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyw4QkFBOEIsQ0FBQztpQkFDcEUsQ0FBQyxDQUFDO2FBQ0o7WUFFRCxzRUFBc0U7WUFDdEUsc0RBQXNEO1lBQ3RELDREQUE0RDtZQUM1RCx1REFBdUQ7WUFDdkQscURBQXFEO1lBQ3JELE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsQ0FBQztZQUUvQyxJQUFJLElBQUksQ0FBQyxNQUFNLENBQUMsZ0JBQWdCLEVBQUU7Z0JBQ2hDLElBQUksQ0FBQyxlQUFlLENBQUMsV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztvQkFDdkQsT0FBTyxFQUFFLENBQUMsR0FBRyxDQUFDO29CQUNkLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztpQkFDakIsQ0FBQyxDQUFDLENBQUM7YUFDTDtTQUNGO1FBRUQsa0NBQWUsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUM7UUFFMUUsTUFBTSxvQkFBb0IsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLGVBQWU7WUFDdEQsSUFBSSxDQUFDLE1BQU0sQ0FBQyxZQUFZLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFO2dCQUNoQyxRQUFRLENBQUMsRUFBRTtvQkFDVCxLQUFLLGNBQWMsQ0FBQywwQkFBMEIsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLEdBQUcsQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDO29CQUNyRixLQUFLLGNBQWMsQ0FBQywwQkFBMEIsQ0FBQyxhQUFhLENBQUMsQ0FBQyxPQUFPLEdBQUcsQ0FBQyxlQUFlLENBQUMsYUFBYSxDQUFDO29CQUN2RyxLQUFLLGNBQWMsQ0FBQywwQkFBMEIsQ0FBQyxTQUFTLENBQUMsQ0FBQyxPQUFPLEdBQUcsQ0FBQyxlQUFlLENBQUMsU0FBUyxDQUFDO29CQUMvRixLQUFLLGNBQWMsQ0FBQywwQkFBMEIsQ0FBQyxXQUFXLENBQUMsQ0FBQyxPQUFPLEdBQUcsQ0FBQyxlQUFlLENBQUMsV0FBVyxDQUFDO2lCQUNwRztZQUNILENBQUMsQ0FBQyxDQUFDO1FBQ0wsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxnQkFBZ0IsSUFBSSxvQkFBb0IsS0FBSyxTQUFTO1lBQ3JGLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxlQUFlLENBQUMsU0FBUyxDQUFDO1lBQ2pDLENBQUMsQ0FBQyxvQkFBb0IsQ0FBQztRQUV6QixNQUFNLFlBQVksR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDeEQsT0FBTztZQUNMLEdBQUcsWUFBWTtZQUNmLGFBQWEsRUFBRTtnQkFDYixHQUFHLFlBQVksQ0FBQyxhQUFhO2dCQUM3QiwwRUFBMEU7Z0JBQzFFLFlBQVksRUFBRSxvQ0FBaUIsQ0FBQyxZQUFZLENBQUM7Z0JBQzdDLE9BQU8sRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLE9BQU87Z0JBQ3BDLGtCQUFrQixFQUFFLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLGtCQUFrQixDQUFDO2dCQUNwRixxQkFBcUIsRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLHFCQUFxQjtvQkFDdEQsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMscUJBQXFCLENBQUMsUUFBUTtvQkFDNUMsQ0FBQyxDQUFDLFNBQVM7Z0JBQ2IsU0FBUyxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsU0FBUzthQUNqQztTQUNGLENBQUM7S0FDSDtJQUVPLGlCQUFpQixDQUFDLE1BQWM7UUFDdEMsSUFBSSxJQUFJLENBQUMsZUFBZSxFQUFFO1lBQ3hCLE9BQU8sSUFBSSxDQUFDLGVBQWUsQ0FBQztTQUM3QjthQUFNO1lBQ0wsTUFBTSxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsTUFBTSxpREFBaUQsQ0FBQyxDQUFDO1NBQzVGO0tBQ0Y7Q0FDRjtBQWlCRDs7Ozs7R0FLRztBQUNILE1BQWEsMENBQTJDLFNBQVEsMEJBQTBCO0lBR3hGLFlBQVksS0FBc0Q7UUFDaEUsS0FBSyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMscUJBQXFCO1lBQ3RDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQyxRQUFRLENBQUM7WUFDckUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDOzs7Ozs7K0NBTjFCLDBDQUEwQzs7OztRQVFuRCxJQUFJLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQztLQUNyQjtJQUVTLEtBQUssQ0FBQyxLQUFnQixFQUFFLEtBQTBCLEVBQUUsT0FBdUM7Ozs7Ozs7Ozs7OztRQUVuRyxtRkFBbUY7UUFDbkYsTUFBTSxZQUFZLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBRXhELGtDQUFlLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQywyQkFBMkIsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFL0UsT0FBTztZQUNMLEdBQUcsWUFBWTtZQUNmLGFBQWEsRUFBRTtnQkFDYixHQUFHLFlBQVksQ0FBQyxhQUFhO2dCQUM3QixVQUFVLEVBQUUsb0JBQW9CO2dCQUNoQyxhQUFhLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxhQUFhO2dCQUN4QyxZQUFZLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxZQUFZLENBQUMsUUFBUTthQUNoRDtTQUNGLENBQUM7S0FDSDs7QUEzQkgsZ0dBNEJDOzs7QUEyQkQ7Ozs7Ozs7Ozs7Ozs7R0FhRztBQUNILE1BQWEscUNBQXNDLFNBQVEsMEJBQTBCO0lBR25GLFlBQVksS0FBaUQ7UUFDM0QsS0FBSyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMscUJBQXFCO1lBQ3RDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQyxRQUFRLENBQUM7WUFDckUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDOzs7Ozs7K0NBTjFCLHFDQUFxQzs7OztRQVE5QyxJQUFJLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQztLQUNyQjtJQUVTLEtBQUssQ0FBQyxLQUFnQixFQUFFLEtBQTBCLEVBQUUsT0FBdUM7Ozs7Ozs7Ozs7OztRQUVuRyxtRkFBbUY7UUFDbkYsTUFBTSxZQUFZLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBRXhELGtDQUFlLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFMUUsT0FBTztZQUNMLEdBQUcsWUFBWTtZQUNmLGFBQWEsRUFBRTtnQkFDYixHQUFHLFlBQVksQ0FBQyxhQUFhO2dCQUM3QixVQUFVLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsb0JBQW9CLENBQUMsQ0FBQyxDQUFDLGVBQWU7Z0JBQ2pGLFlBQVksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLFlBQVksQ0FBQyxRQUFRO2FBQ2hEO1NBQ0YsQ0FBQztLQUNIOztBQTFCSCxzRkEyQkM7OztBQVFEOzs7OztHQUtHO0FBQ0gsTUFBYSwrQkFBZ0MsU0FBUSwwQkFBMEI7SUFHN0UsWUFBWSxLQUEyQztRQUNyRCxLQUFLLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDOzs7Ozs7K0NBSmYsK0JBQStCOzs7O1FBTXhDLElBQUksQ0FBQyxNQUFNLEdBQUcsS0FBSyxDQUFDO0tBQ3JCO0lBRVMsS0FBSyxDQUFDLEtBQWdCLEVBQUUsS0FBMEIsRUFBRSxPQUF1Qzs7Ozs7Ozs7Ozs7O1FBRW5HLG1GQUFtRjtRQUNuRixNQUFNLFlBQVksR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFFeEQsa0NBQWUsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUVwRSxPQUFPO1lBQ0wsR0FBRyxZQUFZO1lBQ2YsYUFBYSxFQUFFO2dCQUNiLEdBQUcsWUFBWSxDQUFDLGFBQWE7Z0JBQzdCLFVBQVUsRUFBRSxhQUFhO2FBQzFCO1NBQ0YsQ0FBQztLQUNIOztBQXZCSCwwRUF3QkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjbG91ZGZvcm1hdGlvbiBmcm9tICcuLi8uLi8uLi9hd3MtY2xvdWRmb3JtYXRpb24nO1xuaW1wb3J0ICogYXMgY29kZXBpcGVsaW5lIGZyb20gJy4uLy4uLy4uL2F3cy1jb2RlcGlwZWxpbmUnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJy4uLy4uLy4uL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJy4uLy4uLy4uL2NvcmUnO1xuaW1wb3J0IHsgQWN0aW9uIH0gZnJvbSAnLi4vYWN0aW9uJztcbmltcG9ydCB7IHBhcnNlQ2FwYWJpbGl0aWVzLCBTaW5nbGV0b25Qb2xpY3kgfSBmcm9tICcuL3ByaXZhdGUvc2luZ2xldG9uLXBvbGljeSc7XG5cbi8vIGtlZXAgdGhpcyBpbXBvcnQgc2VwYXJhdGUgZnJvbSBvdGhlciBpbXBvcnRzIHRvIHJlZHVjZSBjaGFuY2UgZm9yIG1lcmdlIGNvbmZsaWN0cyB3aXRoIHYyLW1haW5cbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1kdXBsaWNhdGUtaW1wb3J0cywgaW1wb3J0L29yZGVyXG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICcuLi8uLi8uLi9jb3JlJztcblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGNvbW1vbiB0byBhbGwgQ2xvdWRGb3JtYXRpb24gYWN0aW9uc1xuICovXG5pbnRlcmZhY2UgQ2xvdWRGb3JtYXRpb25BY3Rpb25Qcm9wcyBleHRlbmRzIGNvZGVwaXBlbGluZS5Db21tb25Bd3NBY3Rpb25Qcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgc3RhY2sgdG8gYXBwbHkgdGhpcyBhY3Rpb24gdG9cbiAgICovXG4gIHJlYWRvbmx5IHN0YWNrTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBIG5hbWUgZm9yIHRoZSBmaWxlbmFtZSBpbiB0aGUgb3V0cHV0IGFydGlmYWN0IHRvIHN0b3JlIHRoZSBBV1MgQ2xvdWRGb3JtYXRpb24gY2FsbCdzIHJlc3VsdC5cbiAgICpcbiAgICogVGhlIGZpbGUgd2lsbCBjb250YWluIHRoZSByZXN1bHQgb2YgdGhlIGNhbGwgdG8gQVdTIENsb3VkRm9ybWF0aW9uIChmb3IgZXhhbXBsZVxuICAgKiB0aGUgY2FsbCB0byBVcGRhdGVTdGFjayBvciBDcmVhdGVDaGFuZ2VTZXQpLlxuICAgKlxuICAgKiBBV1MgQ29kZVBpcGVsaW5lIGFkZHMgdGhlIGZpbGUgdG8gdGhlIG91dHB1dCBhcnRpZmFjdCBhZnRlciBwZXJmb3JtaW5nXG4gICAqIHRoZSBzcGVjaWZpZWQgYWN0aW9uLlxuICAgKlxuICAgKiBAZGVmYXVsdCBObyBvdXRwdXQgYXJ0aWZhY3QgZ2VuZXJhdGVkXG4gICAqL1xuICByZWFkb25seSBvdXRwdXRGaWxlTmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIG91dHB1dCBhcnRpZmFjdCB0byBnZW5lcmF0ZVxuICAgKlxuICAgKiBPbmx5IGFwcGxpZWQgaWYgYG91dHB1dEZpbGVOYW1lYCBpcyBzZXQgYXMgd2VsbC5cbiAgICpcbiAgICogQGRlZmF1bHQgQXV0b21hdGljYWxseSBnZW5lcmF0ZWQgYXJ0aWZhY3QgbmFtZS5cbiAgICovXG4gIHJlYWRvbmx5IG91dHB1dD86IGNvZGVwaXBlbGluZS5BcnRpZmFjdDtcblxuICAvKipcbiAgICogVGhlIEFXUyByZWdpb24gdGhlIGdpdmVuIEFjdGlvbiByZXNpZGVzIGluLlxuICAgKiBOb3RlIHRoYXQgYSBjcm9zcy1yZWdpb24gUGlwZWxpbmUgcmVxdWlyZXMgcmVwbGljYXRpb24gYnVja2V0cyB0byBmdW5jdGlvbiBjb3JyZWN0bHkuXG4gICAqIFlvdSBjYW4gcHJvdmlkZSB0aGVpciBuYW1lcyB3aXRoIHRoZSB7QGxpbmsgUGlwZWxpbmVQcm9wcyNjcm9zc1JlZ2lvblJlcGxpY2F0aW9uQnVja2V0c30gcHJvcGVydHkuXG4gICAqIElmIHlvdSBkb24ndCwgdGhlIENvZGVQaXBlbGluZSBDb25zdHJ1Y3Qgd2lsbCBjcmVhdGUgbmV3IFN0YWNrcyBpbiB5b3VyIENESyBhcHAgY29udGFpbmluZyB0aG9zZSBidWNrZXRzLFxuICAgKiB0aGF0IHlvdSB3aWxsIG5lZWQgdG8gYGNkayBkZXBsb3lgIGJlZm9yZSBkZXBsb3lpbmcgdGhlIG1haW4sIFBpcGVsaW5lLWNvbnRhaW5pbmcgU3RhY2suXG4gICAqXG4gICAqIEBkZWZhdWx0IHRoZSBBY3Rpb24gcmVzaWRlcyBpbiB0aGUgc2FtZSByZWdpb24gYXMgdGhlIFBpcGVsaW5lXG4gICAqL1xuICByZWFkb25seSByZWdpb24/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBBV1MgYWNjb3VudCB0aGlzIEFjdGlvbiBpcyBzdXBwb3NlZCB0byBvcGVyYXRlIGluLlxuICAgKiAqKk5vdGUqKjogaWYgeW91IHNwZWNpZnkgdGhlIGByb2xlYCBwcm9wZXJ0eSxcbiAgICogdGhpcyBpcyBpZ25vcmVkIC0gdGhlIGFjdGlvbiB3aWxsIG9wZXJhdGUgaW4gdGhlIHNhbWUgcmVnaW9uIHRoZSBwYXNzZWQgcm9sZSBkb2VzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGFjdGlvbiByZXNpZGVzIGluIHRoZSBzYW1lIGFjY291bnQgYXMgdGhlIHBpcGVsaW5lXG4gICAqL1xuICByZWFkb25seSBhY2NvdW50Pzogc3RyaW5nO1xufVxuXG4vKipcbiAqIEJhc2UgY2xhc3MgZm9yIEFjdGlvbnMgdGhhdCBleGVjdXRlIENsb3VkRm9ybWF0aW9uXG4gKi9cbmFic3RyYWN0IGNsYXNzIENsb3VkRm9ybWF0aW9uQWN0aW9uIGV4dGVuZHMgQWN0aW9uIHtcbiAgcHJpdmF0ZSByZWFkb25seSBwcm9wczogQ2xvdWRGb3JtYXRpb25BY3Rpb25Qcm9wcztcblxuICBjb25zdHJ1Y3Rvcihwcm9wczogQ2xvdWRGb3JtYXRpb25BY3Rpb25Qcm9wcywgaW5wdXRzOiBjb2RlcGlwZWxpbmUuQXJ0aWZhY3RbXSB8IHVuZGVmaW5lZCkge1xuICAgIHN1cGVyKHtcbiAgICAgIC4uLnByb3BzLFxuICAgICAgcHJvdmlkZXI6ICdDbG91ZEZvcm1hdGlvbicsXG4gICAgICBjYXRlZ29yeTogY29kZXBpcGVsaW5lLkFjdGlvbkNhdGVnb3J5LkRFUExPWSxcbiAgICAgIGFydGlmYWN0Qm91bmRzOiB7XG4gICAgICAgIG1pbklucHV0czogMCxcbiAgICAgICAgbWF4SW5wdXRzOiAxMCxcbiAgICAgICAgbWluT3V0cHV0czogMCxcbiAgICAgICAgbWF4T3V0cHV0czogMSxcbiAgICAgIH0sXG4gICAgICBpbnB1dHMsXG4gICAgICBvdXRwdXRzOiBwcm9wcy5vdXRwdXRGaWxlTmFtZVxuICAgICAgICA/IFtwcm9wcy5vdXRwdXQgfHwgbmV3IGNvZGVwaXBlbGluZS5BcnRpZmFjdChgJHtwcm9wcy5hY3Rpb25OYW1lfV8ke3Byb3BzLnN0YWNrTmFtZX1fQXJ0aWZhY3RgKV1cbiAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgfSk7XG5cbiAgICB0aGlzLnByb3BzID0gcHJvcHM7XG4gIH1cblxuICBwcm90ZWN0ZWQgYm91bmQoX3Njb3BlOiBDb25zdHJ1Y3QsIF9zdGFnZTogY29kZXBpcGVsaW5lLklTdGFnZSwgb3B0aW9uczogY29kZXBpcGVsaW5lLkFjdGlvbkJpbmRPcHRpb25zKTpcbiAgY29kZXBpcGVsaW5lLkFjdGlvbkNvbmZpZyB7XG4gICAgY29uc3Qgc2luZ2xldG9uUG9saWN5ID0gU2luZ2xldG9uUG9saWN5LmZvclJvbGUob3B0aW9ucy5yb2xlKTtcblxuICAgIGlmICgodGhpcy5hY3Rpb25Qcm9wZXJ0aWVzLm91dHB1dHMgfHwgW10pLmxlbmd0aCA+IDApIHtcbiAgICAgIG9wdGlvbnMuYnVja2V0LmdyYW50UmVhZFdyaXRlKHNpbmdsZXRvblBvbGljeSk7XG4gICAgfSBlbHNlIGlmICgodGhpcy5hY3Rpb25Qcm9wZXJ0aWVzLmlucHV0cyB8fCBbXSkubGVuZ3RoID4gMCkge1xuICAgICAgb3B0aW9ucy5idWNrZXQuZ3JhbnRSZWFkKHNpbmdsZXRvblBvbGljeSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHtcbiAgICAgIGNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgU3RhY2tOYW1lOiB0aGlzLnByb3BzLnN0YWNrTmFtZSxcbiAgICAgICAgT3V0cHV0RmlsZU5hbWU6IHRoaXMucHJvcHMub3V0cHV0RmlsZU5hbWUsXG4gICAgICB9LFxuICAgIH07XG4gIH1cbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciB0aGUgQ2xvdWRGb3JtYXRpb25FeGVjdXRlQ2hhbmdlU2V0QWN0aW9uLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENsb3VkRm9ybWF0aW9uRXhlY3V0ZUNoYW5nZVNldEFjdGlvblByb3BzIGV4dGVuZHMgQ2xvdWRGb3JtYXRpb25BY3Rpb25Qcm9wcyB7XG4gIC8qKlxuICAgKiBOYW1lIG9mIHRoZSBjaGFuZ2Ugc2V0IHRvIGV4ZWN1dGUuXG4gICAqL1xuICByZWFkb25seSBjaGFuZ2VTZXROYW1lOiBzdHJpbmc7XG59XG5cbi8qKlxuICogQ29kZVBpcGVsaW5lIGFjdGlvbiB0byBleGVjdXRlIGEgcHJlcGFyZWQgY2hhbmdlIHNldC5cbiAqL1xuZXhwb3J0IGNsYXNzIENsb3VkRm9ybWF0aW9uRXhlY3V0ZUNoYW5nZVNldEFjdGlvbiBleHRlbmRzIENsb3VkRm9ybWF0aW9uQWN0aW9uIHtcbiAgcHJpdmF0ZSByZWFkb25seSBwcm9wczI6IENsb3VkRm9ybWF0aW9uRXhlY3V0ZUNoYW5nZVNldEFjdGlvblByb3BzO1xuXG4gIGNvbnN0cnVjdG9yKHByb3BzOiBDbG91ZEZvcm1hdGlvbkV4ZWN1dGVDaGFuZ2VTZXRBY3Rpb25Qcm9wcykge1xuICAgIHN1cGVyKHByb3BzLCB1bmRlZmluZWQpO1xuXG4gICAgdGhpcy5wcm9wczIgPSBwcm9wcztcbiAgfVxuXG4gIHByb3RlY3RlZCBib3VuZChzY29wZTogQ29uc3RydWN0LCBzdGFnZTogY29kZXBpcGVsaW5lLklTdGFnZSwgb3B0aW9uczogY29kZXBpcGVsaW5lLkFjdGlvbkJpbmRPcHRpb25zKTpcbiAgY29kZXBpcGVsaW5lLkFjdGlvbkNvbmZpZyB7XG4gICAgU2luZ2xldG9uUG9saWN5LmZvclJvbGUob3B0aW9ucy5yb2xlKS5ncmFudEV4ZWN1dGVDaGFuZ2VTZXQodGhpcy5wcm9wczIpO1xuXG4gICAgY29uc3QgYWN0aW9uQ29uZmlnID0gc3VwZXIuYm91bmQoc2NvcGUsIHN0YWdlLCBvcHRpb25zKTtcbiAgICByZXR1cm4ge1xuICAgICAgLi4uYWN0aW9uQ29uZmlnLFxuICAgICAgY29uZmlndXJhdGlvbjoge1xuICAgICAgICAuLi5hY3Rpb25Db25maWcuY29uZmlndXJhdGlvbixcbiAgICAgICAgQWN0aW9uTW9kZTogJ0NIQU5HRV9TRVRfRVhFQ1VURScsXG4gICAgICAgIENoYW5nZVNldE5hbWU6IHRoaXMucHJvcHMyLmNoYW5nZVNldE5hbWUsXG4gICAgICB9LFxuICAgIH07XG4gIH1cbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGNvbW1vbiB0byBDbG91ZEZvcm1hdGlvbiBhY3Rpb25zIHRoYXQgc3RhZ2UgZGVwbG95bWVudHNcbiAqL1xuaW50ZXJmYWNlIENsb3VkRm9ybWF0aW9uRGVwbG95QWN0aW9uUHJvcHMgZXh0ZW5kcyBDbG91ZEZvcm1hdGlvbkFjdGlvblByb3BzIHtcbiAgLyoqXG4gICAqIElBTSByb2xlIHRvIGFzc3VtZSB3aGVuIGRlcGxveWluZyBjaGFuZ2VzLlxuICAgKlxuICAgKiBJZiBub3Qgc3BlY2lmaWVkLCBhIGZyZXNoIHJvbGUgaXMgY3JlYXRlZC4gVGhlIHJvbGUgaXMgY3JlYXRlZCB3aXRoIHplcm9cbiAgICogcGVybWlzc2lvbnMgdW5sZXNzIGBhZG1pblBlcm1pc3Npb25zYCBpcyB0cnVlLCBpbiB3aGljaCBjYXNlIHRoZSByb2xlIHdpbGwgaGF2ZVxuICAgKiBmdWxsIHBlcm1pc3Npb25zLlxuICAgKlxuICAgKiBAZGVmYXVsdCBBIGZyZXNoIHJvbGUgd2l0aCBmdWxsIG9yIG5vIHBlcm1pc3Npb25zIChkZXBlbmRpbmcgb24gdGhlIHZhbHVlIG9mIGBhZG1pblBlcm1pc3Npb25zYCkuXG4gICAqL1xuICByZWFkb25seSBkZXBsb3ltZW50Um9sZT86IGlhbS5JUm9sZTtcblxuICAvKipcbiAgICogQWNrbm93bGVkZ2UgY2VydGFpbiBjaGFuZ2VzIG1hZGUgYXMgcGFydCBvZiBkZXBsb3ltZW50XG4gICAqXG4gICAqIEZvciBzdGFja3MgdGhhdCBjb250YWluIGNlcnRhaW4gcmVzb3VyY2VzLCBleHBsaWNpdCBhY2tub3dsZWRnZW1lbnQgdGhhdCBBV1MgQ2xvdWRGb3JtYXRpb25cbiAgICogbWlnaHQgY3JlYXRlIG9yIHVwZGF0ZSB0aG9zZSByZXNvdXJjZXMuIEZvciBleGFtcGxlLCB5b3UgbXVzdCBzcGVjaWZ5IGBBbm9ueW1vdXNJQU1gIG9yIGBOYW1lZElBTWBcbiAgICogaWYgeW91ciBzdGFjayB0ZW1wbGF0ZSBjb250YWlucyBBV1MgSWRlbnRpdHkgYW5kIEFjY2VzcyBNYW5hZ2VtZW50IChJQU0pIHJlc291cmNlcy4gRm9yIG1vcmVcbiAgICogaW5mb3JtYXRpb24gc2VlIHRoZSBsaW5rIGJlbG93LlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL3VzaW5nLWlhbS10ZW1wbGF0ZS5odG1sI3VzaW5nLWlhbS1jYXBhYmlsaXRpZXNcbiAgICogQGRlZmF1bHQgTm9uZSwgdW5sZXNzIGBhZG1pblBlcm1pc3Npb25zYCBpcyB0cnVlXG4gICAqIEBkZXByZWNhdGVkIHVzZSB7QGxpbmsgY2ZuQ2FwYWJpbGl0aWVzfSBpbnN0ZWFkXG4gICAqL1xuICByZWFkb25seSBjYXBhYmlsaXRpZXM/OiBjbG91ZGZvcm1hdGlvbi5DbG91ZEZvcm1hdGlvbkNhcGFiaWxpdGllc1tdO1xuXG4gIC8qKlxuICAgKiBBY2tub3dsZWRnZSBjZXJ0YWluIGNoYW5nZXMgbWFkZSBhcyBwYXJ0IG9mIGRlcGxveW1lbnQuXG4gICAqXG4gICAqIEZvciBzdGFja3MgdGhhdCBjb250YWluIGNlcnRhaW4gcmVzb3VyY2VzLFxuICAgKiBleHBsaWNpdCBhY2tub3dsZWRnZW1lbnQgaXMgcmVxdWlyZWQgdGhhdCBBV1MgQ2xvdWRGb3JtYXRpb24gbWlnaHQgY3JlYXRlIG9yIHVwZGF0ZSB0aG9zZSByZXNvdXJjZXMuXG4gICAqIEZvciBleGFtcGxlLCB5b3UgbXVzdCBzcGVjaWZ5IGBBTk9OWU1PVVNfSUFNYCBvciBgTkFNRURfSUFNYCBpZiB5b3VyIHN0YWNrIHRlbXBsYXRlIGNvbnRhaW5zIEFXU1xuICAgKiBJZGVudGl0eSBhbmQgQWNjZXNzIE1hbmFnZW1lbnQgKElBTSkgcmVzb3VyY2VzLlxuICAgKiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgc2VlIHRoZSBsaW5rIGJlbG93LlxuICAgKlxuICAgKiBAZGVmYXVsdCBOb25lLCB1bmxlc3MgYGFkbWluUGVybWlzc2lvbnNgIGlzIHRydWVcbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTQ2xvdWRGb3JtYXRpb24vbGF0ZXN0L1VzZXJHdWlkZS91c2luZy1pYW0tdGVtcGxhdGUuaHRtbCN1c2luZy1pYW0tY2FwYWJpbGl0aWVzXG4gICAqL1xuICByZWFkb25seSBjZm5DYXBhYmlsaXRpZXM/OiBjZGsuQ2ZuQ2FwYWJpbGl0aWVzW107XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdG8gZ3JhbnQgZnVsbCBwZXJtaXNzaW9ucyB0byBDbG91ZEZvcm1hdGlvbiB3aGlsZSBkZXBsb3lpbmcgdGhpcyB0ZW1wbGF0ZS5cbiAgICpcbiAgICogU2V0dGluZyB0aGlzIHRvIGB0cnVlYCBhZmZlY3RzIHRoZSBkZWZhdWx0cyBmb3IgYHJvbGVgIGFuZCBgY2FwYWJpbGl0aWVzYCwgaWYgeW91XG4gICAqIGRvbid0IHNwZWNpZnkgYW55IGFsdGVybmF0aXZlcy5cbiAgICpcbiAgICogVGhlIGRlZmF1bHQgcm9sZSB0aGF0IHdpbGwgYmUgY3JlYXRlZCBmb3IgeW91IHdpbGwgaGF2ZSBmdWxsIChpLmUuLCBgKmApXG4gICAqIHBlcm1pc3Npb25zIG9uIGFsbCByZXNvdXJjZXMsIGFuZCB0aGUgZGVwbG95bWVudCB3aWxsIGhhdmUgbmFtZWQgSUFNXG4gICAqIGNhcGFiaWxpdGllcyAoaS5lLiwgYWJsZSB0byBjcmVhdGUgYWxsIElBTSByZXNvdXJjZXMpLlxuICAgKlxuICAgKiBUaGlzIGlzIGEgc2hvcnRoYW5kIHRoYXQgeW91IGNhbiB1c2UgaWYgeW91IGZ1bGx5IHRydXN0IHRoZSB0ZW1wbGF0ZXMgdGhhdFxuICAgKiBhcmUgZGVwbG95ZWQgaW4gdGhpcyBwaXBlbGluZS4gSWYgeW91IHdhbnQgbW9yZSBmaW5lLWdyYWluZWQgcGVybWlzc2lvbnMsXG4gICAqIHVzZSBgYWRkVG9Sb2xlUG9saWN5YCBhbmQgYGNhcGFiaWxpdGllc2AgdG8gY29udHJvbCB3aGF0IHRoZSBDbG91ZEZvcm1hdGlvblxuICAgKiBkZXBsb3ltZW50IGlzIGFsbG93ZWQgdG8gZG8uXG4gICAqL1xuICByZWFkb25seSBhZG1pblBlcm1pc3Npb25zOiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBJbnB1dCBhcnRpZmFjdCB0byB1c2UgZm9yIHRlbXBsYXRlIHBhcmFtZXRlcnMgdmFsdWVzIGFuZCBzdGFjayBwb2xpY3kuXG4gICAqXG4gICAqIFRoZSB0ZW1wbGF0ZSBjb25maWd1cmF0aW9uIGZpbGUgc2hvdWxkIGNvbnRhaW4gYSBKU09OIG9iamVjdCB0aGF0IHNob3VsZCBsb29rIGxpa2UgdGhpczpcbiAgICogYHsgXCJQYXJhbWV0ZXJzXCI6IHsuLi59LCBcIlRhZ3NcIjogey4uLn0sIFwiU3RhY2tQb2xpY3lcIjogey4uLiB9fWAuIEZvciBtb3JlIGluZm9ybWF0aW9uLFxuICAgKiBzZWUgW0FXUyBDbG91ZEZvcm1hdGlvbiBBcnRpZmFjdHNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2NvbnRpbnVvdXMtZGVsaXZlcnktY29kZXBpcGVsaW5lLWNmbi1hcnRpZmFjdHMuaHRtbCkuXG4gICAqXG4gICAqIE5vdGUgdGhhdCBpZiB5b3UgaW5jbHVkZSBzZW5zaXRpdmUgaW5mb3JtYXRpb24sIHN1Y2ggYXMgcGFzc3dvcmRzLCByZXN0cmljdCBhY2Nlc3MgdG8gdGhpc1xuICAgKiBmaWxlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBObyB0ZW1wbGF0ZSBjb25maWd1cmF0aW9uIGJhc2VkIG9uIGlucHV0IGFydGlmYWN0c1xuICAgKi9cbiAgcmVhZG9ubHkgdGVtcGxhdGVDb25maWd1cmF0aW9uPzogY29kZXBpcGVsaW5lLkFydGlmYWN0UGF0aDtcblxuICAvKipcbiAgICogQWRkaXRpb25hbCB0ZW1wbGF0ZSBwYXJhbWV0ZXJzLlxuICAgKlxuICAgKiBUZW1wbGF0ZSBwYXJhbWV0ZXJzIHNwZWNpZmllZCBoZXJlIHRha2UgcHJlY2VkZW5jZSBvdmVyIHRlbXBsYXRlIHBhcmFtZXRlcnNcbiAgICogZm91bmQgaW4gdGhlIGFydGlmYWN0IHNwZWNpZmllZCBieSB0aGUgYHRlbXBsYXRlQ29uZmlndXJhdGlvbmAgcHJvcGVydHkuXG4gICAqXG4gICAqIFdlIHJlY29tbWVuZCB0aGF0IHlvdSB1c2UgdGhlIHRlbXBsYXRlIGNvbmZpZ3VyYXRpb24gZmlsZSB0byBzcGVjaWZ5XG4gICAqIG1vc3Qgb2YgeW91ciBwYXJhbWV0ZXIgdmFsdWVzLiBVc2UgcGFyYW1ldGVyIG92ZXJyaWRlcyB0byBzcGVjaWZ5IG9ubHlcbiAgICogZHluYW1pYyBwYXJhbWV0ZXIgdmFsdWVzICh2YWx1ZXMgdGhhdCBhcmUgdW5rbm93biB1bnRpbCB5b3UgcnVuIHRoZVxuICAgKiBwaXBlbGluZSkuXG4gICAqXG4gICAqIEFsbCBwYXJhbWV0ZXIgbmFtZXMgbXVzdCBiZSBwcmVzZW50IGluIHRoZSBzdGFjayB0ZW1wbGF0ZS5cbiAgICpcbiAgICogTm90ZTogdGhlIGVudGlyZSBvYmplY3QgY2Fubm90IGJlIG1vcmUgdGhhbiAxa0IuXG4gICAqXG4gICAqIEBkZWZhdWx0IE5vIG92ZXJyaWRlc1xuICAgKi9cbiAgcmVhZG9ubHkgcGFyYW1ldGVyT3ZlcnJpZGVzPzogeyBbbmFtZTogc3RyaW5nXTogYW55IH07XG5cbiAgLyoqXG4gICAqIFRoZSBsaXN0IG9mIGFkZGl0aW9uYWwgaW5wdXQgQXJ0aWZhY3RzIGZvciB0aGlzIEFjdGlvbi5cbiAgICogVGhpcyBpcyBlc3BlY2lhbGx5IHVzZWZ1bCB3aGVuIHVzZWQgaW4gY29uanVuY3Rpb24gd2l0aCB0aGUgYHBhcmFtZXRlck92ZXJyaWRlc2AgcHJvcGVydHkuXG4gICAqIEZvciBleGFtcGxlLCBpZiB5b3UgaGF2ZTpcbiAgICpcbiAgICogICBwYXJhbWV0ZXJPdmVycmlkZXM6IHtcbiAgICogICAgICdQYXJhbTEnOiBhY3Rpb24xLm91dHB1dEFydGlmYWN0LmJ1Y2tldE5hbWUsXG4gICAqICAgICAnUGFyYW0yJzogYWN0aW9uMi5vdXRwdXRBcnRpZmFjdC5vYmplY3RLZXksXG4gICAqICAgfVxuICAgKlxuICAgKiAsIGlmIHRoZSBvdXRwdXQgQXJ0aWZhY3RzIG9mIGBhY3Rpb24xYCBhbmQgYGFjdGlvbjJgIHdlcmUgbm90IHVzZWQgdG9cbiAgICogc2V0IGVpdGhlciB0aGUgYHRlbXBsYXRlQ29uZmlndXJhdGlvbmAgb3IgdGhlIGB0ZW1wbGF0ZVBhdGhgIHByb3BlcnRpZXMsXG4gICAqIHlvdSBuZWVkIHRvIG1ha2Ugc3VyZSB0byBpbmNsdWRlIHRoZW0gaW4gdGhlIGBleHRyYUlucHV0c2AgLVxuICAgKiBvdGhlcndpc2UsIHlvdSdsbCBnZXQgYW4gXCJ1bnJlY29nbml6ZWQgQXJ0aWZhY3RcIiBlcnJvciBkdXJpbmcgeW91ciBQaXBlbGluZSdzIGV4ZWN1dGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGV4dHJhSW5wdXRzPzogY29kZXBpcGVsaW5lLkFydGlmYWN0W107XG59XG5cbi8qKlxuICogQmFzZSBjbGFzcyBmb3IgYWxsIENsb3VkRm9ybWF0aW9uIGFjdGlvbnMgdGhhdCBleGVjdXRlIG9yIHN0YWdlIGRlcGxveW1lbnRzLlxuICovXG5hYnN0cmFjdCBjbGFzcyBDbG91ZEZvcm1hdGlvbkRlcGxveUFjdGlvbiBleHRlbmRzIENsb3VkRm9ybWF0aW9uQWN0aW9uIHtcbiAgcHJpdmF0ZSBfZGVwbG95bWVudFJvbGU/OiBpYW0uSVJvbGU7XG4gIHByaXZhdGUgcmVhZG9ubHkgcHJvcHMyOiBDbG91ZEZvcm1hdGlvbkRlcGxveUFjdGlvblByb3BzO1xuXG4gIGNvbnN0cnVjdG9yKHByb3BzOiBDbG91ZEZvcm1hdGlvbkRlcGxveUFjdGlvblByb3BzLCBpbnB1dHM6IGNvZGVwaXBlbGluZS5BcnRpZmFjdFtdIHwgdW5kZWZpbmVkKSB7XG4gICAgc3VwZXIocHJvcHMsIChwcm9wcy5leHRyYUlucHV0cyB8fCBbXSkuY29uY2F0KGlucHV0cyB8fCBbXSkpO1xuXG4gICAgdGhpcy5wcm9wczIgPSBwcm9wcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgc3RhdGVtZW50IHRvIHRoZSBzZXJ2aWNlIHJvbGUgYXNzdW1lZCBieSBDbG91ZEZvcm1hdGlvbiB3aGlsZSBleGVjdXRpbmcgdGhpcyBhY3Rpb24uXG4gICAqL1xuICBwdWJsaWMgYWRkVG9EZXBsb3ltZW50Um9sZVBvbGljeShzdGF0ZW1lbnQ6IGlhbS5Qb2xpY3lTdGF0ZW1lbnQpIHtcbiAgICByZXR1cm4gdGhpcy5nZXREZXBsb3ltZW50Um9sZSgnbWV0aG9kIGFkZFRvUm9sZVBvbGljeSgpJykuYWRkVG9Qb2xpY3koc3RhdGVtZW50KTtcbiAgfVxuXG4gIHB1YmxpYyBnZXQgZGVwbG95bWVudFJvbGUoKTogaWFtLklSb2xlIHtcbiAgICByZXR1cm4gdGhpcy5nZXREZXBsb3ltZW50Um9sZSgncHJvcGVydHkgcm9sZSgpJyk7XG4gIH1cblxuICBwcm90ZWN0ZWQgYm91bmQoc2NvcGU6IENvbnN0cnVjdCwgc3RhZ2U6IGNvZGVwaXBlbGluZS5JU3RhZ2UsIG9wdGlvbnM6IGNvZGVwaXBlbGluZS5BY3Rpb25CaW5kT3B0aW9ucyk6XG4gIGNvZGVwaXBlbGluZS5BY3Rpb25Db25maWcge1xuICAgIGlmICh0aGlzLnByb3BzMi5kZXBsb3ltZW50Um9sZSkge1xuICAgICAgdGhpcy5fZGVwbG95bWVudFJvbGUgPSB0aGlzLnByb3BzMi5kZXBsb3ltZW50Um9sZTtcbiAgICB9IGVsc2Uge1xuICAgICAgY29uc3Qgcm9sZVN0YWNrID0gY2RrLlN0YWNrLm9mKG9wdGlvbnMucm9sZSk7XG4gICAgICBjb25zdCBwaXBlbGluZVN0YWNrID0gY2RrLlN0YWNrLm9mKHNjb3BlKTtcbiAgICAgIGlmIChyb2xlU3RhY2suYWNjb3VudCAhPT0gcGlwZWxpbmVTdGFjay5hY2NvdW50KSB7XG4gICAgICAgIC8vIHBhc3Mgcm9sZSBpcyBub3QgYWxsb3dlZCBmb3IgY3Jvc3MtYWNjb3VudCBhY2Nlc3MgLSBzbyxcbiAgICAgICAgLy8gY3JlYXRlIHRoZSBkZXBsb3ltZW50IFJvbGUgaW4gdGhlIG90aGVyIGFjY291bnQhXG4gICAgICAgIHRoaXMuX2RlcGxveW1lbnRSb2xlID0gbmV3IGlhbS5Sb2xlKHJvbGVTdGFjayxcbiAgICAgICAgICBgJHtjZGsuTmFtZXMubm9kZVVuaXF1ZUlkKHN0YWdlLnBpcGVsaW5lLm5vZGUpfS0ke3N0YWdlLnN0YWdlTmFtZX0tJHt0aGlzLmFjdGlvblByb3BlcnRpZXMuYWN0aW9uTmFtZX0tRGVwbG95bWVudFJvbGVgLCB7XG4gICAgICAgICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnY2xvdWRmb3JtYXRpb24uYW1hem9uYXdzLmNvbScpLFxuICAgICAgICAgICAgcm9sZU5hbWU6IGNkay5QaHlzaWNhbE5hbWUuR0VORVJBVEVfSUZfTkVFREVELFxuICAgICAgICAgIH0pO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgdGhpcy5fZGVwbG95bWVudFJvbGUgPSBuZXcgaWFtLlJvbGUoc2NvcGUsICdSb2xlJywge1xuICAgICAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdjbG91ZGZvcm1hdGlvbi5hbWF6b25hd3MuY29tJyksXG4gICAgICAgIH0pO1xuICAgICAgfVxuXG4gICAgICAvLyB0aGUgZGVwbG95bWVudCByb2xlIG1pZ2h0IG5lZWQgcmVhZCBhY2Nlc3MgdG8gdGhlIHBpcGVsaW5lJ3MgYnVja2V0XG4gICAgICAvLyAoZm9yIGV4YW1wbGUsIGlmIGl0J3MgZGVwbG95aW5nIGEgTGFtYmRhIGZ1bmN0aW9uKSxcbiAgICAgIC8vIGFuZCBldmVuIGlmIGl0IGhhcyBhZG1pbiBwZXJtaXNzaW9ucywgaXQgd29uJ3QgYmUgZW5vdWdoLFxuICAgICAgLy8gYXMgaXQgbmVlZHMgdG8gYmUgYWRkZWQgdG8gdGhlIGtleSdzIHJlc291cmNlIHBvbGljeVxuICAgICAgLy8gKGFuZCB0aGUgYnVja2V0J3MsIGlmIHRoZSBhY2Nlc3MgaXMgY3Jvc3MtYWNjb3VudClcbiAgICAgIG9wdGlvbnMuYnVja2V0LmdyYW50UmVhZCh0aGlzLl9kZXBsb3ltZW50Um9sZSk7XG5cbiAgICAgIGlmICh0aGlzLnByb3BzMi5hZG1pblBlcm1pc3Npb25zKSB7XG4gICAgICAgIHRoaXMuX2RlcGxveW1lbnRSb2xlLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBhY3Rpb25zOiBbJyonXSxcbiAgICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgICB9KSk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgU2luZ2xldG9uUG9saWN5LmZvclJvbGUob3B0aW9ucy5yb2xlKS5ncmFudFBhc3NSb2xlKHRoaXMuX2RlcGxveW1lbnRSb2xlKTtcblxuICAgIGNvbnN0IHByb3ZpZGVkQ2FwYWJpbGl0aWVzID0gdGhpcy5wcm9wczIuY2ZuQ2FwYWJpbGl0aWVzID8/XG4gICAgICB0aGlzLnByb3BzMi5jYXBhYmlsaXRpZXM/Lm1hcChjID0+IHtcbiAgICAgICAgc3dpdGNoIChjKSB7XG4gICAgICAgICAgY2FzZSBjbG91ZGZvcm1hdGlvbi5DbG91ZEZvcm1hdGlvbkNhcGFiaWxpdGllcy5OT05FOiByZXR1cm4gY2RrLkNmbkNhcGFiaWxpdGllcy5OT05FO1xuICAgICAgICAgIGNhc2UgY2xvdWRmb3JtYXRpb24uQ2xvdWRGb3JtYXRpb25DYXBhYmlsaXRpZXMuQU5PTllNT1VTX0lBTTogcmV0dXJuIGNkay5DZm5DYXBhYmlsaXRpZXMuQU5PTllNT1VTX0lBTTtcbiAgICAgICAgICBjYXNlIGNsb3VkZm9ybWF0aW9uLkNsb3VkRm9ybWF0aW9uQ2FwYWJpbGl0aWVzLk5BTUVEX0lBTTogcmV0dXJuIGNkay5DZm5DYXBhYmlsaXRpZXMuTkFNRURfSUFNO1xuICAgICAgICAgIGNhc2UgY2xvdWRmb3JtYXRpb24uQ2xvdWRGb3JtYXRpb25DYXBhYmlsaXRpZXMuQVVUT19FWFBBTkQ6IHJldHVybiBjZGsuQ2ZuQ2FwYWJpbGl0aWVzLkFVVE9fRVhQQU5EO1xuICAgICAgICB9XG4gICAgICB9KTtcbiAgICBjb25zdCBjYXBhYmlsaXRpZXMgPSB0aGlzLnByb3BzMi5hZG1pblBlcm1pc3Npb25zICYmIHByb3ZpZGVkQ2FwYWJpbGl0aWVzID09PSB1bmRlZmluZWRcbiAgICAgID8gW2Nkay5DZm5DYXBhYmlsaXRpZXMuTkFNRURfSUFNXVxuICAgICAgOiBwcm92aWRlZENhcGFiaWxpdGllcztcblxuICAgIGNvbnN0IGFjdGlvbkNvbmZpZyA9IHN1cGVyLmJvdW5kKHNjb3BlLCBzdGFnZSwgb3B0aW9ucyk7XG4gICAgcmV0dXJuIHtcbiAgICAgIC4uLmFjdGlvbkNvbmZpZyxcbiAgICAgIGNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgLi4uYWN0aW9uQ29uZmlnLmNvbmZpZ3VyYXRpb24sXG4gICAgICAgIC8vIE5vbmUgZXZhbHVhdGVzIHRvIGVtcHR5IHN0cmluZyB3aGljaCBpcyBmYWxzZXkgYW5kIHJlc3VsdHMgaW4gdW5kZWZpbmVkXG4gICAgICAgIENhcGFiaWxpdGllczogcGFyc2VDYXBhYmlsaXRpZXMoY2FwYWJpbGl0aWVzKSxcbiAgICAgICAgUm9sZUFybjogdGhpcy5kZXBsb3ltZW50Um9sZS5yb2xlQXJuLFxuICAgICAgICBQYXJhbWV0ZXJPdmVycmlkZXM6IGNkay5TdGFjay5vZihzY29wZSkudG9Kc29uU3RyaW5nKHRoaXMucHJvcHMyLnBhcmFtZXRlck92ZXJyaWRlcyksXG4gICAgICAgIFRlbXBsYXRlQ29uZmlndXJhdGlvbjogdGhpcy5wcm9wczIudGVtcGxhdGVDb25maWd1cmF0aW9uXG4gICAgICAgICAgPyB0aGlzLnByb3BzMi50ZW1wbGF0ZUNvbmZpZ3VyYXRpb24ubG9jYXRpb25cbiAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgICAgU3RhY2tOYW1lOiB0aGlzLnByb3BzMi5zdGFja05hbWUsXG4gICAgICB9LFxuICAgIH07XG4gIH1cblxuICBwcml2YXRlIGdldERlcGxveW1lbnRSb2xlKG1lbWJlcjogc3RyaW5nKTogaWFtLklSb2xlIHtcbiAgICBpZiAodGhpcy5fZGVwbG95bWVudFJvbGUpIHtcbiAgICAgIHJldHVybiB0aGlzLl9kZXBsb3ltZW50Um9sZTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBDYW5ub3QgdXNlIHRoZSAke21lbWJlcn0gYmVmb3JlIHRoZSBBY3Rpb24gaGFzIGJlZW4gYWRkZWQgdG8gYSBQaXBlbGluZWApO1xuICAgIH1cbiAgfVxufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIHRoZSBDbG91ZEZvcm1hdGlvbkNyZWF0ZVJlcGxhY2VDaGFuZ2VTZXRBY3Rpb24uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ2xvdWRGb3JtYXRpb25DcmVhdGVSZXBsYWNlQ2hhbmdlU2V0QWN0aW9uUHJvcHMgZXh0ZW5kcyBDbG91ZEZvcm1hdGlvbkRlcGxveUFjdGlvblByb3BzIHtcbiAgLyoqXG4gICAqIE5hbWUgb2YgdGhlIGNoYW5nZSBzZXQgdG8gY3JlYXRlIG9yIHVwZGF0ZS5cbiAgICovXG4gIHJlYWRvbmx5IGNoYW5nZVNldE5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogSW5wdXQgYXJ0aWZhY3Qgd2l0aCB0aGUgQ2hhbmdlU2V0J3MgQ2xvdWRGb3JtYXRpb24gdGVtcGxhdGVcbiAgICovXG4gIHJlYWRvbmx5IHRlbXBsYXRlUGF0aDogY29kZXBpcGVsaW5lLkFydGlmYWN0UGF0aDtcbn1cblxuLyoqXG4gKiBDb2RlUGlwZWxpbmUgYWN0aW9uIHRvIHByZXBhcmUgYSBjaGFuZ2Ugc2V0LlxuICpcbiAqIENyZWF0ZXMgdGhlIGNoYW5nZSBzZXQgaWYgaXQgZG9lc24ndCBleGlzdCBiYXNlZCBvbiB0aGUgc3RhY2sgbmFtZSBhbmQgdGVtcGxhdGUgdGhhdCB5b3Ugc3VibWl0LlxuICogSWYgdGhlIGNoYW5nZSBzZXQgZXhpc3RzLCBBV1MgQ2xvdWRGb3JtYXRpb24gZGVsZXRlcyBpdCwgYW5kIHRoZW4gY3JlYXRlcyBhIG5ldyBvbmUuXG4gKi9cbmV4cG9ydCBjbGFzcyBDbG91ZEZvcm1hdGlvbkNyZWF0ZVJlcGxhY2VDaGFuZ2VTZXRBY3Rpb24gZXh0ZW5kcyBDbG91ZEZvcm1hdGlvbkRlcGxveUFjdGlvbiB7XG4gIHByaXZhdGUgcmVhZG9ubHkgcHJvcHMzOiBDbG91ZEZvcm1hdGlvbkNyZWF0ZVJlcGxhY2VDaGFuZ2VTZXRBY3Rpb25Qcm9wcztcblxuICBjb25zdHJ1Y3Rvcihwcm9wczogQ2xvdWRGb3JtYXRpb25DcmVhdGVSZXBsYWNlQ2hhbmdlU2V0QWN0aW9uUHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcywgcHJvcHMudGVtcGxhdGVDb25maWd1cmF0aW9uXG4gICAgICA/IFtwcm9wcy50ZW1wbGF0ZVBhdGguYXJ0aWZhY3QsIHByb3BzLnRlbXBsYXRlQ29uZmlndXJhdGlvbi5hcnRpZmFjdF1cbiAgICAgIDogW3Byb3BzLnRlbXBsYXRlUGF0aC5hcnRpZmFjdF0pO1xuXG4gICAgdGhpcy5wcm9wczMgPSBwcm9wcztcbiAgfVxuXG4gIHByb3RlY3RlZCBib3VuZChzY29wZTogQ29uc3RydWN0LCBzdGFnZTogY29kZXBpcGVsaW5lLklTdGFnZSwgb3B0aW9uczogY29kZXBpcGVsaW5lLkFjdGlvbkJpbmRPcHRpb25zKTpcbiAgY29kZXBpcGVsaW5lLkFjdGlvbkNvbmZpZyB7XG4gICAgLy8gdGhlIHN1cGVyIGNhbGwgb3JkZXIgaXMgdG8gcHJlc2VydmUgdGhlIGV4aXN0aW5nIG9yZGVyIG9mIHN0YXRlbWVudHMgaW4gcG9saWNpZXNcbiAgICBjb25zdCBhY3Rpb25Db25maWcgPSBzdXBlci5ib3VuZChzY29wZSwgc3RhZ2UsIG9wdGlvbnMpO1xuXG4gICAgU2luZ2xldG9uUG9saWN5LmZvclJvbGUob3B0aW9ucy5yb2xlKS5ncmFudENyZWF0ZVJlcGxhY2VDaGFuZ2VTZXQodGhpcy5wcm9wczMpO1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIC4uLmFjdGlvbkNvbmZpZyxcbiAgICAgIGNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgLi4uYWN0aW9uQ29uZmlnLmNvbmZpZ3VyYXRpb24sXG4gICAgICAgIEFjdGlvbk1vZGU6ICdDSEFOR0VfU0VUX1JFUExBQ0UnLFxuICAgICAgICBDaGFuZ2VTZXROYW1lOiB0aGlzLnByb3BzMy5jaGFuZ2VTZXROYW1lLFxuICAgICAgICBUZW1wbGF0ZVBhdGg6IHRoaXMucHJvcHMzLnRlbXBsYXRlUGF0aC5sb2NhdGlvbixcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIHRoZSBDbG91ZEZvcm1hdGlvbkNyZWF0ZVVwZGF0ZVN0YWNrQWN0aW9uLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENsb3VkRm9ybWF0aW9uQ3JlYXRlVXBkYXRlU3RhY2tBY3Rpb25Qcm9wcyBleHRlbmRzIENsb3VkRm9ybWF0aW9uRGVwbG95QWN0aW9uUHJvcHMge1xuICAvKipcbiAgICogSW5wdXQgYXJ0aWZhY3Qgd2l0aCB0aGUgQ2xvdWRGb3JtYXRpb24gdGVtcGxhdGUgdG8gZGVwbG95XG4gICAqL1xuICByZWFkb25seSB0ZW1wbGF0ZVBhdGg6IGNvZGVwaXBlbGluZS5BcnRpZmFjdFBhdGg7XG5cbiAgLyoqXG4gICAqIFJlcGxhY2UgdGhlIHN0YWNrIGlmIGl0J3MgaW4gYSBmYWlsZWQgc3RhdGUuXG4gICAqXG4gICAqIElmIHRoaXMgaXMgc2V0IHRvIHRydWUgYW5kIHRoZSBzdGFjayBpcyBpbiBhIGZhaWxlZCBzdGF0ZSAob25lIG9mXG4gICAqIFJPTExCQUNLX0NPTVBMRVRFLCBST0xMQkFDS19GQUlMRUQsIENSRUFURV9GQUlMRUQsIERFTEVURV9GQUlMRUQsIG9yXG4gICAqIFVQREFURV9ST0xMQkFDS19GQUlMRUQpLCBBV1MgQ2xvdWRGb3JtYXRpb24gZGVsZXRlcyB0aGUgc3RhY2sgYW5kIHRoZW5cbiAgICogY3JlYXRlcyBhIG5ldyBzdGFjay5cbiAgICpcbiAgICogSWYgdGhpcyBpcyBub3Qgc2V0IHRvIHRydWUgYW5kIHRoZSBzdGFjayBpcyBpbiBhIGZhaWxlZCBzdGF0ZSxcbiAgICogdGhlIGRlcGxveW1lbnQgZmFpbHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSByZXBsYWNlT25GYWlsdXJlPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBDb2RlUGlwZWxpbmUgYWN0aW9uIHRvIGRlcGxveSBhIHN0YWNrLlxuICpcbiAqIENyZWF0ZXMgdGhlIHN0YWNrIGlmIHRoZSBzcGVjaWZpZWQgc3RhY2sgZG9lc24ndCBleGlzdC4gSWYgdGhlIHN0YWNrIGV4aXN0cyxcbiAqIEFXUyBDbG91ZEZvcm1hdGlvbiB1cGRhdGVzIHRoZSBzdGFjay4gVXNlIHRoaXMgYWN0aW9uIHRvIHVwZGF0ZSBleGlzdGluZ1xuICogc3RhY2tzLlxuICpcbiAqIEFXUyBDb2RlUGlwZWxpbmUgd29uJ3QgcmVwbGFjZSB0aGUgc3RhY2ssIGFuZCB3aWxsIGZhaWwgZGVwbG95bWVudCBpZiB0aGVcbiAqIHN0YWNrIGlzIGluIGEgZmFpbGVkIHN0YXRlLiBVc2UgYFJlcGxhY2VPbkZhaWx1cmVgIGZvciBhbiBhY3Rpb24gdGhhdFxuICogd2lsbCBkZWxldGUgYW5kIHJlY3JlYXRlIHRoZSBzdGFjayB0byB0cnkgYW5kIHJlY292ZXIgZnJvbSBmYWlsZWQgc3RhdGVzLlxuICpcbiAqIFVzZSB0aGlzIGFjdGlvbiB0byBhdXRvbWF0aWNhbGx5IHJlcGxhY2UgZmFpbGVkIHN0YWNrcyB3aXRob3V0IHJlY292ZXJpbmcgb3JcbiAqIHRyb3VibGVzaG9vdGluZyB0aGVtLiBZb3Ugd291bGQgdHlwaWNhbGx5IGNob29zZSB0aGlzIG1vZGUgZm9yIHRlc3RpbmcuXG4gKi9cbmV4cG9ydCBjbGFzcyBDbG91ZEZvcm1hdGlvbkNyZWF0ZVVwZGF0ZVN0YWNrQWN0aW9uIGV4dGVuZHMgQ2xvdWRGb3JtYXRpb25EZXBsb3lBY3Rpb24ge1xuICBwcml2YXRlIHJlYWRvbmx5IHByb3BzMzogQ2xvdWRGb3JtYXRpb25DcmVhdGVVcGRhdGVTdGFja0FjdGlvblByb3BzO1xuXG4gIGNvbnN0cnVjdG9yKHByb3BzOiBDbG91ZEZvcm1hdGlvbkNyZWF0ZVVwZGF0ZVN0YWNrQWN0aW9uUHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcywgcHJvcHMudGVtcGxhdGVDb25maWd1cmF0aW9uXG4gICAgICA/IFtwcm9wcy50ZW1wbGF0ZVBhdGguYXJ0aWZhY3QsIHByb3BzLnRlbXBsYXRlQ29uZmlndXJhdGlvbi5hcnRpZmFjdF1cbiAgICAgIDogW3Byb3BzLnRlbXBsYXRlUGF0aC5hcnRpZmFjdF0pO1xuXG4gICAgdGhpcy5wcm9wczMgPSBwcm9wcztcbiAgfVxuXG4gIHByb3RlY3RlZCBib3VuZChzY29wZTogQ29uc3RydWN0LCBzdGFnZTogY29kZXBpcGVsaW5lLklTdGFnZSwgb3B0aW9uczogY29kZXBpcGVsaW5lLkFjdGlvbkJpbmRPcHRpb25zKTpcbiAgY29kZXBpcGVsaW5lLkFjdGlvbkNvbmZpZyB7XG4gICAgLy8gdGhlIHN1cGVyIGNhbGwgb3JkZXIgaXMgdG8gcHJlc2VydmUgdGhlIGV4aXN0aW5nIG9yZGVyIG9mIHN0YXRlbWVudHMgaW4gcG9saWNpZXNcbiAgICBjb25zdCBhY3Rpb25Db25maWcgPSBzdXBlci5ib3VuZChzY29wZSwgc3RhZ2UsIG9wdGlvbnMpO1xuXG4gICAgU2luZ2xldG9uUG9saWN5LmZvclJvbGUob3B0aW9ucy5yb2xlKS5ncmFudENyZWF0ZVVwZGF0ZVN0YWNrKHRoaXMucHJvcHMzKTtcblxuICAgIHJldHVybiB7XG4gICAgICAuLi5hY3Rpb25Db25maWcsXG4gICAgICBjb25maWd1cmF0aW9uOiB7XG4gICAgICAgIC4uLmFjdGlvbkNvbmZpZy5jb25maWd1cmF0aW9uLFxuICAgICAgICBBY3Rpb25Nb2RlOiB0aGlzLnByb3BzMy5yZXBsYWNlT25GYWlsdXJlID8gJ1JFUExBQ0VfT05fRkFJTFVSRScgOiAnQ1JFQVRFX1VQREFURScsXG4gICAgICAgIFRlbXBsYXRlUGF0aDogdGhpcy5wcm9wczMudGVtcGxhdGVQYXRoLmxvY2F0aW9uLFxuICAgICAgfSxcbiAgICB9O1xuICB9XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgdGhlIENsb3VkRm9ybWF0aW9uRGVsZXRlU3RhY2tBY3Rpb24uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ2xvdWRGb3JtYXRpb25EZWxldGVTdGFja0FjdGlvblByb3BzIGV4dGVuZHMgQ2xvdWRGb3JtYXRpb25EZXBsb3lBY3Rpb25Qcm9wcyB7XG59XG5cbi8qKlxuICogQ29kZVBpcGVsaW5lIGFjdGlvbiB0byBkZWxldGUgYSBzdGFjay5cbiAqXG4gKiBEZWxldGVzIGEgc3RhY2suIElmIHlvdSBzcGVjaWZ5IGEgc3RhY2sgdGhhdCBkb2Vzbid0IGV4aXN0LCB0aGUgYWN0aW9uIGNvbXBsZXRlcyBzdWNjZXNzZnVsbHlcbiAqIHdpdGhvdXQgZGVsZXRpbmcgYSBzdGFjay5cbiAqL1xuZXhwb3J0IGNsYXNzIENsb3VkRm9ybWF0aW9uRGVsZXRlU3RhY2tBY3Rpb24gZXh0ZW5kcyBDbG91ZEZvcm1hdGlvbkRlcGxveUFjdGlvbiB7XG4gIHByaXZhdGUgcmVhZG9ubHkgcHJvcHMzOiBDbG91ZEZvcm1hdGlvbkRlbGV0ZVN0YWNrQWN0aW9uUHJvcHM7XG5cbiAgY29uc3RydWN0b3IocHJvcHM6IENsb3VkRm9ybWF0aW9uRGVsZXRlU3RhY2tBY3Rpb25Qcm9wcykge1xuICAgIHN1cGVyKHByb3BzLCB1bmRlZmluZWQpO1xuXG4gICAgdGhpcy5wcm9wczMgPSBwcm9wcztcbiAgfVxuXG4gIHByb3RlY3RlZCBib3VuZChzY29wZTogQ29uc3RydWN0LCBzdGFnZTogY29kZXBpcGVsaW5lLklTdGFnZSwgb3B0aW9uczogY29kZXBpcGVsaW5lLkFjdGlvbkJpbmRPcHRpb25zKTpcbiAgY29kZXBpcGVsaW5lLkFjdGlvbkNvbmZpZyB7XG4gICAgLy8gdGhlIHN1cGVyIGNhbGwgb3JkZXIgaXMgdG8gcHJlc2VydmUgdGhlIGV4aXN0aW5nIG9yZGVyIG9mIHN0YXRlbWVudHMgaW4gcG9saWNpZXNcbiAgICBjb25zdCBhY3Rpb25Db25maWcgPSBzdXBlci5ib3VuZChzY29wZSwgc3RhZ2UsIG9wdGlvbnMpO1xuXG4gICAgU2luZ2xldG9uUG9saWN5LmZvclJvbGUob3B0aW9ucy5yb2xlKS5ncmFudERlbGV0ZVN0YWNrKHRoaXMucHJvcHMzKTtcblxuICAgIHJldHVybiB7XG4gICAgICAuLi5hY3Rpb25Db25maWcsXG4gICAgICBjb25maWd1cmF0aW9uOiB7XG4gICAgICAgIC4uLmFjdGlvbkNvbmZpZy5jb25maWd1cmF0aW9uLFxuICAgICAgICBBY3Rpb25Nb2RlOiAnREVMRVRFX09OTFknLFxuICAgICAgfSxcbiAgICB9O1xuICB9XG59XG4iXX0=