"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseCluster = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const logs = require("../../aws-logs");
const secretsmanager = require("../../aws-secretsmanager");
const core_1 = require("../../core");
const database_secret_1 = require("./database-secret");
const docdb_generated_1 = require("./docdb.generated");
const endpoint_1 = require("./endpoint");
/**
 * A new or imported clustered database.
 */
class DatabaseClusterBase extends core_1.Resource {
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterIdentifier,
            targetType: secretsmanager.AttachmentTargetType.DOCDB_DB_CLUSTER,
        };
    }
}
/**
 * Create a clustered database with a given number of instances.
 *
 * @resource AWS::DocDB::DBCluster
 */
class DatabaseCluster extends DatabaseClusterBase {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Identifiers of the replicas
         */
        this.instanceIdentifiers = [];
        /**
         * Endpoints which address each individual replica.
         */
        this.instanceEndpoints = [];
        try {
            jsiiDeprecationWarnings.monocdk_aws_docdb_DatabaseClusterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, DatabaseCluster);
            }
            throw error;
        }
        this.vpc = props.vpc;
        this.vpcSubnets = props.vpcSubnets;
        // Determine the subnet(s) to deploy the DocDB cluster to
        const { subnetIds, internetConnectivityEstablished } = this.vpc.selectSubnets(this.vpcSubnets);
        // DocDB clusters require a subnet group with subnets from at least two AZs.
        // We cannot test whether the subnets are in different AZs, but at least we can test the amount.
        // See https://docs.aws.amazon.com/documentdb/latest/developerguide/replication.html#replication.high-availability
        if (subnetIds.length < 2) {
            throw new Error(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        const subnetGroup = new docdb_generated_1.CfnDBSubnetGroup(this, 'Subnets', {
            dbSubnetGroupDescription: `Subnets for ${id} database`,
            subnetIds,
        });
        // Create the security group for the DB cluster
        let securityGroup;
        if (props.securityGroup) {
            securityGroup = props.securityGroup;
        }
        else {
            securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'DocumentDB security group',
                vpc: this.vpc,
            });
            // HACK: Use an escape-hatch to apply a consistent removal policy to the
            // security group so we don't get errors when trying to delete the stack
            securityGroup.node.defaultChild.applyRemovalPolicy(props.removalPolicy, {
                applyToUpdateReplacePolicy: true,
            });
        }
        this.securityGroupId = securityGroup.securityGroupId;
        // Create the CloudwatchLogsConfiguratoin
        const enableCloudwatchLogsExports = [];
        if (props.exportAuditLogsToCloudWatch) {
            enableCloudwatchLogsExports.push('audit');
        }
        if (props.exportProfilerLogsToCloudWatch) {
            enableCloudwatchLogsExports.push('profiler');
        }
        // Create the secret manager secret if no password is specified
        let secret;
        if (!props.masterUser.password) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUser.username,
                encryptionKey: props.masterUser.kmsKey,
                excludeCharacters: props.masterUser.excludeCharacters,
                secretName: props.masterUser.secretName,
            });
        }
        // Default to encrypted storage
        const storageEncrypted = props.storageEncrypted ?? true;
        if (props.kmsKey && !storageEncrypted) {
            throw new Error('KMS key supplied but storageEncrypted is false');
        }
        // Create the DocDB cluster
        this.cluster = new docdb_generated_1.CfnDBCluster(this, 'Resource', {
            // Basic
            engineVersion: props.engineVersion,
            dbClusterIdentifier: props.dbClusterName,
            dbSubnetGroupName: subnetGroup.ref,
            port: props.port,
            vpcSecurityGroupIds: [this.securityGroupId],
            dbClusterParameterGroupName: props.parameterGroup?.parameterGroupName,
            deletionProtection: props.deletionProtection,
            // Admin
            masterUsername: secret ? secret.secretValueFromJson('username').unsafeUnwrap() : props.masterUser.username,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').unsafeUnwrap()
                : props.masterUser.password.unsafeUnwrap(),
            // Backup
            backupRetentionPeriod: props.backup?.retention?.toDays(),
            preferredBackupWindow: props.backup?.preferredWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            // EnableCloudwatchLogsExports
            enableCloudwatchLogsExports: enableCloudwatchLogsExports.length > 0 ? enableCloudwatchLogsExports : undefined,
            // Encryption
            kmsKeyId: props.kmsKey?.keyArn,
            storageEncrypted,
        });
        this.cluster.applyRemovalPolicy(props.removalPolicy, {
            applyToUpdateReplacePolicy: true,
        });
        this.clusterIdentifier = this.cluster.ref;
        this.clusterResourceIdentifier = this.cluster.attrClusterResourceId;
        const port = core_1.Token.asNumber(this.cluster.attrPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(this.cluster.attrEndpoint, port);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(this.cluster.attrReadEndpoint, port);
        this.setLogRetention(this, props, enableCloudwatchLogsExports);
        if (secret) {
            this.secret = secret.attach(this);
        }
        // Create the instances
        const instanceCount = props.instances ?? DatabaseCluster.DEFAULT_NUM_INSTANCES;
        if (instanceCount < 1) {
            throw new Error('At least one instance is required');
        }
        for (let i = 0; i < instanceCount; i++) {
            const instanceIndex = i + 1;
            const instanceIdentifier = props.instanceIdentifierBase != null ? `${props.instanceIdentifierBase}${instanceIndex}`
                : props.dbClusterName != null ? `${props.dbClusterName}instance${instanceIndex}` : undefined;
            const instance = new docdb_generated_1.CfnDBInstance(this, `Instance${instanceIndex}`, {
                // Link to cluster
                dbClusterIdentifier: this.cluster.ref,
                dbInstanceIdentifier: instanceIdentifier,
                // Instance properties
                dbInstanceClass: databaseInstanceType(props.instanceType),
            });
            instance.applyRemovalPolicy(props.removalPolicy, {
                applyToUpdateReplacePolicy: true,
            });
            // We must have a dependency on the NAT gateway provider here to create
            // things in the right order.
            instance.node.addDependency(internetConnectivityEstablished);
            this.instanceIdentifiers.push(instance.ref);
            this.instanceEndpoints.push(new endpoint_1.Endpoint(instance.attrEndpoint, port));
        }
        this.connections = new ec2.Connections({
            defaultPort: ec2.Port.tcp(port),
            securityGroups: [securityGroup],
        });
    }
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromDatabaseClusterAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_docdb_DatabaseClusterAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromDatabaseClusterAttributes);
            }
            throw error;
        }
        class Import extends DatabaseClusterBase {
            constructor() {
                super(...arguments);
                this.defaultPort = typeof attrs.port !== 'undefined' ? ec2.Port.tcp(attrs.port) : undefined;
                this.connections = new ec2.Connections({
                    securityGroups: attrs.securityGroup ? [attrs.securityGroup] : undefined,
                    defaultPort: this.defaultPort,
                });
                this.clusterIdentifier = attrs.clusterIdentifier;
                this._instanceIdentifiers = attrs.instanceIdentifiers;
                this._clusterEndpoint = attrs.clusterEndpointAddress && typeof attrs.port !== 'undefined' ?
                    new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port) : undefined;
                this._clusterReadEndpoint = attrs.readerEndpointAddress && typeof attrs.port !== 'undefined' ?
                    new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port) : undefined;
                this._instanceEndpoints = attrs.instanceEndpointAddresses && typeof attrs.port !== 'undefined' ?
                    attrs.instanceEndpointAddresses.map(addr => new endpoint_1.Endpoint(addr, attrs.port)) : undefined;
                this._securityGroupId = attrs.securityGroup?.securityGroupId;
            }
            get instanceIdentifiers() {
                if (!this._instanceIdentifiers) {
                    throw new Error('Cannot access `instanceIdentifiers` of an imported cluster without provided instanceIdentifiers');
                }
                return this._instanceIdentifiers;
            }
            get clusterEndpoint() {
                if (!this._clusterEndpoint) {
                    throw new Error('Cannot access `clusterEndpoint` of an imported cluster without an endpoint address and port');
                }
                return this._clusterEndpoint;
            }
            get clusterReadEndpoint() {
                if (!this._clusterReadEndpoint) {
                    throw new Error('Cannot access `clusterReadEndpoint` of an imported cluster without a readerEndpointAddress and port');
                }
                return this._clusterReadEndpoint;
            }
            get instanceEndpoints() {
                if (!this._instanceEndpoints) {
                    throw new Error('Cannot access `instanceEndpoints` of an imported cluster without instanceEndpointAddresses and port');
                }
                return this._instanceEndpoints;
            }
            get securityGroupId() {
                if (!this._securityGroupId) {
                    throw new Error('Cannot access `securityGroupId` of an imported cluster without securityGroupId');
                }
                return this._securityGroupId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Sets up CloudWatch log retention if configured.
     */
    setLogRetention(cluster, props, cloudwatchLogsExports) {
        if (props.cloudWatchLogsRetention) {
            for (const log of cloudwatchLogsExports) {
                new logs.LogRetention(cluster, `LogRetention${log}`, {
                    logGroupName: `/aws/docdb/${cluster.clusterIdentifier}/${log}`,
                    retention: props.cloudWatchLogsRetention,
                    role: props.cloudWatchLogsRetentionRole,
                });
            }
        }
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        try {
            jsiiDeprecationWarnings.monocdk_Duration(automaticallyAfter);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addRotationSingleUser);
            }
            throw error;
        }
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: DatabaseCluster.SINGLE_USER_ROTATION_APPLICATION,
            excludeCharacters: this.node.tryFindChild('Secret')._excludedCharacters,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id, options) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_docdb_RotationMultiUserOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addRotationMultiUser);
            }
            throw error;
        }
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            excludeCharacters: this.node.tryFindChild('Secret')._excludedCharacters,
            application: DatabaseCluster.MULTI_USER_ROTATION_APPLICATION,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    /**
     * Adds security groups to this cluster.
     * @param securityGroups The security groups to add.
     */
    addSecurityGroups(...securityGroups) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ec2_ISecurityGroup(securityGroups);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addSecurityGroups);
            }
            throw error;
        }
        if (this.cluster.vpcSecurityGroupIds === undefined) {
            this.cluster.vpcSecurityGroupIds = [];
        }
        this.cluster.vpcSecurityGroupIds.push(...securityGroups.map(sg => sg.securityGroupId));
    }
}
exports.DatabaseCluster = DatabaseCluster;
_a = JSII_RTTI_SYMBOL_1;
DatabaseCluster[_a] = { fqn: "monocdk.aws_docdb.DatabaseCluster", version: "1.191.0" };
/**
 * The default number of instances in the DocDB cluster if none are
 * specified
 */
DatabaseCluster.DEFAULT_NUM_INSTANCES = 1;
/**
 * The default port Document DB listens on
 */
DatabaseCluster.DEFAULT_PORT = 27017;
/**
 * The single user secret rotation application.
 */
DatabaseCluster.SINGLE_USER_ROTATION_APPLICATION = secretsmanager.SecretRotationApplication.MONGODB_ROTATION_SINGLE_USER;
/**
 * The multi user secret rotation application.
 */
DatabaseCluster.MULTI_USER_ROTATION_APPLICATION = secretsmanager.SecretRotationApplication.MONGODB_ROTATION_MULTI_USER;
/**
 * Turn a regular instance type into a database instance type
 */
function databaseInstanceType(instanceType) {
    return 'db.' + instanceType.toString();
}
//# sourceMappingURL=data:application/json;base64,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