"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.isExternalCompatible = exports.isFargateCompatible = exports.isEc2Compatible = exports.Compatibility = exports.Scope = exports.PidMode = exports.IpcMode = exports.NetworkMode = exports.TaskDefinition = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../../aws-ec2");
const iam = require("../../../aws-iam");
const core_1 = require("../../../core");
const container_definition_1 = require("../container-definition");
const ecs_generated_1 = require("../ecs.generated");
const firelens_log_router_1 = require("../firelens-log-router");
const aws_log_driver_1 = require("../log-drivers/aws-log-driver");
const _imported_task_definition_1 = require("./_imported-task-definition");
class TaskDefinitionBase extends core_1.Resource {
    /**
     * Return true if the task definition can be run on an EC2 cluster
     */
    get isEc2Compatible() {
        return isEc2Compatible(this.compatibility);
    }
    /**
     * Return true if the task definition can be run on a Fargate cluster
     */
    get isFargateCompatible() {
        return isFargateCompatible(this.compatibility);
    }
    /**
     * Return true if the task definition can be run on a ECS anywhere cluster
     */
    get isExternalCompatible() {
        return isExternalCompatible(this.compatibility);
    }
}
/**
 * The base class for all task definitions.
 */
class TaskDefinition extends TaskDefinitionBase {
    /**
     * Constructs a new instance of the TaskDefinition class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * The container definitions.
         */
        this.containers = new Array();
        /**
         * All volumes
         */
        this.volumes = [];
        /**
         * Placement constraints for task instances
         */
        this.placementConstraints = new Array();
        /**
         * Inference accelerators for task instances
         */
        this._inferenceAccelerators = [];
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_TaskDefinitionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, TaskDefinition);
            }
            throw error;
        }
        this.family = props.family || core_1.Names.uniqueId(this);
        this.compatibility = props.compatibility;
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        this.networkMode = props.networkMode ?? (this.isFargateCompatible ? NetworkMode.AWS_VPC : NetworkMode.BRIDGE);
        if (this.isFargateCompatible && this.networkMode !== NetworkMode.AWS_VPC) {
            throw new Error(`Fargate tasks can only have AwsVpc network mode, got: ${this.networkMode}`);
        }
        if (props.proxyConfiguration && this.networkMode !== NetworkMode.AWS_VPC) {
            throw new Error(`ProxyConfiguration can only be used with AwsVpc network mode, got: ${this.networkMode}`);
        }
        if (props.placementConstraints && props.placementConstraints.length > 0 && this.isFargateCompatible) {
            throw new Error('Cannot set placement constraints on tasks that run on Fargate');
        }
        if (this.isFargateCompatible && (!props.cpu || !props.memoryMiB)) {
            throw new Error(`Fargate-compatible tasks require both CPU (${props.cpu}) and memory (${props.memoryMiB}) specifications`);
        }
        if (props.inferenceAccelerators && props.inferenceAccelerators.length > 0 && this.isFargateCompatible) {
            throw new Error('Cannot use inference accelerators on tasks that run on Fargate');
        }
        if (this.isExternalCompatible && this.networkMode !== NetworkMode.BRIDGE) {
            throw new Error(`External tasks can only have Bridge network mode, got: ${this.networkMode}`);
        }
        if (!this.isFargateCompatible && props.runtimePlatform) {
            throw new Error('Cannot specify runtimePlatform in non-Fargate compatible tasks');
        }
        this._executionRole = props.executionRole;
        this.taskRole = props.taskRole || new iam.Role(this, 'TaskRole', {
            assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
        });
        if (props.inferenceAccelerators) {
            props.inferenceAccelerators.forEach(ia => this.addInferenceAccelerator(ia));
        }
        this.ephemeralStorageGiB = props.ephemeralStorageGiB;
        // validate the cpu and memory size for the Windows operation system family.
        if (props.runtimePlatform?.operatingSystemFamily?._operatingSystemFamily.includes('WINDOWS')) {
            // We know that props.cpu and props.memoryMiB are defined because an error would have been thrown previously if they were not.
            // But, typescript is not able to figure this out, so using the `!` operator here to let the type-checker know they are defined.
            this.checkFargateWindowsBasedTasksSize(props.cpu, props.memoryMiB, props.runtimePlatform);
        }
        this.runtimePlatform = props.runtimePlatform;
        const taskDef = new ecs_generated_1.CfnTaskDefinition(this, 'Resource', {
            containerDefinitions: core_1.Lazy.any({ produce: () => this.renderContainers() }, { omitEmptyArray: true }),
            volumes: core_1.Lazy.any({ produce: () => this.renderVolumes() }, { omitEmptyArray: true }),
            executionRoleArn: core_1.Lazy.string({ produce: () => this.executionRole && this.executionRole.roleArn }),
            family: this.family,
            taskRoleArn: this.taskRole.roleArn,
            requiresCompatibilities: [
                ...(isEc2Compatible(props.compatibility) ? ['EC2'] : []),
                ...(isFargateCompatible(props.compatibility) ? ['FARGATE'] : []),
                ...(isExternalCompatible(props.compatibility) ? ['EXTERNAL'] : []),
            ],
            networkMode: this.renderNetworkMode(this.networkMode),
            placementConstraints: core_1.Lazy.any({
                produce: () => !isFargateCompatible(this.compatibility) ? this.placementConstraints : undefined,
            }, { omitEmptyArray: true }),
            proxyConfiguration: props.proxyConfiguration ? props.proxyConfiguration.bind(this.stack, this) : undefined,
            cpu: props.cpu,
            memory: props.memoryMiB,
            ipcMode: props.ipcMode,
            pidMode: props.pidMode,
            inferenceAccelerators: core_1.Lazy.any({
                produce: () => !isFargateCompatible(this.compatibility) ? this.renderInferenceAccelerators() : undefined,
            }, { omitEmptyArray: true }),
            ephemeralStorage: this.ephemeralStorageGiB ? {
                sizeInGiB: this.ephemeralStorageGiB,
            } : undefined,
            runtimePlatform: this.isFargateCompatible && this.runtimePlatform ? {
                cpuArchitecture: this.runtimePlatform?.cpuArchitecture?._cpuArchitecture,
                operatingSystemFamily: this.runtimePlatform?.operatingSystemFamily?._operatingSystemFamily,
            } : undefined,
        });
        if (props.placementConstraints) {
            props.placementConstraints.forEach(pc => this.addPlacementConstraint(pc));
        }
        this.taskDefinitionArn = taskDef.ref;
    }
    /**
     * Imports a task definition from the specified task definition ARN.
     *
     * The task will have a compatibility of EC2+Fargate.
     */
    static fromTaskDefinitionArn(scope, id, taskDefinitionArn) {
        return new _imported_task_definition_1.ImportedTaskDefinition(scope, id, { taskDefinitionArn: taskDefinitionArn });
    }
    /**
     * Create a task definition from a task definition reference
     */
    static fromTaskDefinitionAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_TaskDefinitionAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromTaskDefinitionAttributes);
            }
            throw error;
        }
        return new _imported_task_definition_1.ImportedTaskDefinition(scope, id, {
            taskDefinitionArn: attrs.taskDefinitionArn,
            compatibility: attrs.compatibility,
            networkMode: attrs.networkMode,
            taskRole: attrs.taskRole,
        });
    }
    get executionRole() {
        return this._executionRole;
    }
    /**
     * Public getter method to access list of inference accelerators attached to the instance.
     */
    get inferenceAccelerators() {
        return this._inferenceAccelerators;
    }
    renderVolumes() {
        return this.volumes.map(renderVolume);
        function renderVolume(spec) {
            return {
                host: spec.host,
                name: spec.name,
                dockerVolumeConfiguration: spec.dockerVolumeConfiguration && {
                    autoprovision: spec.dockerVolumeConfiguration.autoprovision,
                    driver: spec.dockerVolumeConfiguration.driver,
                    driverOpts: spec.dockerVolumeConfiguration.driverOpts,
                    labels: spec.dockerVolumeConfiguration.labels,
                    scope: spec.dockerVolumeConfiguration.scope,
                },
                efsVolumeConfiguration: spec.efsVolumeConfiguration && {
                    filesystemId: spec.efsVolumeConfiguration.fileSystemId,
                    authorizationConfig: spec.efsVolumeConfiguration.authorizationConfig,
                    rootDirectory: spec.efsVolumeConfiguration.rootDirectory,
                    transitEncryption: spec.efsVolumeConfiguration.transitEncryption,
                    transitEncryptionPort: spec.efsVolumeConfiguration.transitEncryptionPort,
                },
            };
        }
    }
    renderInferenceAccelerators() {
        return this._inferenceAccelerators.map(renderInferenceAccelerator);
        function renderInferenceAccelerator(inferenceAccelerator) {
            return {
                deviceName: inferenceAccelerator.deviceName,
                deviceType: inferenceAccelerator.deviceType,
            };
        }
    }
    /**
     * Validate the existence of the input target and set default values.
     *
     * @internal
     */
    _validateTarget(options) {
        const targetContainer = this.findContainer(options.containerName);
        if (targetContainer === undefined) {
            throw new Error(`No container named '${options.containerName}'. Did you call "addContainer()"?`);
        }
        const targetProtocol = options.protocol || container_definition_1.Protocol.TCP;
        const targetContainerPort = options.containerPort || targetContainer.containerPort;
        const portMapping = targetContainer.findPortMapping(targetContainerPort, targetProtocol);
        if (portMapping === undefined) {
            // eslint-disable-next-line max-len
            throw new Error(`Container '${targetContainer}' has no mapping for port ${options.containerPort} and protocol ${targetProtocol}. Did you call "container.addPortMappings()"?`);
        }
        return {
            containerName: options.containerName,
            portMapping,
        };
    }
    /**
     * Returns the port range to be opened that match the provided container name and container port.
     *
     * @internal
     */
    _portRangeFromPortMapping(portMapping) {
        if (portMapping.hostPort !== undefined && portMapping.hostPort !== 0) {
            return portMapping.protocol === container_definition_1.Protocol.UDP ? ec2.Port.udp(portMapping.hostPort) : ec2.Port.tcp(portMapping.hostPort);
        }
        if (this.networkMode === NetworkMode.BRIDGE || this.networkMode === NetworkMode.NAT) {
            return EPHEMERAL_PORT_RANGE;
        }
        return portMapping.protocol === container_definition_1.Protocol.UDP ? ec2.Port.udp(portMapping.containerPort) : ec2.Port.tcp(portMapping.containerPort);
    }
    /**
     * Adds a policy statement to the task IAM role.
     */
    addToTaskRolePolicy(statement) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_iam_PolicyStatement(statement);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToTaskRolePolicy);
            }
            throw error;
        }
        this.taskRole.addToPrincipalPolicy(statement);
    }
    /**
     * Adds a policy statement to the task execution IAM role.
     */
    addToExecutionRolePolicy(statement) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_iam_PolicyStatement(statement);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToExecutionRolePolicy);
            }
            throw error;
        }
        this.obtainExecutionRole().addToPrincipalPolicy(statement);
    }
    /**
     * Adds a new container to the task definition.
     */
    addContainer(id, props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_ContainerDefinitionOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addContainer);
            }
            throw error;
        }
        return new container_definition_1.ContainerDefinition(this, id, { taskDefinition: this, ...props });
    }
    /**
     * Adds a firelens log router to the task definition.
     */
    addFirelensLogRouter(id, props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_FirelensLogRouterDefinitionOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addFirelensLogRouter);
            }
            throw error;
        }
        // only one firelens log router is allowed in each task.
        if (this.containers.find(x => x instanceof firelens_log_router_1.FirelensLogRouter)) {
            throw new Error('Firelens log router is already added in this task.');
        }
        return new firelens_log_router_1.FirelensLogRouter(this, id, { taskDefinition: this, ...props });
    }
    /**
     * Links a container to this task definition.
     * @internal
     */
    _linkContainer(container) {
        this.containers.push(container);
        if (this.defaultContainer === undefined && container.essential) {
            this.defaultContainer = container;
        }
        if (container.referencesSecretJsonField) {
            this._referencesSecretJsonField = true;
        }
    }
    /**
     * Adds a volume to the task definition.
     */
    addVolume(volume) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_Volume(volume);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addVolume);
            }
            throw error;
        }
        this.volumes.push(volume);
    }
    /**
     * Adds the specified placement constraint to the task definition.
     */
    addPlacementConstraint(constraint) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_PlacementConstraint(constraint);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addPlacementConstraint);
            }
            throw error;
        }
        if (isFargateCompatible(this.compatibility)) {
            throw new Error('Cannot set placement constraints on tasks that run on Fargate');
        }
        this.placementConstraints.push(...constraint.toJson());
    }
    /**
     * Adds the specified extension to the task definition.
     *
     * Extension can be used to apply a packaged modification to
     * a task definition.
     */
    addExtension(extension) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_ITaskDefinitionExtension(extension);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addExtension);
            }
            throw error;
        }
        extension.extend(this);
    }
    /**
     * Adds an inference accelerator to the task definition.
     */
    addInferenceAccelerator(inferenceAccelerator) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_InferenceAccelerator(inferenceAccelerator);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addInferenceAccelerator);
            }
            throw error;
        }
        if (isFargateCompatible(this.compatibility)) {
            throw new Error('Cannot use inference accelerators on tasks that run on Fargate');
        }
        this._inferenceAccelerators.push(inferenceAccelerator);
    }
    /**
     * Creates the task execution IAM role if it doesn't already exist.
     */
    obtainExecutionRole() {
        if (!this._executionRole) {
            this._executionRole = new iam.Role(this, 'ExecutionRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
                // needed for cross-account access with TagParameterContainerImage
                roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            });
        }
        return this._executionRole;
    }
    /**
     * Whether this task definition has at least a container that references a
     * specific JSON field of a secret stored in Secrets Manager.
     */
    get referencesSecretJsonField() {
        return this._referencesSecretJsonField;
    }
    /**
     * Validates the task definition.
     */
    validate() {
        const ret = super.validate();
        if (isEc2Compatible(this.compatibility)) {
            // EC2 mode validations
            // Container sizes
            for (const container of this.containers) {
                if (!container.memoryLimitSpecified) {
                    ret.push(`ECS Container ${container.containerName} must have at least one of 'memoryLimitMiB' or 'memoryReservationMiB' specified`);
                }
            }
        }
        return ret;
    }
    /**
     * Returns the container that match the provided containerName.
     */
    findContainer(containerName) {
        return this.containers.find(c => c.containerName === containerName);
    }
    renderNetworkMode(networkMode) {
        return (networkMode === NetworkMode.NAT) ? undefined : networkMode;
    }
    renderContainers() {
        // add firelens log router container if any application container is using firelens log driver,
        // also check if already created log router container
        for (const container of this.containers) {
            if (container.logDriverConfig && container.logDriverConfig.logDriver === 'awsfirelens'
                && !this.containers.find(x => x instanceof firelens_log_router_1.FirelensLogRouter)) {
                this.addFirelensLogRouter('log-router', {
                    image: firelens_log_router_1.obtainDefaultFluentBitECRImage(this, container.logDriverConfig),
                    firelensConfig: {
                        type: firelens_log_router_1.FirelensLogRouterType.FLUENTBIT,
                    },
                    logging: new aws_log_driver_1.AwsLogDriver({ streamPrefix: 'firelens' }),
                    memoryReservationMiB: 50,
                });
                break;
            }
        }
        return this.containers.map(x => x.renderContainerDefinition());
    }
    checkFargateWindowsBasedTasksSize(cpu, memory, runtimePlatform) {
        if (Number(cpu) === 1024) {
            if (Number(memory) < 1024 || Number(memory) > 8192 || (Number(memory) % 1024 !== 0)) {
                throw new Error(`If provided cpu is ${cpu}, then memoryMiB must have a min of 1024 and a max of 8192, in 1024 increments. Provided memoryMiB was ${Number(memory)}.`);
            }
        }
        else if (Number(cpu) === 2048) {
            if (Number(memory) < 4096 || Number(memory) > 16384 || (Number(memory) % 1024 !== 0)) {
                throw new Error(`If provided cpu is ${cpu}, then memoryMiB must have a min of 4096 and max of 16384, in 1024 increments. Provided memoryMiB ${Number(memory)}.`);
            }
        }
        else if (Number(cpu) === 4096) {
            if (Number(memory) < 8192 || Number(memory) > 30720 || (Number(memory) % 1024 !== 0)) {
                throw new Error(`If provided cpu is ${cpu}, then memoryMiB must have a min of 8192 and a max of 30720, in 1024 increments.Provided memoryMiB was ${Number(memory)}.`);
            }
        }
        else {
            throw new Error(`If operatingSystemFamily is ${runtimePlatform.operatingSystemFamily._operatingSystemFamily}, then cpu must be in 1024 (1 vCPU), 2048 (2 vCPU), or 4096 (4 vCPU). Provided value was: ${cpu}`);
        }
    }
    ;
}
exports.TaskDefinition = TaskDefinition;
_a = JSII_RTTI_SYMBOL_1;
TaskDefinition[_a] = { fqn: "monocdk.aws_ecs.TaskDefinition", version: "1.191.0" };
/**
 * The port range to open up for dynamic port mapping
 */
const EPHEMERAL_PORT_RANGE = ec2.Port.tcpRange(32768, 65535);
/**
 * The networking mode to use for the containers in the task.
 */
var NetworkMode;
(function (NetworkMode) {
    /**
     * The task's containers do not have external connectivity and port mappings can't be specified in the container definition.
     */
    NetworkMode["NONE"] = "none";
    /**
     * The task utilizes Docker's built-in virtual network which runs inside each container instance.
     */
    NetworkMode["BRIDGE"] = "bridge";
    /**
     * The task is allocated an elastic network interface.
     */
    NetworkMode["AWS_VPC"] = "awsvpc";
    /**
     * The task bypasses Docker's built-in virtual network and maps container ports directly to the EC2 instance's network interface directly.
     *
     * In this mode, you can't run multiple instantiations of the same task on a
     * single container instance when port mappings are used.
     */
    NetworkMode["HOST"] = "host";
    /**
     * The task utilizes NAT network mode required by Windows containers.
     *
     * This is the only supported network mode for Windows containers. For more information, see
     * [Task Definition Parameters](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html#network_mode).
     */
    NetworkMode["NAT"] = "nat";
})(NetworkMode = exports.NetworkMode || (exports.NetworkMode = {}));
/**
 * The IPC resource namespace to use for the containers in the task.
 */
var IpcMode;
(function (IpcMode) {
    /**
     * If none is specified, then IPC resources within the containers of a task are private and not
     * shared with other containers in a task or on the container instance
     */
    IpcMode["NONE"] = "none";
    /**
     * If host is specified, then all containers within the tasks that specified the host IPC mode on
     * the same container instance share the same IPC resources with the host Amazon EC2 instance.
     */
    IpcMode["HOST"] = "host";
    /**
     * If task is specified, all containers within the specified task share the same IPC resources.
     */
    IpcMode["TASK"] = "task";
})(IpcMode = exports.IpcMode || (exports.IpcMode = {}));
/**
 * The process namespace to use for the containers in the task.
 */
var PidMode;
(function (PidMode) {
    /**
     * If host is specified, then all containers within the tasks that specified the host PID mode
     * on the same container instance share the same process namespace with the host Amazon EC2 instance.
     */
    PidMode["HOST"] = "host";
    /**
     * If task is specified, all containers within the specified task share the same process namespace.
     */
    PidMode["TASK"] = "task";
})(PidMode = exports.PidMode || (exports.PidMode = {}));
/**
 * The scope for the Docker volume that determines its lifecycle.
 * Docker volumes that are scoped to a task are automatically provisioned when the task starts and destroyed when the task stops.
 * Docker volumes that are scoped as shared persist after the task stops.
 */
var Scope;
(function (Scope) {
    /**
     * Docker volumes that are scoped to a task are automatically provisioned when the task starts and destroyed when the task stops.
     */
    Scope["TASK"] = "task";
    /**
     * Docker volumes that are scoped as shared persist after the task stops.
     */
    Scope["SHARED"] = "shared";
})(Scope = exports.Scope || (exports.Scope = {}));
/**
 * The task launch type compatibility requirement.
 */
var Compatibility;
(function (Compatibility) {
    /**
     * The task should specify the EC2 launch type.
     */
    Compatibility[Compatibility["EC2"] = 0] = "EC2";
    /**
     * The task should specify the Fargate launch type.
     */
    Compatibility[Compatibility["FARGATE"] = 1] = "FARGATE";
    /**
     * The task can specify either the EC2 or Fargate launch types.
     */
    Compatibility[Compatibility["EC2_AND_FARGATE"] = 2] = "EC2_AND_FARGATE";
    /**
     * The task should specify the External launch type.
     */
    Compatibility[Compatibility["EXTERNAL"] = 3] = "EXTERNAL";
})(Compatibility = exports.Compatibility || (exports.Compatibility = {}));
/**
 * Return true if the given task definition can be run on an EC2 cluster
 */
function isEc2Compatible(compatibility) {
    return [Compatibility.EC2, Compatibility.EC2_AND_FARGATE].includes(compatibility);
}
exports.isEc2Compatible = isEc2Compatible;
/**
 * Return true if the given task definition can be run on a Fargate cluster
 */
function isFargateCompatible(compatibility) {
    return [Compatibility.FARGATE, Compatibility.EC2_AND_FARGATE].includes(compatibility);
}
exports.isFargateCompatible = isFargateCompatible;
/**
 * Return true if the given task definition can be run on a ECS Anywhere cluster
 */
function isExternalCompatible(compatibility) {
    return [Compatibility.EXTERNAL].includes(compatibility);
}
exports.isExternalCompatible = isExternalCompatible;
//# sourceMappingURL=data:application/json;base64,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