"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AsgCapacityProvider = exports.ExecuteCommandLogging = exports.Cluster = exports.MachineImageType = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const autoscaling = require("../../aws-autoscaling");
const cloudwatch = require("../../aws-cloudwatch");
const ec2 = require("../../aws-ec2");
const iam = require("../../aws-iam");
const cloudmap = require("../../aws-servicediscovery");
const core_1 = require("../../core");
const amis_1 = require("./amis");
const instance_drain_hook_1 = require("./drain-hook/instance-drain-hook");
const ecs_canned_metrics_generated_1 = require("./ecs-canned-metrics.generated");
const ecs_generated_1 = require("./ecs.generated");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("../../core");
/**
 * The machine image type
 */
var MachineImageType;
(function (MachineImageType) {
    /**
     * Amazon ECS-optimized Amazon Linux 2 AMI
     */
    MachineImageType[MachineImageType["AMAZON_LINUX_2"] = 0] = "AMAZON_LINUX_2";
    /**
     * Bottlerocket AMI
     */
    MachineImageType[MachineImageType["BOTTLEROCKET"] = 1] = "BOTTLEROCKET";
})(MachineImageType = exports.MachineImageType || (exports.MachineImageType = {}));
/**
 * A regional grouping of one or more container instances on which you can run tasks and services.
 */
class Cluster extends core_1.Resource {
    /**
     * Constructs a new instance of the Cluster class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.clusterName,
        });
        /**
         * Manage the allowed network connections for the cluster with Security Groups.
         */
        this.connections = new ec2.Connections();
        /**
         * The names of both ASG and Fargate capacity providers associated with the cluster.
         */
        this._capacityProviderNames = [];
        /**
         * Specifies whether the cluster has EC2 instance capacity.
         */
        this._hasEc2Capacity = false;
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_ClusterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Cluster);
            }
            throw error;
        }
        /**
         * clusterSettings needs to be undefined if containerInsights is not explicitly set in order to allow any
         * containerInsights settings on the account to apply.  See:
         * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-clustersettings.html#cfn-ecs-cluster-clustersettings-value
        */
        let clusterSettings = undefined;
        if (props.containerInsights !== undefined) {
            clusterSettings = [{ name: 'containerInsights', value: props.containerInsights ? ContainerInsights.ENABLED : ContainerInsights.DISABLED }];
        }
        this._capacityProviderNames = props.capacityProviders ?? [];
        if (props.enableFargateCapacityProviders) {
            this.enableFargateCapacityProviders();
        }
        if (props.executeCommandConfiguration) {
            if ((props.executeCommandConfiguration.logging === ExecuteCommandLogging.OVERRIDE) !==
                (props.executeCommandConfiguration.logConfiguration !== undefined)) {
                throw new Error('Execute command log configuration must only be specified when logging is OVERRIDE.');
            }
            this._executeCommandConfiguration = props.executeCommandConfiguration;
        }
        const cluster = new ecs_generated_1.CfnCluster(this, 'Resource', {
            clusterName: this.physicalName,
            clusterSettings,
            configuration: this._executeCommandConfiguration && this.renderExecuteCommandConfiguration(),
        });
        this.clusterArn = this.getResourceArnAttribute(cluster.attrArn, {
            service: 'ecs',
            resource: 'cluster',
            resourceName: this.physicalName,
        });
        this.clusterName = this.getResourceNameAttribute(cluster.ref);
        this.vpc = props.vpc || new ec2.Vpc(this, 'Vpc', { maxAzs: 2 });
        this._defaultCloudMapNamespace = props.defaultCloudMapNamespace !== undefined
            ? this.addDefaultCloudMapNamespace(props.defaultCloudMapNamespace)
            : undefined;
        this._autoscalingGroup = props.capacity !== undefined
            ? this.addCapacity('DefaultAutoScalingGroup', props.capacity)
            : undefined;
        // Only create cluster capacity provider associations if there are any EC2
        // capacity providers. Ordinarily we'd just add the construct to the tree
        // since it's harmless, but we'd prefer not to add unexpected new
        // resources to the stack which could surprise users working with
        // brown-field CDK apps and stacks.
        core_1.Aspects.of(this).add(new MaybeCreateCapacityProviderAssociations(this, id, this._capacityProviderNames));
    }
    /**
     * Import an existing cluster to the stack from its attributes.
     */
    static fromClusterAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_ClusterAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromClusterAttributes);
            }
            throw error;
        }
        return new ImportedCluster(scope, id, attrs);
    }
    /**
     * Import an existing cluster to the stack from the cluster ARN.
     * This does not provide access to the vpc, hasEc2Capacity, or connections -
     * use the `fromClusterAttributes` method to access those properties.
     */
    static fromClusterArn(scope, id, clusterArn) {
        const stack = core_1.Stack.of(scope);
        const arn = stack.splitArn(clusterArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        const clusterName = arn.resourceName;
        if (!clusterName) {
            throw new Error(`Missing required Cluster Name from Cluster ARN: ${clusterArn}`);
        }
        const errorSuffix = 'is not available for a Cluster imported using fromClusterArn(), please use fromClusterAttributes() instead.';
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.clusterArn = clusterArn;
                this.clusterName = clusterName;
            }
            get hasEc2Capacity() {
                throw new Error(`hasEc2Capacity ${errorSuffix}`);
            }
            get connections() {
                throw new Error(`connections ${errorSuffix}`);
            }
            get vpc() {
                throw new Error(`vpc ${errorSuffix}`);
            }
        }
        return new Import(scope, id, {
            environmentFromArn: clusterArn,
        });
    }
    /**
     * Enable the Fargate capacity providers for this cluster.
     */
    enableFargateCapacityProviders() {
        for (const provider of ['FARGATE', 'FARGATE_SPOT']) {
            if (!this._capacityProviderNames.includes(provider)) {
                this._capacityProviderNames.push(provider);
            }
        }
    }
    renderExecuteCommandConfiguration() {
        return {
            executeCommandConfiguration: {
                kmsKeyId: this._executeCommandConfiguration?.kmsKey?.keyArn,
                logConfiguration: this._executeCommandConfiguration?.logConfiguration && this.renderExecuteCommandLogConfiguration(),
                logging: this._executeCommandConfiguration?.logging,
            },
        };
    }
    renderExecuteCommandLogConfiguration() {
        const logConfiguration = this._executeCommandConfiguration?.logConfiguration;
        if (logConfiguration?.s3EncryptionEnabled && !logConfiguration?.s3Bucket) {
            throw new Error('You must specify an S3 bucket name in the execute command log configuration to enable S3 encryption.');
        }
        if (logConfiguration?.cloudWatchEncryptionEnabled && !logConfiguration?.cloudWatchLogGroup) {
            throw new Error('You must specify a CloudWatch log group in the execute command log configuration to enable CloudWatch encryption.');
        }
        return {
            cloudWatchEncryptionEnabled: logConfiguration?.cloudWatchEncryptionEnabled,
            cloudWatchLogGroupName: logConfiguration?.cloudWatchLogGroup?.logGroupName,
            s3BucketName: logConfiguration?.s3Bucket?.bucketName,
            s3EncryptionEnabled: logConfiguration?.s3EncryptionEnabled,
            s3KeyPrefix: logConfiguration?.s3KeyPrefix,
        };
    }
    /**
     * Add an AWS Cloud Map DNS namespace for this cluster.
     * NOTE: HttpNamespaces are not supported, as ECS always requires a DNSConfig when registering an instance to a Cloud
     * Map service.
     */
    addDefaultCloudMapNamespace(options) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_CloudMapNamespaceOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addDefaultCloudMapNamespace);
            }
            throw error;
        }
        if (this._defaultCloudMapNamespace !== undefined) {
            throw new Error('Can only add default namespace once.');
        }
        const namespaceType = options.type !== undefined
            ? options.type
            : cloudmap.NamespaceType.DNS_PRIVATE;
        const sdNamespace = namespaceType === cloudmap.NamespaceType.DNS_PRIVATE ?
            new cloudmap.PrivateDnsNamespace(this, 'DefaultServiceDiscoveryNamespace', {
                name: options.name,
                vpc: this.vpc,
            }) :
            new cloudmap.PublicDnsNamespace(this, 'DefaultServiceDiscoveryNamespace', {
                name: options.name,
            });
        this._defaultCloudMapNamespace = sdNamespace;
        return sdNamespace;
    }
    /**
     * Getter for namespace added to cluster
     */
    get defaultCloudMapNamespace() {
        return this._defaultCloudMapNamespace;
    }
    /**
     * It is highly recommended to use {@link Cluster.addAsgCapacityProvider} instead of this method.
     *
     * This method adds compute capacity to a cluster by creating an AutoScalingGroup with the specified options.
     *
     * Returns the AutoScalingGroup so you can add autoscaling settings to it.
     */
    addCapacity(id, options) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_AddCapacityOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addCapacity);
            }
            throw error;
        }
        // Do 2-way defaulting here: if the machineImageType is BOTTLEROCKET, pick the right AMI.
        // Otherwise, determine the machineImageType from the given AMI.
        const machineImage = options.machineImage ??
            (options.machineImageType === MachineImageType.BOTTLEROCKET ? new amis_1.BottleRocketImage({
                architecture: options.instanceType.architecture,
            }) : new amis_1.EcsOptimizedAmi());
        const machineImageType = options.machineImageType ??
            (isBottleRocketImage(machineImage) ? MachineImageType.BOTTLEROCKET : MachineImageType.AMAZON_LINUX_2);
        const autoScalingGroup = new autoscaling.AutoScalingGroup(this, id, {
            vpc: this.vpc,
            machineImage,
            updateType: options.updateType || autoscaling.UpdateType.REPLACING_UPDATE,
            ...options,
        });
        this.addAutoScalingGroup(autoScalingGroup, {
            machineImageType: machineImageType,
            ...options,
        });
        return autoScalingGroup;
    }
    /**
     * This method adds an Auto Scaling Group Capacity Provider to a cluster.
     *
     * @param provider the capacity provider to add to this cluster.
     */
    addAsgCapacityProvider(provider, options = {}) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_AsgCapacityProvider(provider);
            jsiiDeprecationWarnings.monocdk_aws_ecs_AddAutoScalingGroupCapacityOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addAsgCapacityProvider);
            }
            throw error;
        }
        // Don't add the same capacity provider more than once.
        if (this._capacityProviderNames.includes(provider.capacityProviderName)) {
            return;
        }
        this._hasEc2Capacity = true;
        this.configureAutoScalingGroup(provider.autoScalingGroup, {
            ...options,
            machineImageType: provider.machineImageType,
            // Don't enable the instance-draining lifecycle hook if managed termination protection is enabled
            taskDrainTime: provider.enableManagedTerminationProtection ? core_1.Duration.seconds(0) : options.taskDrainTime,
            canContainersAccessInstanceRole: options.canContainersAccessInstanceRole ?? provider.canContainersAccessInstanceRole,
        });
        this._capacityProviderNames.push(provider.capacityProviderName);
    }
    /**
     * This method adds compute capacity to a cluster using the specified AutoScalingGroup.
     *
     * @deprecated Use {@link Cluster.addAsgCapacityProvider} instead.
     * @param autoScalingGroup the ASG to add to this cluster.
     * [disable-awslint:ref-via-interface] is needed in order to install the ECS
     * agent by updating the ASGs user data.
     */
    addAutoScalingGroup(autoScalingGroup, options = {}) {
        try {
            jsiiDeprecationWarnings.print("monocdk.aws_ecs.Cluster#addAutoScalingGroup", "Use {@link Cluster.addAsgCapacityProvider} instead.");
            jsiiDeprecationWarnings.monocdk_aws_autoscaling_AutoScalingGroup(autoScalingGroup);
            jsiiDeprecationWarnings.monocdk_aws_ecs_AddAutoScalingGroupCapacityOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addAutoScalingGroup);
            }
            throw error;
        }
        this._hasEc2Capacity = true;
        this.connections.connections.addSecurityGroup(...autoScalingGroup.connections.securityGroups);
        this.configureAutoScalingGroup(autoScalingGroup, options);
    }
    configureAutoScalingGroup(autoScalingGroup, options = {}) {
        if (autoScalingGroup.osType === ec2.OperatingSystemType.WINDOWS) {
            this.configureWindowsAutoScalingGroup(autoScalingGroup, options);
        }
        else {
            // Tie instances to cluster
            switch (options.machineImageType) {
                // Bottlerocket AMI
                case MachineImageType.BOTTLEROCKET: {
                    autoScalingGroup.addUserData(
                    // Connect to the cluster
                    // Source: https://github.com/bottlerocket-os/bottlerocket/blob/develop/QUICKSTART-ECS.md#connecting-to-your-cluster
                    '[settings.ecs]', `cluster = "${this.clusterName}"`);
                    // Enabling SSM
                    // Source: https://github.com/bottlerocket-os/bottlerocket/blob/develop/QUICKSTART-ECS.md#enabling-ssm
                    autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
                    // required managed policy
                    autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2ContainerServiceforEC2Role'));
                    break;
                }
                default:
                    // Amazon ECS-optimized AMI for Amazon Linux 2
                    autoScalingGroup.addUserData(`echo ECS_CLUSTER=${this.clusterName} >> /etc/ecs/ecs.config`);
                    if (!options.canContainersAccessInstanceRole) {
                        // Deny containers access to instance metadata service
                        // Source: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/instance_IAM_role.html
                        autoScalingGroup.addUserData('sudo iptables --insert FORWARD 1 --in-interface docker+ --destination 169.254.169.254/32 --jump DROP');
                        autoScalingGroup.addUserData('sudo service iptables save');
                        // The following is only for AwsVpc networking mode, but doesn't hurt for the other modes.
                        autoScalingGroup.addUserData('echo ECS_AWSVPC_BLOCK_IMDS=true >> /etc/ecs/ecs.config');
                    }
                    if (autoScalingGroup.spotPrice && options.spotInstanceDraining) {
                        autoScalingGroup.addUserData('echo ECS_ENABLE_SPOT_INSTANCE_DRAINING=true >> /etc/ecs/ecs.config');
                    }
            }
        }
        // ECS instances must be able to do these things
        // Source: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/instance_IAM_role.html
        // But, scoped down to minimal permissions required.
        //  Notes:
        //   - 'ecs:CreateCluster' removed. The cluster already exists.
        autoScalingGroup.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ecs:DeregisterContainerInstance',
                'ecs:RegisterContainerInstance',
                'ecs:Submit*',
            ],
            resources: [
                this.clusterArn,
            ],
        }));
        autoScalingGroup.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                // These act on a cluster instance, and the instance doesn't exist until the service starts.
                // Thus, scope to the cluster using a condition.
                // See: https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticcontainerservice.html
                'ecs:Poll',
                'ecs:StartTelemetrySession',
            ],
            resources: ['*'],
            conditions: {
                ArnEquals: { 'ecs:cluster': this.clusterArn },
            },
        }));
        autoScalingGroup.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                // These do not support resource constraints, and must be resource '*'
                'ecs:DiscoverPollEndpoint',
                'ecr:GetAuthorizationToken',
                // Preserved for backwards compatibility.
                // Users are able to enable cloudwatch agent using CDK. Existing
                // customers might be installing CW agent as part of user-data so if we
                // remove these permissions we will break that customer use cases.
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: ['*'],
        }));
        // 0 disables, otherwise forward to underlying implementation which picks the sane default
        if (!options.taskDrainTime || options.taskDrainTime.toSeconds() !== 0) {
            new instance_drain_hook_1.InstanceDrainHook(autoScalingGroup, 'DrainECSHook', {
                autoScalingGroup,
                cluster: this,
                drainTime: options.taskDrainTime,
                topicEncryptionKey: options.topicEncryptionKey,
            });
        }
    }
    /**
     * This method enables the Fargate or Fargate Spot capacity providers on the cluster.
     *
     * @param provider the capacity provider to add to this cluster.
     * @deprecated Use {@link enableFargateCapacityProviders} instead.
     * @see {@link addAsgCapacityProvider} to add an Auto Scaling Group capacity provider to the cluster.
     */
    addCapacityProvider(provider) {
        try {
            jsiiDeprecationWarnings.print("monocdk.aws_ecs.Cluster#addCapacityProvider", "Use {@link enableFargateCapacityProviders} instead.");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addCapacityProvider);
            }
            throw error;
        }
        if (!(provider === 'FARGATE' || provider === 'FARGATE_SPOT')) {
            throw new Error('CapacityProvider not supported');
        }
        if (!this._capacityProviderNames.includes(provider)) {
            this._capacityProviderNames.push(provider);
        }
    }
    configureWindowsAutoScalingGroup(autoScalingGroup, options = {}) {
        // clear the cache of the agent
        autoScalingGroup.addUserData('Remove-Item -Recurse C:\\ProgramData\\Amazon\\ECS\\Cache');
        // pull the latest ECS Tools
        autoScalingGroup.addUserData('Import-Module ECSTools');
        // set the cluster name environment variable
        autoScalingGroup.addUserData(`[Environment]::SetEnvironmentVariable("ECS_CLUSTER", "${this.clusterName}", "Machine")`);
        autoScalingGroup.addUserData('[Environment]::SetEnvironmentVariable("ECS_ENABLE_AWSLOGS_EXECUTIONROLE_OVERRIDE", "true", "Machine")');
        // tslint:disable-next-line: max-line-length
        autoScalingGroup.addUserData('[Environment]::SetEnvironmentVariable("ECS_AVAILABLE_LOGGING_DRIVERS", \'["json-file","awslogs"]\', "Machine")');
        // enable instance draining
        if (autoScalingGroup.spotPrice && options.spotInstanceDraining) {
            autoScalingGroup.addUserData('[Environment]::SetEnvironmentVariable("ECS_ENABLE_SPOT_INSTANCE_DRAINING", "true", "Machine")');
        }
        // enable task iam role
        if (!options.canContainersAccessInstanceRole) {
            autoScalingGroup.addUserData('[Environment]::SetEnvironmentVariable("ECS_ENABLE_TASK_IAM_ROLE", "true", "Machine")');
            autoScalingGroup.addUserData(`Initialize-ECSAgent -Cluster '${this.clusterName}' -EnableTaskIAMRole`);
        }
        else {
            autoScalingGroup.addUserData(`Initialize-ECSAgent -Cluster '${this.clusterName}'`);
        }
    }
    /**
     * Getter for autoscaling group added to cluster
     */
    get autoscalingGroup() {
        return this._autoscalingGroup;
    }
    /**
     * Whether the cluster has EC2 capacity associated with it
     */
    get hasEc2Capacity() {
        return this._hasEc2Capacity;
    }
    /**
     * Getter for execute command configuration associated with the cluster.
     */
    get executeCommandConfiguration() {
        return this._executeCommandConfiguration;
    }
    /**
     * This method returns the CloudWatch metric for this clusters CPU reservation.
     *
     * @default average over 5 minutes
     */
    metricCpuReservation(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricCpuReservation);
            }
            throw error;
        }
        return this.cannedMetric(ecs_canned_metrics_generated_1.ECSMetrics.cpuReservationAverage, props);
    }
    /**
     * This method returns the CloudWatch metric for this clusters CPU utilization.
     *
     * @default average over 5 minutes
     */
    metricCpuUtilization(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricCpuUtilization);
            }
            throw error;
        }
        return this.cannedMetric(ecs_canned_metrics_generated_1.ECSMetrics.cpuUtilizationAverage, props);
    }
    /**
     * This method returns the CloudWatch metric for this clusters memory reservation.
     *
     * @default average over 5 minutes
     */
    metricMemoryReservation(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricMemoryReservation);
            }
            throw error;
        }
        return this.cannedMetric(ecs_canned_metrics_generated_1.ECSMetrics.memoryReservationAverage, props);
    }
    /**
     * This method returns the CloudWatch metric for this clusters memory utilization.
     *
     * @default average over 5 minutes
     */
    metricMemoryUtilization(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricMemoryUtilization);
            }
            throw error;
        }
        return this.cannedMetric(ecs_canned_metrics_generated_1.ECSMetrics.memoryUtilizationAverage, props);
    }
    /**
     * This method returns the specifed CloudWatch metric for this cluster.
     */
    metric(metricName, props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metric);
            }
            throw error;
        }
        return new cloudwatch.Metric({
            namespace: 'AWS/ECS',
            metricName,
            dimensionsMap: { ClusterName: this.clusterName },
            ...props,
        }).attachTo(this);
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({ ClusterName: this.clusterName }),
            ...props,
        }).attachTo(this);
    }
}
exports.Cluster = Cluster;
_a = JSII_RTTI_SYMBOL_1;
Cluster[_a] = { fqn: "monocdk.aws_ecs.Cluster", version: "1.191.0" };
/**
 * An Cluster that has been imported
 */
class ImportedCluster extends core_1.Resource {
    /**
     * Constructs a new instance of the ImportedCluster class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Security group of the cluster instances
         */
        this.connections = new ec2.Connections();
        this.clusterName = props.clusterName;
        this.vpc = props.vpc;
        this.hasEc2Capacity = props.hasEc2Capacity !== false;
        this._defaultCloudMapNamespace = props.defaultCloudMapNamespace;
        this._executeCommandConfiguration = props.executeCommandConfiguration;
        this.clusterArn = props.clusterArn ?? core_1.Stack.of(this).formatArn({
            service: 'ecs',
            resource: 'cluster',
            resourceName: props.clusterName,
        });
        this.connections = new ec2.Connections({
            securityGroups: props.securityGroups,
        });
    }
    get defaultCloudMapNamespace() {
        return this._defaultCloudMapNamespace;
    }
    get executeCommandConfiguration() {
        return this._executeCommandConfiguration;
    }
}
var ContainerInsights;
(function (ContainerInsights) {
    /**
     * Enable CloudWatch Container Insights for the cluster
     */
    ContainerInsights["ENABLED"] = "enabled";
    /**
     * Disable CloudWatch Container Insights for the cluster
     */
    ContainerInsights["DISABLED"] = "disabled";
})(ContainerInsights || (ContainerInsights = {}));
/**
 * The log settings to use to for logging the execute command session. For more information, see
 * [Logging] https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandconfiguration.html#cfn-ecs-cluster-executecommandconfiguration-logging
 */
var ExecuteCommandLogging;
(function (ExecuteCommandLogging) {
    /**
     * The execute command session is not logged.
     */
    ExecuteCommandLogging["NONE"] = "NONE";
    /**
     * The awslogs configuration in the task definition is used. If no logging parameter is specified, it defaults to this value. If no awslogs log driver is configured in the task definition, the output won't be logged.
     */
    ExecuteCommandLogging["DEFAULT"] = "DEFAULT";
    /**
     * Specify the logging details as a part of logConfiguration.
     */
    ExecuteCommandLogging["OVERRIDE"] = "OVERRIDE";
})(ExecuteCommandLogging = exports.ExecuteCommandLogging || (exports.ExecuteCommandLogging = {}));
/**
 * An Auto Scaling Group Capacity Provider. This allows an ECS cluster to target
 * a specific EC2 Auto Scaling Group for the placement of tasks. Optionally (and
 * recommended), ECS can manage the number of instances in the ASG to fit the
 * tasks, and can ensure that instances are not prematurely terminated while
 * there are still tasks running on them.
 */
class AsgCapacityProvider extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_AsgCapacityProviderProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, AsgCapacityProvider);
            }
            throw error;
        }
        this.autoScalingGroup = props.autoScalingGroup;
        this.machineImageType = props.machineImageType ?? MachineImageType.AMAZON_LINUX_2;
        this.canContainersAccessInstanceRole = props.canContainersAccessInstanceRole;
        this.enableManagedTerminationProtection =
            props.enableManagedTerminationProtection === undefined ? true : props.enableManagedTerminationProtection;
        if (this.enableManagedTerminationProtection) {
            this.autoScalingGroup.protectNewInstancesFromScaleIn();
        }
        if (props.capacityProviderName) {
            if (!(/^(?!aws|ecs|fargate).+/gm.test(props.capacityProviderName))) {
                throw new Error(`Invalid Capacity Provider Name: ${props.capacityProviderName}, If a name is specified, it cannot start with aws, ecs, or fargate.`);
            }
        }
        const capacityProvider = new ecs_generated_1.CfnCapacityProvider(this, id, {
            name: props.capacityProviderName,
            autoScalingGroupProvider: {
                autoScalingGroupArn: this.autoScalingGroup.autoScalingGroupName,
                managedScaling: props.enableManagedScaling === false ? undefined : {
                    status: 'ENABLED',
                    targetCapacity: props.targetCapacityPercent || 100,
                    maximumScalingStepSize: props.maximumScalingStepSize,
                    minimumScalingStepSize: props.minimumScalingStepSize,
                },
                managedTerminationProtection: this.enableManagedTerminationProtection ? 'ENABLED' : 'DISABLED',
            },
        });
        this.capacityProviderName = capacityProvider.ref;
    }
}
exports.AsgCapacityProvider = AsgCapacityProvider;
_b = JSII_RTTI_SYMBOL_1;
AsgCapacityProvider[_b] = { fqn: "monocdk.aws_ecs.AsgCapacityProvider", version: "1.191.0" };
/**
 * A visitor that adds a capacity provider association to a Cluster only if
 * the caller created any EC2 Capacity Providers.
 */
class MaybeCreateCapacityProviderAssociations {
    constructor(scope, id, capacityProviders) {
        this.scope = scope;
        this.id = id;
        this.capacityProviders = capacityProviders;
    }
    visit(node) {
        if (node instanceof Cluster) {
            if (this.capacityProviders.length > 0 && !this.resource) {
                const resource = new ecs_generated_1.CfnClusterCapacityProviderAssociations(this.scope, this.id, {
                    cluster: node.clusterName,
                    defaultCapacityProviderStrategy: [],
                    capacityProviders: core_1.Lazy.list({ produce: () => this.capacityProviders }),
                });
                this.resource = resource;
            }
        }
    }
}
function isBottleRocketImage(image) {
    return image instanceof amis_1.BottleRocketImage;
}
//# sourceMappingURL=data:application/json;base64,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