"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cognito = require("../../aws-cognito");
const ec2 = require("../../aws-ec2");
const elbv2 = require("../../aws-elasticloadbalancingv2");
const core_1 = require("../../core");
const actions = require("../lib");
class CognitoStack extends core_1.Stack {
    /// !hide
    constructor(scope, id) {
        super(scope, id);
        const vpc = new ec2.Vpc(this, 'Stack', {
            maxAzs: 2,
        });
        const certificate = {
            certificateArn: process.env.SELF_SIGNED_CERT_ARN ?? '',
        };
        /// !show
        const lb = new elbv2.ApplicationLoadBalancer(this, 'LB', {
            vpc,
            internetFacing: true,
        });
        const userPool = new cognito.UserPool(this, 'UserPool');
        const userPoolClient = new cognito.UserPoolClient(this, 'Client', {
            userPool,
            // Required minimal configuration for use with an ELB
            generateSecret: true,
            authFlows: {
                userPassword: true,
            },
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                },
                scopes: [cognito.OAuthScope.EMAIL],
                callbackUrls: [
                    `https://${lb.loadBalancerDnsName}/oauth2/idpresponse`,
                ],
            },
        });
        const cfnClient = userPoolClient.node.defaultChild;
        cfnClient.addPropertyOverride('RefreshTokenValidity', 1);
        cfnClient.addPropertyOverride('SupportedIdentityProviders', ['COGNITO']);
        const userPoolDomain = new cognito.UserPoolDomain(this, 'Domain', {
            userPool,
            cognitoDomain: {
                domainPrefix: 'test-cdk-prefix',
            },
        });
        lb.addListener('Listener', {
            port: 443,
            certificates: [certificate],
            defaultAction: new actions.AuthenticateCognitoAction({
                userPool,
                userPoolClient,
                userPoolDomain,
                next: elbv2.ListenerAction.fixedResponse(200, {
                    contentType: 'text/plain',
                    messageBody: 'Authenticated',
                }),
            }),
        });
        new core_1.CfnOutput(this, 'DNS', {
            value: lb.loadBalancerDnsName,
        });
    }
}
const app = new core_1.App();
new CognitoStack(app, 'integ-cognito');
app.synth();
//# sourceMappingURL=data:application/json;base64,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