"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SqsQueue = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const util_1 = require("./util");
/**
 * Use an SQS Queue as a target for Amazon EventBridge rules.
 *
 * @example
 *   /// fixture=withRepoAndSqsQueue
 *   // publish to an SQS queue every time code is committed
 *   // to a CodeCommit repository
 *   repository.onCommit('onCommit', { target: new targets.SqsQueue(queue) });
 *
 */
class SqsQueue {
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
        try {
            jsiiDeprecationWarnings.monocdk_aws_sqs_IQueue(queue);
            jsiiDeprecationWarnings.monocdk_aws_events_targets_SqsQueueProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, SqsQueue);
            }
            throw error;
        }
        if (props.messageGroupId !== undefined && !queue.fifo) {
            throw new Error('messageGroupId cannot be specified for non-FIFO queues');
        }
    }
    /**
     * Returns a RuleTarget that can be used to trigger this SQS queue as a
     * result from an EventBridge event.
     *
     * @see https://docs.aws.amazon.com/eventbridge/latest/userguide/resource-based-policies-eventbridge.html#sqs-permissions
     */
    bind(rule, _id) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_events_IRule(rule);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        // Only add the rule as a condition if the queue is not encrypted, to avoid circular dependency. See issue #11158.
        const principalOpts = this.queue.encryptionMasterKey ? {} : {
            conditions: {
                ArnEquals: { 'aws:SourceArn': rule.ruleArn },
            },
        };
        // deduplicated automatically
        this.queue.grantSendMessages(new iam.ServicePrincipal('events.amazonaws.com', principalOpts));
        if (this.props.deadLetterQueue) {
            util_1.addToDeadLetterQueueResourcePolicy(rule, this.props.deadLetterQueue);
        }
        return {
            ...util_1.bindBaseTargetConfig(this.props),
            arn: this.queue.queueArn,
            input: this.props.message,
            targetResource: this.queue,
            sqsParameters: this.props.messageGroupId ? { messageGroupId: this.props.messageGroupId } : undefined,
        };
    }
}
exports.SqsQueue = SqsQueue;
_a = JSII_RTTI_SYMBOL_1;
SqsQueue[_a] = { fqn: "monocdk.aws_events_targets.SqsQueue", version: "1.191.0" };
//# sourceMappingURL=data:application/json;base64,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