"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const diff_1 = require("./diff");
const external_1 = require("./external");
async function handler(event) {
    if (event.RequestType === 'Create') {
        return onCreate(event);
    }
    if (event.RequestType === 'Update') {
        return onUpdate(event);
    }
    if (event.RequestType === 'Delete') {
        return onDelete(event);
    }
    throw new Error('invalid request type');
}
exports.handler = handler;
async function onCreate(event) {
    const issuerUrl = event.ResourceProperties.Url;
    const thumbprints = (event.ResourceProperties.ThumbprintList ?? []).sort(); // keep sorted for UPDATE
    const clients = (event.ResourceProperties.ClientIDList ?? []).sort();
    if (thumbprints.length === 0) {
        thumbprints.push(await external_1.external.downloadThumbprint(issuerUrl));
    }
    const resp = await external_1.external.createOpenIDConnectProvider({
        Url: issuerUrl,
        ClientIDList: clients,
        ThumbprintList: thumbprints,
    });
    return {
        PhysicalResourceId: resp.OpenIDConnectProviderArn,
        Data: {
            Thumbprints: JSON.stringify(thumbprints),
        },
    };
}
async function onUpdate(event) {
    const issuerUrl = event.ResourceProperties.Url;
    const thumbprints = (event.ResourceProperties.ThumbprintList ?? []).sort(); // keep sorted for UPDATE
    const clients = (event.ResourceProperties.ClientIDList ?? []).sort();
    // determine which update we are talking about.
    const oldIssuerUrl = event.OldResourceProperties.Url;
    // if this is a URL update, then we basically create a new resource and cfn will delete the old one
    // since the physical resource ID will change.
    if (oldIssuerUrl !== issuerUrl) {
        return onCreate({ ...event, RequestType: 'Create' });
    }
    const providerArn = event.PhysicalResourceId;
    if (thumbprints.length === 0) {
        thumbprints.push(await external_1.external.downloadThumbprint(issuerUrl));
    }
    external_1.external.log('updating thumbprint to', thumbprints);
    await external_1.external.updateOpenIDConnectProviderThumbprint({
        OpenIDConnectProviderArn: providerArn,
        ThumbprintList: thumbprints,
    });
    // if client ID list has changed, determine "diff" because the API is add/remove
    const oldClients = (event.OldResourceProperties.ClientIDList || []).sort();
    const diff = diff_1.arrayDiff(oldClients, clients);
    external_1.external.log(`client ID diff: ${JSON.stringify(diff)}`);
    for (const addClient of diff.adds) {
        external_1.external.log(`adding client id "${addClient}" to provider ${providerArn}`);
        await external_1.external.addClientIDToOpenIDConnectProvider({
            OpenIDConnectProviderArn: providerArn,
            ClientID: addClient,
        });
    }
    for (const deleteClient of diff.deletes) {
        external_1.external.log(`removing client id "${deleteClient}" from provider ${providerArn}`);
        await external_1.external.removeClientIDFromOpenIDConnectProvider({
            OpenIDConnectProviderArn: providerArn,
            ClientID: deleteClient,
        });
    }
    return {
        Data: {
            Thumbprints: JSON.stringify(thumbprints),
        },
    };
}
async function onDelete(deleteEvent) {
    await external_1.external.deleteOpenIDConnectProvider({
        OpenIDConnectProviderArn: deleteEvent.PhysicalResourceId,
    });
}
//# sourceMappingURL=data:application/json;base64,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