"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StreamMode = exports.StreamEncryption = exports.Stream = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("../../aws-cloudwatch");
const iam = require("../../aws-iam");
const kms = require("../../aws-kms");
const core_1 = require("../../core");
const kinesis_fixed_canned_metrics_1 = require("./kinesis-fixed-canned-metrics");
const kinesis_generated_1 = require("./kinesis.generated");
const READ_OPERATIONS = [
    'kinesis:DescribeStreamSummary',
    'kinesis:GetRecords',
    'kinesis:GetShardIterator',
    'kinesis:ListShards',
    'kinesis:SubscribeToShard',
    'kinesis:DescribeStream',
    'kinesis:ListStreams',
];
const WRITE_OPERATIONS = [
    'kinesis:ListShards',
    'kinesis:PutRecord',
    'kinesis:PutRecords',
];
/**
 * Represents a Kinesis Stream.
 */
class StreamBase extends core_1.Resource {
    /**
     * Grant read permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to ues the key to decrypt the
     * contents of the stream will also be granted.
     */
    grantRead(grantee) {
        const ret = this.grant(grantee, ...READ_OPERATIONS);
        if (this.encryptionKey) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        return ret;
    }
    /**
     * Grant write permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to ues the key to encrypt the
     * contents of the stream will also be granted.
     */
    grantWrite(grantee) {
        const ret = this.grant(grantee, ...WRITE_OPERATIONS);
        this.encryptionKey?.grantEncrypt(grantee);
        return ret;
    }
    /**
     * Grants read/write permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to use the key for
     * encrypt/decrypt will also be granted.
     */
    grantReadWrite(grantee) {
        const ret = this.grant(grantee, ...Array.from(new Set([...READ_OPERATIONS, ...WRITE_OPERATIONS])));
        this.encryptionKey?.grantEncryptDecrypt(grantee);
        return ret;
    }
    /**
     * Grant the indicated permissions on this stream to the given IAM principal (Role/Group/User).
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.streamArn],
            scope: this,
        });
    }
    /**
     * Return stream metric based from its metric name
     *
     * @param metricName name of the stream metric
     * @param props properties of the metric
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Kinesis',
            metricName,
            dimensionsMap: {
                StreamName: this.streamName,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * The number of bytes retrieved from the Kinesis stream, measured over the specified time period. Minimum, Maximum,
     * and Average statistics represent the bytes in a single GetRecords operation for the stream in the specified time
     * period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricGetRecordsBytes(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.getRecordsBytesAverage, props);
    }
    /**
     * The age of the last record in all GetRecords calls made against a Kinesis stream, measured over the specified time
     * period. Age is the difference between the current time and when the last record of the GetRecords call was written
     * to the stream. The Minimum and Maximum statistics can be used to track the progress of Kinesis consumer
     * applications. A value of zero indicates that the records being read are completely caught up with the stream.
     *
     * The metric defaults to maximum over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricGetRecordsIteratorAgeMilliseconds(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.getRecordsIteratorAgeMillisecondsMaximum, props);
    }
    /**
     * The number of successful GetRecords operations per stream, measured over the specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricGetRecordsSuccess(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.getRecordsSuccessAverage, props);
    }
    /**
     * The number of records retrieved from the shard, measured over the specified time period. Minimum, Maximum, and
     * Average statistics represent the records in a single GetRecords operation for the stream in the specified time
     * period.
     *
     * average
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricGetRecords(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.getRecordsRecordsAverage, props);
    }
    /**
     * The number of successful GetRecords operations per stream, measured over the specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricGetRecordsLatency(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.getRecordsLatencyAverage, props);
    }
    /**
     * The number of bytes put to the Kinesis stream using the PutRecord operation over the specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricPutRecordBytes(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.putRecordBytesAverage, props);
    }
    /**
     * The time taken per PutRecord operation, measured over the specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricPutRecordLatency(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.putRecordLatencyAverage, props);
    }
    /**
     * The number of successful PutRecord operations per Kinesis stream, measured over the specified time period. Average
     * reflects the percentage of successful writes to a stream.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricPutRecordSuccess(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.putRecordSuccessAverage, props);
    }
    /**
     * The number of bytes put to the Kinesis stream using the PutRecords operation over the specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricPutRecordsBytes(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.putRecordsBytesAverage, props);
    }
    /**
     * The time taken per PutRecords operation, measured over the specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricPutRecordsLatency(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.putRecordsLatencyAverage, props);
    }
    /**
     *  The number of PutRecords operations where at least one record succeeded, per Kinesis stream, measured over the
     *  specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricPutRecordsSuccess(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.putRecordsSuccessAverage, props);
    }
    /**
     * The total number of records sent in a PutRecords operation per Kinesis data stream, measured over the specified
     * time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricPutRecordsTotalRecords(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.putRecordsTotalRecordsAverage, props);
    }
    /**
     * The number of successful records in a PutRecords operation per Kinesis data stream, measured over the specified
     * time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricPutRecordsSuccessfulRecords(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.putRecordsSuccessfulRecordsAverage, props);
    }
    /**
     * The number of records rejected due to internal failures in a PutRecords operation per Kinesis data stream,
     * measured over the specified time period. Occasional internal failures are to be expected and should be retried.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricPutRecordsFailedRecords(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.putRecordsFailedRecordsAverage, props);
    }
    /**
     * The number of records rejected due to throttling in a PutRecords operation per Kinesis data stream, measured over
     * the specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricPutRecordsThrottledRecords(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.putRecordsThrottledRecordsAverage, props);
    }
    /**
     * The number of bytes successfully put to the Kinesis stream over the specified time period. This metric includes
     * bytes from PutRecord and PutRecords operations. Minimum, Maximum, and Average statistics represent the bytes in a
     * single put operation for the stream in the specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricIncomingBytes(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.incomingBytesAverage, props);
    }
    /**
     * The number of records successfully put to the Kinesis stream over the specified time period. This metric includes
     * record counts from PutRecord and PutRecords operations. Minimum, Maximum, and Average statistics represent the
     * records in a single put operation for the stream in the specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricIncomingRecords(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.incomingRecordsAverage, props);
    }
    /**
     * The number of GetRecords calls throttled for the stream over the specified time period. The most commonly used
     * statistic for this metric is Average.
     *
     * When the Minimum statistic has a value of 1, all records were throttled for the stream during the specified time
     * period.
     *
     * When the Maximum statistic has a value of 0 (zero), no records were throttled for the stream during the specified
     * time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties
     *
     * @param props properties of the metric
     *
     */
    metricReadProvisionedThroughputExceeded(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.readProvisionedThroughputExceededAverage, props);
    }
    /**
     * The number of records rejected due to throttling for the stream over the specified time period. This metric
     * includes throttling from PutRecord and PutRecords operations.
     *
     * When the Minimum statistic has a non-zero value, records were being throttled for the stream during the specified
     * time period.
     *
     * When the Maximum statistic has a value of 0 (zero), no records were being throttled for the stream during the
     * specified time period.
     *
     * The metric defaults to average over 5 minutes, it can be changed by passing `statistic` and `period` properties.
     *
     * @param props properties of the metric
     */
    metricWriteProvisionedThroughputExceeded(props) {
        return this.metricFromCannedFunction(kinesis_fixed_canned_metrics_1.KinesisMetrics.writeProvisionedThroughputExceededAverage, props);
    }
    // create metrics based on generated KinesisMetrics static methods
    metricFromCannedFunction(createCannedProps, props) {
        return new cloudwatch.Metric({
            ...createCannedProps({ StreamName: this.streamName }),
            ...props,
        }).attachTo(this);
    }
}
/**
 * A Kinesis stream. Can be encrypted with a KMS key.
 */
class Stream extends StreamBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.streamName,
        });
        try {
            jsiiDeprecationWarnings.monocdk_aws_kinesis_StreamProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Stream);
            }
            throw error;
        }
        let shardCount = props.shardCount;
        const streamMode = props.streamMode ?? StreamMode.PROVISIONED;
        if (streamMode === StreamMode.ON_DEMAND && shardCount !== undefined) {
            throw new Error(`streamMode must be set to ${StreamMode.PROVISIONED} (default) when specifying shardCount`);
        }
        if (streamMode === StreamMode.PROVISIONED && shardCount === undefined) {
            shardCount = 1;
        }
        const retentionPeriodHours = props.retentionPeriod?.toHours() ?? 24;
        if (!core_1.Token.isUnresolved(retentionPeriodHours)) {
            if (retentionPeriodHours < 24 || retentionPeriodHours > 8760) {
                throw new Error(`retentionPeriod must be between 24 and 8760 hours. Received ${retentionPeriodHours}`);
            }
        }
        const { streamEncryption, encryptionKey } = this.parseEncryption(props);
        this.stream = new kinesis_generated_1.CfnStream(this, 'Resource', {
            name: this.physicalName,
            retentionPeriodHours,
            shardCount,
            streamEncryption,
            streamModeDetails: streamMode ? { streamMode } : undefined,
        });
        this.streamArn = this.getResourceArnAttribute(this.stream.attrArn, {
            service: 'kinesis',
            resource: 'stream',
            resourceName: this.physicalName,
        });
        this.streamName = this.getResourceNameAttribute(this.stream.ref);
        this.encryptionKey = encryptionKey;
    }
    /**
     * Import an existing Kinesis Stream provided an ARN
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name
     * @param streamArn Stream ARN (i.e. arn:aws:kinesis:<region>:<account-id>:stream/Foo)
     */
    static fromStreamArn(scope, id, streamArn) {
        return Stream.fromStreamAttributes(scope, id, { streamArn });
    }
    /**
     * Creates a Stream construct that represents an external stream.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs Stream import properties
     */
    static fromStreamAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_kinesis_StreamAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromStreamAttributes);
            }
            throw error;
        }
        class Import extends StreamBase {
            constructor() {
                super(...arguments);
                this.streamArn = attrs.streamArn;
                this.streamName = core_1.Stack.of(scope).splitArn(attrs.streamArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Set up key properties and return the Stream encryption property from the
     * user's configuration.
     */
    parseEncryption(props) {
        // if encryption properties are not set, default to KMS in regions where KMS is available
        if (!props.encryption && !props.encryptionKey) {
            const conditionName = 'AwsCdkKinesisEncryptedStreamsUnsupportedRegions';
            const existing = core_1.Stack.of(this).node.tryFindChild(conditionName);
            // create a single condition for the Stack
            if (!existing) {
                new core_1.CfnCondition(core_1.Stack.of(this), conditionName, {
                    expression: core_1.Fn.conditionOr(core_1.Fn.conditionEquals(core_1.Aws.REGION, 'cn-north-1'), core_1.Fn.conditionEquals(core_1.Aws.REGION, 'cn-northwest-1')),
                });
            }
            return {
                streamEncryption: core_1.Fn.conditionIf(conditionName, core_1.Aws.NO_VALUE, { EncryptionType: 'KMS', KeyId: 'alias/aws/kinesis' }),
            };
        }
        // default based on whether encryption key is specified
        const encryptionType = props.encryption ??
            (props.encryptionKey ? StreamEncryption.KMS : StreamEncryption.UNENCRYPTED);
        // if encryption key is set, encryption must be set to KMS.
        if (encryptionType !== StreamEncryption.KMS && props.encryptionKey) {
            throw new Error(`encryptionKey is specified, so 'encryption' must be set to KMS (value: ${encryptionType})`);
        }
        if (encryptionType === StreamEncryption.UNENCRYPTED) {
            return {};
        }
        if (encryptionType === StreamEncryption.MANAGED) {
            const encryption = { encryptionType: 'KMS', keyId: 'alias/aws/kinesis' };
            return { streamEncryption: encryption };
        }
        if (encryptionType === StreamEncryption.KMS) {
            const encryptionKey = props.encryptionKey || new kms.Key(this, 'Key', {
                description: `Created by ${this.node.path}`,
            });
            const streamEncryption = {
                encryptionType: 'KMS',
                keyId: encryptionKey.keyArn,
            };
            return { encryptionKey, streamEncryption };
        }
        throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
    }
}
exports.Stream = Stream;
_a = JSII_RTTI_SYMBOL_1;
Stream[_a] = { fqn: "monocdk.aws_kinesis.Stream", version: "1.191.0" };
/**
 * What kind of server-side encryption to apply to this stream
 */
var StreamEncryption;
(function (StreamEncryption) {
    /**
     * Records in the stream are not encrypted.
     */
    StreamEncryption["UNENCRYPTED"] = "NONE";
    /**
     * Server-side encryption with a KMS key managed by the user.
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    StreamEncryption["KMS"] = "KMS";
    /**
     * Server-side encryption with a master key managed by Amazon Kinesis
     */
    StreamEncryption["MANAGED"] = "MANAGED";
})(StreamEncryption = exports.StreamEncryption || (exports.StreamEncryption = {}));
/**
 * Specifies the capacity mode to apply to this stream.
 */
var StreamMode;
(function (StreamMode) {
    /**
     * Specify the provisioned capacity mode. The stream will have `shardCount` shards unless
     * modified and will be billed according to the provisioned capacity.
     */
    StreamMode["PROVISIONED"] = "PROVISIONED";
    /**
     * Specify the on-demand capacity mode. The stream will autoscale and be billed according to the
     * volume of data ingested and retrieved.
     */
    StreamMode["ON_DEMAND"] = "ON_DEMAND";
})(StreamMode = exports.StreamMode || (exports.StreamMode = {}));
//# sourceMappingURL=data:application/json;base64,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