#!/usr/bin/env node
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/// !cdk-integ pragma:ignore-assets
const path = require("path");
const firehose = require("../../aws-kinesisfirehose");
const kms = require("../../aws-kms");
const lambdanodejs = require("../../aws-lambda-nodejs");
const logs = require("../../aws-logs");
const s3 = require("../../aws-s3");
const cdk = require("../../core");
const destinations = require("../lib");
const app = new cdk.App();
const stack = new cdk.Stack(app, 'aws-cdk-firehose-delivery-stream-s3-all-properties');
const bucket = new s3.Bucket(stack, 'Bucket', {
    removalPolicy: cdk.RemovalPolicy.DESTROY,
    autoDeleteObjects: true,
});
const backupBucket = new s3.Bucket(stack, 'BackupBucket', {
    removalPolicy: cdk.RemovalPolicy.DESTROY,
    autoDeleteObjects: true,
});
const logGroup = new logs.LogGroup(stack, 'LogGroup', {
    removalPolicy: cdk.RemovalPolicy.DESTROY,
});
const dataProcessorFunction = new lambdanodejs.NodejsFunction(stack, 'DataProcessorFunction', {
    entry: path.join(__dirname, 'lambda-data-processor.js'),
    timeout: cdk.Duration.minutes(1),
});
const processor = new firehose.LambdaFunctionProcessor(dataProcessorFunction, {
    bufferInterval: cdk.Duration.seconds(60),
    bufferSize: cdk.Size.mebibytes(1),
    retries: 1,
});
const key = new kms.Key(stack, 'Key', {
    removalPolicy: cdk.RemovalPolicy.DESTROY,
});
const backupKey = new kms.Key(stack, 'BackupKey', {
    removalPolicy: cdk.RemovalPolicy.DESTROY,
});
new firehose.DeliveryStream(stack, 'Delivery Stream', {
    destinations: [new destinations.S3Bucket(bucket, {
            logging: true,
            logGroup: logGroup,
            processor: processor,
            compression: destinations.Compression.GZIP,
            dataOutputPrefix: 'regularPrefix',
            errorOutputPrefix: 'errorPrefix',
            bufferingInterval: cdk.Duration.seconds(60),
            bufferingSize: cdk.Size.mebibytes(1),
            encryptionKey: key,
            s3Backup: {
                mode: destinations.BackupMode.ALL,
                bucket: backupBucket,
                compression: destinations.Compression.ZIP,
                dataOutputPrefix: 'backupPrefix',
                errorOutputPrefix: 'backupErrorPrefix',
                bufferingInterval: cdk.Duration.seconds(60),
                bufferingSize: cdk.Size.mebibytes(1),
                encryptionKey: backupKey,
            },
        })],
});
app.synth();
//# sourceMappingURL=data:application/json;base64,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