"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("../../aws-lambda");
const cdk = require("../../core");
const package_installation_1 = require("./package-installation");
const package_manager_1 = require("./package-manager");
const types_1 = require("./types");
const util_1 = require("./util");
const ESBUILD_MAJOR_VERSION = '0';
/**
 * Bundling with esbuild
 */
class Bundling {
    constructor(props) {
        this.props = props;
        this.packageManager = package_manager_1.PackageManager.fromLockFile(props.depsLockFilePath, props.logLevel);
        Bundling.esbuildInstallation = Bundling.esbuildInstallation ?? package_installation_1.PackageInstallation.detect('esbuild');
        Bundling.tscInstallation = Bundling.tscInstallation ?? package_installation_1.PackageInstallation.detect('typescript');
        this.projectRoot = props.projectRoot;
        this.relativeEntryPath = path.relative(this.projectRoot, path.resolve(props.entry));
        this.relativeDepsLockFilePath = path.relative(this.projectRoot, path.resolve(props.depsLockFilePath));
        if (this.relativeDepsLockFilePath.includes('..')) {
            throw new Error(`Expected depsLockFilePath: ${props.depsLockFilePath} to be under projectRoot: ${this.projectRoot} (${this.relativeDepsLockFilePath})`);
        }
        if (props.tsconfig) {
            this.relativeTsconfigPath = path.relative(this.projectRoot, path.resolve(props.tsconfig));
        }
        if (props.preCompilation && !/\.tsx?$/.test(props.entry)) {
            throw new Error('preCompilation can only be used with typescript files');
        }
        if (props.format === types_1.OutputFormat.ESM
            && (props.runtime === aws_lambda_1.Runtime.NODEJS_10_X || props.runtime === aws_lambda_1.Runtime.NODEJS_12_X)) {
            throw new Error(`ECMAScript module output format is not supported by the ${props.runtime.name} runtime`);
        }
        this.externals = [
            ...props.externalModules ?? ['aws-sdk'],
            ...props.nodeModules ?? [],
        ];
        // Docker bundling
        const shouldBuildImage = props.forceDockerBundling || !Bundling.esbuildInstallation;
        this.image = shouldBuildImage ? props.dockerImage ?? cdk.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
            buildArgs: {
                ...props.buildArgs ?? {},
                IMAGE: props.runtime.bundlingImage.image,
                ESBUILD_VERSION: props.esbuildVersion ?? ESBUILD_MAJOR_VERSION,
            },
            platform: props.architecture.dockerPlatform,
        })
            : cdk.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand({
            inputDir: cdk.AssetStaging.BUNDLING_INPUT_DIR,
            outputDir: cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
            esbuildRunner: 'esbuild',
            tscRunner: 'tsc',
            osPlatform: 'linux',
        });
        this.command = ['bash', '-c', bundlingCommand];
        this.environment = props.environment;
        // Bundling sets the working directory to cdk.AssetStaging.BUNDLING_INPUT_DIR
        // and we want to force npx to use the globally installed esbuild.
        this.workingDirectory = '/';
        // Local bundling
        if (!props.forceDockerBundling) { // only if Docker is not forced
            this.local = this.getLocalBundlingProvider();
        }
    }
    /**
     * esbuild bundled Lambda asset code
     */
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(options.projectRoot, {
            assetHash: options.assetHash,
            assetHashType: options.assetHash ? cdk.AssetHashType.CUSTOM : cdk.AssetHashType.OUTPUT,
            bundling: new Bundling(options),
        });
    }
    static clearEsbuildInstallationCache() {
        this.esbuildInstallation = undefined;
    }
    static clearTscInstallationCache() {
        this.tscInstallation = undefined;
    }
    createBundlingCommand(options) {
        const pathJoin = osPathJoin(options.osPlatform);
        let relativeEntryPath = pathJoin(options.inputDir, this.relativeEntryPath);
        let tscCommand = '';
        if (this.props.preCompilation) {
            const tsconfig = this.props.tsconfig ?? util_1.findUp('tsconfig.json', path.dirname(this.props.entry));
            if (!tsconfig) {
                throw new Error('Cannot find a `tsconfig.json` but `preCompilation` is set to `true`, please specify it via `tsconfig`');
            }
            const compilerOptions = util_1.getTsconfigCompilerOptions(tsconfig);
            tscCommand = `${options.tscRunner} "${relativeEntryPath}" ${compilerOptions}`;
            relativeEntryPath = relativeEntryPath.replace(/\.ts(x?)$/, '.js$1');
        }
        const loaders = Object.entries(this.props.loader ?? {});
        const defines = Object.entries(this.props.define ?? {});
        if (this.props.sourceMap === false && this.props.sourceMapMode) {
            throw new Error('sourceMapMode cannot be used when sourceMap is false');
        }
        const sourceMapEnabled = this.props.sourceMapMode ?? this.props.sourceMap;
        const sourceMapMode = this.props.sourceMapMode ?? types_1.SourceMapMode.DEFAULT;
        const sourceMapValue = sourceMapMode === types_1.SourceMapMode.DEFAULT ? '' : `=${this.props.sourceMapMode}`;
        const sourcesContent = this.props.sourcesContent ?? true;
        const outFile = this.props.format === types_1.OutputFormat.ESM ? 'index.mjs' : 'index.js';
        const esbuildCommand = [
            options.esbuildRunner,
            '--bundle', `"${relativeEntryPath}"`,
            `--target=${this.props.target ?? toTarget(this.props.runtime)}`,
            '--platform=node',
            ...this.props.format ? [`--format=${this.props.format}`] : [],
            `--outfile="${pathJoin(options.outputDir, outFile)}"`,
            ...this.props.minify ? ['--minify'] : [],
            ...sourceMapEnabled ? [`--sourcemap${sourceMapValue}`] : [],
            ...sourcesContent ? [] : [`--sources-content=${sourcesContent}`],
            ...this.externals.map(external => `--external:${external}`),
            ...loaders.map(([ext, name]) => `--loader:${ext}=${name}`),
            ...defines.map(([key, value]) => `--define:${key}=${JSON.stringify(value)}`),
            ...this.props.logLevel ? [`--log-level=${this.props.logLevel}`] : [],
            ...this.props.keepNames ? ['--keep-names'] : [],
            ...this.relativeTsconfigPath ? [`--tsconfig=${pathJoin(options.inputDir, this.relativeTsconfigPath)}`] : [],
            ...this.props.metafile ? [`--metafile=${pathJoin(options.outputDir, 'index.meta.json')}`] : [],
            ...this.props.banner ? [`--banner:js=${JSON.stringify(this.props.banner)}`] : [],
            ...this.props.footer ? [`--footer:js=${JSON.stringify(this.props.footer)}`] : [],
            ...this.props.charset ? [`--charset=${this.props.charset}`] : [],
            ...this.props.mainFields ? [`--main-fields=${this.props.mainFields.join(',')}`] : [],
            ...this.props.inject ? this.props.inject.map(i => `--inject:${i}`) : [],
            ...this.props.esbuildArgs ? [toCliArgs(this.props.esbuildArgs)] : [],
        ];
        let depsCommand = '';
        if (this.props.nodeModules) {
            // Find 'package.json' closest to entry folder, we are going to extract the
            // modules versions from it.
            const pkgPath = util_1.findUp('package.json', path.dirname(this.props.entry));
            if (!pkgPath) {
                throw new Error('Cannot find a `package.json` in this project. Using `nodeModules` requires a `package.json`.');
            }
            // Determine dependencies versions, lock file and installer
            const dependencies = util_1.extractDependencies(pkgPath, this.props.nodeModules);
            const osCommand = new OsCommand(options.osPlatform);
            const lockFilePath = pathJoin(options.inputDir, this.relativeDepsLockFilePath ?? this.packageManager.lockFile);
            // Create dummy package.json, copy lock file if any and then install
            depsCommand = chain([
                osCommand.writeJson(pathJoin(options.outputDir, 'package.json'), { dependencies }),
                osCommand.copy(lockFilePath, pathJoin(options.outputDir, this.packageManager.lockFile)),
                osCommand.changeDirectory(options.outputDir),
                this.packageManager.installCommand.join(' '),
            ]);
        }
        return chain([
            ...this.props.commandHooks?.beforeBundling(options.inputDir, options.outputDir) ?? [],
            tscCommand,
            esbuildCommand.join(' '),
            ...(this.props.nodeModules && this.props.commandHooks?.beforeInstall(options.inputDir, options.outputDir)) ?? [],
            depsCommand,
            ...this.props.commandHooks?.afterBundling(options.inputDir, options.outputDir) ?? [],
        ]);
    }
    getLocalBundlingProvider() {
        const osPlatform = os.platform();
        const createLocalCommand = (outputDir, esbuild, tsc) => this.createBundlingCommand({
            inputDir: this.projectRoot,
            outputDir,
            esbuildRunner: esbuild.isLocal ? this.packageManager.runBinCommand('esbuild') : 'esbuild',
            tscRunner: tsc && (tsc.isLocal ? this.packageManager.runBinCommand('tsc') : 'tsc'),
            osPlatform,
        });
        const environment = this.props.environment ?? {};
        const cwd = this.projectRoot;
        return {
            tryBundle(outputDir) {
                if (!Bundling.esbuildInstallation) {
                    process.stderr.write('esbuild cannot run locally. Switching to Docker bundling.\n');
                    return false;
                }
                if (!Bundling.esbuildInstallation.version.startsWith(`${ESBUILD_MAJOR_VERSION}.`)) {
                    throw new Error(`Expected esbuild version ${ESBUILD_MAJOR_VERSION}.x but got ${Bundling.esbuildInstallation.version}`);
                }
                const localCommand = createLocalCommand(outputDir, Bundling.esbuildInstallation, Bundling.tscInstallation);
                util_1.exec(osPlatform === 'win32' ? 'cmd' : 'bash', [
                    osPlatform === 'win32' ? '/c' : '-c',
                    localCommand,
                ], {
                    env: { ...process.env, ...environment },
                    stdio: [
                        'ignore',
                        process.stderr,
                        'inherit',
                    ],
                    cwd,
                    windowsVerbatimArguments: osPlatform === 'win32',
                });
                return true;
            },
        };
    }
}
exports.Bundling = Bundling;
/**
 * OS agnostic command
 */
class OsCommand {
    constructor(osPlatform) {
        this.osPlatform = osPlatform;
    }
    writeJson(filePath, data) {
        const stringifiedData = JSON.stringify(data);
        if (this.osPlatform === 'win32') {
            return `echo ^${stringifiedData}^ > "${filePath}"`;
        }
        return `echo '${stringifiedData}' > "${filePath}"`;
    }
    copy(src, dest) {
        if (this.osPlatform === 'win32') {
            return `copy "${src}" "${dest}"`;
        }
        return `cp "${src}" "${dest}"`;
    }
    changeDirectory(dir) {
        return `cd "${dir}"`;
    }
}
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
/**
 * Converts a runtime to an esbuild node target
 */
function toTarget(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return `node${match[1]}`;
}
function toCliArgs(esbuildArgs) {
    const args = [];
    for (const [key, value] of Object.entries(esbuildArgs)) {
        if (value === true || value === '') {
            args.push(key);
        }
        else if (value) {
            args.push(`${key}="${value}"`);
        }
    }
    return args.join(' ');
}
//# sourceMappingURL=data:application/json;base64,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