"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = exports.BUNDLER_DEPENDENCIES_CACHE = exports.DEPENDENCY_EXCLUDES = void 0;
const path = require("path");
const aws_lambda_1 = require("../../aws-lambda");
const core_1 = require("../../core");
const packaging_1 = require("./packaging");
/**
 * Dependency files to exclude from the asset hash.
 */
exports.DEPENDENCY_EXCLUDES = ['*.pyc'];
/**
 * The location in the image that the bundler image caches dependencies.
 */
exports.BUNDLER_DEPENDENCIES_CACHE = '/var/dependencies';
/**
 * Produce bundled Lambda asset code
 */
class Bundling {
    constructor(props) {
        const { entry, runtime, architecture = aws_lambda_1.Architecture.X86_64, outputPathSuffix = '', image, } = props;
        const outputPath = path.posix.join(core_1.AssetStaging.BUNDLING_OUTPUT_DIR, outputPathSuffix);
        const bundlingCommands = this.createBundlingCommand({
            entry,
            inputDir: core_1.AssetStaging.BUNDLING_INPUT_DIR,
            outputDir: outputPath,
        });
        this.image = image ?? core_1.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
            buildArgs: {
                ...props.buildArgs,
                IMAGE: runtime.bundlingImage.image,
            },
            platform: architecture.dockerPlatform,
        });
        this.command = ['bash', '-c', chain(bundlingCommands)];
        this.environment = props.environment;
    }
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(options.entry, {
            assetHash: options.assetHash,
            assetHashType: options.assetHashType,
            exclude: exports.DEPENDENCY_EXCLUDES,
            bundling: options.skip ? undefined : new Bundling(options),
        });
    }
    createBundlingCommand(options) {
        const packaging = packaging_1.Packaging.fromEntry(options.entry);
        let bundlingCommands = [];
        bundlingCommands.push(packaging.exportCommand ?? '');
        if (packaging.dependenciesFile) {
            bundlingCommands.push(`python -m pip install -r ${packaging_1.DependenciesFile.PIP} -t ${options.outputDir}`);
        }
        bundlingCommands.push(`cp -rT ${options.inputDir}/ ${options.outputDir}`);
        return bundlingCommands;
    }
}
exports.Bundling = Bundling;
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVuZGxpbmcuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJidW5kbGluZy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw2QkFBNkI7QUFDN0IsaURBQTBFO0FBQzFFLHFDQUE4RjtBQUM5RiwyQ0FBMEQ7QUFHMUQ7O0dBRUc7QUFDVSxRQUFBLG1CQUFtQixHQUFHLENBQUMsT0FBTyxDQUFDLENBQUM7QUFFN0M7O0dBRUc7QUFDVSxRQUFBLDBCQUEwQixHQUFHLG1CQUFtQixDQUFDO0FBK0I5RDs7R0FFRztBQUNILE1BQWEsUUFBUTtJQWNuQixZQUFZLEtBQW9CO1FBQzlCLE1BQU0sRUFDSixLQUFLLEVBQ0wsT0FBTyxFQUNQLFlBQVksR0FBRyx5QkFBWSxDQUFDLE1BQU0sRUFDbEMsZ0JBQWdCLEdBQUcsRUFBRSxFQUNyQixLQUFLLEdBQ04sR0FBRyxLQUFLLENBQUM7UUFFVixNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxtQkFBWSxDQUFDLG1CQUFtQixFQUFFLGdCQUFnQixDQUFDLENBQUM7UUFFdkYsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLENBQUMscUJBQXFCLENBQUM7WUFDbEQsS0FBSztZQUNMLFFBQVEsRUFBRSxtQkFBWSxDQUFDLGtCQUFrQjtZQUN6QyxTQUFTLEVBQUUsVUFBVTtTQUN0QixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsS0FBSyxHQUFHLEtBQUssSUFBSSxrQkFBVyxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUMsRUFBRTtZQUMxRSxTQUFTLEVBQUU7Z0JBQ1QsR0FBRyxLQUFLLENBQUMsU0FBUztnQkFDbEIsS0FBSyxFQUFFLE9BQU8sQ0FBQyxhQUFhLENBQUMsS0FBSzthQUNuQztZQUNELFFBQVEsRUFBRSxZQUFZLENBQUMsY0FBYztTQUN0QyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsT0FBTyxHQUFHLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDO1FBQ3ZELElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQztLQUN0QztJQXZDTSxNQUFNLENBQUMsTUFBTSxDQUFDLE9BQXNCO1FBQ3pDLE9BQU8saUJBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRTtZQUNuQyxTQUFTLEVBQUUsT0FBTyxDQUFDLFNBQVM7WUFDNUIsYUFBYSxFQUFFLE9BQU8sQ0FBQyxhQUFhO1lBQ3BDLE9BQU8sRUFBRSwyQkFBbUI7WUFDNUIsUUFBUSxFQUFFLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxRQUFRLENBQUMsT0FBTyxDQUFDO1NBQzNELENBQUMsQ0FBQztLQUNKO0lBa0NPLHFCQUFxQixDQUFDLE9BQStCO1FBQzNELE1BQU0sU0FBUyxHQUFHLHFCQUFTLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNyRCxJQUFJLGdCQUFnQixHQUFhLEVBQUUsQ0FBQztRQUNwQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLGFBQWEsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUNyRCxJQUFJLFNBQVMsQ0FBQyxnQkFBZ0IsRUFBRTtZQUM5QixnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsNEJBQTRCLDRCQUFnQixDQUFDLEdBQUcsT0FBTyxPQUFPLENBQUMsU0FBUyxFQUFFLENBQUMsQ0FBQztTQUNuRztRQUNELGdCQUFnQixDQUFDLElBQUksQ0FBQyxVQUFVLE9BQU8sQ0FBQyxRQUFRLEtBQUssT0FBTyxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUM7UUFDMUUsT0FBTyxnQkFBZ0IsQ0FBQztLQUN6QjtDQUNGO0FBcERELDRCQW9EQztBQVFEOztHQUVHO0FBQ0gsU0FBUyxLQUFLLENBQUMsUUFBa0I7SUFDL0IsT0FBTyxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztBQUNoRCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCB7IEFyY2hpdGVjdHVyZSwgQXNzZXRDb2RlLCBDb2RlLCBSdW50aW1lIH0gZnJvbSAnLi4vLi4vYXdzLWxhbWJkYSc7XG5pbXBvcnQgeyBBc3NldFN0YWdpbmcsIEJ1bmRsaW5nT3B0aW9ucyBhcyBDZGtCdW5kbGluZ09wdGlvbnMsIERvY2tlckltYWdlIH0gZnJvbSAnLi4vLi4vY29yZSc7XG5pbXBvcnQgeyBQYWNrYWdpbmcsIERlcGVuZGVuY2llc0ZpbGUgfSBmcm9tICcuL3BhY2thZ2luZyc7XG5pbXBvcnQgeyBCdW5kbGluZ09wdGlvbnMgfSBmcm9tICcuL3R5cGVzJztcblxuLyoqXG4gKiBEZXBlbmRlbmN5IGZpbGVzIHRvIGV4Y2x1ZGUgZnJvbSB0aGUgYXNzZXQgaGFzaC5cbiAqL1xuZXhwb3J0IGNvbnN0IERFUEVOREVOQ1lfRVhDTFVERVMgPSBbJyoucHljJ107XG5cbi8qKlxuICogVGhlIGxvY2F0aW9uIGluIHRoZSBpbWFnZSB0aGF0IHRoZSBidW5kbGVyIGltYWdlIGNhY2hlcyBkZXBlbmRlbmNpZXMuXG4gKi9cbmV4cG9ydCBjb25zdCBCVU5ETEVSX0RFUEVOREVOQ0lFU19DQUNIRSA9ICcvdmFyL2RlcGVuZGVuY2llcyc7XG5cbi8qKlxuICogT3B0aW9ucyBmb3IgYnVuZGxpbmdcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBCdW5kbGluZ1Byb3BzIGV4dGVuZHMgQnVuZGxpbmdPcHRpb25zIHtcbiAgLyoqXG4gICAqIEVudHJ5IHBhdGhcbiAgICovXG4gIHJlYWRvbmx5IGVudHJ5OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBydW50aW1lIGVudmlyb25tZW50LlxuICAgKi9cbiAgcmVhZG9ubHkgcnVudGltZTogUnVudGltZTtcblxuICAvKipcbiAgICogVGhlIHN5c3RlbSBhcmNoaXRlY3R1cmUgb2YgdGhlIGxhbWJkYSBmdW5jdGlvblxuICAgKlxuICAgKiBAZGVmYXVsdCBBcmNoaXRlY3R1cmUuWDg2XzY0XG4gICAqL1xuICByZWFkb25seSBhcmNoaXRlY3R1cmU/OiBBcmNoaXRlY3R1cmU7XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgb3Igbm90IHRoZSBidW5kbGluZyBwcm9jZXNzIHNob3VsZCBiZSBza2lwcGVkXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gRG9lcyBub3Qgc2tpcCBidW5kbGluZ1xuICAgKi9cbiAgcmVhZG9ubHkgc2tpcD86IGJvb2xlYW47XG59XG5cbi8qKlxuICogUHJvZHVjZSBidW5kbGVkIExhbWJkYSBhc3NldCBjb2RlXG4gKi9cbmV4cG9ydCBjbGFzcyBCdW5kbGluZyBpbXBsZW1lbnRzIENka0J1bmRsaW5nT3B0aW9ucyB7XG4gIHB1YmxpYyBzdGF0aWMgYnVuZGxlKG9wdGlvbnM6IEJ1bmRsaW5nUHJvcHMpOiBBc3NldENvZGUge1xuICAgIHJldHVybiBDb2RlLmZyb21Bc3NldChvcHRpb25zLmVudHJ5LCB7XG4gICAgICBhc3NldEhhc2g6IG9wdGlvbnMuYXNzZXRIYXNoLFxuICAgICAgYXNzZXRIYXNoVHlwZTogb3B0aW9ucy5hc3NldEhhc2hUeXBlLFxuICAgICAgZXhjbHVkZTogREVQRU5ERU5DWV9FWENMVURFUyxcbiAgICAgIGJ1bmRsaW5nOiBvcHRpb25zLnNraXAgPyB1bmRlZmluZWQgOiBuZXcgQnVuZGxpbmcob3B0aW9ucyksXG4gICAgfSk7XG4gIH1cblxuICBwdWJsaWMgcmVhZG9ubHkgaW1hZ2U6IERvY2tlckltYWdlO1xuICBwdWJsaWMgcmVhZG9ubHkgY29tbWFuZDogc3RyaW5nW107XG4gIHB1YmxpYyByZWFkb25seSBlbnZpcm9ubWVudD86IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH07XG5cbiAgY29uc3RydWN0b3IocHJvcHM6IEJ1bmRsaW5nUHJvcHMpIHtcbiAgICBjb25zdCB7XG4gICAgICBlbnRyeSxcbiAgICAgIHJ1bnRpbWUsXG4gICAgICBhcmNoaXRlY3R1cmUgPSBBcmNoaXRlY3R1cmUuWDg2XzY0LFxuICAgICAgb3V0cHV0UGF0aFN1ZmZpeCA9ICcnLFxuICAgICAgaW1hZ2UsXG4gICAgfSA9IHByb3BzO1xuXG4gICAgY29uc3Qgb3V0cHV0UGF0aCA9IHBhdGgucG9zaXguam9pbihBc3NldFN0YWdpbmcuQlVORExJTkdfT1VUUFVUX0RJUiwgb3V0cHV0UGF0aFN1ZmZpeCk7XG5cbiAgICBjb25zdCBidW5kbGluZ0NvbW1hbmRzID0gdGhpcy5jcmVhdGVCdW5kbGluZ0NvbW1hbmQoe1xuICAgICAgZW50cnksXG4gICAgICBpbnB1dERpcjogQXNzZXRTdGFnaW5nLkJVTkRMSU5HX0lOUFVUX0RJUixcbiAgICAgIG91dHB1dERpcjogb3V0cHV0UGF0aCxcbiAgICB9KTtcblxuICAgIHRoaXMuaW1hZ2UgPSBpbWFnZSA/PyBEb2NrZXJJbWFnZS5mcm9tQnVpbGQocGF0aC5qb2luKF9fZGlybmFtZSwgJy4uL2xpYicpLCB7XG4gICAgICBidWlsZEFyZ3M6IHtcbiAgICAgICAgLi4ucHJvcHMuYnVpbGRBcmdzLFxuICAgICAgICBJTUFHRTogcnVudGltZS5idW5kbGluZ0ltYWdlLmltYWdlLFxuICAgICAgfSxcbiAgICAgIHBsYXRmb3JtOiBhcmNoaXRlY3R1cmUuZG9ja2VyUGxhdGZvcm0sXG4gICAgfSk7XG4gICAgdGhpcy5jb21tYW5kID0gWydiYXNoJywgJy1jJywgY2hhaW4oYnVuZGxpbmdDb21tYW5kcyldO1xuICAgIHRoaXMuZW52aXJvbm1lbnQgPSBwcm9wcy5lbnZpcm9ubWVudDtcbiAgfVxuXG4gIHByaXZhdGUgY3JlYXRlQnVuZGxpbmdDb21tYW5kKG9wdGlvbnM6IEJ1bmRsaW5nQ29tbWFuZE9wdGlvbnMpOiBzdHJpbmdbXSB7XG4gICAgY29uc3QgcGFja2FnaW5nID0gUGFja2FnaW5nLmZyb21FbnRyeShvcHRpb25zLmVudHJ5KTtcbiAgICBsZXQgYnVuZGxpbmdDb21tYW5kczogc3RyaW5nW10gPSBbXTtcbiAgICBidW5kbGluZ0NvbW1hbmRzLnB1c2gocGFja2FnaW5nLmV4cG9ydENvbW1hbmQgPz8gJycpO1xuICAgIGlmIChwYWNrYWdpbmcuZGVwZW5kZW5jaWVzRmlsZSkge1xuICAgICAgYnVuZGxpbmdDb21tYW5kcy5wdXNoKGBweXRob24gLW0gcGlwIGluc3RhbGwgLXIgJHtEZXBlbmRlbmNpZXNGaWxlLlBJUH0gLXQgJHtvcHRpb25zLm91dHB1dERpcn1gKTtcbiAgICB9XG4gICAgYnVuZGxpbmdDb21tYW5kcy5wdXNoKGBjcCAtclQgJHtvcHRpb25zLmlucHV0RGlyfS8gJHtvcHRpb25zLm91dHB1dERpcn1gKTtcbiAgICByZXR1cm4gYnVuZGxpbmdDb21tYW5kcztcbiAgfVxufVxuXG5pbnRlcmZhY2UgQnVuZGxpbmdDb21tYW5kT3B0aW9ucyB7XG4gIHJlYWRvbmx5IGVudHJ5OiBzdHJpbmc7XG4gIHJlYWRvbmx5IGlucHV0RGlyOiBzdHJpbmc7XG4gIHJlYWRvbmx5IG91dHB1dERpcjogc3RyaW5nO1xufVxuXG4vKipcbiAqIENoYWluIGNvbW1hbmRzXG4gKi9cbmZ1bmN0aW9uIGNoYWluKGNvbW1hbmRzOiBzdHJpbmdbXSk6IHN0cmluZyB7XG4gIHJldHVybiBjb21tYW5kcy5maWx0ZXIoYyA9PiAhIWMpLmpvaW4oJyAmJiAnKTtcbn1cbiJdfQ==