"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FunctionVersionUpgrade = exports.verifyCodeConfig = exports.Function = exports.Tracing = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("../../aws-cloudwatch");
const aws_codeguruprofiler_1 = require("../../aws-codeguruprofiler");
const ec2 = require("../../aws-ec2");
const iam = require("../../aws-iam");
const logs = require("../../aws-logs");
const sqs = require("../../aws-sqs");
const core_1 = require("../../core");
const cx_api_1 = require("../../cx-api");
const architecture_1 = require("./architecture");
const function_base_1 = require("./function-base");
const function_hash_1 = require("./function-hash");
const handler_1 = require("./handler");
const lambda_version_1 = require("./lambda-version");
const lambda_generated_1 = require("./lambda.generated");
const layers_1 = require("./layers");
const runtime_1 = require("./runtime");
const util_1 = require("./util");
/**
 * X-Ray Tracing Modes (https://docs.aws.amazon.com/lambda/latest/dg/API_TracingConfig.html)
 */
var Tracing;
(function (Tracing) {
    /**
     * Lambda will respect any tracing header it receives from an upstream service.
     * If no tracing header is received, Lambda will call X-Ray for a tracing decision.
     */
    Tracing["ACTIVE"] = "Active";
    /**
     * Lambda will only trace the request from an upstream service
     * if it contains a tracing header with "sampled=1"
     */
    Tracing["PASS_THROUGH"] = "PassThrough";
    /**
     * Lambda will not trace any request.
     */
    Tracing["DISABLED"] = "Disabled";
})(Tracing = exports.Tracing || (exports.Tracing = {}));
/**
 * Deploys a file from inside the construct library as a function.
 *
 * The supplied file is subject to the 4096 bytes limit of being embedded in a
 * CloudFormation template.
 *
 * The construct includes an associated role with the lambda.
 *
 * This construct does not yet reproduce all features from the underlying resource
 * library.
 */
class Function extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.functionName,
        });
        this.permissionsNode = this.node;
        this.canCreatePermissions = true;
        /** @internal */
        this._layers = [];
        /**
         * Environment variables for this function
         */
        this.environment = {};
        try {
            jsiiDeprecationWarnings.monocdk_aws_lambda_FunctionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Function);
            }
            throw error;
        }
        if (props.functionName && !core_1.Token.isUnresolved(props.functionName)) {
            if (props.functionName.length > 64) {
                throw new Error(`Function name can not be longer than 64 characters but has ${props.functionName.length} characters.`);
            }
            if (!/^[a-zA-Z0-9-_]+$/.test(props.functionName)) {
                throw new Error(`Function name ${props.functionName} can contain only letters, numbers, hyphens, or underscores with no spaces.`);
            }
        }
        if (props.description && !core_1.Token.isUnresolved(props.description)) {
            if (props.description.length > 256) {
                throw new Error(`Function description can not be longer than 256 characters but has ${props.description.length} characters.`);
            }
        }
        const managedPolicies = new Array();
        // the arn is in the form of - arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
        managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        if (props.vpc) {
            // Policy that will have ENI creation permissions
            managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'));
        }
        this.role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies,
        });
        this.grantPrincipal = this.role;
        // add additional managed policies when necessary
        if (props.filesystem) {
            const config = props.filesystem.config;
            if (config.policies) {
                config.policies.forEach(p => {
                    this.role?.addToPrincipalPolicy(p);
                });
            }
        }
        for (const statement of (props.initialPolicy || [])) {
            this.role.addToPrincipalPolicy(statement);
        }
        const code = props.code.bind(this);
        verifyCodeConfig(code, props);
        let profilingGroupEnvironmentVariables = {};
        if (props.profilingGroup && props.profiling !== false) {
            this.validateProfiling(props);
            props.profilingGroup.grantPublish(this.role);
            profilingGroupEnvironmentVariables = {
                AWS_CODEGURU_PROFILER_GROUP_ARN: core_1.Stack.of(scope).formatArn({
                    service: 'codeguru-profiler',
                    resource: 'profilingGroup',
                    resourceName: props.profilingGroup.profilingGroupName,
                }),
                AWS_CODEGURU_PROFILER_ENABLED: 'TRUE',
            };
        }
        else if (props.profiling) {
            this.validateProfiling(props);
            const profilingGroup = new aws_codeguruprofiler_1.ProfilingGroup(this, 'ProfilingGroup', {
                computePlatform: aws_codeguruprofiler_1.ComputePlatform.AWS_LAMBDA,
            });
            profilingGroup.grantPublish(this.role);
            profilingGroupEnvironmentVariables = {
                AWS_CODEGURU_PROFILER_GROUP_ARN: profilingGroup.profilingGroupArn,
                AWS_CODEGURU_PROFILER_ENABLED: 'TRUE',
            };
        }
        const env = { ...profilingGroupEnvironmentVariables, ...props.environment };
        for (const [key, value] of Object.entries(env)) {
            this.addEnvironment(key, value);
        }
        // DLQ can be either sns.ITopic or sqs.IQueue
        const dlqTopicOrQueue = this.buildDeadLetterQueue(props);
        if (dlqTopicOrQueue !== undefined) {
            if (this.isQueue(dlqTopicOrQueue)) {
                this.deadLetterQueue = dlqTopicOrQueue;
            }
            else {
                this.deadLetterTopic = dlqTopicOrQueue;
            }
        }
        let fileSystemConfigs = undefined;
        if (props.filesystem) {
            fileSystemConfigs = [{
                    arn: props.filesystem.config.arn,
                    localMountPath: props.filesystem.config.localMountPath,
                }];
        }
        if (props.architecture && props.architectures !== undefined) {
            throw new Error('Either architecture or architectures must be specified but not both.');
        }
        if (props.architectures && props.architectures.length > 1) {
            throw new Error('Only one architecture must be specified.');
        }
        this._architecture = props.architecture ?? (props.architectures && props.architectures[0]);
        if (props.ephemeralStorageSize && !props.ephemeralStorageSize.isUnresolved()
            && (props.ephemeralStorageSize.toMebibytes() < 512 || props.ephemeralStorageSize.toMebibytes() > 10240)) {
            throw new Error(`Ephemeral storage size must be between 512 and 10240 MB, received ${props.ephemeralStorageSize}.`);
        }
        const resource = new lambda_generated_1.CfnFunction(this, 'Resource', {
            functionName: this.physicalName,
            description: props.description,
            code: {
                s3Bucket: code.s3Location && code.s3Location.bucketName,
                s3Key: code.s3Location && code.s3Location.objectKey,
                s3ObjectVersion: code.s3Location && code.s3Location.objectVersion,
                zipFile: code.inlineCode,
                imageUri: code.image?.imageUri,
            },
            layers: core_1.Lazy.list({ produce: () => this.renderLayers() }),
            handler: props.handler === handler_1.Handler.FROM_IMAGE ? undefined : props.handler,
            timeout: props.timeout && props.timeout.toSeconds(),
            packageType: props.runtime === runtime_1.Runtime.FROM_IMAGE ? 'Image' : undefined,
            runtime: props.runtime === runtime_1.Runtime.FROM_IMAGE ? undefined : props.runtime.name,
            role: this.role.roleArn,
            // Uncached because calling '_checkEdgeCompatibility', which gets called in the resolve of another
            // Token, actually *modifies* the 'environment' map.
            environment: core_1.Lazy.uncachedAny({ produce: () => this.renderEnvironment() }),
            memorySize: props.memorySize,
            ephemeralStorage: props.ephemeralStorageSize ? {
                size: props.ephemeralStorageSize.toMebibytes(),
            } : undefined,
            vpcConfig: this.configureVpc(props),
            deadLetterConfig: this.buildDeadLetterConfig(dlqTopicOrQueue),
            tracingConfig: this.buildTracingConfig(props),
            reservedConcurrentExecutions: props.reservedConcurrentExecutions,
            imageConfig: undefinedIfNoKeys({
                command: code.image?.cmd,
                entryPoint: code.image?.entrypoint,
                workingDirectory: code.image?.workingDirectory,
            }),
            kmsKeyArn: props.environmentEncryption?.keyArn,
            fileSystemConfigs,
            codeSigningConfigArn: props.codeSigningConfig?.codeSigningConfigArn,
            architectures: this._architecture ? [this._architecture.name] : undefined,
        });
        resource.node.addDependency(this.role);
        this.functionName = this.getResourceNameAttribute(resource.ref);
        this.functionArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'lambda',
            resource: 'function',
            resourceName: this.physicalName,
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        });
        this.runtime = props.runtime;
        this.timeout = props.timeout;
        this.architecture = props.architecture ?? architecture_1.Architecture.X86_64;
        if (props.layers) {
            if (props.runtime === runtime_1.Runtime.FROM_IMAGE) {
                throw new Error('Layers are not supported for container image functions');
            }
            this.addLayers(...props.layers);
        }
        for (const event of props.events || []) {
            this.addEventSource(event);
        }
        // Log retention
        if (props.logRetention) {
            const logRetention = new logs.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: props.logRetention,
                role: props.logRetentionRole,
                logRetentionRetryOptions: props.logRetentionRetryOptions,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, 'LogGroup', logRetention.logGroupArn);
        }
        props.code.bindToResource(resource);
        // Event Invoke Config
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
        this.currentVersionOptions = props.currentVersionOptions;
        if (props.filesystem) {
            if (!props.vpc) {
                throw new Error('Cannot configure \'filesystem\' without configuring a VPC.');
            }
            const config = props.filesystem.config;
            if (config.dependency) {
                this.node.addDependency(...config.dependency);
            }
            // There could be a race if the Lambda is used in a CustomResource. It is possible for the Lambda to
            // fail to attach to a given FileSystem if we do not have a dependency on the SecurityGroup ingress/egress
            // rules that were created between this Lambda's SG & the Filesystem SG.
            this.connections.securityGroups.forEach(sg => {
                sg.node.findAll().forEach(child => {
                    if (child instanceof core_1.CfnResource && child.cfnResourceType === 'AWS::EC2::SecurityGroupEgress') {
                        resource.node.addDependency(child);
                    }
                });
            });
            config.connections?.securityGroups.forEach(sg => {
                sg.node.findAll().forEach(child => {
                    if (child instanceof core_1.CfnResource && child.cfnResourceType === 'AWS::EC2::SecurityGroupIngress') {
                        resource.node.addDependency(child);
                    }
                });
            });
        }
        // Configure Lambda insights
        this.configureLambdaInsights(props);
    }
    /**
     * Returns a `lambda.Version` which represents the current version of this
     * Lambda function. A new version will be created every time the function's
     * configuration changes.
     *
     * You can specify options for this version using the `currentVersionOptions`
     * prop when initializing the `lambda.Function`.
     */
    get currentVersion() {
        if (this._currentVersion) {
            return this._currentVersion;
        }
        if (this._warnIfCurrentVersionCalled) {
            this.warnInvokeFunctionPermissions(this);
        }
        ;
        this._currentVersion = new lambda_version_1.Version(this, 'CurrentVersion', {
            lambda: this,
            ...this.currentVersionOptions,
        });
        // override the version's logical ID with a lazy string which includes the
        // hash of the function itself, so a new version resource is created when
        // the function configuration changes.
        const cfn = this._currentVersion.node.defaultChild;
        const originalLogicalId = this.stack.resolve(cfn.logicalId);
        cfn.overrideLogicalId(core_1.Lazy.uncachedString({
            produce: () => {
                const hash = function_hash_1.calculateFunctionHash(this);
                const logicalId = function_hash_1.trimFromStart(originalLogicalId, 255 - 32);
                return `${logicalId}${hash}`;
            },
        }));
        return this._currentVersion;
    }
    get resourceArnsForGrantInvoke() {
        return [this.functionArn, `${this.functionArn}:*`];
    }
    /**
     * Record whether specific properties in the `AWS::Lambda::Function` resource should
     * also be associated to the Version resource.
     * See 'currentVersion' section in the module README for more details.
     * @param propertyName The property to classify
     * @param locked whether the property should be associated to the version or not.
     */
    static classifyVersionProperty(propertyName, locked) {
        this._VER_PROPS[propertyName] = locked;
    }
    /**
     * Import a lambda function into the CDK using its name
     */
    static fromFunctionName(scope, id, functionName) {
        return Function.fromFunctionAttributes(scope, id, {
            functionArn: core_1.Stack.of(scope).formatArn({
                service: 'lambda',
                resource: 'function',
                resourceName: functionName,
                arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
            }),
        });
    }
    /**
     * Import a lambda function into the CDK using its ARN
     */
    static fromFunctionArn(scope, id, functionArn) {
        return Function.fromFunctionAttributes(scope, id, { functionArn });
    }
    /**
     * Creates a Lambda function object which represents a function not defined
     * within this stack.
     *
     * @param scope The parent construct
     * @param id The name of the lambda construct
     * @param attrs the attributes of the function to import
     */
    static fromFunctionAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_lambda_FunctionAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromFunctionAttributes);
            }
            throw error;
        }
        const functionArn = attrs.functionArn;
        const functionName = extractNameFromArn(attrs.functionArn);
        const role = attrs.role;
        class Import extends function_base_1.FunctionBase {
            constructor(s, i) {
                super(s, i, {
                    environmentFromArn: functionArn,
                });
                this.functionName = functionName;
                this.functionArn = functionArn;
                this.role = role;
                this.permissionsNode = this.node;
                this.architecture = attrs.architecture ?? architecture_1.Architecture.X86_64;
                this.resourceArnsForGrantInvoke = [this.functionArn, `${this.functionArn}:*`];
                this.canCreatePermissions = attrs.sameEnvironment ?? this._isStackAccount();
                this._skipPermissions = attrs.skipPermissions ?? false;
                this.grantPrincipal = role || new iam.UnknownPrincipal({ resource: this });
                if (attrs.securityGroup) {
                    this._connections = new ec2.Connections({
                        securityGroups: [attrs.securityGroup],
                    });
                }
                else if (attrs.securityGroupId) {
                    this._connections = new ec2.Connections({
                        securityGroups: [ec2.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', attrs.securityGroupId)],
                    });
                }
            }
        }
        return new Import(scope, id);
    }
    /**
     * Return the given named metric for this Lambda
     */
    static metricAll(metricName, props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricAll);
            }
            throw error;
        }
        return new cloudwatch.Metric({
            namespace: 'AWS/Lambda',
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the number of Errors executing all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllErrors(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricAllErrors);
            }
            throw error;
        }
        return this.metricAll('Errors', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the Duration executing all Lambdas
     *
     * @default average over 5 minutes
     */
    static metricAllDuration(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricAllDuration);
            }
            throw error;
        }
        return this.metricAll('Duration', props);
    }
    /**
     * Metric for the number of invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllInvocations(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricAllInvocations);
            }
            throw error;
        }
        return this.metricAll('Invocations', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of throttled invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllThrottles(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricAllThrottles);
            }
            throw error;
        }
        return this.metricAll('Throttles', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllConcurrentExecutions(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricAllConcurrentExecutions);
            }
            throw error;
        }
        // Mini-FAQ: why max? This metric is a gauge that is emitted every
        // minute, so either max or avg or a percentile make sense (but sum
        // doesn't). Max is more sensitive to spiky load changes which is
        // probably what you're interested in if you're looking at this metric
        // (Load spikes may lead to concurrent execution errors that would
        // otherwise not be visible in the avg)
        return this.metricAll('ConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Metric for the number of unreserved concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllUnreservedConcurrentExecutions(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricAllUnreservedConcurrentExecutions);
            }
            throw error;
        }
        return this.metricAll('UnreservedConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Adds an environment variable to this Lambda function.
     * If this is a ref to a Lambda function, this operation results in a no-op.
     * @param key The environment variable key.
     * @param value The environment variable's value.
     * @param options Environment variable options.
     */
    addEnvironment(key, value, options) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_lambda_EnvironmentOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addEnvironment);
            }
            throw error;
        }
        this.environment[key] = { value, ...options };
        return this;
    }
    /**
     * Adds one or more Lambda Layers to this Lambda function.
     *
     * @param layers the layers to be added.
     *
     * @throws if there are already 5 layers on this function, or the layer is incompatible with this function's runtime.
     */
    addLayers(...layers) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_lambda_ILayerVersion(layers);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addLayers);
            }
            throw error;
        }
        for (const layer of layers) {
            if (this._layers.length === 5) {
                throw new Error('Unable to add layer: this lambda function already uses 5 layers.');
            }
            if (layer.compatibleRuntimes && !layer.compatibleRuntimes.find(runtime => runtime.runtimeEquals(this.runtime))) {
                const runtimes = layer.compatibleRuntimes.map(runtime => runtime.name).join(', ');
                throw new Error(`This lambda function uses a runtime that is incompatible with this layer (${this.runtime.name} is not in [${runtimes}])`);
            }
            // Currently no validations for compatible architectures since Lambda service
            // allows layers configured with one architecture to be used with a Lambda function
            // from another architecture.
            this._layers.push(layer);
        }
    }
    /**
     * Add a new version for this Lambda
     *
     * If you want to deploy through CloudFormation and use aliases, you need to
     * add a new version (with a new name) to your Lambda every time you want to
     * deploy an update. An alias can then refer to the newly created Version.
     *
     * All versions should have distinct names, and you should not delete versions
     * as long as your Alias needs to refer to them.
     *
     * @param name A unique name for this version.
     * @param codeSha256 The SHA-256 hash of the most recently deployed Lambda
     *  source code, or omit to skip validation.
     * @param description A description for this version.
     * @param provisionedExecutions A provisioned concurrency configuration for a
     * function's version.
     * @param asyncInvokeConfig configuration for this version when it is invoked
     * asynchronously.
     * @returns A new Version object.
     *
     * @deprecated This method will create an AWS::Lambda::Version resource which
     * snapshots the AWS Lambda function *at the time of its creation* and it
     * won't get updated when the function changes. Instead, use
     * `this.currentVersion` to obtain a reference to a version resource that gets
     * automatically recreated when the function configuration (or code) changes.
     */
    addVersion(name, codeSha256, description, provisionedExecutions, asyncInvokeConfig = {}) {
        try {
            jsiiDeprecationWarnings.print("monocdk.aws_lambda.Function#addVersion", "This method will create an AWS::Lambda::Version resource which\nsnapshots the AWS Lambda function *at the time of its creation* and it\nwon't get updated when the function changes. Instead, use\n`this.currentVersion` to obtain a reference to a version resource that gets\nautomatically recreated when the function configuration (or code) changes.");
            jsiiDeprecationWarnings.monocdk_aws_lambda_EventInvokeConfigOptions(asyncInvokeConfig);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addVersion);
            }
            throw error;
        }
        return new lambda_version_1.Version(this, 'Version' + name, {
            lambda: this,
            codeSha256,
            description,
            provisionedConcurrentExecutions: provisionedExecutions,
            ...asyncInvokeConfig,
        });
    }
    /**
     * Defines an alias for this function.
     *
     * The alias will automatically be updated to point to the latest version of
     * the function as it is being updated during a deployment.
     *
     * ```ts
     * declare const fn: lambda.Function;
     *
     * fn.addAlias('Live');
     *
     * // Is equivalent to
     *
     * new lambda.Alias(this, 'AliasLive', {
     *   aliasName: 'Live',
     *   version: fn.currentVersion,
     * });
     *
     * @param aliasName The name of the alias
     * @param options Alias options
     */
    addAlias(aliasName, options) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_lambda_AliasOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addAlias);
            }
            throw error;
        }
        return util_1.addAlias(this, this.currentVersion, aliasName, options);
    }
    /**
     * The LogGroup where the Lambda function's logs are made available.
     *
     * If either `logRetention` is set or this property is called, a CloudFormation custom resource is added to the stack that
     * pre-creates the log group as part of the stack deployment, if it already doesn't exist, and sets the correct log retention
     * period (never expire, by default).
     *
     * Further, if the log group already exists and the `logRetention` is not set, the custom resource will reset the log retention
     * to never expire even if it was configured with a different value.
     */
    get logGroup() {
        if (!this._logGroup) {
            const logRetention = new logs.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: logs.RetentionDays.INFINITE,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, `${this.node.id}-LogGroup`, logRetention.logGroupArn);
        }
        return this._logGroup;
    }
    /** @internal */
    _checkEdgeCompatibility() {
        // Check env vars
        const envEntries = Object.entries(this.environment);
        for (const [key, config] of envEntries) {
            if (config.removeInEdge) {
                delete this.environment[key];
                core_1.Annotations.of(this).addInfo(`Removed ${key} environment variable for Lambda@Edge compatibility`);
            }
        }
        const envKeys = Object.keys(this.environment);
        if (envKeys.length !== 0) {
            throw new Error(`The function ${this.node.path} contains environment variables [${envKeys}] and is not compatible with Lambda@Edge. \
Environment variables can be marked for removal when used in Lambda@Edge by setting the \'removeInEdge\' property in the \'addEnvironment()\' API.`);
        }
        return;
    }
    /**
     * Configured lambda insights on the function if specified. This is acheived by adding an imported layer which is added to the
     * list of lambda layers on synthesis.
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Lambda-Insights-extension-versions.html
     */
    configureLambdaInsights(props) {
        if (props.insightsVersion === undefined) {
            return;
        }
        if (props.runtime !== runtime_1.Runtime.FROM_IMAGE) {
            // Layers cannot be added to Lambda container images. The image should have the insights agent installed.
            // See https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Lambda-Insights-Getting-Started-docker.html
            this.addLayers(layers_1.LayerVersion.fromLayerVersionArn(this, 'LambdaInsightsLayer', props.insightsVersion._bind(this, this).arn));
        }
        this.role?.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('CloudWatchLambdaInsightsExecutionRolePolicy'));
    }
    renderLayers() {
        if (!this._layers || this._layers.length === 0) {
            return undefined;
        }
        if (core_1.FeatureFlags.of(this).isEnabled(cx_api_1.LAMBDA_RECOGNIZE_LAYER_VERSION)) {
            this._layers.sort();
        }
        return this._layers.map(layer => layer.layerVersionArn);
    }
    renderEnvironment() {
        if (!this.environment || Object.keys(this.environment).length === 0) {
            return undefined;
        }
        const variables = {};
        // Sort environment so the hash of the function used to create
        // `currentVersion` is not affected by key order (this is how lambda does
        // it). For backwards compatibility we do not sort environment variables in case
        // _currentVersion is not defined. Otherwise, this would have invalidated
        // the template, and for example, may cause unneeded updates for nested
        // stacks.
        const keys = this._currentVersion
            ? Object.keys(this.environment).sort()
            : Object.keys(this.environment);
        for (const key of keys) {
            variables[key] = this.environment[key].value;
        }
        return { variables };
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * Lambda creation properties.
     */
    configureVpc(props) {
        if ((props.securityGroup || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroup\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if (props.securityGroup && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroup.');
        }
        let securityGroups;
        if (props.securityGroup && props.securityGroups) {
            throw new Error('Only one of the function props, securityGroup or securityGroups, is allowed');
        }
        if (props.securityGroups) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic security group for Lambda Function ' + core_1.Names.uniqueId(this),
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        this._connections = new ec2.Connections({ securityGroups });
        if (props.filesystem) {
            if (props.filesystem.config.connections) {
                props.filesystem.config.connections.allowDefaultPortFrom(this);
            }
        }
        const allowPublicSubnet = props.allowPublicSubnet ?? false;
        const { subnetIds } = props.vpc.selectSubnets(props.vpcSubnets);
        const publicSubnetIds = new Set(props.vpc.publicSubnets.map(s => s.subnetId));
        for (const subnetId of subnetIds) {
            if (publicSubnetIds.has(subnetId) && !allowPublicSubnet) {
                throw new Error('Lambda Functions in a public subnet can NOT access the internet. ' +
                    'If you are aware of this limitation and would still like to place the function in a public subnet, set `allowPublicSubnet` to true');
            }
        }
        // List can't be empty here, if we got this far you intended to put your Lambda
        // in subnets. We're going to guarantee that we get the nice error message by
        // making VpcNetwork do the selection again.
        return {
            subnetIds,
            securityGroupIds: securityGroups.map(sg => sg.securityGroupId),
        };
    }
    isQueue(deadLetterQueue) {
        return deadLetterQueue.queueArn !== undefined;
    }
    buildDeadLetterQueue(props) {
        if (!props.deadLetterQueue && !props.deadLetterQueueEnabled && !props.deadLetterTopic) {
            return undefined;
        }
        if (props.deadLetterQueue && props.deadLetterQueueEnabled === false) {
            throw Error('deadLetterQueue defined but deadLetterQueueEnabled explicitly set to false');
        }
        if (props.deadLetterTopic && (props.deadLetterQueue || props.deadLetterQueueEnabled !== undefined)) {
            throw new Error('deadLetterQueue and deadLetterTopic cannot be specified together at the same time');
        }
        let deadLetterQueue;
        if (props.deadLetterTopic) {
            deadLetterQueue = props.deadLetterTopic;
            this.addToRolePolicy(new iam.PolicyStatement({
                actions: ['sns:Publish'],
                resources: [deadLetterQueue.topicArn],
            }));
        }
        else {
            deadLetterQueue = props.deadLetterQueue || new sqs.Queue(this, 'DeadLetterQueue', {
                retentionPeriod: core_1.Duration.days(14),
            });
            this.addToRolePolicy(new iam.PolicyStatement({
                actions: ['sqs:SendMessage'],
                resources: [deadLetterQueue.queueArn],
            }));
        }
        return deadLetterQueue;
    }
    buildDeadLetterConfig(deadLetterQueue) {
        if (deadLetterQueue) {
            return {
                targetArn: this.isQueue(deadLetterQueue) ? deadLetterQueue.queueArn : deadLetterQueue.topicArn,
            };
        }
        else {
            return undefined;
        }
    }
    buildTracingConfig(props) {
        if (props.tracing === undefined || props.tracing === Tracing.DISABLED) {
            return undefined;
        }
        this.addToRolePolicy(new iam.PolicyStatement({
            actions: ['xray:PutTraceSegments', 'xray:PutTelemetryRecords'],
            resources: ['*'],
        }));
        return {
            mode: props.tracing,
        };
    }
    validateProfiling(props) {
        if (!props.runtime.supportsCodeGuruProfiling) {
            throw new Error(`CodeGuru profiling is not supported by runtime ${props.runtime.name}`);
        }
        if (props.environment && (props.environment.AWS_CODEGURU_PROFILER_GROUP_ARN || props.environment.AWS_CODEGURU_PROFILER_ENABLED)) {
            throw new Error('AWS_CODEGURU_PROFILER_GROUP_ARN and AWS_CODEGURU_PROFILER_ENABLED must not be set when profiling options enabled');
        }
    }
}
exports.Function = Function;
_a = JSII_RTTI_SYMBOL_1;
Function[_a] = { fqn: "monocdk.aws_lambda.Function", version: "1.191.0" };
/** @internal */
Function._VER_PROPS = {};
/**
 * Given an opaque (token) ARN, returns a CloudFormation expression that extracts the function
 * name from the ARN.
 *
 * Function ARNs look like this:
 *
 *   arn:aws:lambda:region:account-id:function:function-name
 *
 * ..which means that in order to extract the `function-name` component from the ARN, we can
 * split the ARN using ":" and select the component in index 6.
 *
 * @returns `FnSelect(6, FnSplit(':', arn))`
 */
function extractNameFromArn(arn) {
    return core_1.Fn.select(6, core_1.Fn.split(':', arn));
}
function verifyCodeConfig(code, props) {
    // mutually exclusive
    const codeType = [code.inlineCode, code.s3Location, code.image];
    if (codeType.filter(x => !!x).length !== 1) {
        throw new Error('lambda.Code must specify exactly one of: "inlineCode", "s3Location", or "image"');
    }
    if (!!code.image === (props.handler !== handler_1.Handler.FROM_IMAGE)) {
        throw new Error('handler must be `Handler.FROM_IMAGE` when using image asset for Lambda function');
    }
    if (!!code.image === (props.runtime !== runtime_1.Runtime.FROM_IMAGE)) {
        throw new Error('runtime must be `Runtime.FROM_IMAGE` when using image asset for Lambda function');
    }
    // if this is inline code, check that the runtime supports
    if (code.inlineCode && !props.runtime.supportsInlineCode) {
        throw new Error(`Inline source not allowed for ${props.runtime.name}`);
    }
}
exports.verifyCodeConfig = verifyCodeConfig;
function undefinedIfNoKeys(struct) {
    const allUndefined = Object.values(struct).every(val => val === undefined);
    return allUndefined ? undefined : struct;
}
/**
 * Aspect for upgrading function versions when the feature flag
 * provided feature flag present. This can be necessary when the feature flag
 * changes the function hash, as such changes must be associated with a new
 * version. This aspect will change the function description in these cases,
 * which "validates" the new function hash.
 */
class FunctionVersionUpgrade {
    constructor(featureFlag, enabled = true) {
        this.featureFlag = featureFlag;
        this.enabled = enabled;
    }
    visit(node) {
        try {
            jsiiDeprecationWarnings.monocdk_IConstruct(node);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.visit);
            }
            throw error;
        }
        if (node instanceof Function &&
            this.enabled === core_1.FeatureFlags.of(node).isEnabled(this.featureFlag)) {
            const cfnFunction = node.node.defaultChild;
            const desc = cfnFunction.description ? `${cfnFunction.description} ` : '';
            cfnFunction.addPropertyOverride('Description', `${desc}version-hash:${function_hash_1.calculateFunctionHash(node)}`);
        }
    }
    ;
}
exports.FunctionVersionUpgrade = FunctionVersionUpgrade;
_b = JSII_RTTI_SYMBOL_1;
FunctionVersionUpgrade[_b] = { fqn: "monocdk.aws_lambda.FunctionVersionUpgrade", version: "1.191.0" };
//# sourceMappingURL=data:application/json;base64,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