import * as acmpca from '../../aws-acmpca';
import * as ec2 from '../../aws-ec2';
import * as kms from '../../aws-kms';
import * as logs from '../../aws-logs';
import * as s3 from '../../aws-s3';
import * as core from '../../core';
import * as constructs from 'constructs';
import { KafkaVersion } from './';
/**
 * Represents a MSK Cluster
 */
export interface ICluster extends core.IResource, ec2.IConnectable {
    /**
     * The ARN of cluster.
     *
     * @attribute
     */
    readonly clusterArn: string;
    /**
     * The physical name of the cluster.
     *
     * @attribute
     */
    readonly clusterName: string;
}
/**
 * A new or imported MSK Cluster.
 */
declare abstract class ClusterBase extends core.Resource implements ICluster {
    abstract readonly clusterArn: string;
    abstract readonly clusterName: string;
    /** @internal */
    protected _connections: ec2.Connections | undefined;
    /** Manages connections for the cluster */
    get connections(): ec2.Connections;
}
/**
 *  Properties for a MSK Cluster
 */
export interface ClusterProps {
    /**
     * The physical name of the cluster.
     */
    readonly clusterName: string;
    /**
     * The version of Apache Kafka.
     */
    readonly kafkaVersion: KafkaVersion;
    /**
     * Number of Apache Kafka brokers deployed in each Availability Zone.
     *
     * @default 1
     */
    readonly numberOfBrokerNodes?: number;
    /**
     * Defines the virtual networking environment for this cluster.
     * Must have at least 2 subnets in two different AZs.
     */
    readonly vpc: ec2.IVpc;
    /**
     * Where to place the nodes within the VPC.
     * Amazon MSK distributes the broker nodes evenly across the subnets that you specify.
     * The subnets that you specify must be in distinct Availability Zones.
     * Client subnets can't be in Availability Zone us-east-1e.
     *
     * @default - the Vpc default strategy if not specified.
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * The EC2 instance type that you want Amazon MSK to use when it creates your brokers.
     *
     * @see https://docs.aws.amazon.com/msk/latest/developerguide/msk-create-cluster.html#broker-instance-types
     * @default kafka.m5.large
     */
    readonly instanceType?: ec2.InstanceType;
    /**
     * The AWS security groups to associate with the elastic network interfaces in order to specify who can
     * connect to and communicate with the Amazon MSK cluster.
     *
     * @default - create new security group
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * Information about storage volumes attached to MSK broker nodes.
     *
     * @default - 1000 GiB EBS volume
     */
    readonly ebsStorageInfo?: EbsStorageInfo;
    /**
     * The Amazon MSK configuration to use for the cluster.
     *
     * @default - none
     */
    readonly configurationInfo?: ClusterConfigurationInfo;
    /**
     * Cluster monitoring configuration.
     *
     * @default - DEFAULT monitoring level
     */
    readonly monitoring?: MonitoringConfiguration;
    /**
     * Configure your MSK cluster to send broker logs to different destination types.
     *
     * @default - disabled
     */
    readonly logging?: BrokerLogging;
    /**
     * Config details for encryption in transit.
     *
     * @default - enabled
     */
    readonly encryptionInTransit?: EncryptionInTransitConfig;
    /**
     * Configuration properties for client authentication.
     * MSK supports using private TLS certificates or SASL/SCRAM to authenticate the identity of clients.
     *
     * @default - disabled
     */
    readonly clientAuthentication?: ClientAuthentication;
    /**
     * What to do when this resource is deleted from a stack.
     *
     * @default RemovalPolicy.RETAIN
     */
    readonly removalPolicy?: core.RemovalPolicy;
}
/**
 * EBS volume information.
 */
export interface EbsStorageInfo {
    /**
     * The size in GiB of the EBS volume for the data drive on each broker node.
     *
     * @default 1000
     */
    readonly volumeSize?: number;
    /**
     * The AWS KMS key for encrypting data at rest.
     *
     * @default Uses AWS managed CMK (aws/kafka)
     */
    readonly encryptionKey?: kms.IKey;
}
/**
 * The Amazon MSK configuration to use for the cluster.
 * Note: There is currently no Cloudformation Resource to create a Configuration
 */
export interface ClusterConfigurationInfo {
    /**
     * The Amazon Resource Name (ARN) of the MSK configuration to use.
     * For example, arn:aws:kafka:us-east-1:123456789012:configuration/example-configuration-name/abcdabcd-1234-abcd-1234-abcd123e8e8e-1.
     */
    readonly arn: string;
    /**
     * The revision of the Amazon MSK configuration to use.
     */
    readonly revision: number;
}
/**
 * The level of monitoring for the MSK cluster
 *
 * @see https://docs.aws.amazon.com/msk/latest/developerguide/monitoring.html#metrics-details
 */
export declare enum ClusterMonitoringLevel {
    /**
     * Default metrics are the essential metrics to monitor.
     */
    DEFAULT = "DEFAULT",
    /**
     * Per Broker metrics give you metrics at the broker level.
     */
    PER_BROKER = "PER_BROKER",
    /**
     * Per Topic Per Broker metrics help you understand volume at the topic level.
     */
    PER_TOPIC_PER_BROKER = "PER_TOPIC_PER_BROKER",
    /**
     * Per Topic Per Partition metrics help you understand consumer group lag at the topic partition level.
     */
    PER_TOPIC_PER_PARTITION = "PER_TOPIC_PER_PARTITION"
}
/**
 * Monitoring Configuration
 */
export interface MonitoringConfiguration {
    /**
     * Specifies the level of monitoring for the MSK cluster.
     *
     * @default DEFAULT
     */
    readonly clusterMonitoringLevel?: ClusterMonitoringLevel;
    /**
     * Indicates whether you want to enable or disable the JMX Exporter.
     *
     * @default false
     */
    readonly enablePrometheusJmxExporter?: boolean;
    /**
     * Indicates whether you want to enable or disable the Prometheus Node Exporter.
     *
     * You can use the Prometheus Node Exporter to get CPU and disk metrics for the broker nodes.
     *
     * @default false
     */
    readonly enablePrometheusNodeExporter?: boolean;
}
/**
 * Configuration details related to broker logs.
 */
export interface BrokerLogging {
    /**
     * The Kinesis Data Firehose delivery stream that is the destination for broker logs.
     *
     * @default - disabled
     */
    readonly firehoseDeliveryStreamName?: string;
    /**
     * The CloudWatch Logs group that is the destination for broker logs.
     *
     * @default - disabled
     */
    readonly cloudwatchLogGroup?: logs.ILogGroup;
    /**
     * Details of the Amazon S3 destination for broker logs.
     *
     * @default - disabled
     */
    readonly s3?: S3LoggingConfiguration;
}
/**
 * Details of the Amazon S3 destination for broker logs.
 */
export interface S3LoggingConfiguration {
    /**
     * The S3 bucket that is the destination for broker logs.
     */
    readonly bucket: s3.IBucket;
    /**
     * The S3 prefix that is the destination for broker logs.
     *
     * @default - no prefix
     */
    readonly prefix?: string;
}
/**
 * Indicates the encryption setting for data in transit between clients and brokers.
 */
export declare enum ClientBrokerEncryption {
    /**
     * TLS means that client-broker communication is enabled with TLS only.
     */
    TLS = "TLS",
    /**
     * TLS_PLAINTEXT means that client-broker communication is enabled for both TLS-encrypted, as well as plaintext data.
     */
    TLS_PLAINTEXT = "TLS_PLAINTEXT",
    /**
     * PLAINTEXT means that client-broker communication is enabled in plaintext only.
     */
    PLAINTEXT = "PLAINTEXT"
}
/**
 * The settings for encrypting data in transit.
 *
 * @see https://docs.aws.amazon.com/msk/latest/developerguide/msk-encryption.html#msk-encryption-in-transit
 */
export interface EncryptionInTransitConfig {
    /**
     * Indicates the encryption setting for data in transit between clients and brokers.
     *
     * @default - TLS
     */
    readonly clientBroker?: ClientBrokerEncryption;
    /**
     * Indicates that data communication among the broker nodes of the cluster is encrypted.
     *
     * @default true
     */
    readonly enableInCluster?: boolean;
}
/**
 * SASL authentication properties
 */
export interface SaslAuthProps {
    /**
     * Enable SASL/SCRAM authentication.
     *
     * @default false
     */
    readonly scram?: boolean;
    /**
     * Enable IAM access control.
     *
     * @default false
     */
    readonly iam?: boolean;
    /**
     * KMS Key to encrypt SASL/SCRAM secrets.
     *
     * You must use a customer master key (CMK) when creating users in secrets manager.
     * You cannot use a Secret with Amazon MSK that uses the default Secrets Manager encryption key.
     *
     * @default - CMK will be created with alias msk/{clusterName}/sasl/scram
     */
    readonly key?: kms.IKey;
}
/**
 * TLS authentication properties
 */
export interface TlsAuthProps {
    /**
     * List of ACM Certificate Authorities to enable TLS authentication.
     *
     * @default - none
     */
    readonly certificateAuthorities?: acmpca.ICertificateAuthority[];
}
/**
 * Configuration properties for client authentication.
 */
export declare class ClientAuthentication {
    readonly saslProps?: SaslAuthProps | undefined;
    readonly tlsProps?: TlsAuthProps | undefined;
    /**
     * SASL authentication
     */
    static sasl(props: SaslAuthProps): ClientAuthentication;
    /**
     * TLS authentication
     */
    static tls(props: TlsAuthProps): ClientAuthentication;
    /**
     * @param saslProps - properties for SASL authentication
     * @param tlsProps - properties for TLS authentication
     */
    private constructor();
}
/**
 * Create a MSK Cluster.
 *
 * @resource AWS::MSK::Cluster
 */
export declare class Cluster extends ClusterBase {
    /**
     * Reference an existing cluster, defined outside of the CDK code, by name.
     */
    static fromClusterArn(scope: constructs.Construct, id: string, clusterArn: string): ICluster;
    readonly clusterArn: string;
    readonly clusterName: string;
    /** Key used to encrypt SASL/SCRAM users */
    readonly saslScramAuthenticationKey?: kms.IKey;
    private _clusterDescription?;
    private _clusterBootstrapBrokers?;
    constructor(scope: constructs.Construct, id: string, props: ClusterProps);
    private mskInstanceType;
    /**
     * Get the ZooKeeper Connection string
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @param responseField Field to return from API call. eg. ZookeeperConnectString, ZookeeperConnectStringTls
     * @returns - The connection string to use to connect to the Apache ZooKeeper cluster.
     */
    private _zookeeperConnectionString;
    /**
     * Get the ZooKeeper Connection string
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @returns - The connection string to use to connect to the Apache ZooKeeper cluster.
     */
    get zookeeperConnectionString(): string;
    /**
     * Get the ZooKeeper Connection string for a TLS enabled cluster
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @returns - The connection string to use to connect to zookeeper cluster on TLS port.
     */
    get zookeeperConnectionStringTls(): string;
    /**
     * Get the list of brokers that a client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @param responseField Field to return from API call. eg. BootstrapBrokerStringSaslScram, BootstrapBrokerString
     * @returns - A string containing one or more hostname:port pairs.
     */
    private _bootstrapBrokers;
    /**
     * Get the list of brokers that a client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more hostname:port pairs.
     */
    get bootstrapBrokers(): string;
    /**
     * Get the list of brokers that a TLS authenticated client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more DNS names (or IP) and TLS port pairs.
     */
    get bootstrapBrokersTls(): string;
    /**
     * Get the list of brokers that a SASL/SCRAM authenticated client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more dns name (or IP) and SASL SCRAM port pairs.
     */
    get bootstrapBrokersSaslScram(): string;
    /**
     * A list of usersnames to register with the cluster. The password will automatically be generated using Secrets
     * Manager and the { username, password } JSON object stored in Secrets Manager as `AmazonMSK_username`.
     *
     * Must be using the SASL/SCRAM authentication mechanism.
     *
     * @param usernames - username(s) to register with the cluster
     */
    addUser(...usernames: string[]): void;
}
export {};
