"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenSearchAccessPolicy = void 0;
const iam = require("../../aws-iam");
const cdk = require("../../core");
const cr = require("../../custom-resources");
/**
 * Creates LogGroup resource policies.
 */
class OpenSearchAccessPolicy extends cr.AwsCustomResource {
    constructor(scope, id, props) {
        super(scope, id, {
            resourceType: 'Custom::OpenSearchAccessPolicy',
            onUpdate: {
                action: 'updateDomainConfig',
                service: 'OpenSearch',
                parameters: {
                    DomainName: props.domainName,
                    AccessPolicies: cdk.Lazy.string({
                        produce: () => JSON.stringify(new iam.PolicyDocument({
                            statements: this.accessPolicyStatements,
                        }).toJSON()),
                    }),
                },
                // this is needed to limit the response body, otherwise it exceeds the CFN 4k limit
                outputPaths: ['DomainConfig.AccessPolicies'],
                physicalResourceId: cr.PhysicalResourceId.of(`${props.domainName}AccessPolicy`),
            },
            policy: cr.AwsCustomResourcePolicy.fromStatements([new iam.PolicyStatement({ actions: ['es:UpdateDomainConfig'], resources: [props.domainArn] })]),
        });
        this.accessPolicyStatements = [];
        this.addAccessPolicies(...props.accessPolicies);
    }
    /**
     * Add policy statements to the domain access policy
     */
    addAccessPolicies(...accessPolicyStatements) {
        this.accessPolicyStatements.push(...accessPolicyStatements);
    }
}
exports.OpenSearchAccessPolicy = OpenSearchAccessPolicy;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib3BlbnNlYXJjaC1hY2Nlc3MtcG9saWN5LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsib3BlbnNlYXJjaC1hY2Nlc3MtcG9saWN5LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHFDQUFxQztBQUNyQyxrQ0FBa0M7QUFDbEMsNkNBQTZDO0FBdUI3Qzs7R0FFRztBQUNILE1BQWEsc0JBQXVCLFNBQVEsRUFBRSxDQUFDLGlCQUFpQjtJQUk5RCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWtDO1FBQzFFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsWUFBWSxFQUFFLGdDQUFnQztZQUM5QyxRQUFRLEVBQUU7Z0JBQ1IsTUFBTSxFQUFFLG9CQUFvQjtnQkFDNUIsT0FBTyxFQUFFLFlBQVk7Z0JBQ3JCLFVBQVUsRUFBRTtvQkFDVixVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7b0JBQzVCLGNBQWMsRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQzt3QkFDOUIsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQzNCLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQzs0QkFDckIsVUFBVSxFQUFFLElBQUksQ0FBQyxzQkFBc0I7eUJBQ3hDLENBQUMsQ0FBQyxNQUFNLEVBQUUsQ0FDWjtxQkFDRixDQUFDO2lCQUNIO2dCQUNELG1GQUFtRjtnQkFDbkYsV0FBVyxFQUFFLENBQUMsNkJBQTZCLENBQUM7Z0JBQzVDLGtCQUFrQixFQUFFLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsR0FBRyxLQUFLLENBQUMsVUFBVSxjQUFjLENBQUM7YUFDaEY7WUFDRCxNQUFNLEVBQUUsRUFBRSxDQUFDLHVCQUF1QixDQUFDLGNBQWMsQ0FBQyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQyxFQUFFLE9BQU8sRUFBRSxDQUFDLHVCQUF1QixDQUFDLEVBQUUsU0FBUyxFQUFFLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1NBQ25KLENBQUMsQ0FBQztRQXZCRywyQkFBc0IsR0FBMEIsRUFBRSxDQUFDO1FBeUJ6RCxJQUFJLENBQUMsaUJBQWlCLENBQUMsR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDLENBQUM7S0FDakQ7SUFFRDs7T0FFRztJQUNJLGlCQUFpQixDQUFDLEdBQUcsc0JBQTZDO1FBQ3ZFLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsR0FBRyxzQkFBc0IsQ0FBQyxDQUFDO0tBQzdEO0NBQ0Y7QUFwQ0Qsd0RBb0NDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgaWFtIGZyb20gJy4uLy4uL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJy4uLy4uL2NvcmUnO1xuaW1wb3J0ICogYXMgY3IgZnJvbSAnLi4vLi4vY3VzdG9tLXJlc291cmNlcyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcblxuLyoqXG4gKiBDb25zdHJ1Y3Rpb24gcHJvcGVydGllcyBmb3IgT3BlblNlYXJjaEFjY2Vzc1BvbGljeVxuICovXG5leHBvcnQgaW50ZXJmYWNlIE9wZW5TZWFyY2hBY2Nlc3NQb2xpY3lQcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgT3BlblNlYXJjaCBEb21haW4gbmFtZVxuICAgKi9cbiAgcmVhZG9ubHkgZG9tYWluTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgT3BlblNlYXJjaCBEb21haW4gQVJOXG4gICAqL1xuICByZWFkb25seSBkb21haW5Bcm46IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGFjY2VzcyBwb2xpY3kgc3RhdGVtZW50cyBmb3IgdGhlIE9wZW5TZWFyY2ggY2x1c3RlclxuICAgKi9cbiAgcmVhZG9ubHkgYWNjZXNzUG9saWNpZXM6IGlhbS5Qb2xpY3lTdGF0ZW1lbnRbXTtcbn1cblxuLyoqXG4gKiBDcmVhdGVzIExvZ0dyb3VwIHJlc291cmNlIHBvbGljaWVzLlxuICovXG5leHBvcnQgY2xhc3MgT3BlblNlYXJjaEFjY2Vzc1BvbGljeSBleHRlbmRzIGNyLkF3c0N1c3RvbVJlc291cmNlIHtcblxuICBwcml2YXRlIGFjY2Vzc1BvbGljeVN0YXRlbWVudHM6IGlhbS5Qb2xpY3lTdGF0ZW1lbnRbXSA9IFtdO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBPcGVuU2VhcmNoQWNjZXNzUG9saWN5UHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHJlc291cmNlVHlwZTogJ0N1c3RvbTo6T3BlblNlYXJjaEFjY2Vzc1BvbGljeScsXG4gICAgICBvblVwZGF0ZToge1xuICAgICAgICBhY3Rpb246ICd1cGRhdGVEb21haW5Db25maWcnLFxuICAgICAgICBzZXJ2aWNlOiAnT3BlblNlYXJjaCcsXG4gICAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgICBEb21haW5OYW1lOiBwcm9wcy5kb21haW5OYW1lLFxuICAgICAgICAgIEFjY2Vzc1BvbGljaWVzOiBjZGsuTGF6eS5zdHJpbmcoe1xuICAgICAgICAgICAgcHJvZHVjZTogKCkgPT4gSlNPTi5zdHJpbmdpZnkoXG4gICAgICAgICAgICAgIG5ldyBpYW0uUG9saWN5RG9jdW1lbnQoe1xuICAgICAgICAgICAgICAgIHN0YXRlbWVudHM6IHRoaXMuYWNjZXNzUG9saWN5U3RhdGVtZW50cyxcbiAgICAgICAgICAgICAgfSkudG9KU09OKCksXG4gICAgICAgICAgICApLFxuICAgICAgICAgIH0pLFxuICAgICAgICB9LFxuICAgICAgICAvLyB0aGlzIGlzIG5lZWRlZCB0byBsaW1pdCB0aGUgcmVzcG9uc2UgYm9keSwgb3RoZXJ3aXNlIGl0IGV4Y2VlZHMgdGhlIENGTiA0ayBsaW1pdFxuICAgICAgICBvdXRwdXRQYXRoczogWydEb21haW5Db25maWcuQWNjZXNzUG9saWNpZXMnXSxcbiAgICAgICAgcGh5c2ljYWxSZXNvdXJjZUlkOiBjci5QaHlzaWNhbFJlc291cmNlSWQub2YoYCR7cHJvcHMuZG9tYWluTmFtZX1BY2Nlc3NQb2xpY3lgKSxcbiAgICAgIH0sXG4gICAgICBwb2xpY3k6IGNyLkF3c0N1c3RvbVJlc291cmNlUG9saWN5LmZyb21TdGF0ZW1lbnRzKFtuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7IGFjdGlvbnM6IFsnZXM6VXBkYXRlRG9tYWluQ29uZmlnJ10sIHJlc291cmNlczogW3Byb3BzLmRvbWFpbkFybl0gfSldKSxcbiAgICB9KTtcblxuICAgIHRoaXMuYWRkQWNjZXNzUG9saWNpZXMoLi4ucHJvcHMuYWNjZXNzUG9saWNpZXMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBwb2xpY3kgc3RhdGVtZW50cyB0byB0aGUgZG9tYWluIGFjY2VzcyBwb2xpY3lcbiAgICovXG4gIHB1YmxpYyBhZGRBY2Nlc3NQb2xpY2llcyguLi5hY2Nlc3NQb2xpY3lTdGF0ZW1lbnRzOiBpYW0uUG9saWN5U3RhdGVtZW50W10pIHtcbiAgICB0aGlzLmFjY2Vzc1BvbGljeVN0YXRlbWVudHMucHVzaCguLi5hY2Nlc3NQb2xpY3lTdGF0ZW1lbnRzKTtcbiAgfVxufVxuIl19