"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OptionGroup = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const core_1 = require("../../core");
const rds_generated_1 = require("./rds.generated");
/**
 * An option group
 */
class OptionGroup extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * The connections object for the options.
         */
        this.optionConnections = {};
        this.configurations = [];
        try {
            jsiiDeprecationWarnings.monocdk_aws_rds_OptionGroupProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, OptionGroup);
            }
            throw error;
        }
        const majorEngineVersion = props.engine.engineVersion?.majorVersion;
        if (!majorEngineVersion) {
            throw new Error("OptionGroup cannot be used with an engine that doesn't specify a version");
        }
        props.configurations.forEach(config => this.addConfiguration(config));
        const optionGroup = new rds_generated_1.CfnOptionGroup(this, 'Resource', {
            engineName: props.engine.engineType,
            majorEngineVersion,
            optionGroupDescription: props.description || `Option group for ${props.engine.engineType} ${majorEngineVersion}`,
            optionConfigurations: core_1.Lazy.any({ produce: () => this.renderConfigurations(this.configurations) }),
        });
        this.optionGroupName = optionGroup.ref;
    }
    /**
     * Import an existing option group.
     */
    static fromOptionGroupName(scope, id, optionGroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.optionGroupName = optionGroupName;
            }
            addConfiguration(_) { return false; }
        }
        return new Import(scope, id);
    }
    addConfiguration(configuration) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_rds_OptionConfiguration(configuration);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addConfiguration);
            }
            throw error;
        }
        this.configurations.push(configuration);
        if (configuration.port) {
            if (!configuration.vpc) {
                throw new Error('`port` and `vpc` must be specified together.');
            }
            const securityGroups = configuration.securityGroups && configuration.securityGroups.length > 0
                ? configuration.securityGroups
                : [new ec2.SecurityGroup(this, `SecurityGroup${configuration.name}`, {
                        description: `Security group for ${configuration.name} option`,
                        vpc: configuration.vpc,
                    })];
            this.optionConnections[configuration.name] = new ec2.Connections({
                securityGroups: securityGroups,
                defaultPort: ec2.Port.tcp(configuration.port),
            });
        }
        return true;
    }
    /**
     * Renders the option configurations specifications.
     */
    renderConfigurations(configurations) {
        const configs = [];
        for (const config of configurations) {
            const securityGroups = config.vpc
                ? this.optionConnections[config.name].securityGroups.map(sg => sg.securityGroupId)
                : undefined;
            configs.push({
                optionName: config.name,
                optionSettings: config.settings && Object.entries(config.settings).map(([name, value]) => ({ name, value })),
                optionVersion: config.version,
                port: config.port,
                vpcSecurityGroupMemberships: securityGroups,
            });
        }
        return configs;
    }
}
exports.OptionGroup = OptionGroup;
_a = JSII_RTTI_SYMBOL_1;
OptionGroup[_a] = { fqn: "monocdk.aws_rds.OptionGroup", version: "1.191.0" };
//# sourceMappingURL=data:application/json;base64,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