"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/// !cdk-integ pragma:ignore-assets
const cloudwatch = require("../../aws-cloudwatch");
const ec2 = require("../../aws-ec2");
const targets = require("../../aws-events-targets");
const lambda = require("../../aws-lambda");
const logs = require("../../aws-logs");
const cdk = require("../../core");
const core_1 = require("../../core");
const rds = require("../lib");
const app = new cdk.App();
class DatabaseInstanceStack extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, props);
        const vpc = new ec2.Vpc(this, 'VPC', { maxAzs: 2 });
        /// !show
        // Set open cursors with parameter group
        const parameterGroup = new rds.ParameterGroup(this, 'ParameterGroup', {
            engine: rds.DatabaseInstanceEngine.oracleSe2({ version: rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1 }),
            parameters: {
                open_cursors: '2500',
            },
        });
        /// Add XMLDB and OEM with option group
        const optionGroup = new rds.OptionGroup(this, 'OptionGroup', {
            engine: rds.DatabaseInstanceEngine.oracleSe2({ version: rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1 }),
            configurations: [
                {
                    name: 'LOCATOR',
                },
                {
                    name: 'OEM',
                    port: 1158,
                    vpc,
                },
            ],
        });
        // Allow connections to OEM
        optionGroup.optionConnections.OEM.connections.allowDefaultPortFromAnyIpv4();
        // Database instance with production values
        const instance = new rds.DatabaseInstance(this, 'Instance', {
            engine: rds.DatabaseInstanceEngine.oracleSe2({ version: rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1 }),
            licenseModel: rds.LicenseModel.BRING_YOUR_OWN_LICENSE,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE3, ec2.InstanceSize.MEDIUM),
            multiAz: true,
            storageType: rds.StorageType.IO1,
            credentials: rds.Credentials.fromUsername('syscdk'),
            vpc,
            databaseName: 'ORCL',
            storageEncrypted: true,
            backupRetention: cdk.Duration.days(7),
            monitoringInterval: cdk.Duration.seconds(60),
            enablePerformanceInsights: true,
            cloudwatchLogsExports: [
                'trace',
                'audit',
                'alert',
                'listener',
            ],
            cloudwatchLogsRetention: logs.RetentionDays.ONE_MONTH,
            autoMinorVersionUpgrade: true,
            optionGroup,
            parameterGroup,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        // Allow connections on default port from any IPV4
        instance.connections.allowDefaultPortFromAnyIpv4();
        // Rotate the master user password every 30 days
        instance.addRotationSingleUser();
        // Add alarm for high CPU
        new cloudwatch.Alarm(this, 'HighCPU', {
            metric: instance.metricCPUUtilization(),
            threshold: 90,
            evaluationPeriods: 1,
        });
        // Trigger Lambda function on instance availability events
        const fn = new lambda.Function(this, 'Function', {
            code: lambda.Code.fromInline('exports.handler = (event) => console.log(event);'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_14_X,
        });
        const availabilityRule = instance.onEvent('Availability', { target: new targets.LambdaFunction(fn) });
        availabilityRule.addEventPattern({
            detail: {
                EventCategories: [
                    'availability',
                ],
            },
        });
    }
}
new DatabaseInstanceStack(app, 'aws-cdk-rds-instance');
app.synth();
//# sourceMappingURL=data:application/json;base64,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