"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
// eslint-disable-next-line import/no-extraneous-dependencies
const aws_sdk_1 = require("aws-sdk");
async function handler(event) {
    const resourceProps = event.ResourceProperties;
    switch (event.RequestType) {
        case 'Create':
        case 'Update':
            return cfnEventHandler(resourceProps, false);
        case 'Delete':
            return cfnEventHandler(resourceProps, true);
    }
}
exports.handler = handler;
async function cfnEventHandler(props, isDeleteEvent) {
    const { AssumeRoleArn, ParentZoneId, ParentZoneName, DelegatedZoneName, DelegatedZoneNameServers, TTL } = props;
    if (!ParentZoneId && !ParentZoneName) {
        throw Error('One of ParentZoneId or ParentZoneName must be specified');
    }
    const credentials = await getCrossAccountCredentials(AssumeRoleArn);
    const route53 = new aws_sdk_1.Route53({ credentials });
    const parentZoneId = ParentZoneId ?? await getHostedZoneIdByName(ParentZoneName, route53);
    await route53.changeResourceRecordSets({
        HostedZoneId: parentZoneId,
        ChangeBatch: {
            Changes: [{
                    Action: isDeleteEvent ? 'DELETE' : 'UPSERT',
                    ResourceRecordSet: {
                        Name: DelegatedZoneName,
                        Type: 'NS',
                        TTL,
                        ResourceRecords: DelegatedZoneNameServers.map(ns => ({ Value: ns })),
                    },
                }],
        },
    }).promise();
}
async function getCrossAccountCredentials(roleArn) {
    const sts = new aws_sdk_1.STS();
    const timestamp = (new Date()).getTime();
    const { Credentials: assumedCredentials } = await sts
        .assumeRole({
        RoleArn: roleArn,
        RoleSessionName: `cross-account-zone-delegation-${timestamp}`,
    })
        .promise();
    if (!assumedCredentials) {
        throw Error('Error getting assume role credentials');
    }
    return new aws_sdk_1.Credentials({
        accessKeyId: assumedCredentials.AccessKeyId,
        secretAccessKey: assumedCredentials.SecretAccessKey,
        sessionToken: assumedCredentials.SessionToken,
    });
}
async function getHostedZoneIdByName(name, route53) {
    const zones = await route53.listHostedZonesByName({ DNSName: name }).promise();
    const matchedZones = zones.HostedZones.filter(zone => zone.Name === `${name}.`);
    if (matchedZones.length !== 1) {
        throw Error(`Expected one hosted zone to match the given name but found ${matchedZones.length}`);
    }
    return matchedZones[0].Id;
}
//# sourceMappingURL=data:application/json;base64,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