import { IBucket } from '../../aws-s3';
import { IResource, Resource } from '../../core';
import { Construct } from 'constructs';
/**
 * A Firewall Domain List
 */
export interface IFirewallDomainList extends IResource {
    /**
     * The ID of the domain list
     *
     * @attribute
     */
    readonly firewallDomainListId: string;
}
/**
 * Properties for a Firewall Domain List
 */
export interface FirewallDomainListProps {
    /**
     * A name for the domain list
     *
     * @default - a CloudFormation generated name
     */
    readonly name?: string;
    /**
     * A list of domains
     */
    readonly domains: FirewallDomains;
}
/**
 * A list of domains
 */
export declare abstract class FirewallDomains {
    /**
     * Firewall domains created from a list of domains
     *
     * @param list the list of domains
     */
    static fromList(list: string[]): FirewallDomains;
    /**
     * Firewall domains created from the URL of a file stored in Amazon S3.
     * The file must be a text file and must contain a single domain per line.
     * The content type of the S3 object must be `plain/text`.
     *
     * @param url S3 bucket url (s3://bucket/prefix/objet).
     */
    static fromS3Url(url: string): FirewallDomains;
    /**
     * Firewall domains created from a file stored in Amazon S3.
     * The file must be a text file and must contain a single domain per line.
     * The content type of the S3 object must be `plain/text`.
     *
     * @param bucket S3 bucket
     * @param key S3 key
     */
    static fromS3(bucket: IBucket, key: string): FirewallDomains;
    /**
     * Firewall domains created from a local disk path to a text file.
     * The file must be a text file (`.txt` extension) and must contain a single
     * domain per line. It will be uploaded to S3.
     *
     * @param assetPath path to the text file
     */
    static fromAsset(assetPath: string): FirewallDomains;
    /** Binds the domains to a domain list */
    abstract bind(scope: Construct): DomainsConfig;
}
/**
 * Domains configuration
 */
export interface DomainsConfig {
    /**
     * The fully qualified URL or URI of the file stored in Amazon S3 that contains
     * the list of domains to import. The file must be a text file and must contain
     * a single domain per line. The content type of the S3 object must be `plain/text`.
     *
     * @default - use `domains`
     */
    readonly domainFileUrl?: string;
    /**
     * A list of domains
     *
     * @default - use `domainFileUrl`
     */
    readonly domains?: string[];
}
/**
 * A Firewall Domain List
 */
export declare class FirewallDomainList extends Resource implements IFirewallDomainList {
    /**
     * Import an existing Firewall Rule Group
     */
    static fromFirewallDomainListId(scope: Construct, id: string, firewallDomainListId: string): IFirewallDomainList;
    readonly firewallDomainListId: string;
    /**
     * The ARN (Amazon Resource Name) of the domain list
     * @attribute
     */
    readonly firewallDomainListArn: string;
    /**
      * The date and time that the domain list was created
      * @attribute
      */
    readonly firewallDomainListCreationTime: string;
    /**
      * The creator request ID
      * @attribute
      */
    readonly firewallDomainListCreatorRequestId: string;
    /**
      * The number of domains in the list
      * @attribute
      */
    readonly firewallDomainListDomainCount: number;
    /**
      * The owner of the list, used only for lists that are not managed by you.
      * For example, the managed domain list `AWSManagedDomainsMalwareDomainList`
      * has the managed owner name `Route 53 Resolver DNS Firewall`.
      * @attribute
      */
    readonly firewallDomainListManagedOwnerName: string;
    /**
      * The date and time that the domain list was last modified
      * @attribute
      */
    readonly firewallDomainListModificationTime: string;
    /**
      * The status of the domain list
      * @attribute
      */
    readonly firewallDomainListStatus: string;
    /**
      * Additional information about the status of the rule group
      * @attribute
      */
    readonly firewallDomainListStatusMessage: string;
    constructor(scope: Construct, id: string, props: FirewallDomainListProps);
}
