import { Duration, IResource, Resource } from '../../core';
import { Construct } from 'constructs';
import { IFirewallDomainList } from './firewall-domain-list';
import { FirewallRuleGroupAssociation, FirewallRuleGroupAssociationOptions } from './firewall-rule-group-association';
/**
 * A Firewall Rule Group
 */
export interface IFirewallRuleGroup extends IResource {
    /**
     * The ID of the rule group
     *
     * @attribute
     */
    readonly firewallRuleGroupId: string;
}
/**
 * Properties for a Firewall Rule Group
 */
export interface FirewallRuleGroupProps {
    /**
     * The name of the rule group.
     *
     * @default - a CloudFormation generated name
     */
    readonly name?: string;
    /**
     * A list of rules for this group
     *
     * @default - no rules
     */
    readonly rules?: FirewallRule[];
}
/**
 * A Firewall Rule
 */
export interface FirewallRule {
    /**
     * The action for this rule
     */
    readonly action: FirewallRuleAction;
    /**
     * The domain list for this rule
     */
    readonly firewallDomainList: IFirewallDomainList;
    /**
     * The priority of the rule in the rule group. This value must be unique within
     * the rule group.
     */
    readonly priority: number;
}
/**
 * A Firewall Rule
 */
export declare abstract class FirewallRuleAction {
    /**
     * Permit the request to go through
     */
    static allow(): FirewallRuleAction;
    /**
     * Permit the request to go through but send an alert to the logs
     */
    static alert(): FirewallRuleAction;
    /**
     * Disallow the request
     *
     * @param [response=DnsBlockResponse.noData()] The way that you want DNS Firewall to block the request
     */
    static block(response?: DnsBlockResponse): FirewallRuleAction;
    /**
     * The action that DNS Firewall should take on a DNS query when it matches
     * one of the domains in the rule's domain list
     */
    abstract readonly action: string;
    /**
     * The way that you want DNS Firewall to block the request
     */
    abstract readonly blockResponse?: DnsBlockResponse;
}
/**
 * The way that you want DNS Firewall to block the request
 */
export declare abstract class DnsBlockResponse {
    /**
     * Respond indicating that the query was successful, but no
     * response is available for it.
     */
    static noData(): DnsBlockResponse;
    /**
     * Respond indicating that the domain name that's in the query
     * doesn't exist.
     */
    static nxDomain(): DnsBlockResponse;
    /**
     * Provides a custom override response to the query
     *
     * @param domain The custom DNS record to send back in response to the query
     * @param [ttl=0] The recommended amount of time for the DNS resolver or
     *   web browser to cache the provided override record
     */
    static override(domain: string, ttl?: Duration): DnsBlockResponse;
    /** The DNS record's type */
    abstract readonly blockOverrideDnsType?: string;
    /** The custom DNS record to send back in response to the query */
    abstract readonly blockOverrideDomain?: string;
    /**
     * The recommended amount of time for the DNS resolver or
     * web browser to cache the provided override record
     */
    abstract readonly blockOverrideTtl?: Duration;
    /** The way that you want DNS Firewall to block the request */
    abstract readonly blockResponse?: string;
}
/**
 * A Firewall Rule Group
 */
export declare class FirewallRuleGroup extends Resource implements IFirewallRuleGroup {
    /**
     * Import an existing Firewall Rule Group
     */
    static fromFirewallRuleGroupId(scope: Construct, id: string, firewallRuleGroupId: string): IFirewallRuleGroup;
    readonly firewallRuleGroupId: string;
    /**
     * The ARN (Amazon Resource Name) of the rule group
     * @attribute
     */
    readonly firewallRuleGroupArn: string;
    /**
     * The date and time that the rule group was created
     * @attribute
     */
    readonly firewallRuleGroupCreationTime: string;
    /**
     * The creator request ID
     * @attribute
     */
    readonly firewallRuleGroupCreatorRequestId: string;
    /**
     * The date and time that the rule group was last modified
     * @attribute
     */
    readonly firewallRuleGroupModificationTime: string;
    /**
     * The AWS account ID for the account that created the rule group
     * @attribute
     */
    readonly firewallRuleGroupOwnerId: string;
    /**
     * The number of rules in the rule group
     * @attribute
     */
    readonly firewallRuleGroupRuleCount: number;
    /**
     * Whether the rule group is shared with other AWS accounts,
     * or was shared with the current account by another AWS account
     * @attribute
     */
    readonly firewallRuleGroupShareStatus: string;
    /**
     * The status of the rule group
     * @attribute
     */
    readonly firewallRuleGroupStatus: string;
    /**
     * Additional information about the status of the rule group
     * @attribute
     */
    readonly firewallRuleGroupStatusMessage: string;
    private readonly rules;
    constructor(scope: Construct, id: string, props?: FirewallRuleGroupProps);
    /**
     * Adds a rule to this group
     */
    addRule(rule: FirewallRule): FirewallRuleGroup;
    /**
     * Associates this Firewall Rule Group with a VPC
     */
    associate(id: string, props: FirewallRuleGroupAssociationOptions): FirewallRuleGroupAssociation;
}
