"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Source = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path_1 = require("path");
const s3_assets = require("../../aws-s3-assets");
const core_1 = require("../../core");
const render_data_1 = require("./render-data");
/**
 * Specifies bucket deployment source.
 *
 * Usage:
 *
 *     Source.bucket(bucket, key)
 *     Source.asset('/local/path/to/directory')
 *     Source.asset('/local/path/to/a/file.zip')
 *     Source.data('hello/world/file.txt', 'Hello, world!')
 *     Source.data('config.json', { baz: topic.topicArn })
 *
 */
class Source {
    constructor() { }
    /**
     * Uses a .zip file stored in an S3 bucket as the source for the destination bucket contents.
     *
     * Make sure you trust the producer of the archive.
     *
     * @param bucket The S3 Bucket
     * @param zipObjectKey The S3 object key of the zip file with contents
     */
    static bucket(bucket, zipObjectKey) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_s3_IBucket(bucket);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bucket);
            }
            throw error;
        }
        return {
            bind: (_, context) => {
                if (!context) {
                    throw new Error('To use a Source.bucket(), context must be provided');
                }
                bucket.grantRead(context.handlerRole);
                return { bucket, zipObjectKey };
            },
        };
    }
    /**
     * Uses a local asset as the deployment source.
     *
     * If the local asset is a .zip archive, make sure you trust the
     * producer of the archive.
     *
     * @param path The path to a local .zip file or a directory
     */
    static asset(path, options) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_s3_assets_AssetOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.asset);
            }
            throw error;
        }
        return {
            bind(scope, context) {
                if (!context) {
                    throw new Error('To use a Source.asset(), context must be provided');
                }
                let id = 1;
                while (scope.node.tryFindChild(`Asset${id}`)) {
                    id++;
                }
                const asset = new s3_assets.Asset(scope, `Asset${id}`, {
                    path,
                    ...options,
                });
                if (!asset.isZipArchive) {
                    throw new Error('Asset path must be either a .zip file or a directory');
                }
                asset.grantRead(context.handlerRole);
                return {
                    bucket: asset.bucket,
                    zipObjectKey: asset.s3ObjectKey,
                };
            },
        };
    }
    /**
     * Deploys an object with the specified string contents into the bucket. The
     * content can include deploy-time values (such as `snsTopic.topicArn`) that
     * will get resolved only during deployment.
     *
     * To store a JSON object use `Source.jsonData()`.
     *
     * @param objectKey The destination S3 object key (relative to the root of the
     * S3 deployment).
     * @param data The data to be stored in the object.
     */
    static data(objectKey, data) {
        return {
            bind: (scope, context) => {
                const workdir = core_1.FileSystem.mkdtemp('s3-deployment');
                const outputPath = path_1.join(workdir, objectKey);
                const rendered = render_data_1.renderData(scope, data);
                fs.mkdirSync(path_1.dirname(outputPath), { recursive: true });
                fs.writeFileSync(outputPath, rendered.text);
                const asset = this.asset(workdir).bind(scope, context);
                return {
                    bucket: asset.bucket,
                    zipObjectKey: asset.zipObjectKey,
                    markers: rendered.markers,
                };
            },
        };
    }
    /**
     * Deploys an object with the specified JSON object into the bucket. The
     * object can include deploy-time values (such as `snsTopic.topicArn`) that
     * will get resolved only during deployment.
     *
     * @param objectKey The destination S3 object key (relative to the root of the
     * S3 deployment).
     * @param obj A JSON object.
     */
    static jsonData(objectKey, obj) {
        return {
            bind: (scope, context) => {
                return Source.data(objectKey, core_1.Stack.of(scope).toJsonString(obj)).bind(scope, context);
            },
        };
    }
}
exports.Source = Source;
_a = JSII_RTTI_SYMBOL_1;
Source[_a] = { fqn: "monocdk.aws_s3_deployment.Source", version: "1.191.0" };
//# sourceMappingURL=data:application/json;base64,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