"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssociationManager = void 0;
const servicecatalog_generated_1 = require("../servicecatalog.generated");
const util_1 = require("./util");
const validation_1 = require("./validation");
class AssociationManager {
    static associateProductWithPortfolio(portfolio, product, options) {
        validation_1.InputValidator.validateLength(this.prettyPrintAssociation(portfolio, product), 'description', 0, 2000, options?.description);
        const associationKey = util_1.hashValues(portfolio.node.addr, product.node.addr, product.stack.node.addr);
        const constructId = `PortfolioProductAssociation${associationKey}`;
        const existingAssociation = portfolio.node.tryFindChild(constructId);
        const cfnAssociation = existingAssociation
            ? existingAssociation
            : new servicecatalog_generated_1.CfnPortfolioProductAssociation(portfolio, constructId, {
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
            });
        return {
            associationKey: associationKey,
            cfnPortfolioProductAssociation: cfnAssociation,
        };
    }
    static constrainTagUpdates(portfolio, product, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `ResourceUpdateConstraint${association.associationKey}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnResourceUpdateConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                tagUpdateOnProvisionedProduct: options.allow === false ? 'NOT_ALLOWED' : 'ALLOWED',
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot have multiple tag update constraints for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static notifyOnStackEvents(portfolio, product, topic, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `LaunchNotificationConstraint${util_1.hashValues(topic.node.addr, topic.stack.node.addr, association.associationKey)}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnLaunchNotificationConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                notificationArns: [topic.topicArn],
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Topic ${topic.node.path} is already subscribed to association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static constrainCloudFormationParameters(portfolio, product, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `LaunchTemplateConstraint${util_1.hashValues(association.associationKey, options.rule.ruleName)}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnLaunchTemplateConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                rules: this.formatTemplateRule(portfolio.stack, options.rule),
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Provisioning rule ${options.rule.ruleName} already configured on association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static setLaunchRole(portfolio, product, launchRole, options) {
        this.setLaunchRoleConstraint(portfolio, product, options, {
            roleArn: launchRole.roleArn,
        });
    }
    static setLocalLaunchRoleName(portfolio, product, launchRoleName, options) {
        this.setLaunchRoleConstraint(portfolio, product, options, {
            localRoleName: launchRoleName,
        });
    }
    static deployWithStackSets(portfolio, product, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        // Check if a launch role has already been set.
        if (portfolio.node.tryFindChild(this.launchRoleConstraintLogicalId(association.associationKey))) {
            throw new Error(`Cannot configure StackSet deployment when a launch role is already defined for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
        const constructId = this.stackSetConstraintLogicalId(association.associationKey);
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnStackSetConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description ?? '',
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                accountList: options.accounts,
                regionList: options.regions,
                adminRole: options.adminRole.roleArn,
                executionRole: options.executionRoleName,
                stackInstanceControl: options.allowStackSetInstanceOperations ? 'ALLOWED' : 'NOT_ALLOWED',
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot configure multiple StackSet deployment constraints for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static associateTagOptions(resource, resourceId, tagOptions) {
        for (const cfnTagOption of tagOptions._cfnTagOptions) {
            const tagAssocationConstructId = `TagOptionAssociation${util_1.hashValues(cfnTagOption.key, cfnTagOption.value, resource.node.addr)}`;
            if (!resource.node.tryFindChild(tagAssocationConstructId)) {
                new servicecatalog_generated_1.CfnTagOptionAssociation(resource, tagAssocationConstructId, {
                    resourceId: resourceId,
                    tagOptionId: cfnTagOption.ref,
                });
            }
        }
    }
    static setLaunchRoleConstraint(portfolio, product, options, roleOptions) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        // Check if a stackset deployment constraint has already been configured.
        if (portfolio.node.tryFindChild(this.stackSetConstraintLogicalId(association.associationKey))) {
            throw new Error(`Cannot set launch role when a StackSet rule is already defined for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
        const constructId = this.launchRoleConstraintLogicalId(association.associationKey);
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnLaunchRoleConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                roleArn: roleOptions.roleArn,
                localRoleName: roleOptions.localRoleName,
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot set multiple launch roles for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static stackSetConstraintLogicalId(associationKey) {
        return `StackSetConstraint${associationKey}`;
    }
    static launchRoleConstraintLogicalId(associationKey) {
        return `LaunchRoleConstraint${associationKey}`;
    }
    static prettyPrintAssociation(portfolio, product) {
        return `- Portfolio: ${portfolio.node.path} | Product: ${product.node.path}`;
    }
    static formatTemplateRule(stack, rule) {
        return JSON.stringify({
            [rule.ruleName]: {
                Assertions: this.formatAssertions(stack, rule.assertions),
                RuleCondition: rule.condition ? stack.resolve(rule.condition) : undefined,
            },
        });
    }
    static formatAssertions(stack, assertions) {
        return assertions.reduce((formattedAssertions, assertion) => {
            formattedAssertions.push({
                Assert: stack.resolve(assertion.assert),
                AssertDescription: assertion.description,
            });
            return formattedAssertions;
        }, new Array());
    }
    ;
}
exports.AssociationManager = AssociationManager;
//# sourceMappingURL=data:application/json;base64,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