"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.QueueBase = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const policy_1 = require("./policy");
/**
 * Reference to a new or existing Amazon SQS queue
 */
class QueueBase extends core_1.Resource {
    /**
     * Adds a statement to the IAM resource policy associated with this queue.
     *
     * If this queue was created in this stack (`new Queue`), a queue policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the queue is imported (`Queue.import`), then this is a no-op.
     */
    addToResourcePolicy(statement) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_iam_PolicyStatement(statement);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToResourcePolicy);
            }
            throw error;
        }
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.QueuePolicy(this, 'Policy', { queues: [this] });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    validate() {
        const errors = super.validate();
        errors.push(...this.policy?.document.validateForResourcePolicy() || []);
        return errors;
    }
    /**
     * Grant permissions to consume messages from a queue
     *
     * This will grant the following permissions:
     *
     *   - sqs:ChangeMessageVisibility
     *   - sqs:DeleteMessage
     *   - sqs:ReceiveMessage
     *   - sqs:GetQueueAttributes
     *   - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant consume rights to
     */
    grantConsumeMessages(grantee) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_iam_IGrantable(grantee);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantConsumeMessages);
            }
            throw error;
        }
        const ret = this.grant(grantee, 'sqs:ReceiveMessage', 'sqs:ChangeMessageVisibility', 'sqs:GetQueueUrl', 'sqs:DeleteMessage', 'sqs:GetQueueAttributes');
        if (this.encryptionMasterKey) {
            this.encryptionMasterKey.grantDecrypt(grantee);
        }
        return ret;
    }
    /**
     * Grant access to send messages to a queue to the given identity.
     *
     * This will grant the following permissions:
     *
     *  - sqs:SendMessage
     *  - sqs:GetQueueAttributes
     *  - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to
     */
    grantSendMessages(grantee) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_iam_IGrantable(grantee);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantSendMessages);
            }
            throw error;
        }
        const ret = this.grant(grantee, 'sqs:SendMessage', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
        if (this.encryptionMasterKey) {
            // kms:Decrypt necessary to execute grantsendMessages to an SSE enabled SQS queue
            this.encryptionMasterKey.grantEncryptDecrypt(grantee);
        }
        return ret;
    }
    /**
     * Grant an IAM principal permissions to purge all messages from the queue.
     *
     * This will grant the following permissions:
     *
     *  - sqs:PurgeQueue
     *  - sqs:GetQueueAttributes
     *  - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to
     */
    grantPurge(grantee) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_iam_IGrantable(grantee);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantPurge);
            }
            throw error;
        }
        return this.grant(grantee, 'sqs:PurgeQueue', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
    }
    /**
     * Grant the actions defined in queueActions to the identity Principal given
     * on this SQS queue resource.
     *
     * @param grantee Principal to grant right to
     * @param actions The actions to grant
     */
    grant(grantee, ...actions) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_iam_IGrantable(grantee);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grant);
            }
            throw error;
        }
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions,
            resourceArns: [this.queueArn],
            resource: this,
        });
    }
}
exports.QueueBase = QueueBase;
_a = JSII_RTTI_SYMBOL_1;
QueueBase[_a] = { fqn: "monocdk.aws_sqs.QueueBase", version: "1.191.0" };
//# sourceMappingURL=data:application/json;base64,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