"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualClusterInput = exports.ReleaseLabel = exports.Classification = exports.EmrContainersStartJobRun = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const iam = require("../../../aws-iam");
const lambda = require("../../../aws-lambda");
const logs = require("../../../aws-logs");
const s3 = require("../../../aws-s3");
const sfn = require("../../../aws-stepfunctions");
const cdk = require("../../../core");
const cr = require("../../../custom-resources");
const awscli = require("../../../lambda-layer-awscli");
const task_utils_1 = require("../private/task-utils");
/**
 * Starts a job run.
 *
 * A job is a unit of work that you submit to Amazon EMR on EKS for execution.
 * The work performed by the job can be defined by a Spark jar, PySpark script, or SparkSQL query.
 * A job run is an execution of the job on the virtual cluster.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr-eks.html
 */
class EmrContainersStartJobRun extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        /**
         * Render the EMR Containers ConfigurationProperty as JSON
         */
        this.applicationConfigPropertyToJson = (property) => {
            return {
                Classification: cdk.stringToCloudFormation(property.classification.classificationStatement),
                Properties: property.properties ? cdk.objectToCloudFormation(property.properties) : undefined,
                Configurations: property.nestedConfig ? cdk.listMapper(this.applicationConfigPropertyToJson)(property.nestedConfig) : undefined,
            };
        };
        this.assignLogGroup = () => {
            if (this.props.monitoring?.logGroup) {
                return (this.props.monitoring?.logGroup);
            }
            else {
                return (this.props.monitoring?.logging ? new logs.LogGroup(this, 'Monitoring Log Group') : undefined);
            }
        };
        this.assignLogBucket = () => {
            if (this.props.monitoring?.logBucket) {
                return (this.props.monitoring?.logBucket);
            }
            else {
                return (this.props.monitoring?.logging ? new s3.Bucket(this, 'Monitoring Bucket') : undefined);
            }
        };
        try {
            jsiiDeprecationWarnings.monocdk_aws_stepfunctions_tasks_EmrContainersStartJobRunProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, EmrContainersStartJobRun);
            }
            throw error;
        }
        this.integrationPattern = props.integrationPattern ?? sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, EmrContainersStartJobRun.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.props.applicationConfig) {
            this.validateAppConfig(this.props.applicationConfig);
        }
        if (this.props.jobDriver.sparkSubmitJobDriver) {
            this.validateSparkSubmitJobDriver(props.jobDriver.sparkSubmitJobDriver);
        }
        if (this.props.executionRole === undefined
            && sfn.JsonPath.isEncodedJsonPath(props.virtualCluster.id)) {
            throw new Error('Execution role cannot be undefined when the virtual cluster ID is not a concrete value. Provide an execution role with the correct trust policy');
        }
        this.logGroup = this.assignLogGroup();
        this.logBucket = this.assignLogBucket();
        this.role = this.props.executionRole ?? this.createJobExecutionRole();
        this.grantPrincipal = this.role;
        this.grantMonitoringPolicies();
        this.taskPolicies = this.createPolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('emr-containers', 'startJobRun', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                VirtualClusterId: this.props.virtualCluster.id,
                Name: this.props.jobName,
                ExecutionRoleArn: this.role.roleArn,
                ReleaseLabel: this.props.releaseLabel.label,
                JobDriver: {
                    SparkSubmitJobDriver: {
                        EntryPoint: this.props.jobDriver.sparkSubmitJobDriver?.entryPoint.value,
                        EntryPointArguments: this.props.jobDriver.sparkSubmitJobDriver?.entryPointArguments?.value,
                        SparkSubmitParameters: this.props.jobDriver.sparkSubmitJobDriver?.sparkSubmitParameters,
                    },
                },
                ConfigurationOverrides: {
                    ApplicationConfiguration: cdk.listMapper(this.applicationConfigPropertyToJson)(this.props.applicationConfig),
                    MonitoringConfiguration: {
                        CloudWatchMonitoringConfiguration: this.logGroup ? {
                            LogGroupName: this.logGroup.logGroupName,
                            LogStreamNamePrefix: this.props.monitoring.logStreamNamePrefix,
                        } : undefined,
                        PersistentAppUI: (this.props.monitoring?.persistentAppUI === false)
                            ? 'DISABLED'
                            : 'ENABLED',
                        S3MonitoringConfiguration: this.logBucket ? {
                            LogUri: this.logBucket.s3UrlForObject(),
                        } : undefined,
                    },
                },
                Tags: this.props.tags,
            }),
        };
    }
    validateAppConfigPropertiesLength(appConfig) {
        if (appConfig?.properties === undefined) {
            return;
        }
        else if (Object.keys(appConfig.properties).length > 100) {
            throw new Error(`Application configuration properties must have 100 or fewer entries. Received ${Object.keys(appConfig.properties).length}`);
        }
    }
    validatePropertiesNestedAppConfigBothNotUndefined(appConfig) {
        if (appConfig?.properties === undefined && appConfig?.nestedConfig === undefined) {
            throw new Error('Application configuration must have either properties or nested app configurations defined.');
        }
    }
    validateAppConfig(config) {
        if (config === undefined) {
            return;
        }
        else if (config.length > 100) {
            throw new Error(`Application configuration array must have 100 or fewer entries. Received ${config.length}`);
        }
        else {
            config.forEach(element => this.validateAppConfig(element.nestedConfig));
            config.forEach(element => this.validateAppConfigPropertiesLength(element));
            config.forEach(element => this.validatePropertiesNestedAppConfigBothNotUndefined(element));
        }
    }
    isArrayOfStrings(value) {
        return Array.isArray(value) && value.every(item => typeof item === 'string');
    }
    validateEntryPointArguments(entryPointArguments) {
        if (typeof entryPointArguments.value === 'string' && !sfn.JsonPath.isEncodedJsonPath(entryPointArguments.value)) {
            throw new Error(`Entry point arguments must be a string array or encoded JSON path, but received a non JSON path string');
      .`);
        }
        if (!this.isArrayOfStrings(entryPointArguments.value)) {
            throw new Error(`Entry point arguments must be a string array or encoded JSON path but received ${typeof entryPointArguments.value}.`);
        }
    }
    validateEntryPointArgumentsLength(entryPointArguments) {
        if (this.isArrayOfStrings(entryPointArguments.value)
            && (entryPointArguments.value.length > 10280 || entryPointArguments.value.length < 1)) {
            throw new Error(`Entry point arguments must be a string array between 1 and 10280 in length. Received ${entryPointArguments.value.length}.`);
        }
    }
    validateSparkSubmitParametersLength(sparkSubmitParameters) {
        if (sparkSubmitParameters.length > 102400 || sparkSubmitParameters.length < 1) {
            throw new Error(`Spark submit parameters must be between 1 and 102400 characters in length. Received ${sparkSubmitParameters.length}.`);
        }
    }
    validateEntryPoint(entryPoint) {
        if (!sfn.JsonPath.isEncodedJsonPath(entryPoint.value) && (entryPoint.value.length > 256 || entryPoint.value.length < 1)) {
            throw new Error(`Entry point must be between 1 and 256 characters in length. Received ${entryPoint.value.length}.`);
        }
    }
    validateSparkSubmitJobDriver(driver) {
        this.validateEntryPoint(driver.entryPoint);
        if (driver.entryPointArguments) {
            this.validateEntryPointArguments(driver.entryPointArguments);
            this.validateEntryPointArgumentsLength(driver.entryPointArguments);
        }
        if (driver.sparkSubmitParameters) {
            this.validateSparkSubmitParametersLength(driver.sparkSubmitParameters);
        }
    }
    // https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/creating-job-execution-role.html
    createJobExecutionRole() {
        const jobExecutionRole = new iam.Role(this, 'Job-Execution-Role', {
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('emr-containers.amazonaws.com'), new iam.ServicePrincipal('states.amazonaws.com')),
        });
        this.logBucket?.grantReadWrite(jobExecutionRole);
        this.logGroup?.grantWrite(jobExecutionRole);
        this.logGroup?.grant(jobExecutionRole, 'logs:DescribeLogStreams');
        jobExecutionRole.addToPrincipalPolicy(new iam.PolicyStatement({
            resources: [
                'arn:aws:logs:*:*:*',
            ],
            actions: [
                'logs:DescribeLogGroups',
            ],
        }));
        this.updateRoleTrustPolicy(jobExecutionRole);
        return jobExecutionRole;
    }
    grantMonitoringPolicies() {
        this.logBucket?.grantReadWrite(this.role);
        this.logGroup?.grantWrite(this.role);
        this.logGroup?.grant(this.role, 'logs:DescribeLogStreams');
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            resources: [
                'arn:aws:logs:*:*:*',
            ],
            actions: [
                'logs:DescribeLogGroups',
            ],
        }));
    }
    /**
     * If an execution role is not provided by user, the automatically generated job execution role must create a trust relationship
     * between itself and the identity of the EMR managed service account in order to run jobs on the Kubernetes namespace.
     *
     * This cannot occur if the user provided virtualClusterId is within an encoded JSON path.
     *
     * The trust relationship can be created by updating the trust policy of the job execution role.
     *
     * @param role the automatically generated job execution role
     */
    updateRoleTrustPolicy(role) {
        const eksClusterInfo = new cr.AwsCustomResource(this, 'GetEksClusterInfo', {
            onCreate: {
                service: 'EMRcontainers',
                action: 'describeVirtualCluster',
                parameters: {
                    id: this.props.virtualCluster.id,
                },
                outputPaths: ['virtualCluster.containerProvider.info.eksInfo.namespace', 'virtualCluster.containerProvider.id'],
                physicalResourceId: cr.PhysicalResourceId.of('id'),
            },
            policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                resources: cr.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
        /* We make use of custom resources to call update-roll-trust-policy as this command is only available through
         * aws cli because this is only used during the initial setup and is not available through the sdk.
         * https://awscli.amazonaws.com/v2/documentation/api/latest/reference/emr-containers/update-role-trust-policy.html
         * Commands available through SDK: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/EMRcontainers.html
         * Commands available through CLI: https://awscli.amazonaws.com/v2/documentation/api/latest/reference/emr-containers/index.html
        */
        const cliLayer = new awscli.AwsCliLayer(this, 'awsclilayer');
        const shellCliLambda = new lambda.SingletonFunction(this, 'Call Update-Role-Trust-Policy', {
            uuid: '8693BB64-9689-44B6-9AAF-B0CC9EB8757C',
            runtime: lambda.Runtime.PYTHON_3_9,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, 'utils/role-policy')),
            timeout: cdk.Duration.seconds(30),
            memorySize: 256,
            layers: [cliLayer],
        });
        shellCliLambda.addToRolePolicy(new iam.PolicyStatement({
            resources: [
                cdk.Stack.of(this).formatArn({
                    service: 'eks',
                    resource: 'cluster',
                    resourceName: eksClusterInfo.getResponseField('virtualCluster.containerProvider.id'),
                }),
            ],
            actions: [
                'eks:DescribeCluster',
            ],
        }));
        shellCliLambda.addToRolePolicy(new iam.PolicyStatement({
            resources: [role.roleArn],
            actions: [
                'iam:GetRole',
                'iam:UpdateAssumeRolePolicy',
            ],
        }));
        const provider = new cr.Provider(this, 'CustomResourceProvider', {
            onEventHandler: shellCliLambda,
        });
        new cdk.CustomResource(this, 'Custom Resource', {
            properties: {
                eksNamespace: eksClusterInfo.getResponseField('virtualCluster.containerProvider.info.eksInfo.namespace'),
                eksClusterId: eksClusterInfo.getResponseField('virtualCluster.containerProvider.id'),
                roleName: role.roleName,
            },
            serviceToken: provider.serviceToken,
        });
    }
    createPolicyStatements() {
        const policyStatements = [
            new iam.PolicyStatement({
                resources: [
                    cdk.Stack.of(this).formatArn({
                        arnFormat: cdk.ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME,
                        service: 'emr-containers',
                        resource: 'virtualclusters',
                        resourceName: sfn.JsonPath.isEncodedJsonPath(this.props.virtualCluster.id) ? '*' : this.props.virtualCluster.id,
                    }),
                ],
                actions: ['emr-containers:StartJobRun'],
                conditions: {
                    StringEquals: {
                        'emr-containers:ExecutionRoleArn': this.role.roleArn,
                    },
                },
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                resources: [
                    cdk.Stack.of(this).formatArn({
                        arnFormat: cdk.ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME,
                        service: 'emr-containers',
                        resource: 'virtualclusters',
                        resourceName: sfn.JsonPath.isEncodedJsonPath(this.props.virtualCluster.id) ? '*' : `${this.props.virtualCluster.id}/jobruns/*`,
                    }),
                ],
                actions: [
                    'emr-containers:DescribeJobRun',
                    'emr-containers:CancelJobRun',
                ],
            }));
        }
        return policyStatements;
    }
}
exports.EmrContainersStartJobRun = EmrContainersStartJobRun;
_a = JSII_RTTI_SYMBOL_1;
EmrContainersStartJobRun[_a] = { fqn: "monocdk.aws_stepfunctions_tasks.EmrContainersStartJobRun", version: "1.191.0" };
EmrContainersStartJobRun.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
/**
 * The classification within a EMR Containers application configuration.
 * Class can be extended to add other classifications.
 * For example, new Classification('xxx-yyy');
 */
class Classification {
    /**
     * Creates a new Classification
     *
     * @param classificationStatement A literal string in case a new EMR classification is released, if not already defined.
     */
    constructor(classificationStatement) {
        this.classificationStatement = classificationStatement;
    }
}
exports.Classification = Classification;
_b = JSII_RTTI_SYMBOL_1;
Classification[_b] = { fqn: "monocdk.aws_stepfunctions_tasks.Classification", version: "1.191.0" };
/**
 * Sets the maximizeResourceAllocation property to true or false.
 * When true, Amazon EMR automatically configures spark-defaults properties based on cluster hardware configuration.
 *
 * For more info:
 * @see https://docs.aws.amazon.com/emr/latest/ReleaseGuide/emr-spark-configure.html#emr-spark-maximizeresourceallocation
 */
Classification.SPARK = new Classification('spark');
/**
 * Sets values in the spark-defaults.conf file.
 *
 * For more info:
 * @see https://spark.apache.org/docs/latest/configuration.html
 */
Classification.SPARK_DEFAULTS = new Classification('spark-defaults');
/**
 * Sets values in the spark-env.sh file.
 *
 * For more info:
 * @see https://spark.apache.org/docs/latest/configuration.html#environment-variables
 */
Classification.SPARK_ENV = new Classification('spark-env');
/**
 * Sets values in the hive-site.xml for Spark.
 */
Classification.SPARK_HIVE_SITE = new Classification('spark-hive-site');
/**
 * Sets values in the log4j.properties file.
 *
 * For more settings and info:
 * @see https://github.com/apache/spark/blob/master/conf/log4j.properties.template
 */
Classification.SPARK_LOG4J = new Classification('spark-log4j');
/**
 * Sets values in the metrics.properties file.
 *
 * For more settings and info:
 * @see https://github.com/apache/spark/blob/master/conf/metrics.properties.template
 */
Classification.SPARK_METRICS = new Classification('spark-metrics');
/**
 * The Amazon EMR release version to use for the job run.
 *
 * Can be extended to include new EMR releases
 *
 * For example, `new ReleaseLabel('emr-x.xx.x-latest');`
 */
class ReleaseLabel {
    /**
     * Initializes the label string.
     *
     * @param label A literal string that contains the release-version ex. 'emr-x.x.x-latest'
     */
    constructor(label) {
        this.label = label;
    }
}
exports.ReleaseLabel = ReleaseLabel;
_c = JSII_RTTI_SYMBOL_1;
ReleaseLabel[_c] = { fqn: "monocdk.aws_stepfunctions_tasks.ReleaseLabel", version: "1.191.0" };
/**
 * EMR Release version 5.32.0
 */
ReleaseLabel.EMR_5_32_0 = new ReleaseLabel('emr-5.32.0-latest');
/**
 * EMR Release version 5.33.0
 */
ReleaseLabel.EMR_5_33_0 = new ReleaseLabel('emr-5.33.0-latest');
/**
 * EMR Release version 6.2.0
 */
ReleaseLabel.EMR_6_2_0 = new ReleaseLabel('emr-6.2.0-latest');
/**
 * EMR Release version 6.3.0
 */
ReleaseLabel.EMR_6_3_0 = new ReleaseLabel('emr-6.3.0-latest');
/**
 * Class that returns a virtual cluster's id depending on input type
 */
class VirtualClusterInput {
    /**
     * Initializes the virtual cluster ID.
     *
     * @param id The VirtualCluster Id
     */
    constructor(id) {
        this.id = id;
    }
    /**
     * Input for a virtualClusterId from a Task Input
     */
    static fromTaskInput(taskInput) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_stepfunctions_TaskInput(taskInput);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromTaskInput);
            }
            throw error;
        }
        return new VirtualClusterInput(taskInput.value);
    }
    /**
     * Input for virtualClusterId from a literal string
     */
    static fromVirtualClusterId(virtualClusterId) {
        return new VirtualClusterInput(virtualClusterId);
    }
}
exports.VirtualClusterInput = VirtualClusterInput;
_d = JSII_RTTI_SYMBOL_1;
VirtualClusterInput[_d] = { fqn: "monocdk.aws_stepfunctions_tasks.VirtualClusterInput", version: "1.191.0" };
//# sourceMappingURL=data:application/json;base64,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