"use strict";
var _a, _b, _c, _d, _e;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssembleWith = exports.SplitType = exports.BatchStrategy = exports.AcceleratorType = exports.AcceleratorClass = exports.Mode = exports.ContainerDefinition = exports.CompressionType = exports.InputMode = exports.RecordWrapperType = exports.S3DataDistributionType = exports.S3DataType = exports.DockerImage = exports.S3Location = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ecr_assets_1 = require("../../../aws-ecr-assets");
const iam = require("../../../aws-iam");
const sfn = require("../../../aws-stepfunctions");
/**
 * Constructs `IS3Location` objects.
 *
 */
class S3Location {
    /**
     * An `IS3Location` built with a determined bucket and key prefix.
     *
     * @param bucket    is the bucket where the objects are to be stored.
     * @param keyPrefix is the key prefix used by the location.
     */
    static fromBucket(bucket, keyPrefix) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_s3_IBucket(bucket);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromBucket);
            }
            throw error;
        }
        return new StandardS3Location({ bucket, keyPrefix, uri: bucket.urlForObject(keyPrefix) });
    }
    /**
     * An `IS3Location` determined fully by a JSON Path from the task input.
     *
     * Due to the dynamic nature of those locations, the IAM grants that will be set by `grantRead` and `grantWrite`
     * apply to the `*` resource.
     *
     * @param expression the JSON expression resolving to an S3 location URI.
     */
    static fromJsonExpression(expression) {
        return new StandardS3Location({ uri: sfn.JsonPath.stringAt(expression) });
    }
}
exports.S3Location = S3Location;
_a = JSII_RTTI_SYMBOL_1;
S3Location[_a] = { fqn: "monocdk.aws_stepfunctions_tasks.S3Location", version: "1.191.0" };
/**
 * Creates `IDockerImage` instances.
 *
 */
class DockerImage {
    /**
     * Reference a Docker image stored in an ECR repository.
     *
     * @param repository the ECR repository where the image is hosted.
     * @param tagOrDigest an optional tag or digest (digests must start with `sha256:`)
     */
    static fromEcrRepository(repository, tagOrDigest = 'latest') {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecr_IRepository(repository);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromEcrRepository);
            }
            throw error;
        }
        return new StandardDockerImage({ repository, imageUri: repository.repositoryUriForTagOrDigest(tagOrDigest) });
    }
    /**
     * Reference a Docker image which URI is obtained from the task's input.
     *
     * @param expression           the JSON path expression with the task input.
     * @param allowAnyEcrImagePull whether ECR access should be permitted (set to `false` if the image will never be in ECR).
     */
    static fromJsonExpression(expression, allowAnyEcrImagePull = true) {
        return new StandardDockerImage({ imageUri: expression, allowAnyEcrImagePull });
    }
    /**
     * Reference a Docker image by it's URI.
     *
     * When referencing ECR images, prefer using `inEcr`.
     *
     * @param imageUri the URI to the docker image.
     */
    static fromRegistry(imageUri) {
        return new StandardDockerImage({ imageUri });
    }
    /**
     * Reference a Docker image that is provided as an Asset in the current app.
     *
     * @param scope the scope in which to create the Asset.
     * @param id    the ID for the asset in the construct tree.
     * @param props the configuration props of the asset.
     */
    static fromAsset(scope, id, props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecr_assets_DockerImageAssetProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromAsset);
            }
            throw error;
        }
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new StandardDockerImage({ repository: asset.repository, imageUri: asset.imageUri });
    }
}
exports.DockerImage = DockerImage;
_b = JSII_RTTI_SYMBOL_1;
DockerImage[_b] = { fqn: "monocdk.aws_stepfunctions_tasks.DockerImage", version: "1.191.0" };
/**
 * S3 Data Type.
 *
 */
var S3DataType;
(function (S3DataType) {
    /**
     * Manifest File Data Type
     */
    S3DataType["MANIFEST_FILE"] = "ManifestFile";
    /**
     * S3 Prefix Data Type
     */
    S3DataType["S3_PREFIX"] = "S3Prefix";
    /**
     * Augmented Manifest File Data Type
     */
    S3DataType["AUGMENTED_MANIFEST_FILE"] = "AugmentedManifestFile";
})(S3DataType = exports.S3DataType || (exports.S3DataType = {}));
/**
 * S3 Data Distribution Type.
 *
 */
var S3DataDistributionType;
(function (S3DataDistributionType) {
    /**
     * Fully replicated S3 Data Distribution Type
     */
    S3DataDistributionType["FULLY_REPLICATED"] = "FullyReplicated";
    /**
     * Sharded By S3 Key Data Distribution Type
     */
    S3DataDistributionType["SHARDED_BY_S3_KEY"] = "ShardedByS3Key";
})(S3DataDistributionType = exports.S3DataDistributionType || (exports.S3DataDistributionType = {}));
/**
 * Define the format of the input data.
 *
 */
var RecordWrapperType;
(function (RecordWrapperType) {
    /**
     * None record wrapper type
     */
    RecordWrapperType["NONE"] = "None";
    /**
     * RecordIO record wrapper type
     */
    RecordWrapperType["RECORD_IO"] = "RecordIO";
})(RecordWrapperType = exports.RecordWrapperType || (exports.RecordWrapperType = {}));
/**
 *  Input mode that the algorithm supports.
 *
 */
var InputMode;
(function (InputMode) {
    /**
     * Pipe mode
     */
    InputMode["PIPE"] = "Pipe";
    /**
     * File mode.
     */
    InputMode["FILE"] = "File";
})(InputMode = exports.InputMode || (exports.InputMode = {}));
/**
 * Compression type of the data.
 *
 */
var CompressionType;
(function (CompressionType) {
    /**
     * None compression type
     */
    CompressionType["NONE"] = "None";
    /**
     * Gzip compression type
     */
    CompressionType["GZIP"] = "Gzip";
})(CompressionType = exports.CompressionType || (exports.CompressionType = {}));
/**
 * Describes the container, as part of model definition.
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ContainerDefinition.html
 */
class ContainerDefinition {
    constructor(options) {
        this.options = options;
        try {
            jsiiDeprecationWarnings.monocdk_aws_stepfunctions_tasks_ContainerDefinitionOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ContainerDefinition);
            }
            throw error;
        }
    }
    /**
     * Called when the ContainerDefinition type configured on Sagemaker Task
     */
    bind(task) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_stepfunctions_tasks_ISageMakerTask(task);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        return {
            parameters: {
                ContainerHostname: this.options.containerHostName,
                Image: this.options.image?.bind(task).imageUri,
                Mode: this.options.mode,
                ModelDataUrl: this.options.modelS3Location?.bind(task, { forReading: true }).uri,
                ModelPackageName: this.options.modelPackageName,
                Environment: this.options.environmentVariables?.value,
            },
        };
    }
}
exports.ContainerDefinition = ContainerDefinition;
_c = JSII_RTTI_SYMBOL_1;
ContainerDefinition[_c] = { fqn: "monocdk.aws_stepfunctions_tasks.ContainerDefinition", version: "1.191.0" };
/**
 * Specifies how many models the container hosts
 *
 */
var Mode;
(function (Mode) {
    /**
     * Container hosts a single model
     */
    Mode["SINGLE_MODEL"] = "SingleModel";
    /**
     * Container hosts multiple models
     *
     * @see https://docs.aws.amazon.com/sagemaker/latest/dg/multi-model-endpoints.html
     */
    Mode["MULTI_MODEL"] = "MultiModel";
})(Mode = exports.Mode || (exports.Mode = {}));
/**
 * The generation of Elastic Inference (EI) instance
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
 */
class AcceleratorClass {
    /**
     * @param version - Elastic Inference accelerator generation
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * Custom AcceleratorType
     * @param version - Elastic Inference accelerator generation
    */
    static of(version) { return new AcceleratorClass(version); }
}
exports.AcceleratorClass = AcceleratorClass;
_d = JSII_RTTI_SYMBOL_1;
AcceleratorClass[_d] = { fqn: "monocdk.aws_stepfunctions_tasks.AcceleratorClass", version: "1.191.0" };
/**
 * Elastic Inference accelerator 1st generation
 */
AcceleratorClass.EIA1 = AcceleratorClass.of('eia1');
/**
 * Elastic Inference accelerator 2nd generation
 */
AcceleratorClass.EIA2 = AcceleratorClass.of('eia2');
/**
 * The size of the Elastic Inference (EI) instance to use for the production variant.
 * EI instances provide on-demand GPU computing for inference
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
 */
class AcceleratorType {
    constructor(instanceTypeIdentifier) {
        this.instanceTypeIdentifier = instanceTypeIdentifier;
    }
    /**
     * AcceleratorType
     *
     * This class takes a combination of a class and size.
     */
    static of(acceleratorClass, instanceSize) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_stepfunctions_tasks_AcceleratorClass(acceleratorClass);
            jsiiDeprecationWarnings.monocdk_aws_ec2_InstanceSize(instanceSize);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.of);
            }
            throw error;
        }
        return new AcceleratorType(`ml.${acceleratorClass}.${instanceSize}`);
    }
    /**
     * Return the accelerator type as a dotted string
     */
    toString() {
        return this.instanceTypeIdentifier;
    }
}
exports.AcceleratorType = AcceleratorType;
_e = JSII_RTTI_SYMBOL_1;
AcceleratorType[_e] = { fqn: "monocdk.aws_stepfunctions_tasks.AcceleratorType", version: "1.191.0" };
/**
 * Specifies the number of records to include in a mini-batch for an HTTP inference request.
 *
 */
var BatchStrategy;
(function (BatchStrategy) {
    /**
     * Fits multiple records in a mini-batch.
     */
    BatchStrategy["MULTI_RECORD"] = "MultiRecord";
    /**
     * Use a single record when making an invocation request.
     */
    BatchStrategy["SINGLE_RECORD"] = "SingleRecord";
})(BatchStrategy = exports.BatchStrategy || (exports.BatchStrategy = {}));
/**
 * Method to use to split the transform job's data files into smaller batches.
 *
 */
var SplitType;
(function (SplitType) {
    /**
     * Input data files are not split,
     */
    SplitType["NONE"] = "None";
    /**
     * Split records on a newline character boundary.
     */
    SplitType["LINE"] = "Line";
    /**
     * Split using MXNet RecordIO format.
     */
    SplitType["RECORD_IO"] = "RecordIO";
    /**
     * Split using TensorFlow TFRecord format.
     */
    SplitType["TF_RECORD"] = "TFRecord";
})(SplitType = exports.SplitType || (exports.SplitType = {}));
/**
 * How to assemble the results of the transform job as a single S3 object.
 *
 */
var AssembleWith;
(function (AssembleWith) {
    /**
     * Concatenate the results in binary format.
     */
    AssembleWith["NONE"] = "None";
    /**
     * Add a newline character at the end of every transformed record.
     */
    AssembleWith["LINE"] = "Line";
})(AssembleWith = exports.AssembleWith || (exports.AssembleWith = {}));
class StandardDockerImage extends DockerImage {
    constructor(opts) {
        super();
        this.allowAnyEcrImagePull = !!opts.allowAnyEcrImagePull;
        this.imageUri = opts.imageUri;
        this.repository = opts.repository;
    }
    bind(task) {
        if (this.repository) {
            this.repository.grantPull(task);
        }
        if (this.allowAnyEcrImagePull) {
            task.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: [
                    'ecr:BatchCheckLayerAvailability',
                    'ecr:GetDownloadUrlForLayer',
                    'ecr:BatchGetImage',
                ],
                resources: ['*'],
            }));
        }
        return {
            imageUri: this.imageUri,
        };
    }
}
class StandardS3Location extends S3Location {
    constructor(opts) {
        super();
        this.bucket = opts.bucket;
        this.keyGlob = `${opts.keyPrefix || ''}*`;
        this.uri = opts.uri;
    }
    bind(task, opts) {
        if (this.bucket) {
            if (opts.forReading) {
                this.bucket.grantRead(task, this.keyGlob);
            }
            if (opts.forWriting) {
                this.bucket.grantWrite(task, this.keyGlob);
            }
        }
        else {
            const actions = new Array();
            if (opts.forReading) {
                actions.push('s3:GetObject', 's3:ListBucket');
            }
            if (opts.forWriting) {
                actions.push('s3:PutObject');
            }
            task.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({ actions, resources: ['*'] }));
        }
        return { uri: this.uri };
    }
}
//# sourceMappingURL=data:application/json;base64,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