import { IResolvable } from '../../core';
/**
 * Extract a field from the State Machine data or context
 * that gets passed around between states
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-paths.html
 */
export declare class JsonPath {
    /**
     * Special string value to discard state input, output or result
     */
    static readonly DISCARD = "DISCARD";
    /**
     * Instead of using a literal string, get the value from a JSON path
     */
    static stringAt(path: string): string;
    /**
     * Instead of using a literal string list, get the value from a JSON path
     */
    static listAt(path: string): string[];
    /**
     * Instead of using a literal number, get the value from a JSON path
     */
    static numberAt(path: string): number;
    /**
     * Reference a complete (complex) object in a JSON path location
     */
    static objectAt(path: string): IResolvable;
    /**
     * Use the entire data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entirePayload(): string;
    /**
     * Determines if the indicated string is an encoded JSON path
     *
     * @param value string to be evaluated
     */
    static isEncodedJsonPath(value: string): boolean;
    /**
     * Return the Task Token field
     *
     * External actions will need this token to report step completion
     * back to StepFunctions using the `SendTaskSuccess` or `SendTaskFailure`
     * calls.
     */
    static get taskToken(): string;
    /**
     * Use the entire context data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entireContext(): string;
    /**
     * Make an intrinsic States.Array expression
     *
     * Combine any number of string literals or JsonPath expressions into an array.
     *
     * Use this function if the value of an array element directly has to come
     * from a JSON Path expression (either the State object or the Context object).
     *
     * If the array contains object literals whose values come from a JSON path
     * expression, you do not need to use this function.
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-intrinsic-functions.html
     */
    static array(...values: string[]): string;
    /**
     * Make an intrinsic States.Format expression
     *
     * This can be used to embed JSON Path variables inside a format string.
     *
     * For example:
     *
     * ```ts
     * sfn.JsonPath.format('Hello, my name is {}.', sfn.JsonPath.stringAt('$.name'))
     * ```
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-intrinsic-functions.html
     */
    static format(formatString: string, ...values: string[]): string;
    /**
     * Make an intrinsic States.StringToJson expression
     *
     * During the execution of the Step Functions state machine, parse the given
     * argument as JSON into its object form.
     *
     * For example:
     *
     * ```ts
     * sfn.JsonPath.stringToJson(sfn.JsonPath.stringAt('$.someJsonBody'))
     * ```
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-intrinsic-functions.html
     */
    static stringToJson(jsonString: string): IResolvable;
    /**
     * Make an intrinsic States.JsonToString expression
     *
     * During the execution of the Step Functions state machine, encode the
     * given object into a JSON string.
     *
     * For example:
     *
     * ```ts
     * sfn.JsonPath.jsonToString(sfn.JsonPath.objectAt('$.someObject'))
     * ```
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-intrinsic-functions.html
     */
    static jsonToString(value: any): string;
    private constructor();
}
/**
 * Extract a field from the State Machine data that gets passed around between states
 *
 * @deprecated replaced by `JsonPath`
 */
export declare class Data {
    /**
     * Instead of using a literal string, get the value from a JSON path
     */
    static stringAt(path: string): string;
    /**
     * Instead of using a literal string list, get the value from a JSON path
     */
    static listAt(path: string): string[];
    /**
     * Instead of using a literal number, get the value from a JSON path
     */
    static numberAt(path: string): number;
    /**
     * Use the entire data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entirePayload(): string;
    /**
     * Determines if the indicated string is an encoded JSON path
     *
     * @param value string to be evaluated
     */
    static isJsonPathString(value: string): boolean;
    private constructor();
}
/**
 * Extract a field from the State Machine Context data
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#wait-token-contextobject
 *
 * @deprecated replaced by `JsonPath`
 */
export declare class Context {
    /**
     * Instead of using a literal string, get the value from a JSON path
     */
    static stringAt(path: string): string;
    /**
     * Instead of using a literal number, get the value from a JSON path
     */
    static numberAt(path: string): number;
    /**
     * Return the Task Token field
     *
     * External actions will need this token to report step completion
     * back to StepFunctions using the `SendTaskSuccess` or `SendTaskFailure`
     * calls.
     */
    static get taskToken(): string;
    /**
     * Use the entire context data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entireContext(): string;
    private constructor();
}
/**
 * Helper functions to work with structures containing fields
 */
export declare class FieldUtils {
    /**
     * Render a JSON structure containing fields to the right StepFunctions structure
     */
    static renderObject(obj?: {
        [key: string]: any;
    }): {
        [key: string]: any;
    } | undefined;
    /**
     * Return all JSON paths used in the given structure
     */
    static findReferencedPaths(obj?: {
        [key: string]: any;
    }): string[];
    /**
     * Returns whether the given task structure contains the TaskToken field anywhere
     *
     * The field is considered included if the field itself or one of its containing
     * fields occurs anywhere in the payload.
     */
    static containsTaskToken(obj?: {
        [key: string]: any;
    }): boolean;
    private constructor();
}
