"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WaiterStateMachine = void 0;
const aws_iam_1 = require("../../../aws-iam");
const core_1 = require("../../../core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("../../../core");
/**
 * A very simple StateMachine construct highly customized to the provider framework.
 * This is so that this package does not need to depend on aws-stepfunctions module.
 *
 * The state machine continuously calls the isCompleteHandler, until it succeeds or times out.
 * The handler is called `maxAttempts` times with an `interval` duration and a `backoffRate` rate.
 */
class WaiterStateMachine extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const role = new aws_iam_1.Role(this, 'Role', {
            assumedBy: new aws_iam_1.ServicePrincipal('states.amazonaws.com'),
        });
        props.isCompleteHandler.grantInvoke(role);
        props.timeoutHandler.grantInvoke(role);
        const definition = core_1.Stack.of(this).toJsonString({
            StartAt: 'framework-isComplete-task',
            States: {
                'framework-isComplete-task': {
                    End: true,
                    Retry: [{
                            ErrorEquals: ['States.ALL'],
                            IntervalSeconds: props.interval.toSeconds(),
                            MaxAttempts: props.maxAttempts,
                            BackoffRate: props.backoffRate,
                        }],
                    Catch: [{
                            ErrorEquals: ['States.ALL'],
                            Next: 'framework-onTimeout-task',
                        }],
                    Type: 'Task',
                    Resource: props.isCompleteHandler.functionArn,
                },
                'framework-onTimeout-task': {
                    End: true,
                    Type: 'Task',
                    Resource: props.timeoutHandler.functionArn,
                },
            },
        });
        const resource = new core_1.CfnResource(this, 'Resource', {
            type: 'AWS::StepFunctions::StateMachine',
            properties: {
                DefinitionString: definition,
                RoleArn: role.roleArn,
            },
        });
        resource.node.addDependency(role);
        this.stateMachineArn = resource.ref;
    }
    grantStartExecution(identity) {
        return aws_iam_1.Grant.addToPrincipal({
            grantee: identity,
            actions: ['states:StartExecution'],
            resourceArns: [this.stateMachineArn],
        });
    }
}
exports.WaiterStateMachine = WaiterStateMachine;
//# sourceMappingURL=data:application/json;base64,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