"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StackOutputReference = exports.ShellStep = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../../core");
const javascript_1 = require("../private/javascript");
const file_set_1 = require("./file-set");
const step_1 = require("./step");
/**
 * Run shell script commands in the pipeline. This is a generic step designed
 * to be deployment engine agnostic.
 */
class ShellStep extends step_1.Step {
    constructor(id, props) {
        super(id);
        /**
         * Input FileSets
         *
         * A list of `(FileSet, directory)` pairs, which are a copy of the
         * input properties. This list should not be modified directly.
         */
        this.inputs = [];
        /**
         * Output FileSets
         *
         * A list of `(FileSet, directory)` pairs, which are a copy of the
         * input properties. This list should not be modified directly.
         */
        this.outputs = [];
        this._additionalOutputs = {};
        try {
            jsiiDeprecationWarnings.monocdk_pipelines_ShellStepProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ShellStep);
            }
            throw error;
        }
        this.commands = props.commands;
        this.installCommands = props.installCommands ?? [];
        this.env = props.env ?? {};
        this.envFromCfnOutputs = javascript_1.mapValues(props.envFromCfnOutputs ?? {}, StackOutputReference.fromCfnOutput);
        // 'env' is the only thing that can contain outputs
        this.discoverReferencedOutputs({
            env: this.env,
        });
        // Inputs
        if (props.input) {
            const fileSet = props.input.primaryOutput;
            if (!fileSet) {
                throw new Error(`'${id}': primary input should be a step that has produced a file set, got ${props.input}`);
            }
            this.addDependencyFileSet(fileSet);
            this.inputs.push({ directory: '.', fileSet });
        }
        for (const [directory, step] of Object.entries(props.additionalInputs ?? {})) {
            if (directory === '.') {
                throw new Error(`'${id}': input for directory '.' should be passed via 'input' property`);
            }
            const fileSet = step.primaryOutput;
            if (!fileSet) {
                throw new Error(`'${id}': additionalInput for directory '${directory}' should be a step that has produced a file set, got ${step}`);
            }
            this.addDependencyFileSet(fileSet);
            this.inputs.push({ directory, fileSet });
        }
        // Outputs
        if (props.primaryOutputDirectory) {
            this._primaryOutputDirectory = props.primaryOutputDirectory;
            const fileSet = new file_set_1.FileSet('Output', this);
            this.configurePrimaryOutput(fileSet);
            this.outputs.push({ directory: props.primaryOutputDirectory, fileSet });
        }
    }
    /**
     * Configure the given output directory as primary output
     *
     * If no primary output has been configured yet, this directory
     * will become the primary output of this ShellStep, otherwise this
     * method will throw if the given directory is different than the
     * currently configured primary output directory.
     */
    primaryOutputDirectory(directory) {
        if (this._primaryOutputDirectory !== undefined) {
            if (this._primaryOutputDirectory !== directory) {
                throw new Error(`${this}: primaryOutputDirectory is '${this._primaryOutputDirectory}', cannot be changed to '${directory}'`);
            }
            return this.primaryOutput;
        }
        this._primaryOutputDirectory = directory;
        const fileSet = new file_set_1.FileSet('Output', this);
        this.configurePrimaryOutput(fileSet);
        this.outputs.push({ directory: directory, fileSet });
        return fileSet;
    }
    /**
     * Add an additional output FileSet based on a directory.
     *
     *
     * After running the script, the contents of the given directory
     * will be exported as a `FileSet`. Use the `FileSet` as the
     * input to another step.
     *
     * Multiple calls with the exact same directory name string (not normalized)
     * will return the same FileSet.
     */
    addOutputDirectory(directory) {
        let fileSet = this._additionalOutputs[directory];
        if (!fileSet) {
            fileSet = new file_set_1.FileSet(directory, this);
            this._additionalOutputs[directory] = fileSet;
            this.outputs.push({ directory, fileSet });
        }
        return fileSet;
    }
}
exports.ShellStep = ShellStep;
_a = JSII_RTTI_SYMBOL_1;
ShellStep[_a] = { fqn: "monocdk.pipelines.ShellStep", version: "1.191.0" };
/**
 * A Reference to a Stack Output
 */
class StackOutputReference {
    constructor(
    /** A human-readable description of the producing stack */
    stackDescription, 
    /** Artifact id of the producing stack */
    stackArtifactId, 
    /** Output name of the producing stack */
    outputName) {
        this.stackDescription = stackDescription;
        this.stackArtifactId = stackArtifactId;
        this.outputName = outputName;
    }
    /**
     * Create a StackOutputReference that references the given CfnOutput
     */
    static fromCfnOutput(output) {
        try {
            jsiiDeprecationWarnings.monocdk_CfnOutput(output);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromCfnOutput);
            }
            throw error;
        }
        const stack = core_1.Stack.of(output);
        return new StackOutputReference(stack.node.path, stack.artifactId, stack.resolve(output.logicalId));
    }
    /**
     * Whether or not this stack output is being produced by the given Stack deployment
     */
    isProducedBy(stack) {
        try {
            jsiiDeprecationWarnings.monocdk_pipelines_StackDeployment(stack);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.isProducedBy);
            }
            throw error;
        }
        return stack.stackArtifactId === this.stackArtifactId;
    }
}
exports.StackOutputReference = StackOutputReference;
_b = JSII_RTTI_SYMBOL_1;
StackOutputReference[_b] = { fqn: "monocdk.pipelines.StackOutputReference", version: "1.191.0" };
//# sourceMappingURL=data:application/json;base64,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