import * as cb from '../../../aws-codebuild';
import * as cp from '../../../aws-codepipeline';
import * as ec2 from '../../../aws-ec2';
import * as iam from '../../../aws-iam';
import { Duration } from '../../../core';
import { Construct } from 'constructs';
import { IFileSetProducer } from '../blueprint';
import { DockerCredential } from '../docker-credentials';
import { PipelineBase } from '../main';
/**
 * Properties for a `CodePipeline`
 */
export interface CodePipelineProps {
    /**
     * The build step that produces the CDK Cloud Assembly
     *
     * The primary output of this step needs to be the `cdk.out` directory
     * generated by the `cdk synth` command.
     *
     * If you use a `ShellStep` here and you don't configure an output directory,
     * the output directory will automatically be assumed to be `cdk.out`.
     */
    readonly synth: IFileSetProducer;
    /**
     * The name of the CodePipeline pipeline
     *
     * @default - Automatically generated
     */
    readonly pipelineName?: string;
    /**
     * Create KMS keys for the artifact buckets, allowing cross-account deployments
     *
     * The artifact buckets have to be encrypted to support deploying CDK apps to
     * another account, so if you want to do that or want to have your artifact
     * buckets encrypted, be sure to set this value to `true`.
     *
     * Be aware there is a cost associated with maintaining the KMS keys.
     *
     * @default false
     */
    readonly crossAccountKeys?: boolean;
    /**
     * CDK CLI version to use in self-mutation and asset publishing steps
     *
     * If you want to lock the CDK CLI version used in the pipeline, by steps
     * that are automatically generated for you, specify the version here.
     *
     * We recommend you do not specify this value, as not specifying it always
     * uses the latest CLI version which is backwards compatible with old versions.
     *
     * If you do specify it, be aware that this version should always be equal to or higher than the
     * version of the CDK framework used by the CDK app, when the CDK commands are
     * run during your pipeline execution. When you change this version, the *next
     * time* the `SelfMutate` step runs it will still be using the CLI of the the
     * *previous* version that was in this property: it will only start using the
     * new version after `SelfMutate` completes successfully. That means that if
     * you want to update both framework and CLI version, you should update the
     * CLI version first, commit, push and deploy, and only then update the
     * framework version.
     *
     * @default - Latest version
     */
    readonly cliVersion?: string;
    /**
     * Whether the pipeline will update itself
     *
     * This needs to be set to `true` to allow the pipeline to reconfigure
     * itself when assets or stages are being added to it, and `true` is the
     * recommended setting.
     *
     * You can temporarily set this to `false` while you are iterating
     * on the pipeline itself and prefer to deploy changes using `cdk deploy`.
     *
     * @default true
     */
    readonly selfMutation?: boolean;
    /**
     * Enable Docker for the self-mutate step
     *
     * Set this to true if the pipeline itself uses Docker container assets
     * (for example, if you use `LinuxBuildImage.fromAsset()` as the build
     * image of a CodeBuild step in the pipeline).
     *
     * You do not need to set it if you build Docker image assets in the
     * application Stages and Stacks that are *deployed* by this pipeline.
     *
     * Configures privileged mode for the self-mutation CodeBuild action.
     *
     * If you are about to turn this on in an already-deployed Pipeline,
     * set the value to `true` first, commit and allow the pipeline to
     * self-update, and only then use the Docker asset in the pipeline.
     *
     * @default false
     */
    readonly dockerEnabledForSelfMutation?: boolean;
    /**
     * Enable Docker for the 'synth' step
     *
     * Set this to true if you are using file assets that require
     * "bundling" anywhere in your application (meaning an asset
     * compilation step will be run with the tools provided by
     * a Docker image), both for the Pipeline stack as well as the
     * application stacks.
     *
     * A common way to use bundling assets in your application is by
     * using the `@aws-cdk/aws-lambda-nodejs` library.
     *
     * Configures privileged mode for the synth CodeBuild action.
     *
     * If you are about to turn this on in an already-deployed Pipeline,
     * set the value to `true` first, commit and allow the pipeline to
     * self-update, and only then use the bundled asset.
     *
     * @default false
     */
    readonly dockerEnabledForSynth?: boolean;
    /**
     * Customize the CodeBuild projects created for this pipeline
     *
     * @default - All projects run non-privileged build, SMALL instance, LinuxBuildImage.STANDARD_5_0
     */
    readonly codeBuildDefaults?: CodeBuildOptions;
    /**
     * Additional customizations to apply to the synthesize CodeBuild projects
     *
     * @default - Only `codeBuildDefaults` are applied
     */
    readonly synthCodeBuildDefaults?: CodeBuildOptions;
    /**
     * Additional customizations to apply to the asset publishing CodeBuild projects
     *
     * @default - Only `codeBuildDefaults` are applied
     */
    readonly assetPublishingCodeBuildDefaults?: CodeBuildOptions;
    /**
     * Additional customizations to apply to the self mutation CodeBuild projects
     *
     * @default - Only `codeBuildDefaults` are applied
     */
    readonly selfMutationCodeBuildDefaults?: CodeBuildOptions;
    /**
     * Publish assets in multiple CodeBuild projects
     *
     * If set to false, use one Project per type to publish all assets.
     *
     * Publishing in parallel improves concurrency and may reduce publishing
     * latency, but may also increase overall provisioning time of the CodeBuild
     * projects.
     *
     * Experiment and see what value works best for you.
     *
     * @default true
     */
    readonly publishAssetsInParallel?: boolean;
    /**
     * A list of credentials used to authenticate to Docker registries.
     *
     * Specify any credentials necessary within the pipeline to build, synth, update, or publish assets.
     *
     * @default []
     */
    readonly dockerCredentials?: DockerCredential[];
    /**
     * An existing Pipeline to be reused and built upon.
     *
     * [disable-awslint:ref-via-interface]
     *
     * @default - a new underlying pipeline is created.
     */
    readonly codePipeline?: cp.Pipeline;
    /**
     * Reuse the same cross region support stack for all pipelines in the App.
     *
     * @default - true (Use the same support stack for all pipelines in App)
     */
    readonly reuseCrossRegionSupportStacks?: boolean;
}
/**
 * Options for customizing a single CodeBuild project
 */
export interface CodeBuildOptions {
    /**
     * Partial build environment, will be combined with other build environments that apply
     *
     * @default - Non-privileged build, SMALL instance, LinuxBuildImage.STANDARD_5_0
     */
    readonly buildEnvironment?: cb.BuildEnvironment;
    /**
     * Policy statements to add to role
     *
     * @default - No policy statements added to CodeBuild Project Role
     */
    readonly rolePolicy?: iam.PolicyStatement[];
    /**
     * Partial buildspec, will be combined with other buildspecs that apply
     *
     * The BuildSpec must be available inline--it cannot reference a file
     * on disk.
     *
     * @default - No initial BuildSpec
     */
    readonly partialBuildSpec?: cb.BuildSpec;
    /**
     * Which security group(s) to associate with the project network interfaces.
     *
     * Only used if 'vpc' is supplied.
     *
     * @default - Security group will be automatically created.
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * The VPC where to create the CodeBuild network interfaces in.
     *
     * @default - No VPC
     */
    readonly vpc?: ec2.IVpc;
    /**
     * Which subnets to use.
     *
     * Only used if 'vpc' is supplied.
     *
     * @default - All private subnets.
     */
    readonly subnetSelection?: ec2.SubnetSelection;
    /**
     * The number of minutes after which AWS CodeBuild stops the build if it's
     * not complete. For valid values, see the timeoutInMinutes field in the AWS
     * CodeBuild User Guide.
     *
     * @default Duration.hours(1)
     */
    readonly timeout?: Duration;
}
/**
 * A CDK Pipeline that uses CodePipeline to deploy CDK apps
 *
 * This is a `Pipeline` with its `engine` property set to
 * `CodePipelineEngine`, and exists for nicer ergonomics for
 * users that don't need to switch out engines.
 */
export declare class CodePipeline extends PipelineBase {
    private readonly props;
    private _pipeline?;
    private artifacts;
    private _synthProject?;
    private readonly selfMutation;
    private _myCxAsmRoot?;
    private readonly dockerCredentials;
    /**
     * Asset roles shared for publishing
     */
    private readonly assetCodeBuildRoles;
    /**
     * Per asset type, the target role ARNs that need to be assumed
     */
    private readonly assetPublishingRoles;
    /**
     * This is set to the very first artifact produced in the pipeline
     */
    private _fallbackArtifact?;
    private _cloudAssemblyFileSet?;
    private readonly singlePublisherPerAssetType;
    private readonly cliVersion?;
    constructor(scope: Construct, id: string, props: CodePipelineProps);
    /**
     * The CodeBuild project that performs the Synth
     *
     * Only available after the pipeline has been built.
     */
    get synthProject(): cb.IProject;
    /**
     * The CodePipeline pipeline that deploys the CDK app
     *
     * Only available after the pipeline has been built.
     */
    get pipeline(): cp.Pipeline;
    protected doBuildPipeline(): void;
    private get myCxAsmRoot();
    /**
     * Scope for Assets-related resources.
     *
     * Purely exists for construct tree backwards compatibility with legacy pipelines
     */
    private get assetsScope();
    private pipelineStagesAndActionsFromGraph;
    /**
     * Do additional things after the action got added to the pipeline
     *
     * Some minor state manipulation of CodeBuild projects and pipeline
     * artifacts.
     */
    private postProcessNode;
    /**
     * Make an action from the given node and/or step
     */
    private actionFromNode;
    /**
     * Take a Step and turn it into a CodePipeline Action
     *
     * There are only 3 types of Steps we need to support:
     *
     * - Shell (generic)
     * - ManualApproval (generic)
     * - CodePipelineActionFactory (CodePipeline-specific)
     *
     * The rest is expressed in terms of these 3, or in terms of graph nodes
     * which are handled elsewhere.
     */
    private actionFromStep;
    private createChangeSetAction;
    private executeChangeSetAction;
    private selfMutateAction;
    private publishAssetsAction;
    private nodeTypeFromNode;
    private codeBuildDefaultsFor;
    private roleFromPlaceholderArn;
    /**
     * Non-template config files for CodePipeline actions
     *
     * Currently only supports tags.
     */
    private writeTemplateConfiguration;
    /**
     * This role is used by both the CodePipeline build action and related CodeBuild project. Consolidating these two
     * roles into one, and re-using across all assets, saves significant size of the final synthesized output.
     * Modeled after the CodePipeline role and 'CodePipelineActionRole' roles.
     * Generates one role per asset type to separate file and Docker/image-based permissions.
     */
    private obtainAssetCodeBuildRole;
}
