"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mergeCodeBuildOptions = exports.CodeBuildFactory = void 0;
const fs = require("fs");
const path = require("path");
const codebuild = require("../../../../aws-codebuild");
const codepipeline_actions = require("../../../../aws-codepipeline-actions");
const ec2 = require("../../../../aws-ec2");
const iam = require("../../../../aws-iam");
const core_1 = require("../../../../core");
const constructs_1 = require("constructs");
const pipeline_queries_1 = require("../../helpers-internal/pipeline-queries");
const step_output_1 = require("../../helpers-internal/step-output");
const construct_internals_1 = require("../../private/construct-internals");
const identifiers_1 = require("../../private/identifiers");
const javascript_1 = require("../../private/javascript");
const buildspecs_1 = require("./buildspecs");
/**
 * Produce a CodeBuild project from a ShellStep and some CodeBuild-specific customizations
 *
 * The functionality here is shared between the `CodePipeline` translating a `ShellStep` into
 * a CodeBuild project, as well as the `CodeBuildStep` straight up.
 */
class CodeBuildFactory {
    constructor(constructId, props) {
        this.constructId = constructId;
        this.props = props;
        this.stepId = props.stepId ?? constructId;
    }
    // eslint-disable-next-line max-len
    static fromShellStep(constructId, shellStep, additional) {
        return new CodeBuildFactory(constructId, {
            commands: shellStep.commands,
            env: shellStep.env,
            envFromCfnOutputs: shellStep.envFromCfnOutputs,
            inputs: shellStep.inputs,
            outputs: shellStep.outputs,
            stepId: shellStep.id,
            installCommands: shellStep.installCommands,
            producedStepOutputs: step_output_1.StepOutput.producedStepOutputs(shellStep),
            ...additional,
        });
    }
    static fromCodeBuildStep(constructId, step, additional) {
        const factory = CodeBuildFactory.fromShellStep(constructId, step, {
            projectName: step.projectName,
            role: step.role,
            actionRole: step.actionRole,
            ...additional,
            projectOptions: mergeCodeBuildOptions(additional?.projectOptions, {
                buildEnvironment: step.buildEnvironment,
                rolePolicy: step.rolePolicyStatements,
                securityGroups: step.securityGroups,
                partialBuildSpec: step.partialBuildSpec,
                vpc: step.vpc,
                subnetSelection: step.subnetSelection,
                timeout: step.timeout,
            }),
        });
        return {
            produceAction: (stage, options) => {
                const result = factory.produceAction(stage, options);
                if (result.project) {
                    step._setProject(result.project);
                }
                return result;
            },
        };
    }
    get project() {
        if (!this._project) {
            throw new Error('Project becomes available after produce() has been called');
        }
        return this._project;
    }
    produceAction(stage, options) {
        const projectOptions = mergeCodeBuildOptions(options.codeBuildDefaults, this.props.projectOptions);
        const inputs = this.props.inputs ?? [];
        const outputs = this.props.outputs ?? [];
        const mainInput = inputs.find(x => x.directory === '.');
        const extraInputs = inputs.filter(x => x.directory !== '.');
        const inputArtifact = mainInput
            ? options.artifacts.toCodePipeline(mainInput.fileSet)
            : options.fallbackArtifact;
        const extraInputArtifacts = extraInputs.map(x => options.artifacts.toCodePipeline(x.fileSet));
        const outputArtifacts = outputs.map(x => options.artifacts.toCodePipeline(x.fileSet));
        if (!inputArtifact) {
            // This should actually never happen because CodeBuild projects shouldn't be added before the
            // Source, which always produces at least an artifact.
            throw new Error(`CodeBuild action '${this.stepId}' requires an input (and the pipeline doesn't have a Source to fall back to). Add an input or a pipeline source.`);
        }
        const installCommands = [
            ...generateInputArtifactLinkCommands(options.artifacts, extraInputs),
            ...this.props.installCommands ?? [],
        ];
        const buildSpecHere = codebuild.BuildSpec.fromObject({
            version: '0.2',
            phases: {
                install: (installCommands.length ?? 0) > 0 ? { commands: installCommands } : undefined,
                build: this.props.commands.length > 0 ? { commands: this.props.commands } : undefined,
            },
            artifacts: javascript_1.noEmptyObject(renderArtifactsBuildSpec(options.artifacts, this.props.outputs ?? [])),
        });
        // Partition environment variables into environment variables that can go on the project
        // and environment variables that MUST go in the pipeline (those that reference CodePipeline variables)
        const env = javascript_1.noUndefined(this.props.env ?? {});
        const [actionEnvs, projectEnvs] = javascript_1.partition(Object.entries(env ?? {}), ([, v]) => containsPipelineVariable(v));
        const environment = mergeBuildEnvironments(projectOptions?.buildEnvironment ?? {}, {
            environmentVariables: javascript_1.noEmptyObject(javascript_1.mapValues(javascript_1.mkdict(projectEnvs), value => ({ value }))),
        });
        const fullBuildSpec = projectOptions?.partialBuildSpec
            ? codebuild.mergeBuildSpecs(projectOptions.partialBuildSpec, buildSpecHere)
            : buildSpecHere;
        const osFromEnvironment = environment.buildImage && environment.buildImage instanceof codebuild.WindowsBuildImage
            ? ec2.OperatingSystemType.WINDOWS
            : ec2.OperatingSystemType.LINUX;
        const actualBuildSpec = filterBuildSpecCommands(fullBuildSpec, osFromEnvironment);
        const scope = this.props.scope ?? options.scope;
        let projectBuildSpec;
        if (this.props.passBuildSpecViaCloudAssembly) {
            // Write to disk and replace with a reference
            const relativeSpecFile = `buildspec-${constructs_1.Node.of(scope).addr}-${this.constructId}.yaml`;
            const absSpecFile = path.join(construct_internals_1.cloudAssemblyBuildSpecDir(scope), relativeSpecFile);
            // This should resolve to a pure JSON string. If it resolves to an object, it's a CFN
            // expression, and we can't support that yet. Maybe someday if we think really hard about it.
            const fileContents = core_1.Stack.of(scope).resolve(actualBuildSpec.toBuildSpec());
            if (typeof fileContents !== 'string') {
                throw new Error(`This BuildSpec contains CloudFormation references and is supported by publishInParallel=false: ${JSON.stringify(fileContents, undefined, 2)}`);
            }
            fs.writeFileSync(absSpecFile, fileContents, { encoding: 'utf-8' });
            projectBuildSpec = codebuild.BuildSpec.fromSourceFilename(relativeSpecFile);
        }
        else {
            projectBuildSpec = actualBuildSpec;
        }
        // A hash over the values that make the CodeBuild Project unique (and necessary
        // to restart the pipeline if one of them changes). projectName is not necessary to include
        // here because the pipeline will definitely restart if projectName changes.
        // (Resolve tokens)
        const projectConfigHash = identifiers_1.hash(core_1.Stack.of(scope).resolve({
            environment: serializeBuildEnvironment(environment),
            buildSpecString: actualBuildSpec.toBuildSpec(),
        }));
        const actionName = options.actionName ?? this.stepId;
        let projectScope = scope;
        if (this.props.additionalConstructLevel ?? true) {
            projectScope = construct_internals_1.obtainScope(scope, actionName);
        }
        const safePipelineName = core_1.Token.isUnresolved(options.pipeline.pipeline.pipelineName)
            ? `${core_1.Stack.of(options.pipeline).stackName}/${constructs_1.Node.of(options.pipeline.pipeline).id}`
            : options.pipeline.pipeline.pipelineName;
        const project = new codebuild.PipelineProject(projectScope, this.constructId, {
            projectName: this.props.projectName,
            description: `Pipeline step ${safePipelineName}/${stage.stageName}/${actionName}`.substring(0, 255),
            environment,
            vpc: projectOptions.vpc,
            subnetSelection: projectOptions.subnetSelection,
            securityGroups: projectOptions.securityGroups,
            buildSpec: projectBuildSpec,
            role: this.props.role,
            timeout: projectOptions.timeout,
        });
        if (this.props.additionalDependable) {
            project.node.addDependency(this.props.additionalDependable);
        }
        if (projectOptions.rolePolicy !== undefined) {
            projectOptions.rolePolicy.forEach(policyStatement => {
                project.addToRolePolicy(policyStatement);
            });
        }
        const queries = new pipeline_queries_1.PipelineQueries(options.pipeline);
        const stackOutputEnv = javascript_1.mapValues(this.props.envFromCfnOutputs ?? {}, outputRef => `#{${identifiers_1.stackVariableNamespace(queries.producingStack(outputRef))}.${outputRef.outputName}}`);
        const configHashEnv = options.beforeSelfMutation
            ? { _PROJECT_CONFIG_HASH: projectConfigHash }
            : {};
        // Start all CodeBuild projects from a single (shared) Action Role, so that we don't have to generate an Action Role for each
        // individual CodeBuild Project and blow out the pipeline policy size (and potentially # of resources in the stack).
        const actionRoleCid = 'CodeBuildActionRole';
        const actionRole = this.props.actionRole
            ?? options.pipeline.node.tryFindChild(actionRoleCid)
            ?? new iam.Role(options.pipeline, actionRoleCid, {
                assumedBy: new iam.PrincipalWithConditions(new iam.AccountRootPrincipal(), {
                    Bool: { 'aws:ViaAWSService': iam.ServicePrincipal.servicePrincipalName('codepipeline.amazonaws.com') },
                }),
            });
        stage.addAction(new codepipeline_actions.CodeBuildAction({
            actionName: actionName,
            input: inputArtifact,
            extraInputs: extraInputArtifacts,
            outputs: outputArtifacts,
            project,
            runOrder: options.runOrder,
            variablesNamespace: options.variablesNamespace,
            role: actionRole,
            // Inclusion of the hash here will lead to the pipeline structure for any changes
            // made the config of the underlying CodeBuild Project.
            // Hence, the pipeline will be restarted. This is necessary if the users
            // adds (for example) build or test commands to the buildspec.
            environmentVariables: javascript_1.noEmptyObject(cbEnv({
                ...javascript_1.mkdict(actionEnvs),
                ...configHashEnv,
                ...stackOutputEnv,
            })),
        }));
        this._project = project;
        return { runOrdersConsumed: 1, project };
    }
}
exports.CodeBuildFactory = CodeBuildFactory;
/**
 * Generate commands to move additional input artifacts into the right place
 */
function generateInputArtifactLinkCommands(artifacts, inputs) {
    return inputs.map(input => {
        const fragments = [];
        fragments.push(`[ ! -d "${input.directory}" ] || { echo 'additionalInputs: "${input.directory}" must not exist yet. If you want to merge multiple artifacts, use a "cp" command.'; exit 1; }`);
        const parentDirectory = path.dirname(input.directory);
        if (!['.', '..'].includes(parentDirectory)) {
            fragments.push(`mkdir -p -- "${parentDirectory}"`);
        }
        const artifact = artifacts.toCodePipeline(input.fileSet);
        fragments.push(`ln -s -- "$CODEBUILD_SRC_DIR_${artifact.artifactName}" "${input.directory}"`);
        return fragments.join(' && ');
    });
}
function renderArtifactsBuildSpec(artifactMap, outputs) {
    // save the generated files in the output artifact
    // This part of the buildspec has to look completely different depending on whether we're
    // using secondary artifacts or not.
    if (outputs.length === 0) {
        return {};
    }
    if (outputs.length === 1) {
        return {
            'base-directory': outputs[0].directory,
            'files': '**/*',
        };
    }
    const secondary = {};
    for (const output of outputs) {
        const art = artifactMap.toCodePipeline(output.fileSet);
        if (!art.artifactName) {
            throw new Error('You must give the output artifact a name');
        }
        secondary[art.artifactName] = {
            'base-directory': output.directory,
            'files': '**/*',
        };
    }
    return { 'secondary-artifacts': secondary };
}
function mergeCodeBuildOptions(...opts) {
    const xs = [{}, ...opts.filter(isDefined)];
    while (xs.length > 1) {
        const [a, b] = xs.splice(xs.length - 2, 2);
        xs.push(merge2(a, b));
    }
    return xs[0];
    function merge2(a, b) {
        return {
            buildEnvironment: mergeBuildEnvironments(a.buildEnvironment, b.buildEnvironment),
            rolePolicy: definedArray([...a.rolePolicy ?? [], ...b.rolePolicy ?? []]),
            securityGroups: definedArray([...a.securityGroups ?? [], ...b.securityGroups ?? []]),
            partialBuildSpec: buildspecs_1.mergeBuildSpecs(a.partialBuildSpec, b.partialBuildSpec),
            vpc: b.vpc ?? a.vpc,
            subnetSelection: b.subnetSelection ?? a.subnetSelection,
            timeout: b.timeout ?? a.timeout,
        };
    }
}
exports.mergeCodeBuildOptions = mergeCodeBuildOptions;
function mergeBuildEnvironments(a, b) {
    if (!a || !b) {
        return a ?? b;
    }
    return {
        buildImage: b.buildImage ?? a.buildImage,
        computeType: b.computeType ?? a.computeType,
        environmentVariables: {
            ...a.environmentVariables,
            ...b.environmentVariables,
        },
        privileged: b.privileged ?? a.privileged,
    };
}
function isDefined(x) {
    return x !== undefined;
}
/**
 * Serialize a build environment to data (get rid of constructs & objects), so we can JSON.stringify it
 */
function serializeBuildEnvironment(env) {
    return {
        privileged: env.privileged,
        environmentVariables: env.environmentVariables,
        type: env.buildImage?.type,
        imageId: env.buildImage?.imageId,
        computeType: env.computeType,
        imagePullPrincipalType: env.buildImage?.imagePullPrincipalType,
        secretsManagerArn: env.buildImage?.secretsManagerCredentials?.secretArn,
    };
}
/**
 * Whether the given string contains a reference to a CodePipeline variable
 */
function containsPipelineVariable(s) {
    return !!s.match(/#\{[^}]+\}/) || step_output_1.StepOutput.findAll(s).length > 0;
}
/**
 * Turn a collection into a collection of CodePipeline environment variables
 */
function cbEnv(xs) {
    return javascript_1.mkdict(Object.entries(xs)
        .filter(([, v]) => v !== undefined)
        .map(([k, v]) => [k, { value: v }]));
}
function definedArray(xs) {
    return xs.length > 0 ? xs : undefined;
}
/**
 * If lines in the buildspec start with '!WINDOWS!' or '!LINUX!', only render them on that platform.
 *
 * Very private protocol for now, but may come in handy in other libraries as well.
 */
function filterBuildSpecCommands(buildSpec, osType) {
    if (!buildSpec.isImmediate) {
        return buildSpec;
    }
    const spec = buildSpec.spec;
    const winTag = '!WINDOWS!';
    const linuxTag = '!LINUX!';
    const expectedTag = osType === ec2.OperatingSystemType.WINDOWS ? winTag : linuxTag;
    return codebuild.BuildSpec.fromObject(recurse(spec));
    function recurse(x) {
        if (Array.isArray(x)) {
            const ret = [];
            for (const el of x) {
                const [tag, payload] = extractTag(el);
                if (tag === undefined || tag === expectedTag) {
                    ret.push(payload);
                }
            }
            return ret;
        }
        if (x && typeof x === 'object') {
            return javascript_1.mapValues(x, recurse);
        }
        return x;
    }
    function extractTag(x) {
        if (typeof x !== 'string') {
            return [undefined, x];
        }
        for (const tag of [winTag, linuxTag]) {
            if (x.startsWith(tag)) {
                return [tag, x.slice(tag.length)];
            }
        }
        return [undefined, x];
    }
}
//# sourceMappingURL=data:application/json;base64,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