"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PipelineGraph = void 0;
const blueprint_1 = require("../blueprint");
const graph_1 = require("./graph");
const pipeline_queries_1 = require("./pipeline-queries");
/**
 * Logic to turn the deployment blueprint into a graph
 *
 * This code makes all the decisions on how to lay out the CodePipeline
 */
class PipelineGraph {
    constructor(pipeline, props = {}) {
        this.pipeline = pipeline;
        this.graph = graph_1.Graph.of('', { type: 'group' });
        this.added = new Map();
        this.assetNodes = new Map();
        this.assetNodesByType = new Map();
        this.stackOutputDependencies = new graph_1.DependencyBuilders();
        this._fileAssetCtr = 0;
        this._dockerAssetCtr = 0;
        this.publishTemplate = props.publishTemplate ?? false;
        this.prepareStep = props.prepareStep ?? true;
        this.singlePublisher = props.singlePublisherPerAssetType ?? false;
        this.queries = new pipeline_queries_1.PipelineQueries(pipeline);
        if (pipeline.synth instanceof blueprint_1.Step) {
            this.synthNode = this.addBuildStep(pipeline.synth);
            if (this.synthNode?.data?.type === 'step') {
                this.synthNode.data.isBuildStep = true;
            }
        }
        this.lastPreparationNode = this.synthNode;
        const cloudAssembly = pipeline.synth.primaryOutput?.primaryOutput;
        if (!cloudAssembly) {
            throw new Error(`The synth step must produce the cloud assembly artifact, but doesn't: ${pipeline.synth}`);
        }
        this.cloudAssemblyFileSet = cloudAssembly;
        if (props.selfMutation) {
            const stage = graph_1.Graph.of('UpdatePipeline', { type: 'group' });
            this.graph.add(stage);
            this.selfMutateNode = graph_1.GraphNode.of('SelfMutate', { type: 'self-update' });
            stage.add(this.selfMutateNode);
            this.selfMutateNode.dependOn(this.synthNode);
            this.lastPreparationNode = this.selfMutateNode;
        }
        const waves = pipeline.waves.map(w => this.addWave(w));
        // Make sure the waves deploy sequentially
        for (let i = 1; i < waves.length; i++) {
            waves[i].dependOn(waves[i - 1]);
        }
    }
    isSynthNode(node) {
        return this.synthNode === node;
    }
    addBuildStep(step) {
        return this.addAndRecurse(step, this.topLevelGraph('Build'));
    }
    addWave(wave) {
        // If the wave only has one Stage in it, don't add an additional Graph around it
        const retGraph = wave.stages.length === 1
            ? this.addStage(wave.stages[0])
            : graph_1.Graph.of(wave.id, { type: 'group' }, wave.stages.map(s => this.addStage(s)));
        this.addPrePost(wave.pre, wave.post, retGraph);
        retGraph.dependOn(this.lastPreparationNode);
        this.graph.add(retGraph);
        return retGraph;
    }
    addStage(stage) {
        const retGraph = graph_1.Graph.of(stage.stageName, { type: 'group' });
        const stackGraphs = new Map();
        for (const stack of stage.stacks) {
            const stackGraph = graph_1.Graph.of(this.simpleStackName(stack.stackName, stage.stageName), { type: 'stack-group', stack });
            const prepareNode = this.prepareStep ? graph_1.GraphNode.of('Prepare', { type: 'prepare', stack }) : undefined;
            const deployNode = graph_1.GraphNode.of('Deploy', {
                type: 'execute',
                stack,
                captureOutputs: this.queries.stackOutputsReferenced(stack).length > 0,
            });
            retGraph.add(stackGraph);
            stackGraph.add(deployNode);
            // node or node collection that represents first point of contact in each stack
            let firstDeployNode;
            if (prepareNode) {
                stackGraph.add(prepareNode);
                deployNode.dependOn(prepareNode);
                firstDeployNode = prepareNode;
            }
            else {
                firstDeployNode = deployNode;
            }
            // add changeset steps at the stack level
            if (stack.changeSet.length > 0) {
                if (prepareNode) {
                    this.addChangeSet(stack.changeSet, prepareNode, deployNode, stackGraph);
                }
                else {
                    throw new Error('Your pipeline engine does not support changeSet steps');
                }
            }
            // add pre and post steps at the stack level
            const preNodes = this.addPrePost(stack.pre, stack.post, stackGraph);
            if (preNodes.nodes.length > 0) {
                firstDeployNode = preNodes;
            }
            stackGraphs.set(stack, stackGraph);
            const cloudAssembly = this.cloudAssemblyFileSet;
            firstDeployNode.dependOn(this.addAndRecurse(cloudAssembly.producer, retGraph));
            // add the template asset
            if (this.publishTemplate) {
                if (!stack.templateAsset) {
                    throw new Error(`"publishTemplate" is enabled, but stack ${stack.stackArtifactId} does not have a template asset`);
                }
                firstDeployNode.dependOn(this.publishAsset(stack.templateAsset));
            }
            // Depend on Assets
            // FIXME: Custom Cloud Assembly currently doesn't actually help separating
            // out templates from assets!!!
            for (const asset of stack.assets) {
                const assetNode = this.publishAsset(asset);
                firstDeployNode.dependOn(assetNode);
            }
            // Add stack output synchronization point
            if (this.queries.stackOutputsReferenced(stack).length > 0) {
                this.stackOutputDependencies.get(stack).dependOn(deployNode);
            }
        }
        for (const stack of stage.stacks) {
            for (const dep of stack.stackDependencies) {
                const stackNode = stackGraphs.get(stack);
                const depNode = stackGraphs.get(dep);
                if (!stackNode) {
                    throw new Error(`cannot find node for ${stack.stackName}`);
                }
                if (!depNode) {
                    throw new Error(`cannot find node for ${dep.stackName}`);
                }
                stackNode.dependOn(depNode);
            }
        }
        this.addPrePost(stage.pre, stage.post, retGraph);
        return retGraph;
    }
    addChangeSet(changeSet, prepareNode, deployNode, graph) {
        for (const c of changeSet) {
            const changeSetNode = this.addAndRecurse(c, graph);
            changeSetNode?.dependOn(prepareNode);
            deployNode.dependOn(changeSetNode);
        }
    }
    addPrePost(pre, post, parent) {
        const currentNodes = new graph_1.GraphNodeCollection(parent.nodes);
        const preNodes = new graph_1.GraphNodeCollection(new Array());
        for (const p of pre) {
            const preNode = this.addAndRecurse(p, parent);
            currentNodes.dependOn(preNode);
            preNodes.nodes.push(preNode);
        }
        for (const p of post) {
            const postNode = this.addAndRecurse(p, parent);
            postNode?.dependOn(...currentNodes.nodes);
        }
        return preNodes;
    }
    topLevelGraph(name) {
        let ret = this.graph.tryGetChild(name);
        if (!ret) {
            ret = new graph_1.Graph(name);
            this.graph.add(ret);
        }
        return ret;
    }
    addAndRecurse(step, parent) {
        if (step === PipelineGraph.NO_STEP) {
            return undefined;
        }
        const previous = this.added.get(step);
        if (previous) {
            return previous;
        }
        const node = graph_1.GraphNode.of(step.id, { type: 'step', step });
        // If the step is a source step, change the parent to a special "Source" stage
        // (CodePipeline wants it that way)
        if (step.isSource) {
            parent = this.topLevelGraph('Source');
        }
        parent.add(node);
        this.added.set(step, node);
        for (const dep of step.dependencies) {
            const producerNode = this.addAndRecurse(dep, parent);
            node.dependOn(producerNode);
        }
        // Add stack dependencies (by use of the dependency builder this also works
        // if we encounter the Step before the Stack has been properly added yet)
        if (step instanceof blueprint_1.ShellStep) {
            for (const output of Object.values(step.envFromCfnOutputs)) {
                const stack = this.queries.producingStack(output);
                this.stackOutputDependencies.get(stack).dependBy(node);
            }
        }
        return node;
    }
    publishAsset(stackAsset) {
        const assetsGraph = this.topLevelGraph('Assets');
        let assetNode = this.assetNodes.get(stackAsset.assetId);
        if (assetNode) {
            // If there's already a node publishing this asset, add as a new publishing
            // destination to the same node.
        }
        else if (this.singlePublisher && this.assetNodesByType.has(stackAsset.assetType)) {
            // If we're doing a single node per type, lookup by that
            assetNode = this.assetNodesByType.get(stackAsset.assetType);
        }
        else {
            // Otherwise add a new one
            const id = stackAsset.assetType === blueprint_1.AssetType.FILE
                ? (this.singlePublisher ? 'FileAsset' : `FileAsset${++this._fileAssetCtr}`)
                : (this.singlePublisher ? 'DockerAsset' : `DockerAsset${++this._dockerAssetCtr}`);
            assetNode = graph_1.GraphNode.of(id, { type: 'publish-assets', assets: [] });
            assetsGraph.add(assetNode);
            assetNode.dependOn(this.lastPreparationNode);
            this.assetNodesByType.set(stackAsset.assetType, assetNode);
            this.assetNodes.set(stackAsset.assetId, assetNode);
        }
        const data = assetNode.data;
        if (data?.type !== 'publish-assets') {
            throw new Error(`${assetNode} has the wrong data.type: ${data?.type}`);
        }
        if (!data.assets.some(a => a.assetSelector === stackAsset.assetSelector)) {
            data.assets.push(stackAsset);
        }
        return assetNode;
    }
    /**
     * Simplify the stack name by removing the `Stage-` prefix if it exists.
     */
    simpleStackName(stackName, stageName) {
        return stripPrefix(stackName, `${stageName}-`);
    }
}
exports.PipelineGraph = PipelineGraph;
/**
 * A Step object that may be used as the producer of FileSets that should not be represented in the graph
 */
PipelineGraph.NO_STEP = new class extends blueprint_1.Step {
}('NO_STEP');
function stripPrefix(s, prefix) {
    return s.startsWith(prefix) ? s.slice(prefix.length) : s;
}
//# sourceMappingURL=data:application/json;base64,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