"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PipelineQueries = void 0;
const blueprint_1 = require("../blueprint");
/**
 * Answer some questions about a pipeline blueprint
 */
class PipelineQueries {
    constructor(pipeline) {
        this.pipeline = pipeline;
    }
    /**
     * Return the names of all outputs for the given stack that are referenced in this blueprint
     */
    stackOutputsReferenced(stack) {
        const steps = new Array();
        for (const wave of this.pipeline.waves) {
            steps.push(...wave.pre, ...wave.post);
            for (const stage of wave.stages) {
                steps.push(...stage.pre, ...stage.post);
                for (const stackDeployment of stage.stacks) {
                    steps.push(...stackDeployment.pre, ...stackDeployment.changeSet, ...stackDeployment.post);
                }
            }
        }
        const ret = new Array();
        for (const step of steps) {
            if (!(step instanceof blueprint_1.ShellStep)) {
                continue;
            }
            for (const outputRef of Object.values(step.envFromCfnOutputs)) {
                if (outputRef.isProducedBy(stack)) {
                    ret.push(outputRef.outputName);
                }
            }
        }
        return ret;
    }
    /**
     * Find the stack deployment that is producing the given reference
     */
    producingStack(outputReference) {
        for (const wave of this.pipeline.waves) {
            for (const stage of wave.stages) {
                for (const stack of stage.stacks) {
                    if (outputReference.isProducedBy(stack)) {
                        return stack;
                    }
                }
            }
        }
        throw new Error(`Stack '${outputReference.stackDescription}' (producing output '${outputReference.outputName}') is not in the pipeline; call 'addStage()' to add the stack's Stage to the pipeline`);
    }
    /**
     * All assets referenced in all the Stacks of a StageDeployment
     */
    assetsInStage(stage) {
        const assets = new Map();
        for (const stack of stage.stacks) {
            for (const asset of stack.assets) {
                assets.set(asset.assetSelector, asset);
            }
        }
        return Array.from(assets.values());
    }
}
exports.PipelineQueries = PipelineQueries;
//# sourceMappingURL=data:application/json;base64,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