"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkPipeline = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const codepipeline = require("../../../aws-codepipeline");
const iam = require("../../../aws-iam");
const core_1 = require("../../../core");
const asset_type_1 = require("../blueprint/asset-type");
const docker_credentials_1 = require("../docker-credentials");
const application_security_check_1 = require("../private/application-security-check");
const asset_singleton_role_1 = require("../private/asset-singleton-role");
const cli_version_1 = require("../private/cli-version");
const construct_internals_1 = require("../private/construct-internals");
const actions_1 = require("./actions");
const stage_1 = require("./stage");
const synths_1 = require("./synths");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("../../../core");
const CODE_BUILD_LENGTH_LIMIT = 100;
/**
 * A Pipeline to deploy CDK apps
 *
 * Defines an AWS CodePipeline-based Pipeline to deploy CDK applications.
 *
 * Automatically manages the following:
 *
 * - Stack dependency order.
 * - Asset publishing.
 * - Keeping the pipeline up-to-date as the CDK apps change.
 * - Using stack outputs later on in the pipeline.
 *
 * @deprecated This class is part of the old API. Use the API based on the `CodePipeline` class instead
 */
class CdkPipeline extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this._stages = [];
        this._outputArtifacts = {};
        try {
            jsiiDeprecationWarnings.print("monocdk.pipelines.CdkPipeline", "This class is part of the old API. Use the API based on the `CodePipeline` class instead");
            jsiiDeprecationWarnings.monocdk_pipelines_CdkPipelineProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CdkPipeline);
            }
            throw error;
        }
        this.cliVersion = props.cdkCliVersion ?? cli_version_1.preferredCliVersion();
        if (!core_1.App.isApp(this.node.root)) {
            throw new Error('CdkPipeline must be created under an App');
        }
        this._cloudAssemblyArtifact = props.cloudAssemblyArtifact;
        this._dockerCredentials = props.dockerCredentials ?? [];
        const pipelineStack = core_1.Stack.of(this);
        if (props.codePipeline) {
            if (props.pipelineName) {
                throw new Error('Cannot set \'pipelineName\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            if (props.crossAccountKeys !== undefined) {
                throw new Error('Cannot set \'crossAccountKeys\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            if (props.enableKeyRotation !== undefined) {
                throw new Error('Cannot set \'enableKeyRotation\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            this._pipeline = props.codePipeline;
        }
        else {
            this._pipeline = new codepipeline.Pipeline(this, 'Pipeline', {
                pipelineName: props.pipelineName,
                crossAccountKeys: props.crossAccountKeys,
                enableKeyRotation: props.enableKeyRotation,
                restartExecutionOnUpdate: true,
            });
        }
        if (props.sourceAction && !props.synthAction) {
            // Because of ordering limitations, you can: bring your own Source, bring your own
            // Both, or bring your own Nothing. You cannot bring your own Build (which because of the
            // current CodePipeline API must go BEFORE what we're adding) and then having us add a
            // Source after it. That doesn't make any sense.
            throw new Error('When passing a \'sourceAction\' you must also pass a \'synthAction\' (or a \'codePipeline\' that already has both)');
        }
        if (!props.sourceAction && (!props.codePipeline || props.codePipeline.stages.length < 1)) {
            throw new Error('You must pass a \'sourceAction\' (or a \'codePipeline\' that already has a Source stage)');
        }
        if (props.sourceAction) {
            this._pipeline.addStage({
                stageName: 'Source',
                actions: [props.sourceAction],
            });
        }
        if (props.synthAction) {
            if (props.synthAction instanceof synths_1.SimpleSynthAction && this._dockerCredentials.length > 0) {
                props.synthAction._addDockerCredentials(this._dockerCredentials);
            }
            this._pipeline.addStage({
                stageName: 'Build',
                actions: [props.synthAction],
            });
        }
        if (props.selfMutating ?? true) {
            this._pipeline.addStage({
                stageName: 'UpdatePipeline',
                actions: [new actions_1.UpdatePipelineAction(this, 'UpdatePipeline', {
                        cloudAssemblyInput: this._cloudAssemblyArtifact,
                        pipelineStackHierarchicalId: pipelineStack.node.path,
                        cdkCliVersion: this.cliVersion,
                        projectName: maybeSuffix(props.pipelineName, '-selfupdate'),
                        privileged: props.supportDockerAssets,
                        dockerCredentials: this._dockerCredentials,
                        buildSpec: props.selfMutationBuildSpec,
                    })],
            });
        }
        this._assets = new AssetPublishing(this, 'Assets', {
            cloudAssemblyInput: this._cloudAssemblyArtifact,
            cdkCliVersion: this.cliVersion,
            pipeline: this._pipeline,
            projectName: maybeSuffix(props.pipelineName, '-publish'),
            vpc: props.vpc,
            subnetSelection: props.subnetSelection,
            singlePublisherPerType: props.singlePublisherPerType,
            preInstallCommands: props.assetPreInstallCommands,
            buildSpec: props.assetBuildSpec,
            dockerCredentials: this._dockerCredentials,
        });
    }
    /**
     * The underlying CodePipeline object
     *
     * You can use this to add more Stages to the pipeline, or Actions
     * to Stages.
     */
    get codePipeline() {
        try {
            jsiiDeprecationWarnings.print("monocdk.pipelines.CdkPipeline#codePipeline", "This class is part of the old API. Use the API based on the `CodePipeline` class instead");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, jsiiDeprecationWarnings.getPropertyDescriptor(this, "codePipeline").get);
            }
            throw error;
        }
        return this._pipeline;
    }
    /**
     * Access one of the pipeline's stages by stage name
     *
     * You can use this to add more Actions to a stage.
     */
    stage(stageName) {
        try {
            jsiiDeprecationWarnings.print("monocdk.pipelines.CdkPipeline#stage", "This class is part of the old API. Use the API based on the `CodePipeline` class instead");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.stage);
            }
            throw error;
        }
        return this._pipeline.stage(stageName);
    }
    /**
     * Get a cached version of an Application Security Check, which consists of:
     *  - CodeBuild Project to check for security changes in a stage
     *  - Lambda Function that approves the manual approval if no security changes are detected
     *
     * @internal
     */
    _getApplicationSecurityCheck() {
        if (!this._applicationSecurityCheck) {
            this._applicationSecurityCheck = new application_security_check_1.ApplicationSecurityCheck(this, 'PipelineApplicationSecurityCheck', {
                codePipeline: this._pipeline,
            });
        }
        return this._applicationSecurityCheck;
    }
    /**
     * Add pipeline stage that will deploy the given application stage
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     */
    addApplicationStage(appStage, options = {}) {
        try {
            jsiiDeprecationWarnings.print("monocdk.pipelines.CdkPipeline#addApplicationStage", "This class is part of the old API. Use the API based on the `CodePipeline` class instead");
            jsiiDeprecationWarnings.monocdk_Stage(appStage);
            jsiiDeprecationWarnings.monocdk_pipelines_AddStageOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addApplicationStage);
            }
            throw error;
        }
        const stage = this.addStage(appStage.stageName, options);
        stage.addApplication(appStage, options);
        return stage;
    }
    /**
     * Add a new, empty stage to the pipeline
     *
     * Prefer to use `addApplicationStage` if you are intended to deploy a CDK
     * application, but you can use this method if you want to add other kinds of
     * Actions to a pipeline.
     */
    addStage(stageName, options) {
        try {
            jsiiDeprecationWarnings.print("monocdk.pipelines.CdkPipeline#addStage", "This class is part of the old API. Use the API based on the `CodePipeline` class instead");
            jsiiDeprecationWarnings.monocdk_pipelines_BaseStageOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addStage);
            }
            throw error;
        }
        const pipelineStage = this._pipeline.addStage({
            stageName,
        });
        const stage = new stage_1.CdkStage(this, stageName, {
            cloudAssemblyArtifact: this._cloudAssemblyArtifact,
            pipelineStage,
            stageName,
            host: {
                publishAsset: this._assets.addPublishAssetAction.bind(this._assets),
                stackOutputArtifact: (artifactId) => this._outputArtifacts[artifactId],
            },
            ...options,
        });
        this._stages.push(stage);
        return stage;
    }
    /**
     * Get the StackOutput object that holds this CfnOutput's value in this pipeline
     *
     * `StackOutput` can be used in validation actions later in the pipeline.
     */
    stackOutput(cfnOutput) {
        try {
            jsiiDeprecationWarnings.print("monocdk.pipelines.CdkPipeline#stackOutput", "This class is part of the old API. Use the API based on the `CodePipeline` class instead");
            jsiiDeprecationWarnings.monocdk_CfnOutput(cfnOutput);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.stackOutput);
            }
            throw error;
        }
        const stack = core_1.Stack.of(cfnOutput);
        if (!this._outputArtifacts[stack.artifactId]) {
            // We should have stored the ArtifactPath in the map, but its Artifact
            // property isn't publicly readable...
            const artifactName = `${stack.artifactId}_Outputs`;
            const compactName = artifactName.slice(artifactName.length - Math.min(artifactName.length, CODE_BUILD_LENGTH_LIMIT));
            this._outputArtifacts[stack.artifactId] = new codepipeline.Artifact(compactName);
        }
        return new stage_1.StackOutput(this._outputArtifacts[stack.artifactId].atPath('outputs.json'), cfnOutput.logicalId);
    }
    /**
     * Validate that we don't have any stacks violating dependency order in the pipeline
     *
     * Our own convenience methods will never generate a pipeline that does that (although
     * this is a nice verification), but a user can also add the stacks by hand.
     */
    validate() {
        try {
            jsiiDeprecationWarnings.print("monocdk.pipelines.CdkPipeline#validate", "This class is part of the old API. Use the API based on the `CodePipeline` class instead");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.validate);
            }
            throw error;
        }
        const ret = new Array();
        ret.push(...this.validateDeployOrder());
        ret.push(...this.validateRequestedOutputs());
        return ret;
    }
    /**
     * Return all StackDeployActions in an ordered list
     */
    get stackActions() {
        return flatMap(this._pipeline.stages, s => s.actions.filter(isDeployAction));
    }
    *validateDeployOrder() {
        const stackActions = this.stackActions;
        for (const stackAction of stackActions) {
            // For every dependency, it must be executed in an action before this one is prepared.
            for (const depId of stackAction.dependencyStackArtifactIds) {
                const depAction = stackActions.find(s => s.stackArtifactId === depId);
                if (depAction === undefined) {
                    core_1.Annotations.of(this).addWarning(`Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depId}', but that dependency is not deployed through the pipeline!`);
                }
                else if (!(depAction.executeRunOrder < stackAction.prepareRunOrder)) {
                    yield `Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depAction.stackName}', but is deployed before it in the pipeline!`;
                }
            }
        }
    }
    *validateRequestedOutputs() {
        const artifactIds = this.stackActions.map(s => s.stackArtifactId);
        for (const artifactId of Object.keys(this._outputArtifacts)) {
            if (!artifactIds.includes(artifactId)) {
                yield `Trying to use outputs for Stack '${artifactId}', but Stack is not deployed in this pipeline. Add it to the pipeline.`;
            }
        }
    }
}
exports.CdkPipeline = CdkPipeline;
_a = JSII_RTTI_SYMBOL_1;
CdkPipeline[_a] = { fqn: "monocdk.pipelines.CdkPipeline", version: "1.191.0" };
function isDeployAction(a) {
    return a instanceof actions_1.DeployCdkStackAction;
}
function flatMap(xs, f) {
    return Array.prototype.concat([], ...xs.map(f));
}
/**
 * Add appropriate publishing actions to the asset publishing stage
 */
class AssetPublishing extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        // CodePipelines has a hard limit of 50 actions per stage. See https://github.com/aws/aws-cdk/issues/9353
        this.MAX_PUBLISHERS_PER_STAGE = 50;
        this.publishers = {};
        this.assetRoles = {};
        this.assetAttachedPolicies = {};
        this.assetPublishingRoles = {};
        this.stages = [];
        this._fileAssetCtr = 0;
        this._dockerAssetCtr = 0;
        this.myCxAsmRoot = path.resolve(construct_internals_1.assemblyBuilderOf(construct_internals_1.appOf(this)).outdir);
        this.pipeline = this.props.pipeline;
        // Hacks to get access to the innards of Pipeline
        const stages = this.props.pipeline._stages;
        // Any asset publishing stages will be added directly after the last stage that currently exists.
        this.lastStageBeforePublishing = stages.slice(-1)[0];
        this.dockerCredentials = props.dockerCredentials;
    }
    /**
     * Make sure there is an action in the stage to publish the given asset
     *
     * Assets are grouped by asset ID (which represent individual assets) so all assets
     * are published in parallel. For each assets, all destinations are published sequentially
     * so that we can reuse expensive operations between them (mostly: building a Docker image).
     */
    addPublishAssetAction(command) {
        // FIXME: this is silly, we need the relative path here but no easy way to get it
        const relativePath = path.relative(this.myCxAsmRoot, command.assetManifestPath);
        // The path cannot be outside the asm root. I don't really understand how this could ever
        // come to pass, but apparently it has (see https://github.com/aws/aws-cdk/issues/9766).
        // Add a sanity check here so we can catch it more quickly next time.
        if (relativePath.startsWith(`..${path.sep}`)) {
            throw new Error(`The asset manifest (${command.assetManifestPath}) cannot be outside the Cloud Assembly directory (${this.myCxAsmRoot}). Please report this error at https://github.com/aws/aws-cdk/issues to help us debug why this is happening.`);
        }
        // Late-binding here (rather than in the constructor) to prevent creating the role in cases where no asset actions are created.
        if (!this.assetRoles[command.assetType]) {
            this.generateAssetRole(command.assetType);
        }
        this.assetPublishingRoles[command.assetType] = (this.assetPublishingRoles[command.assetType] ?? new Set()).add(command.assetPublishingRoleArn);
        const publisherKey = this.props.singlePublisherPerType ? command.assetType.toString() : command.assetId;
        let action = this.publishers[publisherKey];
        if (!action) {
            // Dynamically create new stages as needed, with `MAX_PUBLISHERS_PER_STAGE` assets per stage.
            const stageIndex = this.props.singlePublisherPerType ? 0 :
                Math.floor((this._fileAssetCtr + this._dockerAssetCtr) / this.MAX_PUBLISHERS_PER_STAGE);
            if (!this.props.singlePublisherPerType && stageIndex >= this.stages.length) {
                const previousStage = this.stages.slice(-1)[0] ?? this.lastStageBeforePublishing;
                this.stages.push(this.pipeline.addStage({
                    stageName: `Assets${stageIndex > 0 ? stageIndex + 1 : ''}`,
                    placement: { justAfter: previousStage },
                }));
            }
            else if (this.props.singlePublisherPerType && this.stages.length == 0) {
                this.stages.push(this.pipeline.addStage({
                    stageName: 'Assets',
                    placement: { justAfter: this.lastStageBeforePublishing },
                }));
            }
            // The asset ID would be a logical candidate for the construct path and project names, but if the asset
            // changes it leads to recreation of a number of Role/Policy/Project resources which is slower than
            // necessary. Number sequentially instead.
            //
            // FIXME: The ultimate best solution is probably to generate a single Project per asset type
            // and reuse that for all assets.
            const id = this.props.singlePublisherPerType ?
                command.assetType === asset_type_1.AssetType.FILE ? 'FileAsset' : 'DockerAsset' :
                command.assetType === asset_type_1.AssetType.FILE ? `FileAsset${++this._fileAssetCtr}` : `DockerAsset${++this._dockerAssetCtr}`;
            const credsInstallCommands = docker_credentials_1.dockerCredentialsInstallCommands(docker_credentials_1.DockerCredentialUsage.ASSET_PUBLISHING, this.dockerCredentials);
            // NOTE: It's important that asset changes don't force a pipeline self-mutation.
            // This can cause an infinite loop of updates (see https://github.com/aws/aws-cdk/issues/9080).
            // For that reason, we use the id as the actionName below, rather than the asset hash.
            action = this.publishers[publisherKey] = new actions_1.PublishAssetsAction(this, id, {
                actionName: id,
                cloudAssemblyInput: this.props.cloudAssemblyInput,
                cdkCliVersion: this.props.cdkCliVersion,
                assetType: command.assetType,
                role: this.assetRoles[command.assetType],
                dependable: this.assetAttachedPolicies[command.assetType],
                vpc: this.props.vpc,
                subnetSelection: this.props.subnetSelection,
                buildSpec: this.props.buildSpec,
                createBuildspecFile: this.props.singlePublisherPerType,
                preInstallCommands: [...(this.props.preInstallCommands ?? []), ...credsInstallCommands],
            });
            this.stages[stageIndex].addAction(action);
        }
        action.addPublishCommand(relativePath, command.assetSelector);
    }
    /**
     * This role is used by both the CodePipeline build action and related CodeBuild project. Consolidating these two
     * roles into one, and re-using across all assets, saves significant size of the final synthesized output.
     * Modeled after the CodePipeline role and 'CodePipelineActionRole' roles.
     * Generates one role per asset type to separate file and Docker/image-based permissions.
     */
    generateAssetRole(assetType) {
        if (this.assetRoles[assetType]) {
            return this.assetRoles[assetType];
        }
        const rolePrefix = assetType === asset_type_1.AssetType.DOCKER_IMAGE ? 'Docker' : 'File';
        const assetRole = new asset_singleton_role_1.AssetSingletonRole(this, `${rolePrefix}Role`, {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('codebuild.amazonaws.com'), new iam.AccountPrincipal(core_1.Stack.of(this).account)),
        });
        // Publishing role access
        // The ARNs include raw AWS pseudo parameters (e.g., ${AWS::Partition}), which need to be substituted.
        // Lazy-evaluated so all asset publishing roles are included.
        assetRole.addToPolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: core_1.Lazy.list({ produce: () => [...this.assetPublishingRoles[assetType]].map(arn => core_1.Fn.sub(arn)) }),
        }));
        // Grant pull access for any ECR registries and secrets that exist
        if (assetType === asset_type_1.AssetType.DOCKER_IMAGE) {
            this.dockerCredentials.forEach(reg => reg.grantRead(assetRole, docker_credentials_1.DockerCredentialUsage.ASSET_PUBLISHING));
        }
        this.assetRoles[assetType] = assetRole;
        return this.assetRoles[assetType];
    }
}
function maybeSuffix(x, suffix) {
    if (x === undefined) {
        return undefined;
    }
    return `${x}${suffix}`;
}
//# sourceMappingURL=data:application/json;base64,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